/***************************************************************************
                          jdxarrays.h  -  description
                             -------------------
    begin                : Mon Jul 5 2004
    copyright            : (C) 2000-2014 by Thies H. Jochimsen
    email                : thies@jochimsen.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef JDXARRAYS_H
#define JDXARRAYS_H

#include <tjutils/tjarray.h>
#include <odinpara/jdxbase.h>
#include <odinpara/jdxnumbers.h>
#include <odinpara/jdxtypes.h>


#define COMPRESSION_THRESHOLD_SIZE 256


/**
  * @addtogroup jcampdx
  * @{
  */


   
/**
  *
  *  JCAMP-DX template class for representing multi-dimensional arrays
  */
template<class A,class J>
class JDXarray : public A, public virtual JcampDxClass  {

 public:

/**
  *  Default constructor
  */
  JDXarray  () : A() {common_init();}

/**
  *  Constructor with the following arguments:
  * - a:             Initial value for the array base class
  * - name:          The label of the JCAMP-DX parameter
  * - userParameter: Whether this is a user defined JCAMP-DX parameter
  * - mode:          Mode for ASCII representation of strings or arrays of strings
  * - parameter_mode: Mode for GUI accesibility of the parameter
  * - parx_equivalent: Equivalent parameter in PARX to which this parameter will be assigned
  * - parx_assign_factor: Scaling factor when assigning to the equivalent parameter in PARX
  * - parx_assign_offset: Offset when assigning to the equivalent parameter in PARX
  */
  JDXarray (const A& a,const STD_string& name="",bool userParameter=true,
            compatMode mode=notBroken,parameterMode parameter_mode=edit,
            const STD_string& parx_equivalent="",
            float parx_assign_factor=1.0,float parx_assign_offset=0.0);

/**
  *  Assignment from scalar, i.e. fill all elements with this value
  */
//  JDXarray (const J& jv) {common_init(); A::operator = (jv);}

/**
  *  Copy constructor
  */
  JDXarray (const JDXarray<A,J>& ja) {common_init(); JDXarray<A,J>::operator = (ja);}

/**
  *  Assignment operator from an array
  */
  JDXarray<A,J>& operator = (const A& a);

/**
  *  Assignment operator from scalar, i.e. fill all elements with this value
  */
  JDXarray<A,J>& operator = (const J& jv) {A::operator = (jv); return *this;}

/**
  *  Copy assignment
  */
  JDXarray<A,J>& operator = (const JDXarray<A,J>& ja);


  // overwriting virtual functions from JcampDxClass
  STD_ostream& print2stream(STD_ostream& os) const;
  STD_string printvalstring() const;
  bool parsevalstring (const STD_string& parstring);
  STD_string get_parx_code(parxCodeType type, const ParxEquiv& equiv) const;
  ParxEquiv get_parx_equiv() const {return parx_equiv;}
  GuiProps get_gui_props() const {return guiprops;}
  JcampDxClass& set_gui_props(const GuiProps& gp) {guiprops=gp; return *this;}
  const char* get_typeInfo() const;
  JcampDxClass* create_copy() const {return new JDXarray<A,J>(*this);}
  A* cast(A*) {return this;}


  // final overrider
  friend STD_ostream& operator << (STD_ostream& s,const JDXarray<A,J>& ja) { return ja.print2stream(s);}
  int write(const STD_string& filename) const {return JcampDxClass::write(filename);}
  int load(const STD_string &filename) {return JcampDxClass::load(filename);}


 private:
  void common_init();
  STD_string get_dim_str() const;

  bool encode(STD_string* ostring, STD_ostream* ostream) const;

  bool use_compression() const {return (get_filemode()==compressed)&&(A::total()>COMPRESSION_THRESHOLD_SIZE);}

  ParxEquiv parx_equiv;

  GuiProps guiprops;

  mutable STD_string typeInfo_cache;

};

//////////////////////////////////////////////////////////////////////////////////
//
// Aliases:
//

/**
  * A JCAMP-DX array of strings
  */
typedef JDXarray<sarray,JDXstring>   JDXstringArr;

/**
  * A JCAMP-DX array of integer numbers
  */
typedef JDXarray<iarray,JDXint>   JDXintArr;

/**
  * A JCAMP-DX array of single-precision floating point numbers
  */
typedef JDXarray<farray,JDXfloat>   JDXfloatArr;

/**
  * A JCAMP-DX array of double-precision floating point numbers
  */
typedef JDXarray<darray,JDXdouble>   JDXdoubleArr;

/**
  * A JCAMP-DX array of single-precision complex numbers
  */
typedef JDXarray<carray,JDXcomplex>   JDXcomplexArr;



/////////////////////////////////////////////////////////////////////////////
//    Specialised array classes:


/**
  *  A JCAMP-DX  vector of 3 float values which is used to store spatial positions, FOVs, etc.
  */
class JDXtriple : public JDXfloatArr {

 public:

/**
  *  Default constructor
  */
  JDXtriple () : JDXfloatArr(farray(3)) {}

/**
  *  Constructor with the following arguments:
  * - xpos: Spatial position in the first dimension
  * - ypos: Spatial position in the second dimension
  * - zpos: Spatial position in the third dimension
  * - name:          The label of the JCAMP-DX parameter
  * - userParameter: Whether this is a user defined JCAMP-DX parameter
  * - mode:          Mode for ASCII representation of strings or arrays of strings
  * - parameter_mode: Mode for GUI accesibility of the parameter
  */
  JDXtriple (float xpos,float ypos, float zpos,
               const STD_string& name="",bool userParameter=true,
               compatMode mode=notBroken,parameterMode parameter_mode=edit);

/**
  *  Copy constructor
  */
  JDXtriple (const JDXtriple& pos) {JDXtriple::operator = (pos);}

/**
  *  Copy assignment
  */
  JDXtriple& operator = (const JDXtriple& pos) {
    JDXfloatArr::operator = (pos);
    return *this;
  }

  // overwriting virtual functions from JcampDxClass
  const char* get_typeInfo() const {return "triple";}
  JcampDxClass* create_copy() const {return new JDXtriple(*this);}
  JDXtriple* cast(JDXtriple*) {return this;}
  farray* cast(farray*) {return 0;} // disable casting to base class

};

/** @}
  */

#endif

