(*
 * Graph: generic graph library
 * Copyright (C) 2004
 * Sylvain Conchon, Jean-Christophe Filliatre and Julien Signoles
 * 
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License version 2, as published by the Free Software Foundation.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * 
 * See the GNU Library General Public License version 2 for more details
 * (enclosed in the file LGPL).
 *)

(* $Id: rand.mli,v 1.6 2004/02/20 16:44:13 signoles Exp $ *)

(** Random graph generation *)

(** {1 Random graphs} *)

module type S = sig

  type graph 
  type vertex
  type edge_label

  val graph : ?loops:bool -> v:int -> e:int -> unit -> graph
    (** [graph v e] generates a random graph with exactly [v] vertices 
        and [e] edges. Vertices are labeled with [0] ... [v-1].
        The boolean [loops] indicates whether loops are allowed; 
        default value is no loop ([false]). 
        Nothing prevents the algorithm from looping if [e] is chosen too
        large. *)

  val labeled : 
    (vertex -> vertex -> edge_label) -> 
    ?loops:bool -> v:int -> e:int -> unit -> graph
    (** [labeled f] is similar to [graph] except that edges are labeled
        using function [f] *)

end

module P (G : Sig.P with type V.label = int) : 
  S with type graph = G.t 
    and type vertex = G.V.t 
    and type edge_label = G.E.label
      (** Random persistent graphs *)

module I (G : Sig.I with type V.label = int) : 
  S with type graph = G.t
    and type vertex = G.V.t     
    and type edge_label = G.E.label
      (** Random imperative graphs *)

(** {1 Random planar graphs} *)

module Planar : sig

  module type S = sig

    type graph 

    val graph : 
      ?loops:bool -> xrange:int*int -> yrange:int*int ->
	prob:float -> int -> graph
      (** [graph xrange yrange prob v] 
        generates a random planar graph with exactly [v] vertices.
        Vertices are labeled with integer coordinates, randomly distributed
        according to [xrange] and [yrange].
        Edges are built as follows: the full Delaunay triangulation is
        constructed and then each edge is discarded with probabiblity [prob]
        (which should lie in [0..1]). In particular [prob = 0.0] gives the 
        full triangulation.
        Edges are labeled with the (rounded) Euclidean distance between
        the two vertices.
        The boolean [loops] indicates whether loops are allowed; 
        default value is no loop ([false]). *)

  end

  module P (G : Sig.P with type V.label = int * int and type E.label = int) : 
    S with type graph = G.t 
	(** Random persistent planar graphs *)

  module I (G : Sig.I with type V.label = int * int and type E.label = int) : 
    S with type graph = G.t
	(** Random imperative planar graphs *)
			      
end
