(*
  Copyright 2003-2006 The Savonet team

  This file is part of Ocaml-vorbis.

  Ocaml-vorbis is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  Ocaml-vorbis is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with Ocaml-vorbis; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(**
  * Decode from or encode to the Ogg Vorbis compressed audio format;
  * or get informations about an Ogg Vorbis file.
  *
  * @author Samuel Mimram, Julien Cristau, David Baelde
  *)

(* $Id: vorbis.mli 2593 2006-07-07 08:03:09Z dbaelde $ *)

(** {1 Exceptions} *)

(** Some parameters are invalid for this function. *)
exception Invalid_parameters

(** The given quality in invalid. *)
exception Invalid_quality

(** The given bitrate is invalid. *)
exception Invalid_bitrate

(** The given number of channels is invalid. *)
exception Invalid_channels

(** The given sampling frequency is invalid. *)
exception Invalid_sample_freq

(** The given file could not be opened. *)
exception Could_not_open_file

(** Bitstream is not Vorbis data. *)
exception Not_vorbis

(** Invalid Vorbis bitstream header. *)
exception Bad_header

(** A read from media returned an error. *)
exception Read_error

(** Internal logic fault; indicates a bug or heap/stack corruption. *)
exception Internal_fault

(** Indicates there was an interruption in the data (one of: garbage between
  * pages, loss of sync followed by recapture, or a corrupt page). *)
exception Hole_in_data

(** Indicates that an invalid stream section was supplied,
  * or the requested link is corrupt. *)
exception Bad_link

(** Invalid Vorbis bitstream header. *)
exception Version_mismatch

(** An unknown error happened (in fact it should not have happened, please
  * report). *)
exception Unknown_error

(** Error while converting utf8. *)
exception Utf8_failure of string

(** {1 Encoding} *)

(** An ogg encoder. *)
type encoder

(** Parameters for encoders, specifying input and output format. *)
type enc_params =
    {
      enc_bitrate : int option; (** ouput birate in kbit/s *)
      enc_min_bitrate : int option; (** minimal output bitrate in kbit/s *)
      enc_max_bitrate : int option; (** maximal output bitrate in kbit/s *)
      enc_quality : float; (** quality of encoding between 0.0 and 1.0 (1.0 ~ 160 kbit/s, the higher the better) *)
      enc_channels : int; (** number of channels (should be 1 or 2) *)
      enc_sample_freq : int option; (** sampling frequency *)
      enc_managed : bool; (** shoud libvorbis manage parameters? *)
      enc_in_channels : int; (** number of channels of input *)
      enc_in_sample_freq : int; (** sampling frequency of input *)
      enc_in_sample_size : int; (** sample size of input in bits *)
      enc_in_big_endian : bool; (** is the input data stored in big endian? *)
    }

(** Set the current charset. Tags will automatically be converted from this
  * charset from / to utf8 (default: US-ASCII). *)
val set_charset : string -> unit

(** Create a new encoder. The second argument is a string containing the header frame. *)
val create_encoder :
  ?title:string ->
  ?artist:string ->
  ?genre:string ->
  ?date:string ->
  ?album:string -> ?tracknum:string -> ?comment:string -> enc_params -> encoder * string

(** Same as [create_encoder] but with option arguments. *)
val create_encoder_opt :
  string option ->
  string option ->
  string option ->
  string option -> string option ->
  string option -> string option -> enc_params -> encoder * string


(** Encode a wav buffer into ogg.
  * WARNING: the size of the input buffer should be less than or equal to 1024 bytes. *)
val encode_buffer : encoder -> string -> string

(** Same as [encode_buffer] excepting that [encode_buffer_part enc buf ofs len]
  * only encodes [len] bytes of buffer [buf] starting at offset [ofs] (in
  * bytes). *)
val encode_buffer_part : encoder -> string -> int -> int -> string


(** {1 Decoding} *)

(** Output wav format parameters for decoders. *)
type dec_params =
    {
      sample_size : int; (** sample size in bits (should be 8 or 16) *)
      big_endian : bool; (** store data in big endian ? (typically [false], little-endian) *)
      signed : bool; (** sign for PCM output ? (typically [true], signed) *)
    }

(** Ogg/vorbis file opened in input (for decoding). *)
type dec_file

(** Open an ogg/vorbis file for decoding. Can raise: [Could_not_open_file] or
  * [Not_a_vorbis_file]. Raises: [Read_error], [Not_vorbis],
  * [Version_mismatch], [Bad_header], [Internal_fault].*)
val open_dec_file : string -> dec_params -> dec_file

(** Same as [open_dec_file] but uses a file descriptor instead of a file name.
  * The file descriptor must be closed by [close_dec_file]. *)
val open_dec_fd : Unix.file_descr -> dec_params -> dec_file

(** [open_dec_stream read_func seek_func close_func tell_func params] opens a
  * stream like [open_dec_file] for decoding but callbacks are used to
  * manipulate the data. [read_func] should return the requested amount of bytes
  * (or less if it is the end of file), [seek_funk] should return 0 if the seek
  * was ok or -1 if the stream is not seekable, [close_func] should close the
  * stream, and [tell_func] should return the current offset or -1 if there is
  * no notion of offset in the stream. Raises: [Read_error], [Not_vorbis],
  * [Version_mismatch], [Bad_header], [Internal_fault]. *)
val open_dec_stream : (int -> string) -> (unit -> int) -> (unit -> unit) -> (unit -> int) -> dec_params -> dec_file

(** [decode dec_file buf ofs len] decodes [len] octets of wav (should be the
  * same size as [buf], a typical value
  * for [len] is 4096) and put them in [buf] starting at position
  * [pos]. The returned value is the actual number of bytes read. [decode] will
  * decode at most one vorbis packet per invocation, so the value returned will
  * generally be less than [len].
  *
  * @raise Unix.End_of_file when the whole file has been decoded.
  * @raise Hole_in_data if there was an interruption of the data.
  * @raise Invalid_parameters if all the data cannot fit in the buffer starting at the given position.
  *)
val decode : dec_file -> string -> int -> int -> int

(** Close an ogg/vorbis file opened for decoding. The file descriptor used for
  * the matching [open_dec_file] or [open_dec_fd] is closed. *)
val close_dec_file : dec_file -> unit

(** Reset the encoder state to a fresh, initialized state with the given metadata *)
val encoder_reset :
  ?title:string ->
  ?artist:string ->
  ?genre:string ->
  ?date:string -> ?album:string -> ?tracknum:string -> ?comment:string -> encoder -> string
val encoder_reset_opt :
  string option ->
  string option ->
  string option -> string option ->
  string option -> string option -> string option -> encoder -> string


(** {1 Informations about files} *)

(** Vorbis informations about a file. *)
type info = {
  vorbis_version : int; (** version of vorbis codec, must be 0 *)
  audio_channels : int; (** number of audio channels *)
  audio_sample_rate : int; (** samplerate in Hertz *)
  bitrate_maximum : int option; (** maximum bitrate in kbit/s *)
  bitrate_nominal : int option; (** nominal bitrate in kbit/s *)
  bitrate_minimum : int option; (** minimal bitrate in kbit/s *)
  blocksize_0 : int;
  blocksize_1 : int;
  duration : int; (** duration of the file in seconds *)
}

(** Get the index of the sequential logical bitstream currently being decoded
  * (incremented at chaining boundaries even for non-seekable streams). For
  * seekable streams, it represents the actual chaining index within the
  * physical bitstream. *)
val get_dec_file_bitstream : dec_file -> int

(** Get the vorbis comments from a vorbis file (see [Info.get_comments]).
  * The second argument is the number of the logical bistream (the current
  * bistream is used if it is set to [None]). *)
val get_dec_file_comments : dec_file -> int option -> string * ((string * string) array)

(** Get informations about a file being decoded. *)
val get_dec_file_info : dec_file -> info

(** Get the vorbis comments from a file (see [Info.get_comments]). *)
val get_comments : string -> string * (string * string) array

(** Get the vorbis information from the file header.
  * One can avoid the computation of the duration, which is long. *)
val get_info : ?duration:bool -> string -> info

(** Get the size of a file (in bytes). *)
val file_size : string -> int

(** Signature of a module to access to files. *)
module type Iofile =
sig
  type file_descr
  type open_flag = O_RDONLY | O_WRONLY | O_RDWR | O_CREAT | O_TRUNC
  type file_perm = int
  type seek_command = SEEK_SET | SEEK_CUR | SEEK_END
  val openfile : string -> open_flag list -> file_perm -> file_descr
  val close : file_descr -> unit
  val read : file_descr -> string -> int -> int -> int
  val lseek : file_descr -> int -> seek_command -> int
end

(** Get info and comments from a vorbis file via the Io module. *)
module Info : functor (Io : Iofile) ->
sig
  (** Get the vorbis comments from a vorbis file.
    * The first string is the vendor string, the elements of the array are
    * of the form (field name, field contents), where 'field name' is
    * uppercase.
    *)
  val get_comments : string -> string * (string*string) array

  (**
    * Get the information in the header of a vorbis file with a safe
    * value of duration.
    * @raise Unix.End_of_file if the file is too small (no complete pages).
    *)
  val get_info : string -> info

  (** Same as get_info but uses a heuristic to find the duration of
    * the vorbis file without reading every page header. *)
  val get_heuristical_info : string -> info

  (** Get the size of a given file (in bytes). *)
  val file_size : string -> int
end
