(*
 Copyright 2003-2006 Savonet team

 This file is part of Ocaml-vorbis.

 Ocaml-vorbis is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 Ocaml-vorbis is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with Ocaml-vorbis; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)
(**
  Decode from or encode to the Ogg Vorbis compressed audio format; or get
  informations about an Ogg Vorbis file.

  @author Samuel Mimram, Julien Cristau, David Baelde
  *)

(* $Id: vorbis.ml 2593 2006-07-07 08:03:09Z dbaelde $ *)

(* TODO: there are too many assert(). Some of them should be handled
 more gracefully *)

type encoder

exception Invalid_parameters
exception Invalid_quality
exception Invalid_bitrate
exception Invalid_samplesize
exception Invalid_channels
exception Invalid_sample_freq
exception Could_not_open_file
exception Not_vorbis
exception Hole_in_data
exception Bad_link
exception Version_mismatch
exception Bad_header
exception Read_error
exception Internal_fault
exception Unknown_error
exception Utf8_failure of string

let _ =
  Callback.register_exception "vorbis_exn_invalid_parameters" Invalid_parameters;
  Callback.register_exception "vorbis_exn_invalid_quality" Invalid_quality;
  Callback.register_exception "vorbis_exn_invalid_bitrate" Invalid_bitrate;
  Callback.register_exception "vorbis_exn_invalid_samplesize" Invalid_samplesize;
  Callback.register_exception "vorbis_exn_invalid_channels" Invalid_channels;
  Callback.register_exception "vorbis_exn_invalid_sample_freq" Invalid_sample_freq;
  Callback.register_exception "vorbis_exn_could_not_open_file" Could_not_open_file;
  Callback.register_exception "vorbis_exn_not_vorbis" Not_vorbis;
  Callback.register_exception "vorbis_exn_hole_in_data" Hole_in_data;
  Callback.register_exception "vorbis_exn_bad_link" Bad_link;
  Callback.register_exception "vorbis_exn_version_mismatch" Version_mismatch;
  Callback.register_exception "vorbis_exn_bad_header" Bad_header;
  Callback.register_exception "vorbis_exn_read_error" Read_error;
  Callback.register_exception "vorbis_exn_internal_fault" Internal_fault;
  Callback.register_exception "vorbis_exn_unknown_error" Unknown_error;
  Callback.register_exception "vorbis_exn_utf8_failure" (Utf8_failure "")

type enc_params =
    {
      enc_bitrate : int option;
      enc_min_bitrate : int option;
      enc_max_bitrate : int option;
      enc_quality : float;
      enc_channels : int;
      enc_sample_freq : int option;
      enc_managed : bool;
      enc_in_channels : int;
      enc_in_sample_freq : int;
      enc_in_sample_size : int;
      enc_in_big_endian : bool;
    }

external create_encoder_pre : enc_params -> string option -> string option -> string option -> string option -> string option -> string option -> string option -> encoder*string = "ocaml_vorbis_create_encoder_byte" "ocaml_vorbis_create_encoder"

let create_encoder ?title ?artist ?genre ?date ?album ?tracknum ?comment params =
  create_encoder_pre params title artist genre date album tracknum comment

let create_encoder_opt title artist genre date album tracknum comment params =
  create_encoder_pre params title artist genre date album tracknum comment

external encode_buffer_part : encoder -> string -> int -> int -> string = "ocaml_vorbis_encode_buffer"

let encode_buffer enc buf =
  encode_buffer_part enc buf 0 (String.length buf)

type dec_params =
    {
      sample_size : int;
      big_endian : bool;
      signed : bool;
    }

type dec_file

external open_dec_fd : Unix.file_descr -> dec_params -> dec_file = "ocaml_vorbis_open_dec_file"

let open_dec_file fname =
  let fd = Unix.openfile fname [Unix.O_RDONLY] 0o400 in
    open_dec_fd fd

external open_dec_stream : (int -> string) -> (unit -> int) -> (unit -> unit) -> (unit -> int) -> dec_params -> dec_file = "ocaml_vorbis_open_dec_stream"

external decode : dec_file -> string -> int -> int -> int = "ocaml_vorbis_decode"

external close_dec_file : dec_file -> unit = "ocaml_vorbis_close_dec_file"


external encoder_reset_pre : encoder -> string option -> string option -> string option -> string option -> string option -> string option -> string option -> string = "ocaml_vorbis_encoder_reset_byte" "ocaml_vorbis_encoder_reset"
let encoder_reset ?title ?artist ?genre ?date ?album ?tracknum ?comment ep = encoder_reset_pre ep title artist genre date album tracknum comment
let encoder_reset_opt title artist genre date album tracknum comment ep = encoder_reset_pre ep title artist genre date album tracknum comment


type info = {
  vorbis_version : int;
  audio_channels : int;
  audio_sample_rate : int;
  bitrate_maximum : int option;
  bitrate_nominal : int option;
  bitrate_minimum : int option;
  blocksize_0 : int;
  blocksize_1 : int;
  duration : int;
}

external get_dec_file_bitstream : dec_file -> int = "ocaml_vorbis_get_dec_file_bitstream"

(** Set charset to encode from / decode to. *)
external utf8_set_charset : string -> unit = "ocaml_vorbis_utf8_set_charset"

let set_charset = utf8_set_charset

external utf8_decode : string -> string = "ocaml_vorbis_utf8_decode"

external utf8_encode : string -> string = "ocaml_vorbis_utf8_encode"

external get_dec_file_comments_ : dec_file -> int option -> string * (string array) = "ocaml_vorbis_get_dec_file_comments"

(* TODO: share this with get_comments *)
let split_comment comment =
  try
    let equal_pos =
      String.index_from comment 0 '='
    in
    let c1 =
      String.uppercase (String.sub comment 0 equal_pos)
    in
    let c2 =
      utf8_decode (String.sub comment (equal_pos + 1) ((String.length comment) - equal_pos - 1))
    in
      c1, c2;
  with Not_found -> comment, ""

let get_dec_file_comments df link =
  let vd, cmts = get_dec_file_comments_ df link in
    vd, (Array.map split_comment cmts)

external get_dec_file_info : dec_file -> info = "ocaml_vorbis_get_dec_file_info"

(* WARNING: possible overflow *)
let read32 s =
  let a = int_of_char(s.[0])
  and b = int_of_char(s.[1])
  and c = int_of_char(s.[2])
  and d = int_of_char(s.[3]) in
    (d lsl 24) lor (c lsl 16)
    lor (b lsl 8) lor a

(* this function should be called when the current position in [file]
 is at the beginning of a page.
 It reads the header of this page and returns a string containing
 the lengths of the ogg segments in this page. When it returns, the
 position in [file] is at the end of the header *)
let get_lacing_values file =
  ignore(Unix.lseek file 26 Unix.SEEK_CUR);
                                         let buf = String.create 1 in
let r = Unix.read file buf 0 1 in
  assert(r = 1);
              let nb_segments = int_of_char buf.[0] in
let segments = String.create nb_segments in
let ofs = ref 0 in
  while !ofs < nb_segments do
    let r = Unix.read file segments !ofs (nb_segments - !ofs) in
      ofs := !ofs + r;
      if r = 0 then raise End_of_file;
  done;
  segments

(* this function should be called when the current position in file is
 at the beginning of a page.
 @returns the granule_pos specified in the current header, and a
 boolean corresponding to the eos flag in this page.
 @raise End_of_file if it can't read() the header
 The position in [file] is unchanged at the end of this function *)
let last_pos file =
  let pos = Unix.lseek file 0 Unix.SEEK_CUR in
let buf = String.create 27 in
let ofs = ref 0 in
  while !ofs < 27 do
    let r = Unix.read file buf !ofs (27 - !ofs) in
  ofs := !ofs + r;
  (* this should not happen, but... *)
  if r = 0 then raise End_of_file;
     done;
     let newpos = Unix.lseek file pos Unix.SEEK_SET in
  assert(newpos = pos);
                     let eos = (int_of_char buf.[5]) land 0b100 in
  (String.sub buf 6 8), (eos <> 0)

(* try to find the number of pcm samples in [filename]: first try to
 find the last page in the stream (where eos is set), and get the
 granule_pos in the header of this page. If we encounter EOF before
 eos, we use the value of granule_pos in the previous page. This
 function should *never* fail. *)
let samples filename =
  let file = Unix.openfile filename [Unix.O_RDONLY] 0o400 in
let granule_pos = ref "" in
let eos = ref false in
  begin try
    while not !eos do
      let newpos, neweos = last_pos file in
  granule_pos := newpos;
  eos := neweos;
  if not !eos then begin (* this page is not the last one *)
    let segments = get_lacing_values file in
let size = ref 0 in
  (* compute the length of the data in this page *)
  for i = 0 to (String.length segments) - 1 do
    size := !size + (int_of_char segments.[i])
done;
(* seek to the beginning of the next page *)
ignore(Unix.lseek file !size Unix.SEEK_CUR)
end;
   done;
   with
     | End_of_file -> () (* we use the value of granule_pos in the
                          last page before EOF *)
         | _ -> assert false (* I don't think other exceptions can be
                              raised here, but I'd like to be sure *)
                  end;
                     Unix.close file;
                     let nb_samples = ref Int64.zero in
  for i = 7 downto 0 do
    nb_samples := Int64.logor (Int64.shift_left !nb_samples 8)
(Int64.of_int (int_of_char !granule_pos.[i]));
                                            done;
                                            !nb_samples

let decode_comments filename =
  let comments = ref "" in
let file = Unix.openfile filename [Unix.O_RDONLY; Unix.O_NONBLOCK] 0o400 in
  begin try
    (* seek to the beginning of the second page (which begins with the
     comments)  *)
    let pos = Unix.lseek file 58 Unix.SEEK_SET in
      assert(pos = 58);
      let found_end = ref false in
        while not !found_end do
          (* get the length of the segments of this page *)
          let segments = get_lacing_values file in
          let i = ref 0 in
          let comments_size = ref 0 in
            while !i < String.length segments
            && (int_of_char segments.[!i]) = 0xFF do
              (* this is not the last segment containing comments *)
              comments_size := !comments_size + 0xFF;
              incr i;
            done; (* either we are at the end of the comments, or at the
                   end of the current page, but comments continue in
                   the next one *)
            if !i < String.length segments then begin
              (* this segment has a size < 255, which indicates the last
               segment of comments *)
              found_end := true;
              comments_size := !comments_size + int_of_char segments.[!i];
            end;
            let buf = String.create !comments_size in
            let ofs = ref 0 in
              while !ofs < !comments_size do
                let r = Unix.read file buf !ofs (!comments_size - !ofs) in
                  ofs := !ofs + r;
                  if r = 0 then raise End_of_file
              done;
              comments := !comments ^ buf
        done; (* repeat for each page until the end of the comments *)
        Unix.close file;
        !comments
          with e -> Unix.close file; raise e
                                       end

                                       let get_comments filename =
                                         let s = decode_comments filename in
                                         let n = String.length s in
                                           if n < 11 || String.sub s 1 6 <> "vorbis" then
                                             raise Not_vorbis;
                                           let vendorlen = read32 (String.sub s 7 4) in
                                           let () = assert(n >= 15 + vendorlen) in
                                           let vendor = utf8_decode(String.sub s 11 vendorlen) in
                                           let nb_comments = read32 (String.sub s (11+vendorlen) 4) in
                                           let comments = Array.make (nb_comments) ("","") in
                                           let offset = ref (11+vendorlen+4) in
                                             for i = 0 to nb_comments-1 do
                                               let () = assert(n >= !offset + 4) in
                                               let length = read32 (String.sub s !offset 4) in
                                               let () = assert(n >= !offset + 4 + length) in
                                                 begin try
                                                   let comment = String.sub s (!offset + 4) length in
                                                   let equal_pos = String.index_from comment 0 '=' in
                                                   let c1 =
                                                     String.uppercase (String.sub comment 0 equal_pos)
                                                   in
                                                   let c2 =
                                                     utf8_decode
                                                       (String.sub comment (equal_pos + 1) (length - equal_pos - 1))
                                                   in
                                                     comments.(i) <- c1, c2;
                                                 with Not_found -> () end;
                                                 offset := !offset + length + 4;
                                             done;
                                             vendor,comments

let get_info ?(duration=true) filename =
  let file = Unix.openfile filename [Unix.O_RDONLY;Unix.O_NONBLOCK] 0o400 in
  let x = Unix.lseek file 35 Unix.SEEK_SET in
  let () = assert(x = 35) in
  let s = String.create 22 in
  let ofs = ref 0 in
    while !ofs < 22 do
      let r = Unix.read file s !ofs (22 - !ofs) in
        ofs := !ofs + r;
        if r = 0 then begin Unix.close file; raise End_of_file end
    done;
    Unix.close file;
    let br_max = read32 (String.sub s 9 4)
    and br_nom = read32 (String.sub s 13 4)
    and br_min = read32 (String.sub s 17 4)
    and blocksize_1 = ((int_of_char s.[21]) lsr 4) land 0xf
    and blocksize_0 = (int_of_char s.[21]) land 0xf
    and sample_rate = read32 (String.sub s 5 4)
    in
      {
        vorbis_version = read32 (String.sub s 0 4);
        audio_channels = int_of_char s.[4];
        audio_sample_rate = sample_rate;
        bitrate_maximum = if br_max>0 then Some br_max else None;
        bitrate_nominal = if br_nom>0 then Some br_nom else None;
        bitrate_minimum = if br_min>0 then Some br_min else None;
        blocksize_0 = blocksize_0;
        blocksize_1 = blocksize_1;
        duration =
          if duration then
            Int64.to_int
              (Int64.div (samples filename) (Int64.of_int sample_rate))
          else
            -1
      }

let file_size filename =
  (Unix.stat filename).Unix.st_size

module type Iofile =
sig
  type file_descr
  type open_flag = O_RDONLY | O_WRONLY | O_RDWR | O_CREAT | O_TRUNC
  type file_perm = int
  type seek_command = SEEK_SET | SEEK_CUR | SEEK_END
  val openfile : string -> open_flag list -> file_perm -> file_descr
  val close : file_descr -> unit
  val read : file_descr -> string -> int -> int -> int
  val lseek : file_descr -> int -> seek_command -> int
end

module Info (Io: Iofile) =
struct

  (* Internal type for info without duration *)
  type header = {
    version : int;
    channels : int;
    sample_rate : int;
    bitmax : int option;
    bitnom : int option;
    bitmin : int option;
    block0 : int;
    block1 : int;
  }

  (* Find the lacing values of the segments in the current ogg page *)
  let get_lacing_values file =
    let _ = Io.lseek file 26 Io.SEEK_CUR in
    let buf = String.create 1 in
      if Io.read file buf 0 1 <> 1 then raise End_of_file;
      let nb_segments = int_of_char buf.[0] in
      let segments = String.create nb_segments in
      let ofs = ref 0 in
        while !ofs < nb_segments do
          let r = Io.read file segments !ofs (nb_segments - !ofs) in
            ofs := !ofs + r;
            if r = 0 then raise End_of_file
        done;
        segments

  (* Returns a buffer containing at least the current page *)
  let get_page file =
    let x = Io.lseek file 26 Io.SEEK_CUR in
    let () = assert (x = 26) in
    let buf = String.create 1 in
      if Io.read file buf 0 1 <> 1 then raise End_of_file;
      let nb_segments = int_of_char buf.[0] in
      (* each segment is at most 256 bytes *)
      let len = 256 * nb_segments in
      let page = String.create len in
      let ofs = ref 0 in
        begin try
          while !ofs < len do
            let r = Io.read file page !ofs (len - !ofs) in
              ofs := !ofs + r;
              if r = 0 then raise End_of_file
          done;
          page
        with
          | End_of_file -> String.sub page 0 !ofs
        end

        (* Return the absolute granule position of the current page, and a
         boolean set to true iff this is the last page *)
        let last_pos file =
          let pos = Io.lseek file 0 Io.SEEK_CUR in
          let buf = String.create 27 in
          let ofs = ref 0 in
            while !ofs < 27 do
              let r = Io.read file buf !ofs (27 - !ofs) in
                ofs := !ofs + r;
                if r = 0 then raise End_of_file
            done;
            let newpos = Io.lseek file pos Io.SEEK_SET in
              assert(newpos = pos);
              let eos = (int_of_char buf.[5]) land 0b100 in
                (String.sub buf 6 8), (eos <> 0)

  (* Return the number of samples in the vorbis file *)
  let samples filename =
    let granule_pos = ref "" in
    let eos = ref false in
    let file = Io.openfile filename [Io.O_RDONLY] 0o400 in
      begin try
        while not !eos do
          let newpos, neweos = last_pos file in
            granule_pos := newpos;
            eos := neweos;
            if not !eos then begin
              let segments = get_lacing_values file in
              let size = ref 0 in
                for i = 0 to (String.length segments) - 1 do
                  size := !size + (int_of_char segments.[i])
                done;
                ignore(Io.lseek file !size Io.SEEK_CUR)
            end;
        done;
      with
        | End_of_file ->
            if !granule_pos = "" then begin
              Io.close file; raise End_of_file
            end else
              ()
        | _ -> assert false
      end;
      Io.close file;
      let nb_samples = ref Int64.zero in
        for i = 7 downto 0 do
          nb_samples := Int64.logor (Int64.shift_left !nb_samples 8) (Int64.of_int(int_of_char !granule_pos.[i]))
        done;
        !nb_samples

  let decode_comments filename =
    let comments = ref "" in
    let file = Io.openfile filename [Io.O_RDONLY] 0o400 in
      begin try
        let pos = Io.lseek file 58 Io.SEEK_SET in
          assert(pos = 58);
          let found_end = ref false in
            while not !found_end do
              let segments = get_lacing_values file in
              let i = ref 0 in
              let size_comments = ref 0 in
                while !i < (String.length segments)
                && (int_of_char segments.[!i]) = 0xFF do
                  size_comments := !size_comments + 0xFF;
                  incr i
                done;
                if !i < (String.length segments) then begin
                  found_end := true;
                  size_comments := !size_comments + (int_of_char segments.[!i])
                end;
                let s = String.create !size_comments in
                let ofs = ref 0 in
                  while !ofs < !size_comments do
                    let r = Io.read file s !ofs (!size_comments - !ofs) in
                      ofs := !ofs + r;
                      if r = 0 then raise End_of_file
                  done;
                  comments := !comments ^ s;
            done;
            Io.close file;
            !comments
      with e -> Io.close file; raise e
      end

  let get_comments filename =
    let s = decode_comments filename in
    let n = String.length s in
      if n < 11 || String.sub s 1 6 <> "vorbis" then
        raise Not_vorbis;
      let vendorlen = read32 (String.sub s 7 4) in
      let () = assert(n >= 15 + vendorlen) in
      let vendor = utf8_decode(String.sub s 11 vendorlen) in
      let nb_comments = read32 (String.sub s (11+vendorlen) 4) in
      let comments = Array.make (nb_comments) ("","") in
      let offset = ref (11+vendorlen+4) in
        for i = 0 to nb_comments - 1 do
          let () = assert(n >= !offset + 4) in
          let length = read32 (String.sub s !offset 4) in
          let () = assert(n >= !offset + 4 + length) in
          let comment = String.sub s (!offset + 4) length in
            begin try
              let equal_pos = String.index_from comment 0 '=' in
              let c1 =
                String.uppercase (String.sub comment 0 equal_pos)
              in
              let c2 =
                utf8_decode
                  (String.sub comment (equal_pos + 1) (length - equal_pos - 1))
              in
                comments.(i) <- c1, c2;
            with Not_found -> () end;
            offset := !offset + length + 4;
        done;
        vendor,comments

  let read_info filename =
    let file = Io.openfile filename [Io.O_RDONLY] 0o400 in
    let x = Io.lseek file 35 Io.SEEK_SET in
      if x <> 35 then begin Io.close file; raise End_of_file end;
      let s = String.create 22 in
      let ofs = ref 0 in
        while !ofs < 22 do
          let r = Io.read file s !ofs (22 - !ofs) in
            ofs := !ofs + r;
            if r = 0 then begin Io.close file; raise End_of_file end
        done;
        Io.close file;
        let br_max = read32 (String.sub s 9 4)
        and br_nom = read32 (String.sub s 13 4)
        and br_min = read32 (String.sub s 17 4)
        and blocksize_1 = ((int_of_char s.[21]) lsr 4) land 0xf
        and blocksize_0 = (int_of_char s.[21]) land 0xf
        and sample_rate = read32 (String.sub s 5 4)
        in
          {
            version = read32 (String.sub s 0 4);
            channels = int_of_char s.[4];
            sample_rate = sample_rate;
            bitmax = if br_max > 0 then Some br_max else None;
            bitnom = if br_nom > 0 then Some br_nom else None;
            bitmin = if br_min > 0 then Some br_min else None;
            block0 = blocksize_0;
            block1 = blocksize_1;
          }

  let file_size filename =
    let file = Io.openfile filename [Io.O_RDONLY] 0o400 in
    let size = Io.lseek file 0 Io.SEEK_END in
      Io.close file; size

  let heuristical_duration filename =
    let size = 5000 in
    let r = Str.regexp "OggS" in
    let file = Io.openfile filename [Io.O_RDONLY] 0o400 in
    let filesize = Io.lseek file 0 Io.SEEK_END in
    let _ =
      if filesize < size then Io.lseek file 0 Io.SEEK_SET
      else Io.lseek file (-size) Io.SEEK_END
    in
    let size = min size filesize in
      if size < 5 then begin Io.close file; raise End_of_file end;
      let buf = String.create size in
      let ofs = ref 0 in
        while !ofs < size do
          let r =
            try Io.read file buf !ofs (size - !ofs)
            with End_of_file -> (try Io.close file with _ -> ()); raise End_of_file
          in
            ofs := !ofs + r;
            if r = 0 then (Io.close file; raise End_of_file)
        done;
        Io.close file;
        begin try
          let pos = Str.search_backward r buf (size-5) in
          let dur = ref Int64.zero in
            for i = pos + 13 downto pos + 6 do
              dur := Int64.add (Int64.shift_left !dur 8)
                       (Int64.of_int (int_of_char buf.[i]));
            done;
            !dur
        with Not_found ->
          samples filename
        end

  let fill_header filename samples =
    let header = read_info filename in
      {
        vorbis_version = header.version;
        audio_channels = header.channels;
        audio_sample_rate = header.sample_rate;
        bitrate_nominal = header.bitnom;
        bitrate_maximum = header.bitmax;
        bitrate_minimum = header.bitmin;
        blocksize_0 = header.block0;
        blocksize_1 = header.block1;
        duration = Int64.to_int(
          Int64.div samples (Int64.of_int header.sample_rate));
      }

  let get_info filename =
    fill_header filename (samples filename)

  let get_heuristical_info filename =
    fill_header filename (heuristical_duration filename)

end
