/* -*- Mode: C; tab-width: 4; indent-tabs-mode: t; c-basic-offset: 4 -*-
 *
 * Copyright (C) 2007-2008 Tadas Dailyda <tadas@dailyda.com>
 * Parts of code taken from osso-gwobex library by:
 * 				Johan Hedberg <johan.hedberg@nokia.com>
 *
 * Licensed under the GNU General Public License Version 2
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "config.h"

#include <fcntl.h>
#include <sys/stat.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <bluetooth/sdp.h>
#include <openobex/obex.h>
#include <openobex/obex_const.h>

#include "ods-common.h"
#include "ods-folder-listing.h"
#include "ods-obex.h"


/* FIXME: is this needed at all? */
static void
get_target_size_and_time (obex_t *handle, obex_object_t *object,
							guint64 *size, time_t *time)
{
	obex_headerdata_t hv;
	uint8_t hi;
	unsigned int hlen;

	*size = 0;
	*time = -1;

	while (OBEX_ObjectGetNextHeader(handle, object, &hi, &hv, &hlen)) {
		switch (hi) {
			case OBEX_HDR_LENGTH:
				*size = hv.bq4; //(gint) g_ntohl(hv.bq4);
				break;
			case OBEX_HDR_TIME:
				*time = ods_parse_iso8601 ((char *)hv.bs, hlen);
				break;
			default:
				break;
		}
	}

	OBEX_ObjectReParseHeaders (handle, object);
}

static gint
ods_obex_send(obex_t *obex_handle, obex_object_t *object)
{
	int err;

	err = OBEX_Request (obex_handle, object);

	if (err == -EBUSY) {
		g_warning ("EBUSY in ods_session_obex_send");
	}

	return err;
}

static gchar *
ods_obex_get_new_path (const gchar *folder, const gchar *filename)
{
	gchar *path;
	gchar *new_path;
	guint iterator = 1;
	gchar *first_part;
	gchar *extension;
	gchar *parentess;
	guint pos;
	
	/* In case we don't know what the filename is (HDR_NAME wasn't received) */
	if (filename == NULL || *filename == '\0')
		filename = "Unknown";
	
	path = g_build_filename (folder, filename, NULL);
	new_path = g_strdup (path);
	
	extension = g_strrstr (path, ".");
	if (!extension)
		extension = "";
	while (g_file_test (new_path, G_FILE_TEST_EXISTS)) {
		if (iterator == 1)
			pos = strlen (new_path) - strlen (extension);
		else {
			parentess = g_strrstr (new_path, "(");
			pos = strlen (new_path) - strlen (parentess);
		}
		first_part = g_strndup (new_path, pos);
		g_free (new_path);
		new_path = g_strdup_printf ("%s(%d)%s", first_part, iterator, extension);
		g_free (first_part);
		iterator++;
	}
	
	g_free (path);
	return new_path;
}

void
ods_obex_transfer_new (OdsObexContext *obex_context, const gchar *local,
							const gchar *remote, const gchar *type)
{
	obex_context->local = g_strdup (local);
	obex_context->remote = g_strdup (remote);
	obex_context->type = g_strdup (type);
	obex_context->target_size = 0;
	obex_context->modtime = -1;
	obex_context->report_progress = TRUE; /* by default */
	obex_context->transfer_started_signal_emitted = FALSE;
	obex_context->suspended_once = FALSE;
	obex_context->buf_size = 0;
	obex_context->buf = NULL;
	obex_context->stream_fd = -1;
	obex_context->counter = 0;
}

void
ods_obex_transfer_close (OdsObexContext *obex_context)
{
	if (obex_context->local) {
		g_free (obex_context->local);
		obex_context->local = NULL;
	}
	if (obex_context->remote) {
		g_free (obex_context->remote);
		obex_context->remote = NULL;
	}
	if (obex_context->type) {
		g_free (obex_context->type);
		obex_context->type = NULL;
	}
	if (obex_context->buf) {
		g_free (obex_context->buf);
		obex_context->buf = NULL;
	}
	if (obex_context->stream_fd >= 0)
		close (obex_context->stream_fd);
}

OdsObexContext*
ods_obex_context_new (void)
{
	OdsObexContext *obex_context = g_new0 (OdsObexContext, 1);
	obex_context->rx_max = ODS_OBEX_RX_MTU;
	obex_context->tx_max = ODS_OBEX_TX_MTU - 200;
	obex_context->connection_id = CONID_INVALID;
	obex_context->stream_fd = -1;
	
	return obex_context;
}

gchar *
ods_obex_get_buffer_as_string (OdsObexContext *obex_context)
{
	/* put /0 in the end */
	obex_context->buf = g_realloc (obex_context->buf, obex_context->buf_size+1);
	obex_context->buf[obex_context->buf_size] = 0;
	return (gchar *)obex_context->buf;
}

gboolean
ods_obex_srv_new_file (OdsObexContext *obex_context, const gchar *path)
{
	/* Get local path */
	obex_context->local = ods_obex_get_new_path (path, obex_context->remote);
	/* open local file for writing */
	obex_context->stream_fd = open (obex_context->local, O_WRONLY | O_CREAT, 0600);
	
	return obex_context->stream_fd >= 0;
}

gint
ods_obex_connect_done (OdsObexContext *obex_context, 
								obex_object_t *object)
{
	obex_headerdata_t hv;
	uint8_t hi;
	unsigned int hlen;
	uint8_t *ptr;

	if (OBEX_ObjectGetNonHdrData (object, &ptr) 
			!= sizeof (obex_connect_hdr_t)) {
		//g_warning ("Invalid packet content.");
		return -1;
	} else {
		obex_connect_hdr_t *nonhdrdata = (obex_connect_hdr_t *) ptr;
		uint16_t mtu = g_ntohs( nonhdrdata->mtu);
		int new_size;
		//g_message ("Version: 0x%02x. Flags: 0x%02x  OBEX packet length: %d\n",
		//		nonhdrdata->version, nonhdrdata->flags, mtu);
		/* Leave space for headers */
		new_size = mtu - 200;
		if (new_size < ODS_OBEX_TX_MTU) {
			//g_message ("Resizing stream chunks to %d\n", new_size);
			obex_context->tx_max = new_size;
		}
	}
	/* parse headers */
	while (OBEX_ObjectGetNextHeader(obex_context->obex_handle, object, 
										&hi, &hv, &hlen)) {
		switch (hi) {
#ifdef DEBUG
			case OBEX_HDR_WHO:
				{
				char *str;
				str = bytestr(hv.bs, hlen);
				//g_message ("WHO header (UUID): %s\n", str);
				g_free(str);
				}
				break;
#endif
			case OBEX_HDR_CONNECTION:
				obex_context->connection_id = hv.bq4;
				//g_message ("got Conection ID: %#x\n", hv.bq4);
				break;
			default:
				//g_message ("Skipped header %02x\n", hi);
				break;
		}
	}
	
	return 0;
}

gint
ods_obex_connect (OdsObexContext *obex_context, const guchar *uuid, 
							guint uuid_length)
{
	obex_object_t *object;
	obex_headerdata_t hd;
	int ret;

	object = OBEX_ObjectNew(obex_context->obex_handle, OBEX_CMD_CONNECT);
	if (!object) {
		return -ENOMEM;
	}

	/* Add target header */
	if (uuid) {
        hd.bs = uuid;

		ret = OBEX_ObjectAddHeader(obex_context->obex_handle, object, 
									OBEX_HDR_TARGET, hd, uuid_length, 
									OBEX_FL_FIT_ONE_PACKET);
		if (ret < 0) {
			OBEX_ObjectDelete(obex_context->obex_handle, object);
			return ret;
		}
	}

	ret = ods_obex_send(obex_context->obex_handle, object);	
	if (ret < 0)
		OBEX_ObjectDelete(obex_context->obex_handle, object);

	return ret;
}

gint
ods_obex_srv_connect (OdsObexContext *obex_context, obex_object_t *object,
							guint service)
{
	obex_headerdata_t	hv;
	uint8_t				hi;
	guint				hlen;
	uint8_t				*ptr;
	const guchar		*target = NULL;
	guint				target_len = 0;
	obex_headerdata_t	hd;
	gint				ret;

	if (OBEX_ObjectGetNonHdrData (object, &ptr) 
			!= sizeof (obex_connect_hdr_t)) {
		//g_warning ("Invalid packet content.");
		return -1;
	} else {
		obex_connect_hdr_t *nonhdrdata = (obex_connect_hdr_t *) ptr;
		uint16_t mtu = g_ntohs (nonhdrdata->mtu);
		int new_size;
		//g_message ("Version: 0x%02x. Flags: 0x%02x  OBEX packet length: %d\n",
		//		nonhdrdata->version, nonhdrdata->flags, mtu);
		/* Leave space for headers */
		new_size = mtu - 200;
		if (new_size < ODS_OBEX_TX_MTU) {
			//g_message ("Resizing stream chunks to %d\n", new_size);
			obex_context->tx_max = new_size;
		}
	}
	/* parse headers */
	while (OBEX_ObjectGetNextHeader(obex_context->obex_handle, object, 
										&hi, &hv, &hlen)) {
		if (hi == OBEX_HDR_TARGET) {
			target = hv.bs;
			target_len = hlen;
		}
	}

	OBEX_ObjectReParseHeaders (obex_context->obex_handle, object);

	switch (service) {
		case ODS_SERVICE_FTP:
			/* Target header must be F9EC7BC4-953C-11D2-984E-525400DC9E09*/
			if (!target || memcmp(OBEX_FTP_UUID, target, hlen) != 0) {
				g_message("Target header Incorrect");
				goto fail;
			}
			break;
		case ODS_SERVICE_PBAP:
			/* Target header must be 796135f0-f0c5-11d8-0966-0800200c9a66 */
			if (!target || memcmp(OBEX_PBAP_UUID, target, hlen) != 0) {
				g_message("Target header Incorrect");
				goto fail;
			}
			break;
		case ODS_SERVICE_OPP:
			/* Target header must not be used */
			if (target) {
				g_message("Target header must not be used");
				goto fail;
			}
			OBEX_ObjectSetRsp (object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);

			return 0;
		default:
			goto fail;
	}

	hd.bs = target;
	ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
								OBEX_HDR_WHO, hd, target_len, 
								OBEX_FL_FIT_ONE_PACKET);
	if (ret < 0) {
		OBEX_ObjectDelete (obex_context->obex_handle, object);
		return ret;
	}

	hd.bs = NULL;
	hd.bq4 = 1; /* Connection ID is always 1 */
	obex_context->connection_id = 1;
	ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
								OBEX_HDR_CONNECTION, hd, 4,
								OBEX_FL_FIT_ONE_PACKET);
	if (ret < 0) {
		OBEX_ObjectDelete (obex_context->obex_handle, object);
		return ret;
	}

	OBEX_ObjectSetRsp (object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);

	return 0;

fail:

	OBEX_ObjectSetRsp (object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);

	return 0;
}

gint
ods_obex_disconnect (OdsObexContext *obex_context)
{
    obex_object_t	*object;
    gint			ret;

	g_message ("Sending CMD_DISCONNECT");
    object = OBEX_ObjectNew(obex_context->obex_handle, OBEX_CMD_DISCONNECT);
    if (!object) {
		return -ENOMEM;
	}

	/* Add connection header */
    if (obex_context->connection_id != CONID_INVALID) {
        obex_headerdata_t hv;
        hv.bq4 = obex_context->connection_id;
        ret = OBEX_ObjectAddHeader(obex_context->obex_handle, object, 
        							OBEX_HDR_CONNECTION, hv, 4, 0);
        if (ret < 0) {
        	OBEX_ObjectDelete(obex_context->obex_handle, object);
        	return ret;
        }
    }

    ret = ods_obex_send (obex_context->obex_handle, object);
    if (ret < 0)
		OBEX_ObjectDelete(obex_context->obex_handle, object);
	
	return ret;
}

gint
ods_obex_readstream (OdsObexContext *obex_context, obex_object_t *object)
{
	const uint8_t	*buf;
	gint			actual;
	gint			written = 0;
	gint			write_ret;
	gint			ret = 0;

	/* FIXME: Is this needed? */
	if (obex_context->target_size == 0 && obex_context->counter == 0 &&
			obex_context->obex_cmd == 0) {
		/* first data came in, get size and time */
		/* (only in client mode) */
		get_target_size_and_time (obex_context->obex_handle, object,
									&obex_context->target_size,
									&obex_context->modtime);
	}

	actual = OBEX_ObjectReadStream (obex_context->obex_handle, object, &buf);
	if (actual > 0) {
		g_message ("There is some data");
		obex_context->counter += actual;

		if (obex_context->stream_fd >= 0) {
			/* write data to file */
			while (written < actual) {

				write_ret = write (obex_context->stream_fd, buf + written, 
									actual - written);
				if (write_ret < 0 && errno == EINTR)
					continue;

				if (write_ret < 0) {
					ret = -errno;
					break;
				}

				written += write_ret;
			}
		} else {
			g_message ("Writing to buf");
			/* write data to internal buffer */
			obex_context->buf = g_realloc (obex_context->buf, 
											obex_context->counter);
			memcpy (&obex_context->buf[obex_context->buf_size], buf, actual);
			obex_context->buf_size = obex_context->counter;
			//printf ((gchar*)obex_context->buf);
		}
	} else {
		/* No data on OBEX stream */
		ret = 1;
	}
	
	if (ret < 0) {
		/* close this transfer */
		g_message ("closing transfer");
		ods_obex_transfer_close (obex_context);
	}
	
	return ret;
}

gint
ods_obex_writestream (OdsObexContext *obex_context, obex_object_t *object)
{
	g_message ("obex_writestream");
	obex_headerdata_t	hv;
	gint				actual = -1;
	gint				ret = 0;

	if (obex_context->stream_fd >= 0) {
		g_message ("writestream from File: %d", obex_context->stream_fd);
		actual = read (obex_context->stream_fd, obex_context->buf, 
						obex_context->tx_max);
		hv.bs = obex_context->buf;
		if (actual > 0) {
			OBEX_ObjectAddHeader (obex_context->obex_handle, object,
									OBEX_HDR_BODY, hv, actual,
									OBEX_FL_STREAM_DATA);
			obex_context->counter += actual;
			/* Everything OK, continue sending data */
			ret = 0;
		} else if (actual == 0) {
			/* EOF */
			OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
									OBEX_HDR_BODY, hv, 0,
									OBEX_FL_STREAM_DATAEND);
			/* transfer done */
			ret = 1;
		} else {
			/* error reading file */
			ret = -errno;
		}
	} else if (obex_context->buf_size > 0) {
		g_message ("writestream from Buffer");
		/* used only in server mode to send folder listings and such */
		actual = obex_context->buf_size - obex_context->counter;
		if (actual > obex_context->tx_max)
			actual = obex_context->tx_max;
		g_message ("buf_size: %" G_GUINT64_FORMAT ", actual: %d",
					obex_context->buf_size, actual);
		hv.bs = &obex_context->buf[obex_context->counter];
		if (actual > 0) {
			OBEX_ObjectAddHeader (obex_context->obex_handle, object,
									OBEX_HDR_BODY, hv, actual,
									OBEX_FL_STREAM_DATA);
			obex_context->counter += actual;
			/* Everything OK, continue sending data */
			ret = 0;
		} else if (actual == 0) {
			/* EOF */
			OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
									OBEX_HDR_BODY, hv, 0,
									OBEX_FL_STREAM_DATAEND);
			/* transfer done */
			ret = 1;
		}
	} else {
		/* shouldn't happen */
		g_warning ("Invalid fd while transfer in progress");
		ret = -1;
	}
	
	if (ret < 0) {
		/* close this transfer */
		ods_obex_transfer_close (obex_context);
	}

	return ret;
}

gint
ods_obex_get (OdsObexContext *obex_context,
						const gchar *local, const gchar *remote,
						const gchar *type)
{
	gint				ret;
	obex_headerdata_t	hv;
	obex_object_t		*object;
	gchar				*uname;
	gsize				uname_len = 0;

	g_assert (remote || type);

	ods_obex_transfer_new (obex_context, local, remote, type);
	obex_context->obex_cmd = OBEX_CMD_GET;

	object = OBEX_ObjectNew (obex_context->obex_handle, OBEX_CMD_GET);
	if (!object) {
		ret = -ENOMEM;
		goto out;
	}

	/* Add connection header */
	if (obex_context->connection_id != CONID_INVALID) {
		hv.bq4 = obex_context->connection_id;
		ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
										OBEX_HDR_CONNECTION, hv, 4, 0);
		if (ret < 0)
			goto out;
	}

	/* Add type header */
	if (type) {
		hv.bs = (guchar *)type;
		ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
									OBEX_HDR_TYPE, hv, strlen (type) + 1, 0);
		if (ret < 0)
			goto out;
	}

	/* Add name header */
	if (remote) {
		

		uname_len = ods_filename_to_utf16 (&uname, remote);
		if (uname == NULL) {
			ret = -EINVAL;
			goto out;
		}

		/* OpenOBEX is buggy and won't append the header unless hv.bs != NULL */
		hv.bs = (guchar *) uname;

		ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object,
									OBEX_HDR_NAME, hv, uname_len, 0);
		if (ret < 0)
			goto out;
	}

	/* Add local name header */
	if (local) {
		obex_context->stream_fd = open (local, O_WRONLY | O_CREAT, 0600);
		if (obex_context->stream_fd < 0) {
			ret = -errno;
			goto out;
		}
	} else {
		/* don't report progress when receiving to internal buffer only */
		obex_context->report_progress = FALSE;
	}
	/* Initiate transfer */
	OBEX_ObjectReadStream (obex_context->obex_handle, object, NULL);
	ret = ods_obex_send (obex_context->obex_handle, object);
out:
	if (uname_len > 0)
		g_free (uname);
	if (ret < 0 && object)
		OBEX_ObjectDelete (obex_context->obex_handle, object);
	if (ret < 0 && obex_context->stream_fd >= 0)
		g_unlink (obex_context->local); /* delete incomplete file */
	if (ret < 0)
		ods_obex_transfer_close (obex_context);
	return ret;
}

gint
ods_obex_srv_get (OdsObexContext *obex_context, obex_object_t *object,
					const gchar *current_path, const gchar *root_path,
					gboolean allow_write)
{
	obex_headerdata_t	hv;
	uint8_t				hi;
	guint				hlen;
	gint				object_size = 0;
	time_t				object_time = -1;
	gint				ret;

	g_message ("stream_fd=%d", obex_context->stream_fd);
	while (OBEX_ObjectGetNextHeader (obex_context->obex_handle, object,
										&hi, &hv, &hlen)) {
		switch (hi) {
			case OBEX_HDR_NAME:
				if (hlen == 0) {
					/* This is GET by Type, leave remote and local = NULL */
					break;
				}
				obex_context->remote = ods_filename_from_utf16 ((gchar *) hv.bs, hlen);
				obex_context->local = g_build_filename (current_path,
														obex_context->remote,
														NULL);
				g_message ("local filename: %s", obex_context->local);
				/* Check if such file exists */
				if (!g_file_test (obex_context->local, G_FILE_TEST_EXISTS)) {
					ret = -1;
					OBEX_ObjectSetRsp (object, OBEX_RSP_NOT_FOUND, OBEX_RSP_NOT_FOUND);
					goto out;
				}
				if (!g_file_test (obex_context->local, G_FILE_TEST_IS_REGULAR)) {
					ret = -1;
					OBEX_ObjectSetRsp (object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
					goto out;
				}
				if (g_access (obex_context->local, R_OK) < 0) {
					ret = -1;
					OBEX_ObjectSetRsp (object, OBEX_RSP_UNAUTHORIZED, OBEX_RSP_UNAUTHORIZED);
					goto out;
				}
				break;
				
			case OBEX_HDR_TYPE:
				if (hv.bs[hlen - 1] != '\0' ||
						!g_utf8_validate ((const gchar *) hv.bs, -1, NULL)) {
					/* invalid type header */
					ret = -1;
					OBEX_ObjectSetRsp (object, OBEX_RSP_BAD_REQUEST, OBEX_RSP_BAD_REQUEST);
					goto out;
				}
				
				obex_context->type = g_strdup ((const gchar *) hv.bs);
				g_message ("HDR_TYPE: %s", obex_context->type);
				break;
			
			case OBEX_HDR_CONNECTION:
				if (obex_context->connection_id != CONID_INVALID &&
						hv.bq4 != obex_context->connection_id) {
					/* wrong connection id */
					ret = -1;
					OBEX_ObjectSetRsp (object, OBEX_RSP_BAD_REQUEST, OBEX_RSP_BAD_REQUEST);
					goto out;
				}
				break;
			default:
				break;
		}
	}
	g_message ("name: %s, type: %s", obex_context->remote, obex_context->type);
	
	OBEX_ObjectReParseHeaders (obex_context->obex_handle, object);
	
	if (obex_context->remote) {
		g_message ("Serving local file");
		/* open local file for reading */
		obex_context->stream_fd = open (obex_context->local, O_RDONLY);
		if (obex_context->stream_fd < 0) {
			ret = -1;
			OBEX_ObjectSetRsp (object, OBEX_RSP_INTERNAL_SERVER_ERROR, OBEX_RSP_INTERNAL_SERVER_ERROR);
			goto out;
		}
		/* allocate buffer */
		obex_context->buf = g_malloc (obex_context->tx_max);
		obex_context->buf_size = obex_context->tx_max;
		
		/* Try to figure out modification time and size */
		struct stat stats;
		if (fstat (obex_context->stream_fd, &stats) == 0) {
			object_size = stats.st_size;
			object_time = stats.st_mtime;
			obex_context->modtime = object_time;
		}

		/* Add a time header */
		if (object_time >= 0) {
			gchar tstr[17];
			gint len;
	
			len = ods_make_iso8601 (object_time, tstr, sizeof (tstr));
			
			if (len >= 0) {
				hv.bs = (guchar *)tstr;
				ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
											OBEX_HDR_TIME, hv, len, 0);
				if (ret < 0) {
					ret = -1;
					OBEX_ObjectSetRsp (object, OBEX_RSP_INTERNAL_SERVER_ERROR, OBEX_RSP_INTERNAL_SERVER_ERROR);
					goto out;
				}
				
			}
		}

		/* Add a length header */
		if (object_size > 0) {
			obex_context->target_size = object_size;
			hv.bq4 = (uint32_t)object_size;
			ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
										OBEX_HDR_LENGTH, hv, 4, 0);
			if (ret < 0) {
				ret = -1;
				OBEX_ObjectSetRsp (object, OBEX_RSP_INTERNAL_SERVER_ERROR, OBEX_RSP_INTERNAL_SERVER_ERROR);
				goto out;
			}
		} else {
			obex_context->target_size = 0;
		}
	} else if (obex_context->type) {
		/* Don't report progress for object GET by type */
		obex_context->report_progress = FALSE;
		if (!strcmp (obex_context->type, LST_TYPE)) {
			g_message ("Serving FOLDER LISTING");
			/* write folder listing to buffer */
			obex_context->buf = (guchar*) get_folder_listing (
												current_path, root_path, allow_write);
			g_message ("Folder listing: %s", (gchar*)obex_context->buf);
			obex_context->buf_size = strlen ((gchar*) obex_context->buf);
			g_message ("Folder listing length: %" G_GUINT64_FORMAT,
						obex_context->buf_size);
		} else {
			/* currently no other types are supported */
			ret = -1;
			OBEX_ObjectSetRsp (object, OBEX_RSP_NOT_IMPLEMENTED, OBEX_RSP_NOT_IMPLEMENTED);
			goto out;
		}
	} else {
		/* neither name nor type was specified */
		ret = -1;
		OBEX_ObjectSetRsp (object, OBEX_RSP_BAD_REQUEST, OBEX_RSP_BAD_REQUEST);
		goto out;
	}
	
	/* Add body header */
	hv.bs = NULL;
	ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
								OBEX_HDR_BODY, hv, 0, OBEX_FL_STREAM_START);
out:
	if (ret < 0)
		ods_obex_transfer_close (obex_context);
	else
		OBEX_ObjectSetRsp (object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
	return ret;
}

gint
ods_obex_put (OdsObexContext *obex_context,
						const gchar *local, const gchar *remote, 
						const gchar *type)
{
	gint				ret;
	obex_headerdata_t	hv;
	obex_object_t		*object = NULL;
	gchar				*uname = NULL;
	gsize				uname_len = 0;
	gint				object_size = 0;
	time_t				object_time = -1;

	g_assert (remote || type);

	ods_obex_transfer_new (obex_context, local, remote, type);
	obex_context->obex_cmd = OBEX_CMD_PUT;
	
	/* get UTF name for remote file */
	if (remote) {
		uname_len = ods_filename_to_utf16 (&uname, remote);
		if (uname == NULL) {
			ret = -EINVAL;
			goto out;
		}
	}

	/* open local file, allocate buffer */
	if (local) {	
		obex_context->stream_fd = open (local, O_RDONLY);
		if (obex_context->stream_fd < 0) {
			if (errno == ENOENT || errno == ENODEV) {
				ret = -errno;
				goto out;
			}
		}
		/* Allocate buffer */
		obex_context->buf = g_malloc (obex_context->tx_max);
		obex_context->buf_size = obex_context->tx_max;
		//debug("Sending %s to %s\n", local, remote ? remote : type);
	} else {
		obex_context->report_progress = FALSE;
		//debug("Deleting %s\n", remote ? remote : type);
    }

	object = OBEX_ObjectNew (obex_context->obex_handle, OBEX_CMD_PUT);
	if (!object) {
		ret = -ENOMEM;
		goto out;
	}

	/* Add connection header */
	if (obex_context->connection_id != CONID_INVALID) { 
		hv.bq4 = obex_context->connection_id;
		ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object,
									OBEX_HDR_CONNECTION, hv, 4, 0);
		if (ret < 0)
			goto out;
	}

	/* Add name header */
	if (uname) {
		hv.bs = (guchar *) uname;
		ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
									OBEX_HDR_NAME, hv, uname_len, 0);
		if (ret < 0)
			goto out;
	}

	/* Add type header */
	if (type) {
		hv.bs = (unsigned char *)type;
		ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
									OBEX_HDR_TYPE, hv, strlen(type) + 1, 0);
		if (ret < 0)
			goto out;
	}

    /* Try to figure out modification time and size */
	if (obex_context->stream_fd >= 0) {
		struct stat stats;
		if (fstat (obex_context->stream_fd, &stats) == 0) {
			object_size = stats.st_size;
			object_time = stats.st_mtime;
			obex_context->modtime = object_time;
		}
	}

	/* Add a time header */
	if (object_time >= 0) {
		gchar tstr[17];
		gint len;

		len = ods_make_iso8601 (object_time, tstr, sizeof (tstr));
		
		if (len >= 0) {
			hv.bs = (guchar *)tstr;
			ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
										OBEX_HDR_TIME, hv, len, 0);
			if (ret < 0)
				goto out;
		}
	}

	/* Add a length header */
	if (object_size > 0) {
		obex_context->target_size = object_size;
		hv.bq4 = (uint32_t)object_size;
		ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
									OBEX_HDR_LENGTH, hv, 4, 0);
		if (ret < 0)
			goto out;
	} else {
		obex_context->target_size = 0;
	} 

	/* Add body header */
	if (obex_context->stream_fd >= 0) {
		hv.bs = NULL;
		ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
									OBEX_HDR_BODY, hv, 0, OBEX_FL_STREAM_START);
		if (ret < 0)
			goto out;
	}

	ret = ods_obex_send (obex_context->obex_handle, object);
out:
	if (uname_len > 0)
		g_free (uname);
	if (ret < 0 && object)
		OBEX_ObjectDelete (obex_context->obex_handle, object);
	if (ret < 0)
		ods_obex_transfer_close (obex_context);

	return ret;
}

gint
ods_obex_srv_put (OdsObexContext *obex_context, obex_object_t *object,
					const gchar *path)
{
	obex_headerdata_t	hv;
	uint8_t				hi;
	guint				hlen;
	gint				ret = 0;
	guint				written = 0;
	gint				write_ret;
	gboolean			is_delete = TRUE;
	
	/* Check if we already have all transfer info
	 * because both EV_REQCHECK and EV_REQ trigger this function */
	if (obex_context->stream_fd >= 0)
		return 1;
	
	while (OBEX_ObjectGetNextHeader(obex_context->obex_handle, object,
									&hi, &hv, &hlen)) {
		g_message ("header: %d", hi);
		switch (hi) {
			case OBEX_HDR_BODY:
				is_delete = FALSE;
				g_message ("HDR_BODY length=%u", hlen);
				break;
				
			case OBEX_HDR_NAME:
				obex_context->remote = ods_filename_from_utf16 ((gchar *) hv.bs, hlen);
				g_message ("HDR_NAME: %s", obex_context->remote);
				break;
				
			case OBEX_HDR_TYPE:
				if (hv.bs[hlen - 1] != '\0' ||
					!g_utf8_validate ((const gchar *) hv.bs, -1, NULL)) {
					/* invalid type header */
					ret = -1;
					OBEX_ObjectSetRsp (object, OBEX_RSP_BAD_REQUEST, OBEX_RSP_BAD_REQUEST);
					goto out;
				}
				
				obex_context->type = g_strdup ((const gchar *) hv.bs);
				g_message ("HDR_TYPE: %s", obex_context->type);
				break;
				
			case OBEX_HDR_LENGTH:
				obex_context->target_size = hv.bq4;
				is_delete = FALSE;
				g_message ("HDR_LENGTH: %" G_GUINT64_FORMAT, obex_context->target_size);
				break;
				
			case OBEX_HDR_TIME:
				obex_context->modtime = ods_parse_iso8601 ((gchar*) hv.bs, hlen);
				g_message ("HDR_TIME");
				break;
				
			case OBEX_HDR_DESCRIPTION:
				/* Not very useful info */
				break;
				
			case OBEX_HDR_COUNT:
				/* This informs us how many objects client is going to send
				 * during this session. We really don't care. */
				break;
			
			case OBEX_HDR_CONNECTION:
				if (obex_context->connection_id != CONID_INVALID &&
						hv.bq4 != obex_context->connection_id) {
					/* wrong connection id */
					ret = -1;
					OBEX_ObjectSetRsp (object, OBEX_RSP_BAD_REQUEST, OBEX_RSP_BAD_REQUEST);
					goto out;
				}
				break;

			default:
				break;
		}
	}
	
	OBEX_ObjectReParseHeaders (obex_context->obex_handle, object);
	
	g_message ("path: %s", path);
	
	/* Open file for writing only if some data was received already */
	if (obex_context->buf_size > 0) {
		obex_context->report_progress = TRUE;
		if (!ods_obex_srv_new_file (obex_context, path)) {
			ret = -1;
			OBEX_ObjectSetRsp (object, OBEX_RSP_UNAUTHORIZED, OBEX_RSP_UNAUTHORIZED);
			goto out;
		}
	} else if (is_delete) {
		/* this is a delete request */
		obex_context->local = g_build_filename (path, obex_context->remote, NULL);

		if (!g_file_test (obex_context->local, G_FILE_TEST_EXISTS)) {
			ret = -1;
			OBEX_ObjectSetRsp (object, OBEX_RSP_NOT_FOUND, OBEX_RSP_NOT_FOUND);
			goto out;
		}
		if (g_access (obex_context->local, W_OK) < 0) {
			ret = -1;
			OBEX_ObjectSetRsp (object, OBEX_RSP_UNAUTHORIZED, OBEX_RSP_UNAUTHORIZED);
			goto out;
		}
		
		g_message ("Deleting: %s", obex_context->local);
		if (g_file_test (obex_context->local, G_FILE_TEST_IS_DIR))
			ret = rmdir (obex_context->local);
		else
			ret = g_unlink (obex_context->local);
						
		if (ret == -1)
			OBEX_ObjectSetRsp (object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
	}

	/* If there is some data received already, write it to file */
	while (written < obex_context->buf_size) {
		write_ret = write (obex_context->stream_fd, obex_context->buf + written, 
							obex_context->buf_size - written);
		if (write_ret < 0 && errno == EINTR)
			continue;

		if (write_ret < 0) {
			ret = -errno;
			OBEX_ObjectSetRsp (object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
			break;
		}

		written += write_ret;
	}

out:
	if (ret < 0)
		ods_obex_transfer_close (obex_context);
	else
		OBEX_ObjectSetRsp (object, OBEX_RSP_CONTINUE, OBEX_RSP_SUCCESS);
	return ret;
}

gint
ods_obex_setpath (OdsObexContext *obex_context, const gchar *path,
							gboolean create)
{
	gint				ret;
	obex_headerdata_t	hv;
	obex_object_t		*object = NULL;
	obex_setpath_hdr_t	nonhdrdata;
	gchar				*uname; 
	gsize				uname_len;

	nonhdrdata.flags = 0x02;
	nonhdrdata.constants = 0;

	if (strcmp (path, "..") == 0) {
		/* move up one directory */
		nonhdrdata.flags = 0x03;
		uname_len = 0;
	} else {
		/* normal directory change */
		uname_len = ods_filename_to_utf16 (&uname, path);
 		if (uname == NULL) {
 			ret = -EINVAL;
			goto out;
 		}
	}

	/* Add create flag */
	if (create)
		nonhdrdata.flags &= ~0x02;

	object = OBEX_ObjectNew (obex_context->obex_handle, OBEX_CMD_SETPATH);
	if (!object) {
		ret = -ENOMEM;
		goto out;
	}
	
	/* Attach flags */
	ret = OBEX_ObjectSetNonHdrData (object, (uint8_t*)&nonhdrdata, 2);
	if (ret < 0)
		goto out;

	/* Add connection header */
	if (obex_context->connection_id != CONID_INVALID) {
		hv.bq4 = obex_context->connection_id;
		ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
									OBEX_HDR_CONNECTION, hv, 4, 0);
		if (ret < 0)
			goto out;
	}

	/* Add name header */
	hv.bs = (guchar *) uname;
    ret = OBEX_ObjectAddHeader (obex_context->obex_handle, object, 
    							OBEX_HDR_NAME, hv, uname_len, 0);
    if (ret < 0)
    	goto out;

	ret = ods_obex_send (obex_context->obex_handle, object);
out:
	if (uname_len > 0 )
		g_free (uname);
	if (ret < 0 && object)
		OBEX_ObjectDelete (obex_context->obex_handle, object);
	return ret;
}

gboolean
ods_obex_srv_setpath (OdsObexContext *obex_context, obex_object_t *object,
						const gchar *root_path, const gchar *current_path,
						gchar **new_path)
{
	uint8_t				*nonhdrdata_dummy = NULL;
	obex_setpath_hdr_t	*nonhdrdata;
	obex_headerdata_t	hv;
	uint8_t				hi;
	guint				hlen;
	gchar				*directory;
	gboolean			create = FALSE;
	gboolean			backup = FALSE;
	
	OBEX_ObjectGetNonHdrData (object, &nonhdrdata_dummy);
	if (!nonhdrdata_dummy) {
		OBEX_ObjectSetRsp (object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
		return FALSE;
	}
	nonhdrdata = (obex_setpath_hdr_t*) nonhdrdata_dummy;
	
	if (nonhdrdata->flags == 0)
		create = TRUE;
	else if (nonhdrdata->flags == 0x03)
		backup = TRUE;
	else if (nonhdrdata->flags != 0x02) {
		OBEX_ObjectSetRsp (object, OBEX_RSP_NOT_FOUND, OBEX_RSP_NOT_FOUND);
		return FALSE;
	}
		
	if (backup) {
		/* we have to go to parent directory */
		/* Check if we can't go up */
		if (strcmp (root_path, current_path) == 0) {
			OBEX_ObjectSetRsp (object, OBEX_RSP_NOT_FOUND, OBEX_RSP_NOT_FOUND);
			return FALSE;
		}
		
		*new_path = g_path_get_dirname (current_path);
		OBEX_ObjectSetRsp (object, OBEX_RSP_SUCCESS, OBEX_RSP_SUCCESS);
		return TRUE;
	}

	while (OBEX_ObjectGetNextHeader (obex_context->obex_handle, object, 
										&hi, &hv, &hlen)) {
		if (hi == OBEX_HDR_NAME) {
			if (hlen > 0) {
				/* Normal directory change */
				directory = ods_filename_from_utf16 ((gchar *) hv.bs, hlen);
				*new_path = g_build_filename (current_path, directory, NULL);
				g_free (directory);
				/* Check if such path exists */
				if (g_file_test (*new_path, G_FILE_TEST_EXISTS)) {
					if (create) {
						g_free (*new_path);
						*new_path = ods_obex_get_new_path (current_path,
															directory);
						
					} else {
						OBEX_ObjectSetRsp (object, OBEX_RSP_SUCCESS,
											OBEX_RSP_SUCCESS);
						return TRUE;
					}
				} else if (!create) {
					
					g_free (*new_path);
					OBEX_ObjectSetRsp (object, OBEX_RSP_NOT_FOUND,
										OBEX_RSP_NOT_FOUND);
					return FALSE;
				}
				/* In case we are Creating new folder */
				if (mkdir (*new_path, 0755) == 0) {
					OBEX_ObjectSetRsp (object, OBEX_RSP_SUCCESS,
										OBEX_RSP_SUCCESS);
					return TRUE;
				} else {
					OBEX_ObjectSetRsp (object, OBEX_RSP_FORBIDDEN,
										OBEX_RSP_FORBIDDEN);
					return FALSE;
				}
			} else {
				/* Name header empty, change path to root */
				*new_path = g_strdup (root_path);
				OBEX_ObjectSetRsp (object, OBEX_RSP_SUCCESS, OBEX_RSP_SUCCESS);
				return TRUE;
			}
		}
	}
	
	/* invalid headers? */
	OBEX_ObjectSetRsp (object, OBEX_RSP_FORBIDDEN, OBEX_RSP_FORBIDDEN);
	return FALSE;
}
