/* NVTV server -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id$
 *
 * Contents:
 *
 * Server main program
 *
 */

#include <stdio.h>
#include <getopt.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "pipe.h"
#include "debug.h"
#include "backend.h"
#include "back_direct.h"
#include "back_null.h"

/* -------- -------- */

BackFuncPtr backend; /* global backend */

Bool opt_null  = FALSE;
Bool opt_nvdev = FALSE;

static const char *short_options = "?hnN";

static struct option long_options[] =
  {{"help",        no_argument,       NULL, 'h'},
   {"null",        no_argument,       NULL, 'n'},
   {"nvdev",       no_argument,       NULL, 'N'},
   {NULL,          0,                 NULL, 0}
};

char *prog; /* Program name for error messages */
   
/* -------- -------- */

void usage (void)
{
  fprintf (stderr,
      "usage:  nvtv [-options ...]\n\n");
  fprintf (stderr,
      "where options include:\n");
  fprintf (stderr,
      "  -h --help           print this message\n");
  fprintf (stderr,
      "  -N --nvdev          enable usage of /dev/nv* devices\n");
  fprintf (stderr,
      "  -n --null           use null backend (for debugging)\n");
}

/* -------- Server state -------- */

static CardPtr srv_root = NULL;
static CardPtr srv_card = NULL;

static FILE *pipe_in = NULL;
static FILE *pipe_out = NULL;

/* -------- Server command routines -------- */

void srv_openCard (void)
{
  CardPtr card;
  int i, index;

  DPRINTF ("srv_open\n");
  index = 0;
  pipeReadArgs (pipe_in, 1, sizeof(index), &index);
  /* convert index to card */
  card = srv_root;
  for (i = 1; i < index; i++) {
    if (card) card = card->next;
  }
  if (index == 0) card = NULL;
  srv_card = card;
  backend->openCard (srv_card);
  pipeWriteCmd (pipe_out, PCmd_OpenCard);
  if (srv_card) {
    pipeWriteList (pipe_out, sizeof (ChipInfo), srv_card->chips);
  } else {
    pipeWriteArgs (pipe_out, 0);
  }
}

void srv_closeCard (void)
{
  DPRINTF ("srv_close\n");
  pipeReadArgs (pipe_in, 0);
  if (srv_card) {
    backend->closeCard ();
  }
  srv_card = NULL;
}

void srv_probeChips (void)
{
  DPRINTF ("srv_probe\n");
  pipeReadArgs (pipe_in, 0);
  backend->probeChips ();
  pipeWriteCmd (pipe_out, PCmd_ProbeChips);
  pipeWriteList (pipe_out, sizeof (ChipInfo), srv_card->chips);
}

void srv_setChip (void)
{
  ChipPtr chip;
  Bool init;
  int i, index;

  DPRINTF ("srv_setChip\n");
  index = 0;
  init = 1;
  pipeReadArgs (pipe_in, 2, sizeof(index), &index, sizeof(init), &init);
  /* convert index to chip */
  chip = srv_card->chips;
  for (i = 1; i < index; i++) {
    if (chip) chip = chip->next;
  }
  if (index == 0) chip = NULL;
  backend->setChip (chip, init);
}

void srv_setSettings (void)
{
  NVSettings set, *pset;

  DPRINTF ("srv_setSettings\n");
  pipeReadArgsOpt (pipe_in, 1, sizeof(NVSettings), &set, &pset);
  backend->setSettings (pset);
}

void srv_getSettings (void)
{
  NVSettings set;

  DPRINTF ("srv_getSettings\n");
  pipeReadArgs (pipe_in, 0);
  backend->getSettings (&set);
  pipeWriteCmd (pipe_out, PCmd_GetSettings);
  pipeWriteArgs (pipe_out, 1, sizeof(NVSettings), &set);
}

void srv_setMode (void)
{
  int flags; 
  NVCrtRegs crt, *pcrt;
  NVTvRegs tv, *ptv;

  DPRINTF ("srv_setMode\n");
  flags = 0;
  pipeReadArgsOpt (pipe_in, 3, sizeof(flags), &flags, NULL,
    sizeof(NVCrtRegs), &crt, &pcrt, sizeof(NVTvRegs), &tv, &ptv);
  backend->setMode (flags, pcrt, ptv);
}

void srv_getMode (void)
{
  NVCrtRegs crt;
  NVTvRegs tv;

  DPRINTF ("srv_getMode\n");
  pipeReadArgs (pipe_in, 0);
  backend->getMode (&crt, &tv);
  pipeWriteCmd (pipe_out, PCmd_GetMode);
  pipeWriteArgs (pipe_out, 2, sizeof(NVCrtRegs), &crt, sizeof(NVTvRegs), &tv);
}

void srv_setModeSettings (void)
{
  int flags; 
  NVCrtRegs crt, *pcrt;
  NVTvRegs tv, *ptv;
  NVSettings set, *pset;

  DPRINTF ("srv_setModeSettings\n");
  flags = 0;
  pipeReadArgsOpt (pipe_in, 4, sizeof(flags), &flags, NULL,
    sizeof(NVCrtRegs), &crt, &pcrt, sizeof(NVTvRegs), &tv, &ptv,
    sizeof(NVSettings), &set, &pset);
  backend->setModeSettings (flags, pcrt, ptv, pset);
}

void srv_setTestImage (void)
{
  NVTvRegs tv, *ptv;
  NVSettings set, *pset;

  DPRINTF ("srv_setTestImage\n");
  pipeReadArgsOpt (pipe_in, 2, sizeof(NVTvRegs), &tv, &ptv,
    sizeof(NVSettings), &set, &pset);
  backend->setTestImage (ptv, pset);
}

void srv_getStatus (void)
{
  int index;
  long l;

  DPRINTF ("srv_getStatus\n");
  index = 0;
  pipeReadArgs (pipe_in, 1, sizeof(index), &index);
  pipeWriteCmd (pipe_out, PCmd_GetStatus);
  l = backend->getStatus (index);
  pipeWriteArgs (pipe_out, 1, sizeof(l), &l);
}

void srv_getConnection (void)
{
  NVConnect c;

  DPRINTF ("srv_getConnection\n");
  pipeReadArgs (pipe_in, 0);
  pipeWriteCmd (pipe_out, PCmd_GetConnection);
  c = backend->getConnection ();
  pipeWriteArgs (pipe_out, 1, sizeof(c), &c);
}

void srv_findBySize (void)
{
  NVSystem system;
  int xres, yres;
  char *size;
  NVMode mode; 
  NVCrtRegs crt;
  NVTvRegs tv;

  DPRINTF ("srv_findBySize\n");
  pipeReadArgs (pipe_in, 4, sizeof(system), &system, 
		sizeof(xres), &xres, sizeof(yres), &yres, 0, &size);
  pipeWriteCmd (pipe_out, PCmd_FindBySize);
  if (backend->findBySize (system, xres, yres, size, &mode, &crt, &tv)) {
    pipeWriteArgs (pipe_out, 3, sizeof(NVMode), &mode,
		   sizeof(NVCrtRegs), &crt, sizeof(NVTvRegs), &tv);
  } else {
    pipeWriteArgs (pipe_out, 1, 0, NULL);
  }
  /* FIXME must transfer crt and tv */
}

void srv_findByOverscan (void)
{
  NVSystem system;
  int xres, yres;
  double hoc, voc;
  NVMode mode; 
  NVCrtRegs crt;
  NVTvRegs tv;

  DPRINTF ("srv_findByOC\n");
  pipeReadArgs (pipe_in, 5, sizeof(system), &system, 
		 sizeof(xres), &xres, sizeof(yres), &yres,
		 sizeof(hoc), &hoc, sizeof(voc), &voc);
  pipeWriteCmd (pipe_out, PCmd_FindByOverscan);
  if (backend->findByOverscan (system, xres, yres, hoc, voc, 
			       &mode, &crt, &tv)) 
  {
    pipeWriteArgs (pipe_out, 3, sizeof(NVMode), &mode,
		   sizeof(NVCrtRegs), &crt, sizeof(NVTvRegs), &tv);
  } else {
    pipeWriteArgs (pipe_out, 1, 0, NULL);
  }
}

void srv_version (void)
{
  int version = PIPE_VERSION;

  pipeReadArgs (pipe_in, 0);
  pipeWriteCmd (pipe_out, PCmd_Version);
  pipeWriteArgs (pipe_out, 1, sizeof(version), &version);
}

void srv_init (void)
{
  pipeReadArgs (pipe_in, 0);
  if (srv_card) {
    backend->closeCard ();
  }
  srv_card = NULL;
  pipeWriteCmd (pipe_in, PCmd_Init);
  pipeWriteList (pipe_out, sizeof (CardInfo), srv_root);
}


void srv_kill (void)
{
  pipeReadArgs (pipe_in, 0);
  sleep (1);
  /* FIXME */
}

void srv_openPipes (void)
{
  /* IMPORTANT: PIPE_OUT is open first for reading (and thus is
     pipe_in for the server). The open blocks until someone is writing
     to the pipe. */

  if (!pipe_in) {
    DPRINTF ("open in pipe.\n");
    pipe_in  = fopen (PIPE_OUT, "r");
  }
  if (!pipe_in) {
    fprintf (stderr, "%s: Cannot open pipe %s\n", prog, PIPE_OUT);
    unlink (PIPE_OUT);
    unlink (PIPE_IN);
    exit (1);
  }
  DPRINTF ("in pipe opened.\n");

  /* Now open the other pipe for writing. */

  if (!pipe_out) {
    DPRINTF ("open out pipe.\n");
    pipe_out = fopen (PIPE_IN, "w");
  }
  if (!pipe_out) {
    fprintf (stderr, "%s: Cannot open pipe %s\n", prog, PIPE_IN);
    unlink (PIPE_OUT);
    unlink (PIPE_IN);
    exit (1);
  }
  DPRINTF ("out pipe opened.\n");
}

void srv_closePipes ()
{
  DPRINTF ("close pipes.\n");
  fclose (pipe_in);
  fclose (pipe_out);
  pipe_in = NULL;
  pipe_out = NULL;
}

void srv_loop (void) 
{
  PipeCmd cmd;

  while (TRUE) {
    srv_openPipes ();
    DPRINTF ("srv_loop read cmd (eof %i)\n", feof (pipe_in));
    cmd = pipeReadCmd (pipe_in);
    DPRINTF ("srv_loop %i (eof %i)\n", cmd, feof (pipe_in));
    if (feof (pipe_in)) srv_closePipes ();
    if (!pipe_in || !pipe_out) continue;
    switch (cmd) {
      case PCmd_Init:
	srv_init ();
	break;
      case PCmd_Kill:
	srv_kill ();
	break;
      case PCmd_Version:
	srv_version ();
	break;
      case PCmd_OpenCard:
	srv_openCard ();
	break;
      case PCmd_CloseCard:
	srv_closeCard ();
	break;
      case PCmd_ProbeChips:
	srv_probeChips ();
	break;
      case PCmd_SetChip:
	srv_setChip ();
	break;
      case PCmd_SetSettings:
	srv_setSettings ();
	break;
      case PCmd_GetSettings:
	srv_getSettings ();
	break;
      case PCmd_SetMode:
	srv_setMode ();
	break;
      case PCmd_GetMode:
	srv_getMode ();
	break;
      case PCmd_SetModeSettings:
	srv_setModeSettings ();
	break;
      case PCmd_SetTestImage:
	srv_setTestImage ();
	break;
      case PCmd_GetStatus:
	srv_getStatus ();
	break;
      case PCmd_GetConnection:
	srv_getConnection ();
	break;
      case PCmd_FindBySize:
	srv_findBySize ();
	break;
      case PCmd_FindByOverscan:
	srv_findByOverscan ();
	break;
      default: /* not understood, but have to clean up pipe */
	pipeReadArgs (pipe_in, 0);
	if (cmd & 1) {
	  pipeWriteCmd (pipe_out, cmd);
	  pipeWriteArgs (pipe_out, 0);
	}
	break;
    } /* switch */
  } /* while */
}

/* -------- Main -------- */

int main (int argc, char *argv[])
{
  int c = '?';

  prog = argv[0];

  opterr = 0;
  while ((c = getopt_long (argc, argv, short_options, 
                long_options, NULL)) != EOF) 
  {
    switch(c) 
    {
      case 'h': /* Print usage */
      case '?':
        usage(); 
	break;
      case 'n':
	opt_null = TRUE;
	break;
      case 'N':
	opt_nvdev = TRUE;
	break;
    }
  }

  unlink (PIPE_OUT);
  unlink (PIPE_IN);

  umask (0000); /* Reset umask */

  if (mkfifo (PIPE_OUT, 0622)) { /* rw--w--w- */
    fprintf (stderr, "%s: Cannot create pipe %s\n", argv[0], PIPE_OUT);
    unlink (PIPE_OUT);
    exit (1);
  }
  if (mkfifo (PIPE_IN, 0644)) { /* rw-r--r-- */
    fprintf (stderr, "%s: Cannot create pipe %s\n", argv[0], PIPE_IN);
    unlink (PIPE_OUT);
    unlink (PIPE_IN);
    exit (1);
  }

  DPRINTF ("pipes made\n");

  if (opt_null) {
    srv_root = back_null_init (); 
  } else {
    if (back_root_avail ()) {
      srv_root = back_root_init ();
    } else if (opt_nvdev && back_nvdev_avail (TRUE)) {
      srv_root = back_nvdev_init ();
    } else {
      fprintf (stderr, "Cannot access nvidia cards. Either you are not root,"
	       "or the NVidia devices\nare not accessible.\n");
      exit (1);
    }
  }

  DPRINTF ("server loop.\n");
  srv_loop ();

  fclose (pipe_in);
  fclose (pipe_out);
  unlink (PIPE_OUT);
  unlink (PIPE_IN);

  return 0;
}

/* Options

-N force nvdev
-n null backend

*/

/* FIXME: Signals kill, hup */

