/* statedebug.c - debugging tool for writing support programs

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#define STANDALONE
#include "upsd.h"
#include "shared-tables.h"

const char *progname = NULL;

static	time_t 	lastmtime = 0;

static const char *fname(int cmdnum)
{
	int	i;

	for (i = 0; netvars[i].name != NULL; i++)
		if (netvars[i].type == cmdnum)
			return(netvars[i].name);

	/* handle some special cases */
	switch(cmdnum) {
		case INFO_MEMBERS: return("MEMBERS");
		case INFO_MTIME: return("MTIME");
		case INFO_INSTCMD: return("INSTCMD");
		case INFO_ENUM: return("ENUM");
		case INFO_MSGID: return("MSGID");
		case INFO_SHMID: return("SHMID");
		case INFO_UNUSED: return("UNUSED");
	}

	return("UNKNOWN");
}

/* return a name for an instant command given its identifier (from shared.h) */
const char *instcmdname(int cmdnum)
{
	int	i;

	for (i = 0; instcmds[i].name != NULL; i++)
		if (instcmds[i].cmd == cmdnum)
			return(instcmds[i].name);

	return("*** Unknown INSTCMD");
}

void printinfo(int pos, itype *info)
{
	if (pos == 0)
		printf("pos| type                  | aux   | value\n");

	printf("%3i: %15s (%04x) [%04x] = ", pos,
	       fname(info->type), info->type, 
	       info->auxdata);

	switch(info->type) {
		case INFO_INSTCMD:
			printf("%s\n", instcmdname(info->auxdata));
			break;
		case INFO_ENUM:
			printf("%s: %s\n", 
			      fname(info->auxdata), info->value);
			break;
		default:
			printf("%s\n", info->value);
	}
}

#ifdef HAVE_SHMAT
void displayshm(int shmid)
{
	itype	*info;
	int	i, num;
	struct	shmid_ds shmbuf;

	info = (itype *) shmat(shmid, 0, 0);
	if (info == (itype *)(-1))
		fatal("Can't attach to shared memory");

	if (info[0].type != INFO_MEMBERS) {
		shmdt(info);
		fatalx("First entry in info isn't INFO_MEMBERS, giving up...");
	}

	if (shmctl(shmid, IPC_STAT, &shmbuf) != 0)
		fatal("shmctl STAT");

	if (shmbuf.shm_ctime != lastmtime) {
		lastmtime = shmbuf.shm_ctime;

		num = atoi(info[0].value);

		printf("Shared memory ID: %d\n", shmid);
		for (i = 0; i < num; i++) {
			printinfo(i, &info[i]);
		}
	}

	shmdt((char *)info);
}
#else
void displayshm(int shmid)
{
	printf("This state file uses shared memory and you didn't compile that in!\n");
	exit(1);
}
#endif	/* HAVE_SHMAT */

static void display(const char *fn)
{
	itype	info;
	int	ifd, ret, i, count;

	ifd = open(fn, O_RDONLY);
	if (ifd < 0)
		fatal("open %s", fn);

	ret = read(ifd, &info, sizeof(itype));
	if (ret < 1)
		fatal("read");

	if (info.type == INFO_SHMID) {
		displayshm(atoi(info.value));
	} else if (info.type != INFO_MEMBERS) {
		fatalx("First entry in file isn't INFO_MEMBERS, giving up...");
	} else {
		struct stat st;

		if (fstat(ifd, &st))
	 		fatal("fstat");

		if (st.st_mtime != lastmtime) {
                        lastmtime = st.st_mtime;

			count = atoi(info.value);
			printinfo(0, &info);
			for (i = 1; i < count; i++) {
				if ((ret = read(ifd, &info, sizeof(itype))) != sizeof(itype))
					fatal("could not read correct size for %d members", count);
				printinfo(i, &info);
			}
                }
	}

	close(ifd);
	return;
}

static void help(void)
{
	printf("Network UPS Tools statedebug %s\n\n", UPS_VERSION);
	printf("usage: %s [-h]\n", progname);
	printf("       %s [-c] [-w <num>] <state file>\n\n", progname);
	printf("Debugging tool for UPS driver state files.\n\n");
	printf("  -h		show this help\n");
	printf("  -c		run continuously\n");
	printf("  -w <num>	delay <num> seconds in continuous mode (default 30s)\n");
	exit(1);
}

int main (int argc, char **argv)
{
        int continuous = 0;
        char *file;
	int c;
	unsigned int delay = 30;

	progname = argv[0];

	while ((c=getopt(argc, argv, "cw:h")) != EOF) {
		switch (c) {
			case 'c':
				continuous = 1;
				break;
			case 'w':
				delay = atoi(optarg);
				break;
			case 'h':
				help();
				break;
			default:
				help();
				break;
		}
	}

	argc -= optind;
	argv += optind;

	if (argc != 1)
		help();

	file = argv[0];

	do {
		display(file);

		if (continuous)
			sleep(delay);
	} while (continuous);

	return 0;
}
