/* ups-trust425+625.c - model specific routines for Trust UPS 425/625
      also known as PowerCom King Pro 425/625 Smart-UPS units

   Copyright (C) 1999  Peter Bieringer <pb@bieringer.de>
     
   based on 
    smartups.c - model specific routines for APC Smart-UPS units
    Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   some hints also taken from
    backups.c - model specific routines for APC Back-UPS units
    Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>


   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/* Here you can define extended syslog debugging and "do not go in background" mode */
/* #define DEBUG_UPSTRUST425625 */

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>
#include <limits.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "ups-trust425+625.h"
#include "common.h"
#include "timehead.h"

	int	shmok = 1;
	int dtr_bit = TIOCM_DTR;
	int rts_bit = TIOCM_RTS;

unsigned char binaryinfo[NUMRECEIVEDBYTES];
time_t timelastvalid;

static int signal_terminate = 0;

/* some conversion functions for the binary field */

/* Binary to AC frequency (input and output) */
float ups_bin2freq(int binary) {
    /* constants of cycle time */
    static const float m = 0.00020997;
    static const float d = 0.00020928;
    float result;
    
    result = 1.0 / (binary * m + d);
    return(result);
}

/* Binary to LoadLevel in OnLine state */
float ups_bin2load(int binary) {
    static const float m = 4.311;
    static const float d = 0.1811;
    float result;
    
    result = binary * m + d;
    return(result);
}

/* Binary to LoadLevel in OnBatt state */
float ups_bin2load_ob(int binary) {
    static const float m = 6.13426854;
    static const float d = -0.38076152;
    float result;
    
    result = binary * m + d;
    return(result);
}

/* Binary to BatteryLevel in OnLine state */
float ups_bin2battlvl(int binary) {
    static const float m = 4.5639;
    static const float d = -835.82;
    float result;
    
    result = binary * m + d;
    return(result);
}

/* Binary to BatteryLevel in OnBatt state */
float ups_bin2battlvl_ob(int binary_battlevel, int binary_loadlevel) {
    static const float m1 = 5;
    static const float m2 = 0.3268;
    static const float d2 = -825;
    float result;
    
    result = binary_battlevel * m1 + (ups_bin2load_ob(binary_loadlevel) * m2 + d2);
    return(result);
}

/* Binary to Voltage (input and output) in OnLine state */
float ups_bin2volt(int binary) {
    static const float m = 1.9216;
    static const float d = -0.0977;
    float result;
    
    result = binary * m + d;
    return(result);
}

/* Binary to Output Voltage in OnBatt state */
/* KNOWN PROBLEM: This value depends on 4 binary values:
    Battery Level, LoadLevel, FreqOut and VoltOut
     -> don't know how to calculate in a right way 
     The given formular isn't ok for all cases! */
float ups_bin2voltout_ob(int binary) {
    static const float m = -4.00349;
    static const float d = 296.7242;
    float result;
    
    result = binary * m + d;
    return(result);
}

/* signals */
void signalTERM(int sig)
{
    upslogx(LOG_WARNING, "Got signal TERM, so terminating");
    signal_terminate = 1;
    return;
}

void signalINT(int sig)
{
    upslogx(LOG_WARNING, "Got signal INT, so terminating");
    signal_terminate = 1;
    return;
}

/* init information structure */
void initinfo (void)
{
	create_info(INFOMAX, shmok);

	/* manufacturer ID - hardcoded in this particular module */
	addinfo(INFO_MFR, "Trust/PowerCom", 0, 0);
	addinfo(INFO_MODEL, "425/625 (KingPro)", 0, 0);
	addinfo(INFO_UTILITY, "", 0, 0);
	addinfo(INFO_BATTPCT, "", 0, 0);
	addinfo(INFO_STATUS, "", 0, 0);
	addinfo(INFO_ACFREQ, "", 0, 0);
	addinfo(INFO_LOADPCT, "", 0, 0);

	info[1].type = INFO_MFR;
	info[2].type = INFO_MODEL;
	info[4].type = INFO_UTILITY;
	info[5].type = INFO_BATTPCT;
	info[6].type = INFO_STATUS;
	info[8].type = INFO_ACFREQ;
	info[9].type = INFO_LOADPCT;
}

/* get binary data from the UPSs */
/*  return value:  0x00: ok
		   0x10: primitive check failed	    
*/
int getbinarydata(void)
{
    int status, i;
    unsigned char temp[NUMRECEIVEDBYTES];
    
    upslogx(LOG_DEBUG, "Send info request to ups");
    
    upssendchar (REQ_BINARYDATA);

    upslogx(LOG_DEBUG, "Read info from ups: start");
    
    status = recvbinary (temp, NUMRECEIVEDBYTES);

    if (status == 0) {
        upslogx(LOG_DEBUG, "Read info from ups: successful");
	/* some primitive checks */
	if ((temp[5] != 0) || (temp[7] != 0) || (temp[8] != 0)) {
	    /* looks like a transfer error in serial data communication */
	    upslogx(LOG_WARNING, "Serial data from ups was invalid!");
	    return (0x10);
	}
	/* copy array */
	for (i = 0; i < NUMRECEIVEDBYTES; i++) {
	    binaryinfo[i] = temp[i];
	}
    }
    else
    {
#ifdef DEBUG_UPSTRUST425625
	upslogx(LOG_WARNING, "Read info from ups: not successful");
#endif
    }
    return(status);
}



/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
{
	char	temp[256];
	float	volt_in, acfreq_in, batt_lev, load_lev;
	int 	status;
	
	status = getbinarydata();
	
	if (status == 0) {
	    timelastvalid = time(NULL);
	
	    volt_in = ups_bin2volt(binaryinfo[2]);
	    acfreq_in = ups_bin2freq(binaryinfo[4]);
	    
	    if ((binaryinfo[9] & 0x01) == 0) {
		/* OnLine state */
		load_lev = ups_bin2load(binaryinfo[0]);
	    	batt_lev = ups_bin2battlvl(binaryinfo[1]);
	    }
	    else {
		/* OnBatt state */
		load_lev = ups_bin2load_ob(binaryinfo[0]);
	    	batt_lev = ups_bin2battlvl_ob(binaryinfo[1], binaryinfo[0]);
	    }

	    if (volt_in < ACFREQ_ZEROTH) {
		acfreq_in = 0;
	    }
	
	    setinfo(INFO_UTILITY, "%03.1f", volt_in);

	    setinfo(INFO_BATTPCT, "%03.1f", batt_lev);
	
	    setinfo(INFO_ACFREQ, "%02.2f", acfreq_in);

	    setinfo(INFO_LOADPCT, "%03.1f", load_lev);
	
	    strcpy (temp, "");
	    if (((binaryinfo[9] & 0x01) == 0) && ((binaryinfo[9] & 0x80) == 0))
		strcat (temp, "OL ");		/* on line */
	    if (((binaryinfo[9] & 0x01) == 0) && ((binaryinfo[9] & 0x80) != 0))
		strcat (temp, "OFF ");		/* on line */
	    if ((binaryinfo[9] & 0x01) != 0)
		strcat (temp, "OB ");		/* on battery */
	    if ((binaryinfo[9] & 0x02) != 0)
		strcat (temp, "LB ");		/* low battery */
	    if (((binaryinfo[9] & 0x08) != 0) && (volt_in < ACLINEVOLTAGE))
		strcat (temp, "BOOST ");	/* boost voltage */	
	    if (((binaryinfo[9] & 0x08)  != 0) && (volt_in > ACLINEVOLTAGE))
		strcat (temp, "TRIM ");	/* trim voltage */	
	    if ((binaryinfo[9] & 0x20) != 0)
		strcat (temp, "OVER ");	/* overload */
	    if ((binaryinfo[10] & 0x02) != 0)
		strcat (temp, "RB ");		/* replace batt */

	    /* lose trailing space if present */
	    if (temp[strlen(temp)-1] == ' ')
		temp[strlen(temp)-1] = 0;

	    setinfo(INFO_STATUS, "%s", temp);
		
	    writeinfo();
	}
}

/* formerly in upscommon-addon.c */

/* set DTR and RTS lines on a serial port to supply a passive serial interface */
/*  here: DTR to 0 (-V), RTS to 1 (+V) */
void set_serialDTR0RTS1(void)
{
    /* set DTR to low and RTS to high */
    ioctl(upsfd, TIOCMBIC, &dtr_bit);
    ioctl(upsfd, TIOCMBIS, &rts_bit);
}

/* wait for an binary answer and get buflen bytes */
int recvbinary (unsigned char *buf, int buflen)
{
	unsigned char in;
	int ret, counter = 0, retval = 0;
	struct sigaction sa;
	sigset_t sigmask;

#ifdef DEBUG_UPSCOMMONADDON
        upslogx(LOG_DEBUG, "start reading %d binary data bytes from ups", buflen);
#endif	

	sa.sa_handler = timeout;
	sigemptyset (&sigmask);
	sa.sa_mask = sigmask;
	sa.sa_flags = 0;
	sigaction (SIGALRM, &sa, NULL);

	alarm (3);

        while (counter < buflen) {
#ifdef DEBUG_UPSCOMMONADDON
            upslogx(LOG_DEBUG, "want to read [%2d] of %2d", counter +1 , buflen); 
#endif	
    	    ret = read (upsfd, &in, 1);
#ifdef DEBUG_UPSCOMMONADDON
	    if (ret == -1) {
		perror("read from serial");
	    }
#endif	
	    
	    if (ret > 0) {
		buf[counter] = in;
		counter ++;
#ifdef DEBUG_UPSCOMMONADDON
                upslogx(LOG_DEBUG, "read [%2d]: 0x%02x", counter,  buflen, in); 
#endif	
		nolongertimeout();	   
	    }
	    else {
	        upslogx(LOG_DEBUG, "error reading from serial device!");
		retval = -1;
		break;
	    }
	}
	
	alarm (0);
	signal (SIGALRM, SIG_IGN);
	
	if (retval == 0) {
	    upslogx(LOG_DEBUG, "got all bytes from ups");
	}
	
        return (retval);
}

int main (int argc, char **argv)
{
	char	*portname;
	int	i;

	printf ("Network UPS Tools - UPS driver 0.04 for Trust or PowerCom-KingPro 425/625\n");

	printf ("***\n");
	printf ("*** Warning: this driver is unsupported and lacks the shutdown\n");
	printf ("*** command (-k) to turn off the load.  If you would like to fix\n");
	printf ("*** this driver, please contact the list (ups@lists.exploits.org)\n");
	printf ("***\n");

	if (argc != 2) {
		printf ("usage: %s <portname>       Example: %s /dev/ttyS0\n", 
			argv[0], argv[0]);
		exit (1);
	}

	droproot();

	openlog ("ups-trust425+625", LOG_PID, LOG_FACILITY);

	portname = NULL;
	for (i = strlen(argv[1]); i >= 0; i--)
		if (argv[1][i] == '/') {
			portname = &argv[1][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[1]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/ups-trust425+625-%s", STATEPATH,
	          portname);
		  
	open_serial (argv[1], B1200);
	
	set_serialDTR0RTS1();

	initinfo();

#ifndef DEBUG_UPSTRUST425625
	background();
#endif

	signal(SIGTERM, signalTERM);
	signal(SIGINT, signalINT);
	
        upslogx(LOG_INFO, "Start reading information");

	while (signal_terminate == 0) {
	    updateinfo();
	    sleep (2);
	}
	
        upslogx(LOG_INFO, "Terminated");
	return (0);
}
