/* upssched.c - upsmon's scheduling helper for offset timers

   Copyright (C) 2000  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/* design notes for the curious: 
 *
 * 1. we get called with a upsname and notifytype from upsmon
 * 2. the config file is searched for an AT condition that matches
 * 3. the condition on the first matching line is parsed
 * 
 * starting a timer: the timer is added to the daemon's timer queue
 * cancelling a timer: the timer is removed from that queue
 *
 * if the daemon is not already running and is required (to start a timer)
 * it will be started automatically
 *
 * when the time arrives, the command associated with a timer will be
 * executed by the daemon (via the cmdscript)
 *
 * timers can be cancelled at any time before they trigger
 *
 * the daemon will shut down automatically when no more timers are active
 * 
 */

#include "common.h"

#include <sys/socket.h>
#include <sys/un.h>

#include "upssched.h"
#include "timehead.h"
#include "parseconf.h"

typedef struct {
	char	*name;
	time_t	etime;
	void	*next;
}	ttype;

	ttype	*thead = NULL;
	char	*cmdscript = NULL, *pipefn = NULL;
	int	verbose = 0;		/* use for debugging */

	/* ups name and notify type (string) as received from upsmon */
	const	char	*upsname, *notify_type;

/* --- server functions --- */

void removetimer (ttype *tfind)
{
	ttype	*tmp, *last;

	last = NULL;
	tmp = thead;

	while (tmp) {
		if (tmp == tfind) {	/* found it */
			if (tmp->name)
				free (tmp->name);

			if (last == NULL)	/* deleting first */
				thead = tmp->next;
			else
				last->next = tmp->next;

			free (tmp);
			return;
		}

		last = tmp;
		tmp = tmp->next;
	}

	/* this is not necessarily an error */
	if (verbose)
		upslogx(LOG_INFO, "Tried to delete nonexistent timer!");
	return;
}

void checktimers (void)
{
	ttype	*tmp, *tmpnext;
	time_t	now;
	char	exec[LARGEBUF];

	/* if the queue is empty then we're done */
	if (!thead) {
		if (verbose)
			upslogx(LOG_INFO, "Timer queue empty, exiting");
		unlink (pipefn);

		exit (0);
	}

	/* flip through LL, look for activity */
	tmp = thead;

	time (&now);
	while (tmp) {
		tmpnext = tmp->next;

		if (now >= tmp->etime) {
			if (verbose)
				upslogx(LOG_INFO, "Event: %s ", tmp->name);

			/* build the call to the cmdscript and run it */
			snprintf (exec, sizeof(exec), "%s %s", cmdscript,
			          tmp->name);
			
			system (exec);

			/* delete from queue */
			removetimer (tmp);
		}

		tmp = tmpnext;
	}
}

void start_timer (char *name, char *ofsstr)
{
	time_t	now;
	int	ofs;
	ttype	*tmp, *last;

	/* get the time */
	time (&now);

	/* add an event for <now> + <time> */
	ofs = strtol (ofsstr, (char **) NULL, 10);

	if (ofs < 0) {
		upslogx(LOG_INFO, "bogus offset for timer, ignoring");
		return;
	}

	if (verbose)
		upslogx(LOG_INFO, "New timer: %s (%d seconds)", name, ofs);

	/* now add to the queue */
	tmp = last = thead;

	while (tmp) {
		last = tmp;
		tmp = tmp->next;
	}

	tmp = xmalloc (sizeof(ttype));
	tmp->name = xstrdup (name);
	tmp->etime = now + ofs;
	tmp->next = NULL;

	if (last)
		last->next = tmp;
	else
		thead = tmp;
}

void cancel_timer (char *name, char *cname)
{
	ttype	*tmp;

	for (tmp = thead; tmp != NULL; tmp = tmp->next) {
		if (!strcmp(tmp->name, name)) {		/* match */
			if (verbose)
				upslogx(LOG_INFO, "Cancelling timer: %s", name);
			removetimer (tmp);
			return;
		}
	}

	/* this is not necessarily an error */
	if (cname && cname[0]) {
		char	exec[LARGEBUF];

		if (verbose)
			upslogx(LOG_INFO, "Cancel %s, event: %s", name, cname);

		/* build the call to the cmdscript and run it */
		snprintf (exec, sizeof(exec), "%s %s", cmdscript, cname);
		
		system (exec);
	} else if (verbose)	
		upslogx(LOG_INFO, "Tried to cancel nonexistent timer %s", name);
}

/* handle received messages from the unix domain socket */
void parsecmd (cmdtype cbuf)
{
	/* basic sanity check */
	if (cbuf.magic != CMDMAGIC) {
		upslogx(LOG_INFO, "Received packet with bad magic");
		return;
	}

	switch (cbuf.cmd) {
		case CMD_START_TIMER:
			start_timer (cbuf.data1, cbuf.data2);
			break;

		case CMD_CANCEL_TIMER:
			cancel_timer (cbuf.data1, cbuf.data2);
			break;

		default:
			upslogx(LOG_INFO, "unknown command num %d", cbuf.cmd);
	}
}

int openlistenpipe (void)
{
	int	ss, ret;
	struct sockaddr_un ssaddr;

	ss = socket (AF_UNIX, SOCK_DGRAM, 0);

	if (ss == -1)
		fatal("socket failed");

	ssaddr.sun_family = AF_UNIX;
	strlcpy(ssaddr.sun_path, pipefn, sizeof(ssaddr.sun_path));
	unlink (pipefn);

	/* keep nosy people out of this file */
	umask(077);

	ret = bind (ss, (struct sockaddr *) &ssaddr, sizeof ssaddr);

	if (ret < 0)
		fatal("bind %s failed", pipefn);

	ret = chmod (pipefn, 0600);

	if (ret < 0)
		fatal("chmod on pipe failed");

	return (ss);
}

void start_daemon (cmdtype inbuf)
{
	int	pid, pipefd, ret;
	struct	timeval	tv;
	fd_set	rfds;
	cmdtype	cbuf;

	if ((pid = fork()) < 0)
		fatal("Unable to enter background");

	if (pid != 0)
		return;			/* parent */

	/* child */

	close (0);
	close (1);
	close (2);

	(void) open ("/dev/null", O_RDWR);
	dup (0);
	dup (0);

	openlog ("upssched", LOG_PID, LOG_DAEMON);

	pipefd = openlistenpipe();

	if (verbose)
		upslogx(LOG_INFO, "Timer daemon running");

	/* handle whatever it was that caused us to start */
	parsecmd (inbuf);

	/* now watch for activity */

	for (;;) {
		/* wait at most 1s so we can check our timers regularly */
		tv.tv_sec = 1;
		tv.tv_usec = 0;

		FD_ZERO (&rfds);
		FD_SET (pipefd, &rfds);

		ret = select(pipefd + 1, &rfds, NULL, NULL, &tv);

		if (ret) {
			memset (&cbuf, '\0', sizeof(cbuf));
			ret = read (pipefd, &cbuf, sizeof(cbuf) - 1);

			if (ret < 0)
				upslog(LOG_ERR, "read error");
			else
				parsecmd (cbuf);
		}
 
		checktimers();
	}
}	

/* --- 'client' functions --- */

void sendcmd(int type, const char *arg1, const char *arg2)
{
	int	pipefd, ret;
	cmdtype	cbuf;
	struct sockaddr_un pipeaddr;

	cbuf.magic = CMDMAGIC;
	cbuf.cmd = type;

	strlcpy(cbuf.data1, arg1 ? arg1 : "", sizeof(cbuf.data1));
	strlcpy(cbuf.data2, arg2 ? arg2 : "", sizeof(cbuf.data2));

	pipefd = socket(AF_UNIX, SOCK_DGRAM, 0);

	if (pipefd == -1)
		fatal("socket");

	pipeaddr.sun_family = AF_UNIX;
	strlcpy(pipeaddr.sun_path, pipefn, sizeof(pipeaddr.sun_path));

	ret = connect(pipefd, (const struct sockaddr *)&pipeaddr, 
	              sizeof(pipeaddr));

	/* if the connect fails, start our own daemon with this info */

	if (ret < 0) {
		close(pipefd);

		/* don't start a daemon just to cancel something */
		if (cbuf.cmd == CMD_CANCEL_TIMER) {
			if (arg2 == NULL || arg2[0] == '\0') {
				if (verbose)
					printf("No daemon running, but none needed...\n");
				exit(0);
			}
		}

		if (verbose)
			printf("Starting background process...\n");

		/* child returns from this call */
		start_daemon(cbuf);

		exit(0);
	}

	ret = write(pipefd, &cbuf, sizeof(cbuf));

	if (ret < 1)
		fatal("write");

	close(pipefd);
}

void parse_at(const char *ntype, const char *un, const char *cmd, 
		const char *ca1, const char *ca2)
{
	/* complain both ways in case we don't have a tty */	

	if (!cmdscript) {
		printf("CMDSCRIPT must be set before any ATs in the config file\n");
		fatalx("CMDSCRIPT must be set before any ATs in the config file");
	}

	if (!pipefn) {
		printf("PIPEFN must be set before any ATs in the config file!\n");
		fatalx("PIPEFN must be set before any ATs in the config file!");
	}

	/* check upsname: does this apply to us? */
	if (strcmp(upsname, un) != 0)
		if (strcmp(un, "*") != 0)
			return;		/* not for us, and not the wildcard */

	/* see if the current notify type matches the one from the .conf */
	if (strcasecmp(notify_type, ntype) != 0)
		return;

	/* if command is valid, send it to the daemon (which may start it) */

	if (!strcmp(cmd, "START-TIMER")) {
		sendcmd(CMD_START_TIMER, ca1, ca2);
		return;
	}

	if (!strcmp(cmd, "CANCEL-TIMER")) {
		sendcmd(CMD_CANCEL_TIMER, ca1, ca2);
		return;
	}

	printf("Invalid command: %s\n", cmd);
}

void conf_arg(int numargs, char **arg)
{
	if (numargs < 2)
		return;

	/* CMDSCRIPT <scriptname> */
	if (!strcmp(arg[0], "CMDSCRIPT")) {
		cmdscript = xstrdup(arg[1]);
		return;
	}

	/* PIPEFN <pipename> */
	if (!strcmp(arg[0], "PIPEFN")) {
		pipefn = xstrdup(arg[1]);
		return;
	}

	if (numargs < 5)
		return;

	/* AT <notifytype> <upsname> <command> <cmdarg1> [<cmdarg2>] */
	if (!strcmp(arg[0], "AT")) {

		/* don't use arg[5] unless we have it... */
		if (numargs >= 5) {
			parse_at(arg[1], arg[2], arg[3], arg[4], arg[5]);
		} else {
			parse_at(arg[1], arg[2], arg[3], arg[4], NULL);
		}
	}
}

void conf_err(int linenum, char *errtext)
{
	if (linenum)
		upslogx(LOG_ERR, "upssched.conf: %s", errtext);
	else
		upslogx(LOG_ERR, "upssched.conf line %d: %s", linenum, errtext);
}

void checkconf(void)
{
	int	ret;
	char	cfn[SMALLBUF];

	snprintf(cfn, sizeof(cfn), "%s/upssched.conf", CONFPATH);

	ret = parseconf(cfn, conf_arg, conf_err);

	if (ret != 0)
		fatalx("Can't parse %s", cfn);
}

int main (int argc, char **argv)
{
	verbose = 1;		/* TODO: remove when done testing */

	upsname = getenv("UPSNAME");
	notify_type = getenv("NOTIFYTYPE");

	if ((!upsname) || (!notify_type)) {
		printf("Error: UPSNAME and NOTIFYTYPE must be set.\n");
		printf("This program should only be run from upsmon.\n");
		exit(1);
	}

	/* see if this matches anything in the config file */
	checkconf();

	return 0;
}
