/*
 * Copyright (c) 1996 University College London
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Computer Science
 *      Department at University College London
 * 4. Neither the name of the University nor of the Department may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#include <math.h>
#include <stdlib.h>
#include "prototypes.h"
int confsize;

/*sliding key used for retransmissions */
sliding_key skey;

/*our local key for matching against the sliding key of others */
u_int16 lkey;

/*default interval between sending tokens in milliseconds */
/*increase this with each successive sliding key cycle with no mods */
#define MAX_PASS 4
int inter_token_interval[MAX_PASS] =
{200, 400, 800, 1600};

/* rtx time store - so that we can time out re-transmissions */
extern struct timeval rtx_time;

/*the file descriptor to send from */
extern int tx_fd;

extern queue sendq;
extern page *p;

void init_key(sliding_key * skey, u_int16 * lkey)
{

    struct timeval t;
    struct timezone tzp;
    gettimeofday(&t, &tzp);
    lbl_srandom(t.tv_usec);
    confsize = INITIAL_CONF_SIZE;
    skey->master = FALSE;
    skey->mods = FALSE;
    skey->counter = RTX_TOKEN_STOP;
    skey->pass = 0;
    *lkey = get_key();
    debug("init key\n");
}

u_int16 get_key()
{
    return ((u_int16) (lbl_random() & 0xffff));
}

void reset_sliding_key(sliding_key * skey, int conf_size)
{

    skey->key = get_key();	/* get new random key */
    if (conf_size > 0)		/* if there are people in the conference */
	skey->mask = (u_int8) ((log10((double) conf_size) / log10(2.0)) + 1.5);
    else
	skey->mask = INITIAL_MASK;	/* conference is empty - not even this host in it? */
    get_last_mod_time(p, &(skey->timestamp));	/* timestamp key with this host's latest - */
    /*   data modification time */
    skey->master = TRUE;	/* set master to true because we don't know of any - */
    /*   other sliding key sequences */
    skey->counter = 0;		/* reset mask counter (no.of times mask sent) */
    send_recent_list(tx_fd);	/* send summary packet */
    debug("Reset sliding key\n");
}

void init_rtx_token(sliding_key * skey)
{
    confsize = get_conf_size();	/* number of people in participants list - an estimate only of */
    /*   the conference's size */
    if (skey->counter == RTX_TOKEN_STOP) {	/* mask counter is set to this value when key is first created */
	debug("init rtx token - counter== rtxtokenstop\n");
	reset_sliding_key(skey, confsize);
	send_rtx_token();
    } else {			/* if this function gets called everytime a modification is made, */
	/*   then this else case would catch that and reset mods and pass */
	debug("init rtx token - counter!= rtxtokenstop\n");
	skey->mods = TRUE;
	skey->pass = 0;
    }
}

void send_rtx_token( /*uses global skey(sliding key) and tx_fd (transmission socket) */ )
{

    ntheader header;
    netrtxadvert token;
    debug("send rtx token\n");
    if (packets_queued(&sendq) > 0) {
	/*if we've already got stuff in the queue, don't send this time */
	/*just reschedule us for the next time */
	skey.event_token =
	    Tcl_CreateTimerHandler(inter_token_interval[skey.pass],
				   (Tcl_TimerProc *) send_rtx_token,
				   (ClientData) 0);
	return;
    }
    get_cur_time(&rtx_time);
    if (skey.counter++ >= TOKENS_PER_MASK_POSN)
	slide_rtx_key(&skey);
    if (skey.counter == RTX_TOKEN_STOP)
	return;
    debug("Transmitting : Key %d, Mask %d\n", skey.key, skey.mask);
    init_header(&header, RTX_ADVERT, &(skey.timestamp));
    token.key = htons(skey.key);
    token.mask = skey.mask;
    xsocksend(tx_fd, &header, (u_char *) & token, sizeof(netrtxadvert));
    skey.event_token =
	Tcl_CreateTimerHandler(inter_token_interval[skey.pass],
			       (Tcl_TimerProc *) send_rtx_token,
			       (ClientData) 0);
    if ((skey.pass == (MAX_PASS - 1)) && (skey.counter == TOKENS_PER_MASK_POSN) && skey.mask == 0) {
	/*send page checksum also because no-one is using our sliding key stuff */
	send_page_info();
    }
}

void slide_rtx_key(sliding_key * skey)
{
    debug("sliding rtx key\n");
    if (skey->mask == 0) {	/* if we have finished with this mask? */
	if ((skey->master == TRUE) || (skey->mods == TRUE)) {
	    /*Once we start a key cycle, we're the master again until
	       we see another sliding key */
	    skey->master = TRUE;
	    /*Reset the mods flag, until more modifications occur */
	    if (skey->mods == TRUE) {
		skey->mods = FALSE;
		skey->pass = 0;
	    } else {
		if (skey->pass < MAX_PASS - 1)
		    skey->pass++;	/* starting a new cycle */
	    }
	    reset_sliding_key(skey, confsize);	/* resets sliding key and also sends summary packet */
	} else {
	    skey->counter = RTX_TOKEN_STOP;
	}
    } else {			/* we are still using this mask */
	skey->mask--;
	skey->counter = 0;
	skey->holds = 0;
    }
}

void seen_another_sliding_key(struct timeval *t)
{

    struct timeval last_mod;

    debug("seen another sliding key :");

    if ((skey.counter == RTX_TOKEN_STOP) || (skey.mods == TRUE)) {
	/* if we have new modifications to transmit, or we are about to start? */

	if (skey.counter == RTX_TOKEN_STOP) {
	    debug("counter = RTX_TOKEN_STOP - returning without changes\n");
	} else {
	    if (skey.mods == TRUE) {
		debug("skey.mods == true - returning without changes\n");
	    }
	}
	return;
    }
    get_last_mod_time(p, &last_mod);
    if (newer(t, &last_mod)) {
	debug("received key is later than ours - setting key master false\n");
	skey.master = FALSE;
    } else {
	debug("received key is not later than ours\n");
	debug("our time: \n");
	debug("%d,%d \n", t->tv_sec, t->tv_usec);
	debug("received time: \n");
	debug("%d,%d \n", last_mod.tv_sec, last_mod.tv_usec);
    }
}

void hold_key()
{
    skey.counter = 0;
    skey.pass = 0;
    skey.holds++;
    if (skey.holds > MAX_NO_OF_HOLDS)
	reset_sliding_key(&skey, confsize);
}
