// nss-multidomain: NSS module for multidomain authentification
// Copyright (C) 2001, 2002 LibertySurf Telecom, Antoniu-George Savu
// 
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
// 02111-1307, USA.

// Berkeley DB file creation utility.

#include <config.h>

#if defined(HAVE_BERKELEY_DB)

#include <db.h>

#if defined(DB_VERSION_MAJOR)  
#if (DB_VERSION_MAJOR == 3) || (DB_VERSION_MAJOR == 4)
#define DB3   
#endif
#if (DB_VERSION_MAJOR == 2)
#define DB2
#endif
#else
#define DB1
#include <fcntl.h>
#endif /* defined(DB_VERSION_MAJOR) */

#include <errno.h>
#include <stdio.h>
#include <string.h>



/* Global */
DB *f;

int create_db(int db_type, char *file)
{ 
	/* Open DB */
#if defined(DB3) /* DB version 3 */
	if ( (db_create(&f, NULL, 0) != 0) || (f->open(f, file, NULL, db_type, DB_CREATE|DB_TRUNCATE , 0664) != 0) ) {
#elif defined(DB2) /* DB version 2 */
	if (db_open(file, db_type, DB_CREATE|DB_TRUNCATE , 0664, NULL, NULL, &f) != 0) {
#else /* DB version 1 */
	if (!(f = dbopen(file, O_CREAT|O_TRUNC, 0664, db_type, NULL))) {
#endif   
		printf("Error opening %s output file. :: %s\n", file, strerror(errno));
		return 0;
	}
	return 1;
}

int write_db(char *key, char *value) {
	
	DBT k,v;	
	memset(&k, 0, sizeof(DBT));
	memset(&v, 0, sizeof(DBT));
	
	k.data=key;
	v.data=value;
	k.size=strlen(k.data);
	v.size=strlen(v.data)+1;

#if defined(DEBUG)
	printf("Writing key: %s value: %s\n", key, value);
#endif /* defined(DEBUG) */
#if defined(DB3) || defined(DB2)
	if (((f->put) (f, NULL, &k, &v, DB_NOOVERWRITE))) {
#else
	if (((f->put) (f, &k, &v, 0))) {
#endif
		return 0;
	} 
	return 1;
}

int close_db(void) {

#if defined(DB2) || defined(DB3)   
	(f->close) (f, 0);
#else
	(f->close) (f);
#endif
	return 0;
} 

#endif /* HAVE_BERKELEY_DB */


int main(int argc, char **argv) {
#if !defined(HAVE_BERKELEY_DB)
	printf("No Berkeley DB support!\n");
	return 0;
#else
    FILE *fin;
    char line[256], *p, *q;
    unsigned int l_read = 0, l_wrote = 0;
	
    if (argc<4) {
		printf("Usage: \n %s [btree|hash] outfile infile\n", argv[0]);
		return -1;
    }

    if (!(fin=fopen(argv[3], "r"))) {
    	printf("Error opening input file %s :: %s\n", argv[3], strerror(errno));
    	return -1;
    }

#if defined(DB_VERSION_MAJOR) && defined(DB_VERSION_MINOR) && defined(DB_VERSION_PATCH) && defined(DB_VERSION_STRING)
    printf("Using libdb version %d.%d.%d (%s)\n", DB_VERSION_MAJOR, DB_VERSION_MINOR, DB_VERSION_PATCH, DB_VERSION_STRING);
#endif
#if defined(DB_BTREEVERSION) && defined(DB_HASHVERSION)
    printf("BTree version %d Hash version %d\n", DB_BTREEVERSION, DB_HASHVERSION);
#else
    printf("Unable to determine libdb version. Broken db.h header?\n");
#endif    	    
    if (!create_db( (!strcasecmp(argv[1], "hash") ? DB_HASH:DB_BTREE), argv[2])) {
    	printf("Error opening DB %s for creation :: %s\n", argv[2], strerror(errno));
    	fclose(fin);
    	return -1;
    }

    
    while (!feof(fin)) {
    	fgets(line, 255, fin);
    	l_read ++;
	/* Get rid of \n */
	line[strlen(line)-1] = '\0';
	p = line;
    	q = strchr(line, ' ');
    	if ( (q) && (*q != '\0') && *(q+1) != '\0') {
    		*q = '\0';
    		q++;
    	}
		    	
    	if ( (p) && (q) ) {
	    	if (!write_db(p, q)) {
    			printf("Error writing record into DB file :: %s\n", strerror(errno));
	    		close_db();
    			fclose(fin);
	    		return -1;
		}
		l_wrote++;
	}
	else if (!feof(fin))
		printf("Malformed line #%d in file %s. Skipping it.\n", l_read, argv[3]);
	/* reset p && q */
	p = NULL;
	q = NULL;
    }    
	
    close_db();
    fclose(fin);
    printf ("Succesfully created %s file %s (%d records) from %d lines.\n", ( !strcasecmp(argv[1], "hash") ? "HASH" : "BTREE" ), argv[2], l_wrote, l_read-1);
    return 0; 
#endif /* !defined(HAVE_BERKELEY_DB) */    
}
