/**
 * @file util.h
 *
 * General utility macros, defines, and types
 */
#ifndef __NJ_LIB_UTIL_H__
#define __NJ_LIB_UTIL_H__
#include <config.h>
#include <sys/types.h>

/** Generic data item to make it less blasphemous to use pointers to hold
 * interger values and such. */
typedef unsigned long nj_generic_t;
/** A general purpose address variable to which we can perform standard 
 * (ie non-pointer) arithmetic, bitwise ops, and comparason */
typedef unsigned long nj_addr_t;

/**@{ @name Alignment macros */
/** Align to a certain number of bytes, rounding up */
/** FIXME If shit breaks on Solaris, look here first */
#define NJ_ALIGN_UP(val, alignto)				\
		((alignto > 1) ? (((u_long)val) + (alignto) - 1) & (~((alignto) - 1)) \
						: ((u_long)val))

#define NJ_ALIGN_DOWN(val, alignto)				\
		((alignto > 1) ? (((u_long)val) & (~((alignto) - 1))) \
						: ((u_long)val))

	
/** Aligned objects should lack the lower bits */
#define NJ_IS_ALIGNED(val, alignto)		\
		!((u_long)(val) & ((alignto) - 1))

/** Overhead due to aligning */
#define NJ_ALIGN_OVERHEAD(val, alignto)	\
		(NJ_ALIGN_UP(val, alignto) - (u_long)(val))

/**
 * Returns true if bitz is a power of 2.
 * The general concept here is that you can't have bits in both these
 * alternating patterns. Then as you space the patterns, you progressivly
 * rule out multiple bits that fit the previous pattern.  
 * 
 * @param align The integer
 * @returns true if bitz=2^n for some n < 32
 */
#define NJ_IS_POWER_OF_TWO(bits) 								\
		(	(!((bits) & 0x55555555 /* 01010101 */) ^				\
			 !((bits) & 0xAAAAAAAA /* 10101010 */)) &				\
			(!((bits) & 0xCCCCCCCC /* 11001100 */) ^				\
			 !((bits) & 0x33333333 /* 00110011 */)) &				\
			(!((bits) & 0xF0F0F0F0 /* 11110000 */) ^				\
			 !((bits) & 0x0F0F0F0F /* 00001111 */)) &				\
			(!((bits) & 0xFF00FF00 /* 11.00.11 */) ^				\
			 !((bits) & 0x00FF00FF /* yada     */)) &				\
			(!((bits) & 0xFFFFFFFF /* yup      */) ^				\
			 !((bits) & 0x00000000 /* seeya!   */))	)
/*@}*/

/**@{ @name  Constants */
/** Number of bits in a megabyte */
#define NJ_SHIFT_MEGABYTE		20
#ifndef TRUE
# define TRUE					1
# define FALSE					0
#endif

#define LINUX_PROC_OVERCOMMIT		"/proc/sys/vm/overcommit_memory"
#define LINUX_PROC_MAX_MAP_COUNT	"/proc/sys/vm/max_map_count"

#define NJ_ADDRESS_SIZE_32		((u_long)0xffffffff)
/** The number of allocations in our address space if we use PROT=segv. 
 * (Two pages per alloc) */
#define NJ_ALLOCS_IN_ADDRESS_SPACE (NJ_ADDRESS_SIZE_32 >> (NJ_PAGE_SHIFT+1))


/** The value to fencepost our buffers with to check for consistancy */
#define NJ_FENCEPOST			(0xdeadbeef)
/*@}*/

#define NJ_MIN(a, b)			((a) < (b) ? (a) : (b))
#define NJ_MAX(a, b)			((a) > (b) ? (a) : (b))

#define NJ_PTR_ADD_ADDR(ptr, i)			((void *)((u_long)ptr) + (i))
#define NJ_UBER_CAST(i, type)			(*(type *)&i)

/**@{ @name Debugging Macros */
#ifdef EDEBUG
# define TRACEF 	write(2, __FUNCTION__"\n", strlen(__FUNCTION__)+1)
# define TR			do \
					{ \
						write(2, __FILE__":"__FUNCTION__"\n", strlen(__FILE__":"__FUNCTION__"\n")); \
						func = __FUNCTION__; line = __LINE__; file = __FILE__; \
					} while(0)
# define PRINTF(x...)	__nj_eprintf( __FUNCTION__ ": " ##x)
# define EPRINTF(x...)  __nj_eprintf( __FUNCTION__ ": " ##x)
# define __USE_UNIX98
#elif DEBUG
# define __USE_UNIX98
# define TRACEF 	write(2, __FUNCTION__"\n", strlen(__FUNCTION__)+1)
# define TR		do { file = __FILE__; func = __FUNCTION__; line = __LINE__; } while(0)
# define PRINTF(x...)	__nj_eprintf( __FUNCTION__ ": " ##x)
# define EPRINTF(x...)
#else
# define TRACEF
# define TR
# define PRINTF(x...)
# define EPRINTF(x...) 
#endif
/*@}*/

/**@{ @name Size calculation macros */	
/** Round len up to nearest page boundry */
#define NJ_ROUND_TO_PAGE(len)			((((len)-1) & NJ_PAGE_MASK) + NJ_PAGE_SIZE)

/** Round len with accounting info to a blocksize usable in NJAMD */
#define NJ_ROUND_TO_BLOCK(len)			((((len)-1) & NJ_PAGE_MASK) + \
															(2<<NJ_PAGE_SHIFT))

/** Round len with accounting info to a page boundary */
#define NJ_ROUND_TO_FAULT(len)			((((len)+NJ_ACCOUNTING_OVERHEAD-1) \
										& NJ_PAGE_MASK) + NJ_PAGE_SIZE)
/*@}*/

/**@{ @name Flag macros */
#define NJ_SET(var, fl)			(var) |= (fl)
#define NJ_ISSET(var, fl)		((var) & (fl))
#define NJ_CLEAR(var, fl)		(var) &= ~(fl)
/*@}*/


int __nj_log2(int);
char *__nj_itoa(int);
char *__nj_ptoa(void *);


	
#endif /*util.h */
// vim:ts=4
