# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003-2005 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import argparse
import urlparse
import urllib
import os
import socket
import sys
import logging
import shutil
import random
import traceback
import cStringIO
import imp
import locale
import multiprocessing
from collections import OrderedDict

if os.name == 'nt': # locale setting for windows
    lang = os.getenv('LANG')
    if lang is None:
        os.environ['LANG'] = locale._build_localename(locale.getdefaultlocale())
import gettext
gettext.install('ninix')

import gtk
import glib
import pango

import ninix.pix
import ninix.home
import ninix.prefs
import ninix.sakura
import ninix.sstp
import ninix.communicate
import ninix.ngm
import ninix.lock
import ninix.install
import ninix.plugin
import ninix.nekodorif
import ninix.kinoko
import ninix.menu
from ninix.metamagic import Holon
from ninix.metamagic import Meme

parser = argparse.ArgumentParser('ninix')
parser.add_argument('--sstp-port', type=int, dest='sstp_port',
                    help='additional port for listening SSTP requests')
parser.add_argument('--debug', action='store_true', help='debug')
parser.add_argument('--logfile', type=str, help='logfile name')

# XXX: check stderr - logger's default destination and redirect it if needed
# (See http://bugs.python.org/issue706263 for more details.)
try:
    tmp = os.dup(sys.__stderr__.fileno())
except:
    sys.stderr = file(os.devnull, 'w')
else:
    os.close(tmp)
logger = logging.getLogger()
logger.setLevel(logging.INFO) # XXX

def handleException(exception_type, value, tb):
    logger.error('Uncaught exception',
                 exc_info=(exception_type, value, tb))
    response_id = 1
    dialog = gtk.MessageDialog(None, 0, gtk.MESSAGE_ERROR, gtk.BUTTONS_NONE,
                               _('A ninix-aya error has been detected.'))
    dialog.set_title(_('Bug Detected'))
    dialog.set_position(gtk.WIN_POS_CENTER)
    dialog.set_gravity(gtk.gdk.GRAVITY_CENTER)
    button = dialog.add_button(_('Show Details'), response_id)
    dialog.add_button(gtk.STOCK_CLOSE, gtk.RESPONSE_CLOSE)
    textview = gtk.TextView()
    textview.set_editable(False)
    left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
    width = scrn_w // 2
    height = scrn_h // 4
    textview.set_size_request(width, height)
    textview.show()
    sw = gtk.ScrolledWindow()
    sw.show()
    sw.set_policy(gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    sw.add(textview)
    frame = gtk.Frame()
    frame.set_shadow_type(gtk.SHADOW_IN)
    frame.add(sw)
    frame.set_border_width(7)
    dialog.vbox.add(frame)
    stringio = cStringIO.StringIO()
    traceback.print_exception(exception_type, value, tb, None, stringio)
    textbuffer = textview.get_buffer()
    textbuffer.set_text(stringio.getvalue())
    while 1:
        if dialog.run() == response_id:
            frame.show()
            button.set_sensitive(0)
        else: # close button
            break
    dialog.destroy()
    raise SystemExit

sys.excepthook = handleException


def main():
    if gtk.pygtk_version < (2,10,0):
        logging.critical('PyGtk 2.10.0 or later required')
        raise SystemExit
    # parse command line arguments
    args = parser.parse_args()
    if args.logfile is not None:
        logger.addHandler(logging.FileHandler(args.logfile))
    # TCP 7743：伺か（未使用）(IANA Registered Port for SSTP)
    # UDP 7743：伺か（未使用）(IANA Registered Port for SSTP)
    # TCP 9801：伺か          (IANA Registered Port for SSTP)
    # UDP 9801：伺か（未使用）(IANA Registered Port for SSTP)
    # TCP 9821：SSP
    # TCP 11000：伺か（廃止） (IANA Registered Port for IRISA)
    sstp_port = [9801]
    # parse command line arguments
    if args.sstp_port is not None:
        if args.sstp_port < 1024 or args.sstp_port > 65535:
            logging.warning('Invalid --sstp-port number (ignored)')
        else:
            sstp_port.append(args.sstp_port)
    if args.debug:
        logger.setLevel(logging.DEBUG)
    home_dir = ninix.home.get_ninix_home()
    if not os.path.exists(home_dir):
        try:
            os.makedirs(home_dir)
        except:
            raise SystemExit, 'Cannot create Home directory (abort)\n'
    lockfile_path = os.path.join(ninix.home.get_ninix_home(), '.lock')
    if os.path.exists(lockfile_path):
        with open(lockfile_path, 'r') as f:
            abend = f.readline()
        if not abend:
            abend = None
    else:
        abend = None
    # aquire Inter Process Mutex (not Global Mutex)
    with open(lockfile_path, 'w') as f:
        try:
            ninix.lock.lockfile(f)
        except:
            raise SystemExit, 'ninix-aya is already running'
        # start
        logging.info('loading...')
        app = Application(f, sstp_port)
        logging.info('done.')
        app.run(abend)
        f.truncate(0)
        try:
            ninix.lock.unlockfile(f)
        except:
            pass


class SSTPControler(object):

    def __init__(self, sstp_port):
        self.request_parent = lambda *a: None # dummy
        self.sstp_port = sstp_port
        self.sstp_servers = []
        self.__sstp_queue = []
        self.__sstp_flag = 0
        self.__current_sender = None

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            }
        handler = handlers.get(event, getattr(self, event, None))
        if handler is None:
            result = self.request_parent(
                event_type, event, *arglist, **argdict)
        else:
            result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def enqueue_request(self, event, script_odict, sender, handle,
                        address, show_sstp_marker, use_translator,
                        entry_db, request_handler):
        self.__sstp_queue.append(
            (event, script_odict, sender, handle, address, show_sstp_marker,
             use_translator, entry_db, request_handler))

    def check_request_queue(self, sender):
        count = 0
        for request in self.__sstp_queue:
            if request[2].split(' / ')[0] == sender.split(' / ')[0]:
                count += 1
        if self.__sstp_flag and \
           self.__current_sender.split(' / ')[0] == sender.split(' / ')[0]:
            count += 1
        return str(count), str(len(self.__sstp_queue))

    def set_sstp_flag(self, sender):
        self.__sstp_flag = 1
        self.__current_sender = sender

    def reset_sstp_flag(self):
        self.__sstp_flag = 0
        self.__current_sender = None        

    def handle_sstp_queue(self):
        if self.__sstp_flag or not self.__sstp_queue:
            return
        event, script_odict, sender, handle, address, \
            show_sstp_marker, use_translator, \
            entry_db, request_handler = self.__sstp_queue.pop(0)
        working = bool(event is not None)
        for if_ghost in script_odict.keys():
            if if_ghost and self.request_parent('GET', 'if_ghost', if_ghost, working):
                self.request_parent('NOTIFY', 'select_current_sakura', if_ghost)
                default_script = script_odict[if_ghost]
                break
        else:
            if not self.request_parent('GET', 'get_preference', 'allowembryo'):
                if event is None:
                    if request_handler:
                        request_handler.send_response(420) # Refuse
                    return
                else:
                    default_script = None
            else:
                if '' in script_odict: # XXX
                    default_script = script_odict['']
                else:
                    default_script = script_odict.values()[0]
        if event is not None:
            script = self.request_parent('GET', 'get_event_response', event)
        else:
            script = None
        if not script:
            script = default_script
        if script is None:
            if request_handler:
                request_handler.send_response(204) # No Content
            return
        self.set_sstp_flag(sender)
        self.request_parent(
            'NOTIFY', 'enqueue_script',
            event, script, sender, handle, address,
            show_sstp_marker, use_translator, entry_db,
            request_handler, temp_mode=True)

    def receive_sstp_request(self):
        try:
            for sstp_server in self.sstp_servers:
                sstp_server.handle_request()
        except socket.error as e:
            code, message = e.args
            logging.error('socket.error: {0} ({1:d})'.format(message, code))
        except ValueError: # may happen when ninix is terminated
            return

    def get_sstp_port(self):
        if not self.sstp_servers:
            return None
        return self.sstp_servers[0].server_address[1]

    def quit(self):
        for server in self.sstp_servers:
            server.close()

    def start_servers(self):
        for port in self.sstp_port:
            try:
                server = ninix.sstp.SSTPServer(('', port))
            except socket.error as e:
                code, message = e.args
                logging.warning(
                    'Port {0:d}: {1} (ignored)'.format(port, message))
                continue
            server.set_responsible(self.handle_request)
            self.sstp_servers.append(server)
            logging.info('Serving SSTP on port {0:d}'.format(port))


class PluginDialog:

    def __init__(self, plugin_dir, queue, message):
        self.plugin_dir = plugin_dir
        self.queue = queue
        dialog = gtk.Window()
        dialog.set_title(plugin_dir)
        ##dialog.set_decorated(False)
        ##dialog.set_resizable(False)
        dialog.connect('delete_event', self.delete)
        dialog.connect('key_press_event', self.key_press)
        dialog.connect('button_press_event', self.button_press)
        dialog.set_events(gtk.gdk.BUTTON_PRESS_MASK)
        ##dialog.set_modal(True)
        dialog.set_position(gtk.WIN_POS_CENTER)
        dialog.realize()
        entry = gtk.Entry()
        entry.connect('activate', self.activate)
        entry.set_size_request(320, -1)
        entry.show()
        box = gtk.VBox(spacing=10)
        box.set_border_width(10)
        label = gtk.Label(message)
        box.pack_start(label, False)
        label.show()
        box.pack_start(entry)
        dialog.add(box)
        box.show()
        self.dialog = dialog

    def open(self):
        self.dialog.show()

    def delete(self, widget, event):
        self.dialog.hide()
        self.cancel()
        return True

    def key_press(self, widget, event):
        if event.keyval == gtk.keysyms.Escape:
            self.dialog.hide()
            self.cancel()
            return True
        return False

    def button_press(self, widget, event):
        if event.button in [1, 2]:
            self.dialog.begin_move_drag(
                event.button, int(event.x_root), int(event.y_root),
                gtk.get_current_event_time())
        return True

    def activate(self, widget):
        self.dialog.hide()
        self.enter(widget.get_text())
        return True

    def enter(self, text): 
        self.queue.put(text)

    def cancel(self):
        self.queue.put('')


class PluginControler:

    def __init__(self):
        self.jobs = {}
        self.queue = {}
        self.data = {}
        self.dialog = {}
        self.request_parent = lambda *a: None # dummy
        
    def set_responsible(self, request_method):
        self.request_parent = request_method

    def save_data(self, plugin_dir):
        if plugin_dir not in self.data:
            return
        home_dir = ninix.home.get_ninix_home()
        target_dir = os.path.join(home_dir, 'plugin', plugin_dir)
        path = os.path.join(target_dir, 'SAVEDATA')
        with open(path, 'w') as f:
            f.write('#plugin: {0:1.1f}\n'.format(ninix.home.PLUGIN_STANDARD[1]))
            for name, value in self.data[plugin_dir].items():
                if value is None:
                    continue
                f.write('{0}:{1}\n'.format(name.encode('utf-8'),
                                           value.encode('utf-8')))

    def load_data(self, plugin_dir):
        home_dir = ninix.home.get_ninix_home()
        target_dir = os.path.join(home_dir, 'plugin', plugin_dir)
        path = os.path.join(target_dir, 'SAVEDATA')
        if not os.path.exists(path):
            return {}
        data = {}
        try:
            with open(path, 'r') as f:
                line = f.readline()
                line = line.rstrip('\r\n')
                if not line.startswith('#plugin:'):
                    return {}
                try:
                    standard = float(line[8:])
                except:
                    return {}
                if standard < ninix.home.PLUGIN_STANDARD[0] or \
                        standard > ninix.home.PLUGIN_STANDARD[1]:
                    return {}
                for line in f:
                    line = line.rstrip('\r\n')
                    line = unicode(line, 'utf-8')
                    key, value = line.split(':', 1)
                    data[key] = value
        except IOError as e:
            code, message = e.args
            logging.error('cannot read {0}'.format(path))
        return data

    def terminate_plugin(self):
        for plugin_dir in self.data.keys():
            self.save_data(plugin_dir)
        return ## FIXME

    def open_dialog(self, plugin_dir, message):
        self.dialog[plugin_dir] = PluginDialog(plugin_dir,
                                               self.queue[plugin_dir],
                                               message)
        self.dialog[plugin_dir].open()

    def check_queue(self):
        for plugin_dir in self.jobs.keys():
            if not self.queue[plugin_dir].empty():
                data = self.queue[plugin_dir].get()
                self.queue[plugin_dir].task_done()
                if isinstance(data, basestring):
                    if data.startswith('DIALOG:'):
                        message = data[7:]
                        self.open_dialog(plugin_dir, message)
                        continue
                elif not isinstance(data, dict):
                    continue
                for name, value in data.items():
                    if not isinstance(name, basestring) or ':' in name or \
                            not (isinstance(value, basestring) or value is None):
                        break
                else:
                    assert plugin_dir in self.data
                    self.data[plugin_dir].update(data)

    def exec_plugin(self, plugin_dir, argv, caller):
        if plugin_dir in self.jobs and self.jobs[plugin_dir].is_alive():
            logging.warning('plugin {0} is already running'.format(plugin_dir))
            return
        module_name, ext = os.path.splitext(argv[0])
        home_dir = ninix.home.get_ninix_home()
        target_dir = os.path.join(home_dir, 'plugin', plugin_dir)
        module = self.__import_module(module_name, target_dir)
        if module is None:
            return
        port = self.request_parent('GET', 'get_sstp_port')
        queue = multiprocessing.JoinableQueue()
        if plugin_dir not in self.data:
            self.data[plugin_dir] = self.load_data(plugin_dir)
        data = self.data[plugin_dir]
        p = module.Plugin(port, target_dir, argv[1:], home_dir, caller,
                          queue, data)
        if not isinstance(p, ninix.plugin.BasePlugin):
            return
        self.jobs[plugin_dir] = p
        self.queue[plugin_dir] = queue
        p.daemon = True
        p.start()
        if os.name == 'nt' and target_dir in sys.path:
            # XXX: an opposite of the BasePlugin.__init__ hack
            sys.path.remove(target_dir)

    def start_plugins(self, plugins):
        for plugin_name, plugin_dir, startup, menu_items in plugins:
            if startup is not None:
                self.exec_plugin(plugin_dir, startup,
                                 {'name': '', 'directory': ''})

    def __import_module(self, name, directory):
        fp = None
        try:
            return reload(sys.modules[name])
        except:
            pass
        try:
            fp, pathname, description = imp.find_module(name, [directory])
        except:
            return None
        try:
            return imp.load_module(name, fp, pathname, description)
        finally:
            if fp:
                fp.close()
        return None


class BalloonMeme(Meme):

    def __init__(self, key):
        Meme.__init__(self, key)
        self.request_parent = lambda *a: None # dummy

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def create_menuitem(self, data):
        desc, balloon = data
        subdir = balloon['balloon_dir'][0]
        name = desc.get('name', subdir)
        home_dir = ninix.home.get_ninix_home()
        thumbnail_path = os.path.join(home_dir, 'balloon',
                                      subdir, 'thumbnail.png')
        if not os.path.exists(thumbnail_path):
            thumbnail_path = None
        return self.request_parent(
            'GET', 'create_balloon_menuitem', name, self.key, thumbnail_path)

    def delete_by_myself(self):
        self.request_parent('NOTIFY', 'delete_balloon', self.key)


class Ghost(Holon):

    def __init__(self, key):
        Holon.__init__(self, key)
        self.request_parent = lambda *a: None # dummy
        
    def set_responsible(self, request_method):
        self.request_parent = request_method

    def create_menuitem(self, data):
        return self.request_parent('GET', 'create_menuitem', self.key, data)

    def delete_by_myself(self):
        self.request_parent('NOTIFY', 'delete_ghost', self.key)

    def create_instance(self, data):
        return self.request_parent('GET', 'create_ghost', data)


class Application(object):

    def __init__(self, lockfile, sstp_port=[9801, 11000]):
        self.lockfile = lockfile
        self.abend = None
        self.loaded = False
        self.confirmed = False
        self.console = Console(self)
        # create preference dialog
        self.prefs = ninix.prefs.PreferenceDialog()
        self.prefs.set_responsible(self.handle_request)
        self.sstp_controler = SSTPControler(sstp_port)
        self.sstp_controler.set_responsible(self.handle_request)
        # create usage dialog
        self.usage_dialog = UsageDialog()
        self.communicate = ninix.communicate.Communicate()
        # create plugin manager
        self.plugin_controler = PluginControler()
        self.plugin_controler.set_responsible(self.handle_request)
        # create ghost manager
        self.__ngm = ninix.ngm.NGM()
        self.__ngm.set_responsible(self.handle_request)
        self.current_sakura = None
        # create installer
        self.installer = ninix.install.Installer()
        # create popup menu
        self.__menu = ninix.menu.Menu()
        self.__menu.set_responsible(self.handle_request)
        self.__menu_owner = None
        self.ghosts = OrderedDict()
        odict_baseinfo = ninix.home.search_ghosts()
        for key, value in odict_baseinfo.items():
            holon = Ghost(key)
            holon.set_responsible(self.handle_request)
            self.ghosts[key] = holon 
            holon.baseinfo = value
        self.balloons = OrderedDict()
        odict_baseinfo = ninix.home.search_balloons()
        for key, value in odict_baseinfo.items():
            meme = BalloonMeme(key)
            meme.set_responsible(self.handle_request)
            self.balloons[key] = meme
            meme.baseinfo = value
        self.balloon_menu = self.create_balloon_menu()
        self.plugins = ninix.home.search_plugins()
        self.nekoninni = ninix.home.search_nekoninni()
        self.katochan = ninix.home.search_katochan()
        self.kinoko = ninix.home.search_kinoko()

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            'close_all': self.close_all_ghosts,
            'edit_preferences': self.prefs.edit_preferences,
            'get_preference': self.prefs.get,
            'get_otherghostname': self.communicate.get_otherghostname,
            'rebuild_ghostdb': self.communicate.rebuild_ghostdb,
            'notify_other': self.communicate.notify_other,
            'reset_sstp_flag': self.sstp_controler.reset_sstp_flag,
            'get_sstp_port': self.sstp_controler.get_sstp_port,
            'get_prefix': self.get_sakura_prefix,
            }
        handler = handlers.get(event,
                               getattr(self, event,
                                       lambda *a: None)) ## FIXME
        result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def set_collisionmode(self, flag, rect=False):
        self.prefs.check_collision_button.set_active(bool(flag))
        self.prefs.check_collision_name_button.set_active(not bool(rect))
        self.prefs.update(commit=True) # XXX
        self.notify_preference_changed()

    def do_install(self, filename):
        try:
            filetype, target_dirs = self.installer.install(
                filename, ninix.home.get_ninix_home())
        except:
            target_dirs = None
        if target_dirs:
            if filetype == 'ghost':
                self.add_sakura(target_dirs[0])
                ninix.sakura.ReadmeDialog().show(
                    target_dirs[0],
                    os.path.join(ninix.home.get_ninix_home(),
                                 'ghost', target_dirs[0]))
                if target_dirs[1]:
                    self.add_balloon(target_dirs[1])
                    ninix.sakura.ReadmeDialog().show(
                        target_dirs[1],
                        os.path.join(ninix.home.get_ninix_home(),
                                     'balloon', target_dirs[1]))
            elif filetype == 'supplement':
                self.add_sakura(target_dirs) # XXX: reload
            elif filetype == 'balloon':
                self.add_balloon(target_dirs)
                ninix.sakura.ReadmeDialog().show(
                    target_dirs,
                    os.path.join(ninix.home.get_ninix_home(),
                                 'balloon', target_dirs))
            elif filetype == 'plugin':
                self.plugins = ninix.home.search_plugins()
            elif filetype == 'nekoninni':
                self.nekoninni = ninix.home.search_nekoninni()
            elif filetype == 'katochan':
                self.katochan = ninix.home.search_katochan()
            elif filetype == 'kinoko':
                self.kinoko = ninix.home.search_kinoko()

    def notify_installedghostname(self, key=None):
        installed = []
        for value in self.ghosts.values():
            sakura = value.instance
            if sakura is None:
                continue
            installed.append(sakura.get_name(default=''))
        if key is not None:
            if key in self.ghosts:
                sakura = self.ghosts[key].instance
                sakura.notify_event('installedghostname', *installed)
        else:
            for sakura in self.get_working_ghost():
                sakura.notify_event('installedghostname', *installed)

    def notify_installedballoonname(self, key=None):
        installed = []
        for value in self.balloons.values():
            desc, balloon = value.baseinfo
            subdir = balloon['balloon_dir'][0]
            installed.append(desc.get('name', subdir))
        if key is not None:
            if key in self.ghosts:
                sakura = self.ghosts[key].instance
                sakura.notify_event('installedballoonname', *installed)
        else:
            for sakura in self.get_working_ghost():
                sakura.notify_event('installedballoonname', *installed)

    @property
    def current_sakura_instance(self):
        return self.ghosts[self.current_sakura].instance

    def create_ghost(self, data):
        ghost = ninix.sakura.Sakura()
        ghost.set_responsible(self.handle_request)
        ghost.new(*data)
        return ghost

    def get_sakura_cantalk(self):
        return self.current_sakura_instance.cantalk

    def get_event_response(self, event, *arglist, **argdict): ## FIXME
        return self.current_sakura_instance.get_event_response(*event)

    def keep_silence(self, quiet):
        self.current_sakura_instance.keep_silence(quiet)

    def get_ghost_name(self): ## FIXME
        sakura = self.current_sakura_instance
        return sakura.get_ifghost()

    def enqueue_event(self, event, *arglist, **argdict):
        self.current_sakura_instance.enqueue_event(event, *arglist, **argdict)

    def enqueue_script(self, event, script, sender, handle,
                       host, show_sstp_marker, use_translator,
                       db=None, request_handler=None, temp_mode=False):
        sakura = self.current_sakura_instance
        if temp_mode:
            sakura.enter_temp_mode()
        sakura.enqueue_script(event, script, sender, handle,
                              host, show_sstp_marker, use_translator,
                              db, request_handler)

    def get_working_ghost(self, cantalk=0):
        for value in self.ghosts.values():
            sakura = value.instance
            if sakura is None:
                continue
            if not sakura.is_running():
                continue
            if cantalk and not sakura.cantalk:
                continue
            yield sakura

    def get_sakura_prefix(self):
        return self.__menu_owner.get_prefix()

    def getstring(self, name):
        return self.__menu_owner.getstring(name)

    def stick_window(self):
        stick = self.__menu.get_stick()
        self.__menu_owner.stick_window(stick)

    def toggle_bind(self, args):
        self.__menu_owner.toggle_bind(args)

    def select_shell(self, key):
        self.__menu_owner.select_shell(key)

    def select_balloon(self, key):
        desc, balloon = self.get_balloon_description(key)
        self.__menu_owner.select_balloon(key, desc, balloon)

    def get_current_balloon_directory(self): ## FIXME
        return self.__menu_owner.get_current_balloon_directory()

    def start_sakura_cb(self, key, caller=None):
        sakura_name = self.ghosts[key].instance.get_selfname(default='')
        name = self.ghosts[key].instance.get_name(default='')
        if caller is None:
            caller = self.__menu_owner
        caller.notify_event('OnGhostCalling', sakura_name, 'manual', name, key)
        self.start_sakura(key, init=1) # XXX

    def select_sakura(self, key):
        if self.__menu_owner.busy():
            gtk.gdk.beep()
            return
        self.change_sakura(self.__menu_owner, key, 'manual')

    def notify_site_selection(self, args):
        self.__menu_owner.notify_site_selection(args)

    def close_sakura(self):
        self.__menu_owner.close()

    def about(self):
        self.__menu_owner.about()

    def vanish(self):
        self.__menu_owner.vanish()

    def network_update(self):
        self.__menu_owner.network_update()

    def open_popup_menu(self, sakura, button, side):
        self.__menu_owner = sakura
        path_background, path_sidebar, path_foreground = \
            self.__menu_owner.get_menu_pixmap()
        self.__menu.set_pixmap(path_background, path_sidebar, path_foreground)
        background, foreground = self.__menu_owner.get_menu_fontcolor()
        self.__menu.set_fontcolor(background, foreground)
        mayuna_menu = self.__menu_owner.get_mayuna_menu()
        self.__menu.create_mayuna_menu(mayuna_menu)
        self.__menu.popup(button, side)

    def get_ghost_menus(self):
        for value in self.ghosts.values():
            yield value.menuitem

    def get_shell_menu(self):
        return self.__menu_owner.get_shell_menu()

    def get_balloon_menu(self):
        current_key = self.get_current_balloon_directory()
        for key in self.balloons:
            menuitem = self.balloons[key].menuitem
            menuitem.set_sensitive(key != current_key) # not working
        return self.balloon_menu

    def create_balloon_menuitem(self, balloon_name, balloon_key, thumbnail):
        return self.__menu.create_meme_menuitem(
            balloon_name, balloon_key, self.select_balloon, thumbnail)

    def create_balloon_menu(self):
        balloon_menuitems = OrderedDict()
        for key in self.balloons.keys():
            balloon_menuitems[key] = self.balloons[key].menuitem
        return self.__menu.create_meme_menu(balloon_menuitems)

    def create_shell_menu(self, menuitems):
        return self.__menu.create_meme_menu(menuitems)

    def create_shell_menuitem(self, shell_name, shell_key, thumbnail):
        return self.__menu.create_meme_menuitem(
            shell_name, shell_key, self.select_shell, thumbnail)

    def create_menuitem(self, key, baseinfo):
        desc = baseinfo[0]
        shiori_dir = baseinfo[1]
        icon = desc.get('icon', None)
        if icon is not None:
            if os.name == 'nt':
                # XXX: path should be encoded in mbcs on Windows
                icon = icon.encode('mbcs')
            icon_path = os.path.join(shiori_dir, icon)
            if not os.path.exists(icon_path):
                icon_path = None
        else:
            icon_path = None
        name = desc.get('name')
        thumbnail_path = os.path.join(shiori_dir, 'thumbnail.png')
        if not os.path.exists(thumbnail_path):
            thumbnail_path = None
        start_menuitem = self.__menu.create_ghost_menuitem(
            name, icon_path, key, self.start_sakura_cb, # XXX
            thumbnail_path)
        select_menuitem = self.__menu.create_ghost_menuitem(
            name, icon_path, key, self.select_sakura,
            thumbnail_path)
        menuitem = {
            'Summon': start_menuitem,
            'Change': select_menuitem,
            }
        return menuitem

    def delete_ghost(self, key):
        assert key in self.ghosts
        del self.ghosts[key]

    def get_balloon_list(self): ## FIXME
        balloon_list = []
        for key in self.balloons.keys():
            desc, balloon = self.balloons[key].baseinfo
            subdir = balloon['balloon_dir'][0]
            name = desc.get('name', subdir)
            balloon_list.append((name, subdir))
        return balloon_list

    def get_plugin_list(self): ## FIXME
        plugin_list = []
        for i, plugin in enumerate(self.plugins):
            plugin_name = plugin[0]
            menu_items = plugin[3]
            if not menu_items:
                continue
            item = {}
            item['name'] = plugin_name
            item['icon'] = None
            item_list = []
            for j, menu_item in enumerate(menu_items):
                label = menu_item[0]
                value = (i, j)
                item_list.append((label, value))
            item['items'] = item_list
            plugin_list.append(item)
        return plugin_list

    def get_nekodorif_list(self): ## FIXME
        nekodorif_list = []
        nekoninni = self.nekoninni
        for nekoninni_name, nekoninni_dir in nekoninni:
            if not nekoninni_name:
                continue
            item = {}
            item['name'] = nekoninni_name
            item['dir'] = nekoninni_dir
            nekodorif_list.append(item)
        return nekodorif_list

    def get_kinoko_list(self): ## FIXME
        return self.kinoko

    def load(self):
        # load user preferences
        self.prefs.load()
        # choose default ghost/shell
        directory = self.prefs.get('sakura_dir')
        name = self.prefs.get('sakura_name') # XXX: backward compat
        default_sakura = self.find_ghost_by_dir(directory) or \
                         self.find_ghost_by_name(name) or \
                         self.choose_default_sakura()
        # load ghost
        self.current_sakura = default_sakura
        ##for i, name in enumerate(self.get_ghost_names()):
        ##    logging.info(
        ##        'GHOST({0:d}): {1}'.format(
        ##            i, name.encode('utf-8', 'ignore')))
        self.start_sakura(self.current_sakura, init=1, abend=self.abend)

    def find_ghost_by_dir(self, directory):
        return directory if directory in self.ghosts else None

    def find_ghost_by_name(self, name):
        for key in self.ghosts:
            sakura = self.ghosts[key].instance
            try:
                if sakura.get_name(default=None) == name:
                    return key
            except: # old preferences(EUC-JP)
                pass
        return None

    def choose_default_sakura(self):
        return list(self.ghosts.keys())[0]

    def find_balloon_by_name(self, name):
        for key in self.balloons:
            desc, balloon = self.balloons[key].baseinfo
            try:
                if desc.get('name') == name:
                    return key
                if balloon['balloon_dir'][0] == ninix.home.get_normalized_path(name): # XXX
                    return key
            except: # old preferences(EUC-JP)
                pass
        return None

    def find_balloon_by_subdir(self, subdir):
        for key in self.balloons:
            desc, balloon = self.balloons[key].baseinfo
            try:
                if balloon['balloon_dir'][0] == subdir:
                    return key
                if ninix.home.get_normalized_path(desc.get('name')) == subdir: # XXX
                    return key
            except: # old preferences(EUC-JP)
                pass
        return None

    def exit_handler(self, sig_no, frame=None):
        self.close_all_ghosts(reason='shutdown')

    def run(self, abend):
        self.abend = abend
        if os.name == "nt":
            # The SIGTERM signal is not generated under Windows NT.
            import win32api
            win32api.SetConsoleCtrlHandler(self.exit_handler, True)
        else:
            import signal
            signal.signal(signal.SIGTERM, self.exit_handler)
        self.timeout_id = glib.timeout_add(100, self.do_idle_tasks) # 100[ms]
        gtk.main()

    def get_ghost_names(self):
        for value in self.ghosts.values():
            if value.instance is not None:
                yield value.instance.get_selfname() ## FIXME

    def if_ghost(self, if_ghost, working=False):
        for sakura in (value.instance for value in self.ghosts.values() \
                           if value.instance is not None):
            if working:
               if not sakura.is_running() or not sakura.cantalk:
                  continue
            if sakura.ifghost(if_ghost):
               return 1
        else:
            return 0

    def update_sakura(self, name, sender):
        key = self.find_ghost_by_name(name)
        if key is None:
            return
        sakura = self.ghosts[key].instance
        if not sakura.is_running():
            self.start_sakura(key, init=1)
        sakura.enqueue_script(None, '\![updatebymyself]\e', sender,
                              None, None, 0, 0, None)

    def select_current_sakura(self, ifghost=None):
        if ifghost is not None:
            for value in self.ghosts.values():
                sakura = value.instance
                if sakura is None:
                    continue
                if sakura.ifghost(ifghost):
                    if not sakura.is_running():
                        self.current_sakura = value.key
                        self.start_sakura(self.current_sakura, init=1, temp=1) ## FIXME
                    else:
                        self.current_sakura = sakura.key
                    break
                else:
                    pass
            else:
                return
        else:
            working_list = list(self.get_working_ghost(cantalk=1))
            if working_list:
                self.current_sakura = random.choice(working_list).key
            else:
                return ## FIXME

    def set_menu_sensitive(self, key, flag):
        menuitems = self.ghosts[key].menuitem
        for item in menuitems.values():
            item.set_sensitive(flag)

    def close_ghost(self, sakura):
        if not any(self.get_working_ghost()):
            self.prefs.set_current_sakura(sakura.key)
            self.quit()
        elif self.current_sakura == sakura.key:
            self.select_current_sakura()

    def close_all_ghosts(self, reason='user'):
        for sakura in self.get_working_ghost():
            sakura.notify_event('OnCloseAll', reason)

    def quit(self):
        glib.source_remove(self.timeout_id)
        self.usage_dialog.close()
        self.sstp_controler.quit() ## FIXME
        self.plugin_controler.terminate_plugin() ## FIXME
        self.save_preferences()
        gtk.main_quit()

    def save_preferences(self):
        try:
            self.prefs.save()
        except IOError:
            logging.error('Cannot write preferences to file (ignored).')
        except:
            pass ## FIXME

    def select_ghost(self, sakura, sequential, event=1, vanished=0):
        keys = list(self.ghosts.keys())
        if len(keys) < 2:
            return
        # select another ghost
        if sequential:
            key = (keys.index(sakura.key) + 1) % len(keys)
        else:
            keys.remove(sakura.key)
            key = random.choice(keys)
        self.change_sakura(sakura, key, 'automatic', event, vanished)

    def select_ghost_by_name(self, sakura, name, event=1, vanished=0):
        key = self.find_ghost_by_name(name)
        if key is None:
            return
        self.change_sakura(sakura, key, 'automatic', event, vanished)

    def change_sakura(self, sakura, key, method, event=1, vanished=0):
        if sakura.key == key: # XXX: needs reloading?
            return
        def proc(self=self, key=key):
            self.stop_sakura(sakura, self.start_sakura, key, sakura.key)
        if vanished:
            sakura.finalize()
            self.start_sakura(key, sakura.key, vanished)
            self.close_ghost(sakura)
        elif not event:
            proc()
        else:
            sakura_name = self.ghosts[key].instance.get_selfname(default='')
            name = self.ghosts[key].instance.get_name(default='')
            sakura.enqueue_event(
                'OnGhostChanging', sakura_name, method, name, key, proc=proc)

    def stop_sakura(self, sakura, starter=None, *args):
        sakura.finalize()
        if starter is not None:
            starter(*args)
        self.set_menu_sensitive(sakura.key, True)
        self.close_ghost(sakura)

    def start_sakura(self, key, prev=None, vanished=0, init=0, temp=0,
                     abend=None):
        sakura = self.ghosts[key].instance
        assert sakura is not None
        if prev is not None:
            assert prev in self.ghosts ## FIXME: vanish case?
            assert self.ghosts[prev].instance is not None
        if init:
            ghost_changed = 0
        else:
            assert prev is not None ## FIXME
            if prev == key:
                ghost_changed = 0
            else:
                ghost_changed = 1
        if ghost_changed:
            self_name = self.ghosts[prev].instance.get_selfname()
            name = self.ghosts[prev].instance.get_name()
            shell = self.ghosts[prev].instance.get_current_shell_name()
            last_script = self.ghosts[prev].instance.last_script
        else:
            self_name = None
            name = None
            shell = None
            last_script = None
        sakura.notify_preference_changed()
        sakura.start(key, init, temp, vanished, ghost_changed,
                     self_name, name, shell, last_script, abend)
        self.notify_installedghostname(key)
        self.notify_installedballoonname(key)
        sakura.notify_installedshellname()
        self.set_menu_sensitive(key, False)

    def update_working(self, ghost_name):
        self.lockfile.truncate(0)
        self.lockfile.seek(0)
        self.lockfile.write(ghost_name)
        self.lockfile.flush()

    def notify_preference_changed(self):
        for sakura in self.get_working_ghost():
            sakura.notify_preference_changed()

    def get_balloon_description(self, subdir): ## FIXME
        key = self.find_balloon_by_subdir(subdir)
        if key is None:
            ##logging.warning('Balloon {0} not found.'.format(subdir))
            default_balloon = self.prefs.get('default_balloon')
            key = self.find_balloon_by_subdir(default_balloon)
        if key is None:
            key = self.balloons.keys()[0]
        return self.balloons[key].baseinfo

    def reload_current_sakura(self, sakura):
        self.save_preferences()
        key = sakura.key
        ghost_dir = os.path.split(sakura.get_prefix())[1] # XXX
        ghost_conf = ninix.home.search_ghosts([ghost_dir])
        if ghost_conf:
            self.ghosts[key].baseinfo = ghost_conf[key]
        else:
            self.close_ghost(sakura) ## FIXME
            del self.ghosts[key]
            return ## FIXME
        self.start_sakura(key, key, init=1) 

    def add_sakura(self, ghost_dir):
        if ghost_dir in self.ghosts:
            exists = 1
            logging.warning('INSTALLED GHOST CHANGED: {0}'.format(ghost_dir))
        else:
            exists = 0
            logging.info('NEW GHOST INSTALLED: {0}'.format(ghost_dir))
        ghost_conf = ninix.home.search_ghosts([ghost_dir])
        if ghost_conf:
            if exists:
                sakura = self.ghosts[ghost_dir].instance
                if sakura.is_running(): # restart if working
                    key = sakura.key
                    def proc(self=self):
                        self.ghosts[ghost_dir].baseinfo = ghost_conf[ghost_dir]
                        logging.info('restarting....')
                        self.start_sakura(key, key, init=1)
                        logging.info('done.')
                    self.stop_sakura(sakura, proc)
            else:
                holon = Ghost(ghost_dir)
                holon.set_responsible(self.handle_request)
                self.ghosts[ghost_dir] = holon
                holon.baseinfo = ghost_conf[ghost_dir]
        else:
            if exists:
                sakura = self.ghosts[ghost_dir].instance
                if sakura.is_running(): # stop if working
                    self.stop_sakura(sakura)
                del self.ghosts[ghost_dir]
        self.notify_installedghostname()

    def add_balloon(self, balloon_dir):
        if balloon_dir in self.balloons:
            exists = 1
            logging.warning('INSTALLED BALLOON CHANGED: {0}'.format(balloon_dir))
        else:
            exists = 0
            logging.info('NEW BALLOON INSTALLED: {0}'.format(balloon_dir))
        balloon_conf = ninix.home.search_balloons([balloon_dir])
        if balloon_conf:
            if exists:
                self.balloons[balloon_dir].baseinfo = balloon_conf[balloon_dir]
            else:
                meme = BalloonMeme(balloon_dir)
                meme.set_responsible(self.handle_request)
                self.balloons[balloon_dir] = meme
                meme.baseinfo = balloon_conf[balloon_dir]
        else:
            if exists:
                del self.balloons[balloon_dir]
        self.balloon_menu = self.create_balloon_menu()

    def vanish_sakura(self, sakura, next_ghost):
        # remove ghost
        prefix = sakura.get_prefix()
        for filename in os.listdir(prefix):
            if os.path.isfile(os.path.join(prefix, filename)):
                if filename != 'HISTORY':
                    try:
                        os.remove(os.path.join(prefix, filename))
                    except:
                        logging.error(
                            '*** REMOVE FAILED *** : {0}'.format(filename))
            else: # dir
                try:
                    shutil.rmtree(os.path.join(prefix, filename))
                except:
                    logging.error(
                        '*** REMOVE FAILED *** : {0}'.format(filename))
        if next_ghost is not None:
            self.select_ghost_by_name(sakura, next_ghost, vanished=1)
        else:
            self.select_ghost(sakura, 0, vanished=1)
        del self.ghosts[sakura.key]

    def select_plugin(self, item):
        target = self.__menu_owner
        i, j = item
        plugin_name, plugin_dir, startup, menu_items = self.plugins[i]
        label, argv = menu_items[j]
        caller = {}
        caller['name'] = target.get_name()
        caller['directory'] = target.get_prefix()
        caller['ifghost'] = target.get_ifghost()
        self.plugin_controler.exec_plugin(plugin_dir, argv, caller)

    def select_nekodorif(self, nekodorif_dir):
        target = self.__menu_owner
        ninix.nekodorif.Nekoninni().load(nekodorif_dir,
                                         self.katochan, target)

    def select_kinoko(self, data):
        target = self.__menu_owner
        ninix.kinoko.Kinoko(self.kinoko).load(data, target)

    def open_console(self):
        self.console.open()

    def open_ghost_manager(self):
        self.__ngm.show_dialog()

    def show_usage(self):
        for sakura in self.get_working_ghost():
            sakura.save_history()
        history = {}
        for key in self.ghosts:
            sakura = self.ghosts[key].instance
            name = sakura.get_name(default=key)
            ghost_time = 0
            prefix = sakura.get_prefix()
            path = os.path.join(prefix, 'HISTORY')
            if os.path.exists(path):
                try:
                    with open(path, 'r') as f:
                        for line in f:
                            if ',' not in line:
                                continue
                            key, value = line.split(',', 1)
                            key = key.strip()
                            if key == 'time':
                                try:
                                    ghost_time = int(value.strip())
                                except:
                                    pass
                except IOError as e:
                    code, message = e.args
                    logging.error('cannot read {0}'.format(path))
            ai_list = []
            dirlist = os.listdir(os.path.join(prefix, 'shell'))
            for subdir in dirlist:
                path = os.path.join(prefix, 'shell', subdir, 'ai.png')
                if os.path.exists(path):
                    ai_list.append(path)
            history[name] = (ghost_time, ai_list)
        self.usage_dialog.open(history)

    def search_ghosts(self): ## FIXME
        balloons = self.balloons ## FIXME
        ghosts = self.ghosts ## FIXME
        if len(ghosts) > 0 and len(balloons) > 0:
            self.confirmed = True
        return len(ghosts), len(balloons)

    def do_idle_tasks(self):
        if not self.confirmed:
            self.console.open()
        else:
            if not self.loaded:
                self.load()
                # start SSTP server
                self.sstp_controler.start_servers()
                # start plugins
                self.plugin_controler.start_plugins(self.plugins)
                self.loaded = True
            else:
                self.sstp_controler.handle_sstp_queue()
                self.sstp_controler.receive_sstp_request()
                self.plugin_controler.check_queue()
        return True


class Console(object):

    def __init__(self, app):
        self.app = app
        self.window = gtk.Dialog()
        self.window.connect('delete_event', self.close)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        # DnD data types
        dnd_targets = [('text/uri-list', 0, 0)]
        self.darea.connect('drag_data_received', self.drag_data_received)
        self.darea.drag_dest_set(gtk.DEST_DEFAULT_ALL, dnd_targets,
                                 gtk.gdk.ACTION_COPY)
        self.darea.drag_dest_add_uri_targets()
        self.size = (330, 110) ## FIXME
        self.darea.set_size_request(*self.size)
        self.darea.connect('configure_event', self.configure)
        self.darea.connect('expose_event', self.redraw)
        self.window.vbox.pack_start(self.darea)
        self.darea.show()
        box = gtk.HButtonBox()
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('Install')
        button.connect('clicked', self.file_chooser)
        box.add(button)
        button.show()
        button = gtk.Button('Close')
        button.connect('clicked', self.close)
        box.add(button)
        button.show()
        self.file_chooser = gtk.FileChooserDialog(
            "Install..",
            None,
            gtk.FILE_CHOOSER_ACTION_OPEN,
            (gtk.STOCK_OPEN, gtk.RESPONSE_OK,
             gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
        self.file_chooser.set_default_response(gtk.RESPONSE_CANCEL)
        filter = gtk.FileFilter()
        filter.set_name("All files")
        filter.add_pattern("*")
        self.file_chooser.add_filter(filter)
        filter = gtk.FileFilter()
        filter.set_name("nar/zip")
        filter.add_mime_type("application/zip")
        filter.add_pattern("*.nar")
        filter.add_pattern("*.zip")
        self.file_chooser.add_filter(filter)
        self.opened = 0

    def update(self): ## FIXME
        self.darea.queue_draw()

    def open(self):
        if self.opened:
            return
        self.window.show()
        self.opened = 1

    def close(self, widget=None, event=None):
        self.window.hide()
        self.opened = 0
        if not self.app.confirmed: ## FIXME
            self.app.quit()
        return True

    def file_chooser(self, widget=None, event=None):
        response = self.file_chooser.run()
        if response == gtk.RESPONSE_OK:
            filename = self.file_chooser.get_filename()
            self.app.do_install(filename)
            self.update()
        elif response == gtk.RESPONSE_CANCEL:
            pass
        self.file_chooser.hide()

    def configure(self, darea, event):
        x, y, w, h = darea.get_allocation()
        self.size = (w, h)

    def draw_message(self, text): ## FIXME
        pass

    def redraw(self, darea, event):
        ghosts, balloons = self.app.search_ghosts() # XXX
        if ghosts > 0 and balloons > 0:
            self.window.set_title(_('Console'))
        else:
            self.window.set_title(_('Nanntokashitekudasai.'))
        layout = pango.Layout(darea.get_pango_context())
        font_desc = pango.FontDescription()
        font_desc.set_size(9 * pango.SCALE)
        font_desc.set_family('Sans') # FIXME
        layout.set_font_description(font_desc)
        cr = darea.window.cairo_create()
        w, h = self.size
        cr.set_source_rgb(0.0, 0.0, 0.0) # black
        cr.paint()
        layout.set_text('Ghosts: {0:d}'.format(ghosts))
        if ghosts == 0:
            cr.set_source_rgb(1.0, 0.2, 0.2) # red
        else:
            cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(20, 15)
        cr.show_layout(layout)
        w, h = layout.get_pixel_size()
        layout.set_text('Balloons: {0:d}'.format(balloons))
        if balloons == 0:
            cr.set_source_rgb(1.0, 0.2, 0.2) # red
        else:
            cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(20, 15 + h)
        cr.show_layout(layout)
        del cr

    def drag_data_received(self, widget, context, x, y, data, info, time):
        logging.info('Content-type: {0}'.format(data.type))
        logging.info('Content-length: {0:d}'.format(data.get_length()))
        filelist = []
        for uri in data.get_uris():
            scheme, host, path, params, query, fragment = \
                urlparse.urlparse(uri)
            pathname = urllib.url2pathname(path)
            if scheme == 'file' and os.path.exists(pathname):
                filelist.append(pathname)
            elif scheme == 'http' or scheme == 'ftp':
                filelist.append(uri)
        if filelist:
            for filename in filelist:
                self.app.do_install(filename)
            self.update()


class UsageDialog(object):

    def __init__(self):
        self.window = gtk.Dialog()
        self.window.set_title('Usage')
        self.window.connect('delete_event', self.close)
        self.darea = gtk.DrawingArea()
        self.darea.set_events(gtk.gdk.EXPOSURE_MASK)
        self.size = (550, 330)
        self.darea.set_size_request(*self.size)
        self.darea.connect('configure_event', self.configure)
        self.darea.connect('expose_event', self.redraw)
        self.window.vbox.pack_start(self.darea)
        self.darea.show()
        box = gtk.HButtonBox()
        box.set_layout(gtk.BUTTONBOX_END)
        self.window.action_area.pack_start(box)
        box.show()
        button = gtk.Button('Close')
        button.connect('clicked', self.close)
        box.add(button)
        button.show()
        self.opened = 0

    def open(self, history):
        if self.opened:
            return
        self.history = history
        self.items = \
            [(name, clock, path) for name, (clock, path) in \
                 self.history.items()]
        self.items[:] = [(x[1], x) for x in self.items]
        self.items.sort()
        self.items[:] = [x for x_1, x in self.items]
        self.items.reverse()
        ai_list = self.items[0][2]
        if ai_list:
            path = random.choice(ai_list)
            assert os.path.exists(path)
            self.pixbuf = ninix.pix.create_pixbuf_from_file(
                path, is_pnr=False)
            self.pixbuf.saturate_and_pixelate(self.pixbuf, 1.0, True)
        else:
            self.pixbuf = None
        self.window.show()
        self.opened = 1

    def close(self, widget=None, event=None):
        self.window.hide()
        self.opened = 0
        return True

    def configure(self, darea, event):
        x, y, w, h = darea.get_allocation()
        self.size = (w, h)

    def redraw(self, darea, event):
        if not self.items:
            return # should not reach here
        total = float(0)
        for name, clock, path in self.items:
            total += clock
        layout = pango.Layout(darea.get_pango_context())
        font_desc = pango.FontDescription()
        font_desc.set_size(9 * pango.SCALE)
        font_desc.set_family('Sans') # FIXME
        layout.set_font_description(font_desc)
        cr = darea.window.cairo_create()
        # redraw graph
        w, h = self.size
        cr.set_source_rgb(1.0, 1.0, 1.0) # white
        cr.paint()
        # ai.png
        if self.pixbuf:
            cr.set_source_pixbuf(self.pixbuf, 16, 32) # XXX
            cr.paint()
        w3 = w4 = 0
        rows = []
        for name, clock, path in self.items[:14]:
            layout.set_text(name)
            name_w, name_h = layout.get_pixel_size()
            rate = '{0:.1f}%'.format(clock / total * 100)
            layout.set_text(rate)
            rate_w, rate_h = layout.get_pixel_size()
            w3 = max(rate_w, w3)
            time = '{0}:{1:02d}'.format(*divmod(clock // 60, 60))
            layout.set_text(time)
            time_w, time_h = layout.get_pixel_size()
            w4 = max(time_w, w4)
            rows.append((clock, name, name_w, name_h, rate, rate_w, rate_h,
                         time, time_w, time_h))
        w1 = 280
        w2 = w - w1 - w3 - w4 - 70
        x = 20
        y = 15
        x += w1 + 10
        label = 'name'
        layout.set_text(label)
        label_name_w, label_name_h = layout.get_pixel_size()
        cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(x, y)
        cr.show_layout(layout)
        x = x + w2 + 10
        label = 'rate'
        layout.set_text(label)
        label_rate_w, label_rate_h = layout.get_pixel_size()
        cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(x + w3 - label_rate_w, y)
        cr.show_layout(layout)
        x += w3 + 10
        label = 'time'
        layout.set_text(label)
        label_time_w, label_time_h = layout.get_pixel_size()
        cr.set_source_rgb(0.8, 0.8, 0.8) # gray
        cr.move_to(x + w4 - label_time_w, y)
        cr.show_layout(layout)
        y += max([label_name_h, label_rate_h, label_time_h]) + 4
        for clock, name, name_w, name_h, rate, rate_w, rate_h, time, time_w, \
                time_h  in rows:
            x = 20
            bw = int(clock / total * w1)
            bh = max([name_h, rate_h, time_h]) - 1
            cr.set_source_rgb(0.8, 0.8, 0.8) # gray
            cr.rectangle(x + 1, y + 1, bw, bh)
            cr.stroke()
            cr.set_source_rgb(1.0, 1.0, 1.0) # white
            cr.rectangle(x, y, bw, bh)
            cr.stroke()
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.rectangle(x, y, bw, bh)
            cr.stroke()
            x += w1 + 10
            layout.set_text(name)
            end = len(name)
            while end > 0:
                w, h = layout.get_pixel_size()
                if w > 168:
                    end -= 1
                    layout.set_text(''.join((name[:end], u'...')))
                else:
                    break
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.move_to(x, y)
            cr.show_layout(layout)
            x += w2 + 10
            layout.set_text(rate)
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.move_to(x + w3 - rate_w, y)
            cr.show_layout(layout)
            x += w3 + 10
            layout.set_text(time)
            cr.set_source_rgb(0.0, 0.0, 0.0) # black
            cr.move_to(x + w4 - time_w, y)
            cr.show_layout(layout)
            y += max([name_h, rate_h, time_h]) + 4
        del cr


if __name__ == '__main__':
    main()
