/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// PublicationMethod.cpp: implementation of the PublicationMethod class.
//
//////////////////////////////////////////////////////////////////////

#include "PublicationMethod.h"
#include <PKI_ERR.h>

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

PublicationMethod::PublicationMethod(const mString & CaName, const PublicationMethodInfo & method)
{
	size_t i;
	const char * Err;

	ERR_clear_error();


	hLib = NULL;
	m_Functions = NULL;

	
	m_CaName = CaName;
	m_Name = method.get_name();
	Type = method.get_type();


	for(i=0; i<method.get_options().size(); i++)
	{
		m_Options.Add(method.get_options()[i].get_name().c_str(), method.get_options()[i].get_value().c_str());
	}


	hLib = DSO_load(NULL, method.get_librarypath().c_str(), DSO_METHOD_dl(), 0);
	if (!hLib)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		throw ExceptionNewPKI();
	}

	m_GetFunctions = (GET_FUNCTIONS)DSO_bind_func(hLib, "GetFunctions");
	if(!m_GetFunctions)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PLUG_LIB);
		ERR_set_error_data("GetFunctions", ERR_TXT_STRING);
		if(hLib) DSO_free(hLib);
		throw ExceptionNewPKI();
	}

	m_Functions = m_GetFunctions();
	if(!m_Functions)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_UNKNOWN);
		if(hLib) DSO_free(hLib);
		throw ExceptionNewPKI();
	}

	if(!m_Functions->InitLib)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PLUG_LIB);
		ERR_set_error_data("InitLib", ERR_TXT_STRING);
		if(hLib) DSO_free(hLib);
		throw ExceptionNewPKI();
	}

	if(!m_Functions->ClearLib)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PLUG_LIB);
		ERR_set_error_data("ClearLib", ERR_TXT_STRING);
		if(hLib) DSO_free(hLib);
		throw ExceptionNewPKI();
	}

	if(!m_Functions->GetLibLastError)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PLUG_LIB);
		ERR_set_error_data("GetLibLastError", ERR_TXT_STRING);
		if(hLib) DSO_free(hLib);
		throw ExceptionNewPKI();
	}

	if(!m_Functions->InitLib(m_Options))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Err = m_Functions->GetLibLastError();
		if(Err)
			ERR_add_error_data(1, Err);
		if(hLib) DSO_free(hLib);
		throw ExceptionNewPKI();
	}
}

PublicationMethod::~PublicationMethod()
{
	if(m_Functions && m_Functions->ClearLib)
		m_Functions->ClearLib(m_Options);
	if(hLib) DSO_free(hLib);
}

const mString & PublicationMethod::GetCaName()
{
	return m_CaName;
}

int PublicationMethod::GetType()
{
	return Type;
}

bool PublicationMethod::OnNewCertificate(PKI_CERT &Certificate, PKI_P7B & P7B, const mString &ldap_uid)
{
	char * Err;

	if(!m_Functions->OnNewCertificate)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PLUG_LIB);
		ERR_set_error_data("OnNewCertificate", ERR_TXT_STRING);
		return false;
	}
	if(!Certificate.LoadAll())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}
	if(!m_Functions->OnNewCertificate(m_Options, Certificate, P7B, ldap_uid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Err = (char*)m_Functions->GetLibLastError();
		if(Err)
		{
			Err = strdup(Err);
			if(Err)
				ERR_set_error_data(Err, ERR_TXT_MALLOCED | ERR_TXT_STRING);
		}
		return false;
	}

	return true;
}

const mString & PublicationMethod::GetName()
{
	return m_Name;
}

bool PublicationMethod::OnNewRevocation(PKI_CERT &Certificate, const mString &ldap_uid)
{
	char * Err;

	if(!m_Functions->OnNewRevocation)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PLUG_LIB);
		ERR_set_error_data("OnNewRevocation", ERR_TXT_STRING);
		return false;
	}

	if(!Certificate.LoadAll())
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		return false;
	}

	if(!m_Functions->OnNewRevocation(m_Options, Certificate, ldap_uid))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Err = (char*)m_Functions->GetLibLastError();
		if(Err)
		{
			Err = strdup(Err);
			if(Err)
				ERR_set_error_data(Err, ERR_TXT_MALLOCED | ERR_TXT_STRING);
		}
		return false;
	}

	return true;
}

bool PublicationMethod::OnNewCrl(const PKI_CRL &Crl)
{
	char * Err;

	if(!m_Functions->OnNewCrl)
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_BAD_PLUG_LIB);
		ERR_set_error_data("OnNewCrl", ERR_TXT_STRING);
		return false;
	}

	if(!m_Functions->OnNewCrl(m_Options, Crl))
	{
		NEWPKIerr(PKI_ERROR_TXT, ERROR_ABORT);
		Err = (char*)m_Functions->GetLibLastError();
		if(Err)
		{
			Err = strdup(Err);
			if(Err)
				ERR_set_error_data(Err, ERR_TXT_MALLOCED | ERR_TXT_STRING);
		}
		return false;
	}

	return true;
}
