/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/



// ClientLDAP.h: interface for the ClientLDAP class.
//
//////////////////////////////////////////////////////////////////////

#ifndef CLIENTLDAP_H
#define CLIENTLDAP_H

#include <openssl/objects.h>
#include <ASN1/Asn1Resp.h>
#include <mString.h>

#ifdef _WIN32
	#define _SCHNLSP_H_
	#define PSecPkgContext_IssuerListInfoEx void *
	#define HCERTSTORE void *
	#define PCCERT_CONTEXT void *

	#include <windows.h>
	#include <winldap.h>
#else
	#include <ldap.h>
#endif


/*!
	This class provides LDAP functionnalities
*/
class ClientLDAP  
{
public:
	/*! \brief This is the constructor.
	 */
	ClientLDAP();

	/*! \brief This is the destructor.
	 */
	virtual ~ClientLDAP();

	/*! \brief Connect to the LDAP server.
	 *  \param ldap_server [IN] The ldap server's address.
	 *  \param ldap_port [IN] The ldap server's port.
	 *  \param ldap_username [IN] The username.
	 *  \param ldap_password [IN] The password.
	 *  \param ldap_base [IN] The base to search in.
	 *  \param ldap_attr_name [IN] The name of the attribute that hold the uid.
	 *  \param utf8 [IN] The LDAP server uses UTF-8 encoding.
	 *  \return true on success, false on failure.
	 */
	bool Connect(const mString & ldap_server, unsigned long ldap_port, const mString & ldap_username, const mString & ldap_password, const mString & ldap_base, const mString & ldap_attr_name, unsigned long utf8);

	/*! \brief Reconnect to the LDAP server.
	 *  \return true on success, false on failure.
	 */
	bool Reconnect();

	/*! \brief Disconnect from the LDAP server.
	 */
	void Disconnect();

	/*! \brief Search the LDAP server.
	 *  \param SearchString [IN] The filters.
	 *  \param Results [OUT] The results.
	 *  \param MaxResults [IN] The maximum number of entries to return.
	 *  \param MaxTime [IN] A timeout value for the search.
	 *  \return true on success, false on failure.
	 */
	bool Search(const mString & SearchString, mVector<LdapResult> & Results, int MaxResults=LDAP_NO_LIMIT, int MaxTime=LDAP_NO_LIMIT);
	
	/*! \brief This function return 1 when LDAP client is operable and 0 when not.
	 *  \return return 1 when LDAP client is operable and 0 when not.
	 */
	operator int();
private:
	#ifdef _WIN32
		#define LDAP_RC_TYPE unsigned long
		#define LDAP_LAST_ERROR LdapGetLastError()
	#else
		#define LDAP_RC_TYPE int
		#define LDAP_LAST_ERROR errno
	#endif
	LDAP * m_Connection;

	void AddObject(mVector<LdapObject> & Objects, char * Name, char * Value);
	bool ObjectAlreadyKnown(const mVector<LdapObject> & Objects, const mString & Name, const mString & Value);
	int GetNid(char *Obj);
	void AddCurrentMessage(mVector<LdapResult> & Results, LDAPMessage * currMsg);

	mString m_ldap_attr_name;
	mString m_ldap_server;
	int m_ldap_port;
	mString m_ldap_username;
	mString m_ldap_password;
	mString m_ldap_base;
	unsigned long m_utf8;
};

#endif
