/*
	Copyright (C) 2003 Frdric Giudicelli (contact_nos@yahoo.com). 
	All rights reserved.

	This product includes cryptographic software written by Eric Young
	(eay@cryptsoft.com)

	This program is released under the GPL with the additional exemption that
	compiling, linking, and/or using OpenSSL is allowed.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the Free
	Software Foundation; either version 2 of the License.

	This program is distributed in the hope that it will be useful, but WITHOUT
	ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
	FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
	more details.

	You should have received a copy of the GNU General Public License along with
	this program; if not, write to the Free Software Foundation, Inc., 59 Temple
	Place, Suite 330, Boston, MA 02111-1307 USA
*/


// DlgCaAdmin.cpp: implementation of the DlgCaAdmin class.
//
//////////////////////////////////////////////////////////////////////

#include "DlgSendMail.h"
#include "DlgCertProperties.h"
#include "DlgP7bProperties.h"
#include "DlgCreateRootCA.h"
#include "DlgCreateChildCA.h"
#include "DlgPublishCSR.h"
#include "DlgCertProperties.h"
#include "DlgCrlProperties.h"
#include "DlgGetPassword.h"
#include "DlgExtension.h"

#include "DlgCaAdmin.h"
#include "dlgs_wdr.h"
#include "clintl.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

BEGIN_EVENT_TABLE(DlgCaAdminListCtrl, wxListCtrl)
    EVT_LIST_ITEM_RIGHT_CLICK(IDC_LIST_CERTS, DlgCaAdminListCtrl::OnItemRightClick)
    EVT_LIST_ITEM_ACTIVATED(IDC_LIST_CERTS, DlgCaAdminListCtrl::OnItemClick)
END_EVENT_TABLE()



DlgCaAdmin * m_DlgCaAdminParent;


#ifdef _WIN32

wxBitmap DlgCaAdminListCtrlhPic = Backgrounds_GetBitmap(0);

long DlgCaAdminListCtrloldProc;
int DlgCaAdminListCtrlcenterX;
int DlgCaAdminListCtrlcenterY;
bool DlgCaAdminListCtrlinProc = false;

LRESULT CALLBACK DlgCaAdminListCtrllstvProc(HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	RECT rRect;
	PAINTSTRUCT ps;
	HDC hDC;
	HDC hDCParent;
	HBITMAP bmp;
	HBITMAP bmpOld;
				
	switch (message) 
	{
		case WM_PAINT:
			GetClientRect(hwnd,&rRect);

			if(DlgCaAdminListCtrlinProc)
			{
				DlgCaAdminListCtrlinProc = false;
				InvalidateRect(hwnd, &rRect, TRUE);
				return TRUE;
			}
			DlgCaAdminListCtrlinProc = true;


			CallWindowProcA((WNDPROC)DlgCaAdminListCtrloldProc, hwnd, message, wParam, lParam);		

			BeginPaint(hwnd, &ps);
			
			hDC=GetDC(hwnd);
			hDCParent=CreateCompatibleDC(hDC);
			bmp = (HBITMAP)DlgCaAdminListCtrlhPic.GetHBITMAP();
			
			
			bmpOld = (HBITMAP)SelectObject(hDCParent, bmp);
			
			
			DlgCaAdminListCtrlcenterX = ((rRect.right - rRect.left) - DlgCaAdminListCtrlhPic.GetWidth()) / 2;
			DlgCaAdminListCtrlcenterY = ((rRect.bottom - rRect.top) - DlgCaAdminListCtrlhPic.GetHeight())  / 2;
			
			
			BitBlt(hDC,DlgCaAdminListCtrlcenterX, DlgCaAdminListCtrlcenterY, DlgCaAdminListCtrlhPic.GetWidth(),DlgCaAdminListCtrlhPic.GetHeight(), hDCParent, 0, 0,SRCAND );
			
			ReleaseDC(hwnd,hDC);
			SelectObject(hDCParent, bmpOld);
			DeleteDC(hDCParent);

			EndPaint(hwnd, &ps);
			return TRUE;
			
		default:
			return CallWindowProcA((WNDPROC)DlgCaAdminListCtrloldProc, hwnd, message, wParam, lParam);
	}
	return FALSE;
}

#endif

DlgCaAdminListCtrl::DlgCaAdminListCtrl(wxWindow *parent, const wxWindowID id, const wxPoint& pos,	const wxSize& size, long style):wxListCtrl(parent, id, pos, size, style)
{
	#ifdef _WIN32
		DlgCaAdminListCtrloldProc = SetWindowLong((HWND)GetHWND(), GWL_WNDPROC, (long)DlgCaAdminListCtrllstvProc);
	#endif
}

DlgCaAdminListCtrl::~DlgCaAdminListCtrl()
{
}

void DlgCaAdminListCtrl::OnItemRightClick(wxListEvent& event)
{
	m_DlgCaAdminParent->OnItemRightClick(event);
}

void DlgCaAdminListCtrl::OnItemClick(wxListEvent& event)
{
	m_DlgCaAdminParent->OnItemClick(event);
}



BEGIN_EVENT_TABLE(DlgCaAdmin, wxFrame)
	EVT_MENU(IDM_QUIT, DlgGUI::OnQuit)
    EVT_MENU(IDM_RELOAD, DlgCaAdmin::OnReload)
    EVT_MENU(IDM_SHOW_LOGS, DlgGUI::OnShowLogs)
    EVT_MENU(IDM_RECONNECT, DlgGUI::OnReconnect)
    EVT_MENU(IDM_SHOW_SSL_CERT, DlgGUI::ShowSSL_Certificate)
    EVT_MENU(IDM_SHOW_ENTITY_CERT, DlgGUI::ShowEntity_Certificate)
    EVT_MENU(IDM_ABOUT, DlgGUI::OnAbout)
    EVT_MENU(IDC_SEND_ADMIN_MAIL, DlgCaAdmin::OnSendAdminMail)
    EVT_MENU(IDM_CREATE_ROOT_CA, DlgCaAdmin::OnCreateRootCa)
    EVT_MENU(IDM_CREATE_CHILD_CA, DlgCaAdmin::OnCreateChildCa)
    EVT_MENU(IDM_IMPORT_CA_CERT, DlgCaAdmin::OnImportChildCa)
    EVT_MENU(IDM_IMPORT_CA_P12, DlgCaAdmin::OnImportCaP12)
    EVT_MENU(IDC_SHOW_PREV, DlgCaAdmin::OnShowPrevEntries)
    EVT_MENU(IDC_REFRESH, DlgCaAdmin::OnRefreshEntries)
    EVT_MENU(IDC_SHOW_NEXT, DlgCaAdmin::OnShowNextEntries)
    EVT_COMBOBOX(IDC_VIEW_TYPE, DlgCaAdmin::OnViewChanged)
    EVT_MENU(IDC_IMPORT_CSR, DlgCaAdmin::OnImportCSR)
    EVT_MENU(IDM_SHOW_DETAILS, DlgCaAdmin::OnShowDetails)
    EVT_MENU(IDM_REVOKE_CERT, DlgCaAdmin::OnRevokeCert)
    EVT_MENU(IDC_GENERATE_CRL, DlgCaAdmin::OnGenerateCRL)
    EVT_MENU(IDM_SHOW_CA_CERT, DlgCaAdmin::OnViewCaP7b)
    EVT_MENU(IDM_SHOW_CERT_EXTS, DlgCaAdmin::OnShowCertExts)
    EVT_MENU(IDM_SHOW_CRL_EXTS, DlgCaAdmin::OnShowCrlExts)
    EVT_MENU(IDM_SUSPEND_CERT, DlgCaAdmin::OnSuspendCert)
    EVT_MENU(IDM_UNSUSPEND_CERT, DlgCaAdmin::OnUnsuspendCert)
END_EVENT_TABLE()


DlgCaAdmin::DlgCaAdmin(wxWindow * wParent, const char * EntityName, const char * UserName, const char * Password, PkiClient * ClientPki):DlgGUI(wParent, ClientPki, EntityName, UserName, Password)
{
	m_DlgCaAdminParent = this;
	CurrentPosition = 0;
	LastSelectedEntry = -1;

    
	wxToolBar * toolBar = new wxToolBar(this, -1);

	wxString mTitle;
	mTitle.sprintf("CA Admin (%s/%s) [%s:%d]", UserName, EntityName, ClientPki->GetRemoteServer().c_str(), ClientPki->GetRemotePort());
	SetTitle(mTitle);

	DlgCaAdmin_SetWindow(this);
	SetMenuBar(DlgCaAdmin_GetMenu());

	toolBar->SetToolBitmapSize(wxSize(32,32));
	DlgCaAdmin_SetToolbar(toolBar);
    SetToolBar(toolBar);


	CenterOnScreen();

	m_imageListSmall = new wxImageList(16, 16, TRUE);

	
	m_listCtrl = (DlgCaAdminListCtrl *)FindWindow(IDC_LIST_CERTS);
	if(!m_listCtrl) return;

	wxIcon ico;
	
	ico.CopyFromBitmap(DlgCaAdmin_GetBitmap(IDB_CA_CERT_ACT));
	CertIconIdAct = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgCaAdmin_GetBitmap(IDB_CA_CERT_REV));
	CertIconIdRev = m_imageListSmall->Add( ico );

	ico.CopyFromBitmap(DlgCaAdmin_GetBitmap(IDB_CA_CRL));
	CrlIconId = m_imageListSmall->Add( ico );


	m_listCtrl->SetImageList(m_imageListSmall, wxIMAGE_LIST_SMALL);

	
	if(!ReloadConf())
	{
		Close(TRUE);
		return;
	}
}

DlgCaAdmin::~DlgCaAdmin()
{
	if(m_listCtrl) m_listCtrl->ClearAll();
	if(m_imageListSmall) delete m_imageListSmall;
}


void DlgCaAdmin::OnReload(wxCommandEvent& event)
{
	ReloadConf();
}

void DlgCaAdmin::OnItemClick(wxListEvent &event)
{

}

void DlgCaAdmin::OnItemRightClick(wxListEvent &event)
{
	wxMenu * menu;

	if(m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED) == -1)
		return;

	wxMenuBar * menu_b = DlgCaAdmin_GetPopup();
	if(!menu_b) return;

	menu = menu_b->GetMenu(0);
	if(menu)
	{
		switch(EntryType)
		{
			case CA_OBJECTS_VIEW_CRL:
				menu->Enable(IDM_REVOKE_CERT, false);
				menu->Enable(IDM_SUSPEND_CERT, false);
				menu->Enable(IDM_UNSUSPEND_CERT, false);
				break;
			case CA_OBJECTS_VIEW_CERT:
				switch(currCertStatus)
				{
					case CERT_STATE_VALID:
						menu->Enable(IDM_UNSUSPEND_CERT, false);
						break;
					case CERT_STATE_SUSPENDED:
						menu->Enable(IDM_SUSPEND_CERT, false);
						break;
					case CERT_STATE_REVOKED:
						menu->Enable(IDM_REVOKE_CERT, false);
						menu->Enable(IDM_SUSPEND_CERT, false);
						menu->Enable(IDM_UNSUSPEND_CERT, false);
						break;
					default:
						break;
				}
				break;
			default:
				break;
		}
		m_listCtrl->PopupMenu(menu, event.GetPoint().x, event.GetPoint().y);
	}

	delete menu_b;
}


bool DlgCaAdmin::ReloadConf()
{
	unsigned long status;
	ASN1_BIT_STRING * my_acl;
	wxMenuBar * menubar;
	wxStringClientData * ClientData;
	wxString strLng;
	
	m_DlgMsg->wShow(_("Loading Configuration..."));

	m_EntityConf.Clear();
	if(!m_ClientPki->GetLocalConf(m_EntityConf))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}
	
	if(!m_ClientPki->GetMyACL(&my_acl))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}

	menubar = GetMenuBar();

	//Clear view type
	((wxComboBox *)FindWindow(IDC_VIEW_TYPE))->Clear();
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_VIEW_LOGS))
	{
		menubar->GetMenu(1)->Enable(IDM_SHOW_LOGS, false);
	}
	else
	{
		menubar->GetMenu(1)->Enable(IDM_SHOW_LOGS, true);
	}
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_SEND_ADMIN_MAIL))
	{
		GetToolBar()->EnableTool( IDC_SEND_ADMIN_MAIL, FALSE );
	}
	else
	{
		GetToolBar()->EnableTool( IDC_SEND_ADMIN_MAIL, TRUE );
	}
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_CREATE_CHILD_CA))
	{
		menubar->GetMenu(2)->Enable(IDM_CREATE_CHILD_CA, false);
		menubar->GetMenu(2)->Enable(IDM_IMPORT_CA_CERT, false);
		menubar->GetMenu(2)->Enable(IDM_IMPORT_CA_P12, false);
	}
	else
	{
		menubar->GetMenu(2)->Enable(IDM_CREATE_CHILD_CA, true);
		menubar->GetMenu(2)->Enable(IDM_IMPORT_CA_CERT, true);
		menubar->GetMenu(2)->Enable(IDM_IMPORT_CA_P12, true);
	}
	
	if(!ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_CREATE_ROOT_CA))
	{

		menubar->GetMenu(2)->Enable(IDM_CREATE_ROOT_CA, false);
	}
	else
	{
		menubar->GetMenu(2)->Enable(IDM_CREATE_ROOT_CA, true);
	}

	if(ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_CA_SIGN_CERT) || ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_CA_REVOKE_CERT))
	{
		ClientData = new wxStringClientData();
		strLng.sprintf("%d", CA_OBJECTS_VIEW_CERT);
		ClientData->SetData(strLng);
		((wxComboBox *)FindWindow(IDC_VIEW_TYPE))->Append(_("Valid Certificates"), (void*)ClientData);

		GetToolBar()->EnableTool( IDC_IMPORT_CSR, TRUE );
	}
	else
	{
		GetToolBar()->EnableTool( IDC_IMPORT_CSR, FALSE );
	}
	
	if(ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_CA_REVOKE_CERT))
	{
		ClientData = new wxStringClientData();
		strLng.sprintf("%d", CA_OBJECTS_VIEW_REV);
		ClientData->SetData(strLng);
		((wxComboBox *)FindWindow(IDC_VIEW_TYPE))->Append(_("Revoked Certificates"), (void*)ClientData);

		ClientData = new wxStringClientData();
		strLng.sprintf("%d", CA_OBJECTS_VIEW_SUSP);
		ClientData->SetData(strLng);
		((wxComboBox *)FindWindow(IDC_VIEW_TYPE))->Append(_("Suspended Certificates"), (void*)ClientData);
	}
	if(ASN1_BIT_STRING_get_bit(my_acl, ACL_TYPE_CA_GENERATE_CRL))
	{
		ClientData = new wxStringClientData();
		strLng.sprintf("%d", CA_OBJECTS_VIEW_CRL);
		ClientData->SetData(strLng);
		((wxComboBox *)FindWindow(IDC_VIEW_TYPE))->Append(_("CRLs"), (void*)ClientData);
		
		GetToolBar()->EnableTool( IDC_GENERATE_CRL, TRUE );
	}
	else
	{
		GetToolBar()->EnableTool( IDC_GENERATE_CRL, FALSE );
	}

	//Do we have some view rights ?
	if(!((wxComboBox *)FindWindow(IDC_VIEW_TYPE))->GetCount())
	{
		GetToolBar()->EnableTool( IDC_SHOW_PREV, FALSE );
		GetToolBar()->EnableTool( IDC_REFRESH, FALSE );
		GetToolBar()->EnableTool( IDC_SHOW_NEXT, FALSE );
		m_listCtrl->Enable(FALSE);
		((wxComboBox *)FindWindow(IDC_VIEW_TYPE))->Enable(FALSE);
		((wxSpinCtrl *)FindWindow(IDC_NUM_ENTRIES))->Enable(FALSE);
	}
	else
	{
		GetToolBar()->EnableTool( IDC_SHOW_PREV, TRUE );
		GetToolBar()->EnableTool( IDC_REFRESH, TRUE );
		GetToolBar()->EnableTool( IDC_SHOW_NEXT, TRUE );
		m_listCtrl->Enable(TRUE);
		((wxComboBox *)FindWindow(IDC_VIEW_TYPE))->Enable(TRUE);
		((wxSpinCtrl *)FindWindow(IDC_NUM_ENTRIES))->Enable(TRUE);
	}
	ASN1_BIT_STRING_free(my_acl);
	((wxComboBox *)FindWindow(IDC_VIEW_TYPE))->SetSelection(0);
	
		
	if(!m_ClientPki->GetCaStatus(status))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}
	
	switch(status)
	{
		case STATUS_CA_OK: //The CA is already initialized
			menubar->GetMenu(2)->Enable(IDM_CREATE_CHILD_CA, false);
			menubar->GetMenu(2)->Enable(IDM_IMPORT_CA_CERT, false);
			menubar->GetMenu(2)->Enable(IDM_CREATE_ROOT_CA, false);
			menubar->GetMenu(2)->Enable(IDM_IMPORT_CA_P12, false);
			menubar->GetMenu(2)->Enable(IDM_SHOW_CA_CERT, true);
			if(!m_ClientPki->GetCaP7b(m_CaP7b))
			{
				m_DlgMsg->wHide();
				HandleError(m_ClientPki->GetError(), this);
				return false;
			}
			break;
		case STATUS_CA_NO_CERT:
			menubar->GetMenu(2)->Enable(IDM_SHOW_CA_CERT, false);
			menubar->GetMenu(2)->Enable(IDM_IMPORT_CA_CERT, false);
			break;
		case STATUS_CA_AWAITING_CERT:
			menubar->GetMenu(2)->Enable(IDM_CREATE_CHILD_CA, false);
			menubar->GetMenu(2)->Enable(IDM_CREATE_ROOT_CA, false);
			menubar->GetMenu(2)->Enable(IDM_SHOW_CA_CERT, false);
			menubar->GetMenu(2)->Enable(IDM_IMPORT_CA_P12, false);
			break;
	}
	m_DlgMsg->wHide();

	wxCommandEvent ee;
	OnViewChanged(ee);

	return true;
}


void DlgCaAdmin::OnSendAdminMail(wxCommandEvent &event)
{
	DlgSendMail Dlg(this, m_ClientPki);
}


void DlgCaAdmin::OnCreateRootCa(wxCommandEvent &event)
{
	DlgCreateRootCA Dlg(this, m_ClientPki);
	if(Dlg.IsOK())
		ReloadConf();
}


void DlgCaAdmin::OnCreateChildCa(wxCommandEvent &event)
{
	DlgCreateChildCA Dlg(this, m_EntityName, m_ClientPki);
	if(Dlg.IsOK())
		ReloadConf();
}

void DlgCaAdmin::OnImportChildCa(wxCommandEvent &event)
{
	PKCS7 * certs;
	PKI_P7B p7b;

	wxFileDialog dialog(this, _("Open PKCS#7"), "", "", _("PKCS#7 File (*.p7b)|*.p7b"), wxOPEN);
	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	}	
	BIO * rd = BIO_new_file(dialog.GetPath(), "rb");
	if(!rd)
	{
		HandleErrorResult(NULL, this, 0);
		return;
	}
	
	certs = d2i_PKCS7_bio(rd, NULL);
	if(!certs)
	{
		HandleErrorResult(NULL, this, 0);
		BIO_free_all(rd);
		return;
	}
	BIO_free_all(rd);

	if(!p7b.Load(certs))
	{
		HandleErrorResult(NULL, this, 0);
		PKCS7_free(certs);
	}
	PKCS7_free(certs);



	m_DlgMsg->wShow(_("Importing Certificate..."));

	if(!m_ClientPki->ImportChildCaCert(p7b))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	ReloadConf();
}

void DlgCaAdmin::OnImportCaP12(wxCommandEvent &event)
{
	PKI_PKCS12 p12;
	mString password;

	wxFileDialog dialog(this, _("Open PKCS#12"), "", "", _("PKCS#12 File (*.p12)|*.p12"), wxOPEN);
	if (dialog.ShowModal() != wxID_OK)
	{
		return;
	}
	if(!p12.LoadFromFile(dialog.GetPath(), NULL))
	{
		HandleErrorResult(NULL, this, 0);
		return;
	}
	DlgGetPassword Dlg("PKCS#12 Password", this);
	if(!Dlg.IsOK())
		return;
	password = Dlg.GetPassword();
	if(!password.size())
		return;

	m_DlgMsg->wShow(_("Importing PKCS#12..."));

	if(!m_ClientPki->ImportCaP12(p12, password.c_str()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	ReloadConf();
}

void DlgCaAdmin::ShowNextEntries(int NumEntries)
{
	CurrentPosition = CurrentPosition + NumEntries;

	switch(EntryType)
	{
		case CA_OBJECTS_VIEW_CRL:
			LoadCRL(NumEntries);
			break;
		case CA_OBJECTS_VIEW_CERT:
			LoadCERT(NumEntries);
			break;
		default:
			return;
	}
	if(!m_listCtrl->GetItemCount()) ShowPrevEntries(NumEntries);
}

void DlgCaAdmin::ShowPrevEntries(int NumEntries)
{
	CurrentPosition = CurrentPosition - NumEntries;
	if(CurrentPosition < 0) CurrentPosition = 0;

	switch(EntryType)
	{
		case CA_OBJECTS_VIEW_CRL:
			LoadCRL(NumEntries);
			break;
		case CA_OBJECTS_VIEW_CERT:
			LoadCERT(NumEntries);
			break;
		default:
			return;
			break;
	}
}

void DlgCaAdmin::RefreshDisplay(int NumEntries)
{
	switch(EntryType)
	{
		case CA_OBJECTS_VIEW_CERT:
			LoadCERT(NumEntries);
			break;
		case CA_OBJECTS_VIEW_CRL:
			LoadCRL(NumEntries);
			break;
		default:
			return;
	}
}

bool DlgCaAdmin::LoadCERT(int NumEntries)
{
	m_DlgMsg->wShow(_("Loading Certificates List..."));

	wxString Page;
	Page.sprintf("%ld", (CurrentPosition / NumEntries) + 1);
	((wxStaticText *)FindWindow(IDC_PAGE))->SetLabel(Page);

	if(!m_ClientPki->EnumCERT(CurrentPosition, NumEntries, currCertStatus, CertsList))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}

	PrintCERT();
	m_DlgMsg->wHide();
	return true;
}

bool DlgCaAdmin::LoadCRL(int NumEntries)
{
	m_DlgMsg->wShow(_("Loading CRL(s) List..."));

	wxString Page;
	Page.sprintf("%ld", (CurrentPosition / NumEntries) + 1);
	((wxStaticText *)FindWindow(IDC_PAGE))->SetLabel(Page);

	if(!m_ClientPki->EnumCrls(CurrentPosition, NumEntries, CrlsList))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return false;
	}
	PrintCRL();

	m_DlgMsg->wHide();
	return true;
}

bool DlgCaAdmin::DisplayCRL(int NumEntries)
{
	CurrentPosition=0;
	EntryType = CA_OBJECTS_VIEW_CRL;

	m_listCtrl->ClearAll();

	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Begin Date"), wxLIST_ALIGN_LEFT, 150);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("End Date"), wxLIST_ALIGN_LEFT, 150);

	return LoadCRL(NumEntries);
}

bool DlgCaAdmin::DisplayCERT(CERT_STATE State, int NumEntries)
{
	CurrentPosition=0;
	EntryType = CA_OBJECTS_VIEW_CERT;
	currCertStatus = State;

	m_listCtrl->ClearAll();

	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Serial"), wxLIST_ALIGN_LEFT, 95);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("DN"), wxLIST_ALIGN_LEFT, 200);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Begin Date"), wxLIST_ALIGN_LEFT, 150);
	m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("End Date"), wxLIST_ALIGN_LEFT, 150);

	if(State == CERT_STATE_REVOKED)
		m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Revocation Date"), wxLIST_ALIGN_LEFT, 150);
	else if(State == CERT_STATE_SUSPENDED)
		m_listCtrl->InsertColumn(m_listCtrl->GetColumnCount(), _("Suspension Date"), wxLIST_ALIGN_LEFT, 150);


	return LoadCERT(NumEntries);
}

void DlgCaAdmin::OnViewChanged(wxCommandEvent &event)
{
	wxStringClientData * ClientDatas;
	wxString strCmb;
	wxComboBox * view_combo = (wxComboBox *)FindWindow(IDC_VIEW_TYPE);
	if(!view_combo) return;


	if(!view_combo->IsEnabled())
	{
		view_combo->SetSelection(LastSelectedEntry);
		return;
	}

	view_combo->Enable(false);

	long Type;
	int SelectedItem;

	SelectedItem = view_combo->GetSelection();
	if(SelectedItem == -1 || SelectedItem == LastSelectedEntry)
	{
		view_combo->Enable(true);
		return;
	}
	LastSelectedEntry = SelectedItem;

	ClientDatas = (wxStringClientData*)view_combo->GetClientData(SelectedItem);
	ClientDatas->GetData().ToLong(&Type);

	m_listCtrl->DeleteAllItems();

	CertsList.clear();
	CrlsList.clear();

	switch(Type)
	{
		case CA_OBJECTS_VIEW_CERT:
			DisplayCERT(CERT_STATE_VALID, ((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
			break;
		
		case CA_OBJECTS_VIEW_REV:
			DisplayCERT(CERT_STATE_REVOKED, ((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
			break;

		case CA_OBJECTS_VIEW_SUSP:
			DisplayCERT(CERT_STATE_SUSPENDED, ((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
			break;

		case CA_OBJECTS_VIEW_CRL:
			DisplayCRL(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
			break;
		default:
			return;
	}
    view_combo->Enable(true);
}

void DlgCaAdmin::OnShowNextEntries(wxCommandEvent &event)
{
	ShowNextEntries(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgCaAdmin::OnShowPrevEntries(wxCommandEvent &event)
{
	ShowPrevEntries(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgCaAdmin::OnRefreshEntries(wxCommandEvent &event)
{
	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}


void DlgCaAdmin::PrintCRL()
{
	size_t i;
	m_listCtrl->DeleteAllItems();


	for(i=0; i<CrlsList.size(); i++)
	{
		m_listCtrl->InsertItem(i , FormatTime(CrlsList[i].GetStartTime()), CrlIconId);
		m_listCtrl->SetItemData(i, (long)i);

		m_listCtrl->SetItem(i, 1, FormatTime(CrlsList[i].GetEndTime()));
	}
}

void DlgCaAdmin::PrintCERT()
{
	size_t i;
	wxString tmpDatas;
	int CurrCol;
	int IconId;

	m_listCtrl->DeleteAllItems();
	
	for(i=0; i<CertsList.size(); i++)
	{
		CurrCol = 0;

		switch(CertsList[i].get_state())
		{
			case CERT_STATE_VALID:
				IconId = CertIconIdAct;
				break;
			case CERT_STATE_REVOKED:
			case CERT_STATE_SUSPENDED:
				IconId = CertIconIdRev;
				break;
			default:
				IconId = CertIconIdAct;
				break;
		}

		tmpDatas.sprintf("%ld", CertsList[i].get_serial());
		m_listCtrl->InsertItem(i , tmpDatas, IconId);
		m_listCtrl->SetItemData(i, (long)i);
		CurrCol++;

		m_listCtrl->SetItem(i, CurrCol, CertsList[i].get_dn().c_str());
		CurrCol++;

		m_listCtrl->SetItem(i, CurrCol, FormatTime(CertsList[i].get_beginDate()));
		CurrCol ++;

		m_listCtrl->SetItem(i, CurrCol, FormatTime(CertsList[i].get_endDate()));
		CurrCol ++;

		if(currCertStatus == CERT_STATE_REVOKED)
		{
			m_listCtrl->SetItem(i, CurrCol, FormatTime(CertsList[i].get_revDate()));
			CurrCol ++;
		}
		else if(currCertStatus == CERT_STATE_SUSPENDED)
		{
			m_listCtrl->SetItem(i, CurrCol, FormatTime(CertsList[i].get_suspDate()));
			CurrCol ++;
		}
	}
}

void DlgCaAdmin::OnImportCSR(wxCommandEvent &event)
{
	DlgPublishCSR Dlg(this, m_ClientPki);
	if(Dlg.IsOk())
		RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgCaAdmin::OnRevokeCert(wxCommandEvent &event)
{
	size_t mCertIndex;
	int SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;

	if(EntryType != CA_OBJECTS_VIEW_CERT) return;

	mCertIndex = (size_t)m_listCtrl->GetItemData(SelectedItem);;

	if(DisplayMessage(this, _("Are you sure, you want to revoke this certificate ?"), wxYES_NO) == wxNO) return;
	
	m_DlgMsg->wShow(_("Revoking Certificate..."));
	if(!m_ClientPki->RevokeCert(CertsList[mCertIndex].get_serial()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgCaAdmin::OnSuspendCert(wxCommandEvent &event)
{
	size_t mCertIndex;
	int SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;

	if(EntryType != CA_OBJECTS_VIEW_CERT) return;

	mCertIndex = (size_t)m_listCtrl->GetItemData(SelectedItem);;

	if(DisplayMessage(this, _("Are you sure, you want to suspend this certificate ?"), wxYES_NO) == wxNO) return;
	
	m_DlgMsg->wShow(_("Suspending Certificate..."));
	if(!m_ClientPki->SuspendCert(CertsList[mCertIndex].get_serial()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgCaAdmin::OnUnsuspendCert(wxCommandEvent &event)
{
	size_t mCertIndex;
	int SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;

	if(EntryType != CA_OBJECTS_VIEW_CERT) return;

	mCertIndex = (size_t)m_listCtrl->GetItemData(SelectedItem);;

	if(DisplayMessage(this, _("Are you sure, you want to unsuspend this certificate ?"), wxYES_NO) == wxNO) return;
	
	m_DlgMsg->wShow(_("Unsuspending Certificate..."));
	if(!m_ClientPki->UnsuspendCert(CertsList[mCertIndex].get_serial()))
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();

	RefreshDisplay(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}

void DlgCaAdmin::OnShowDetails(wxCommandEvent &event)
{
	int SelectedItem;
	size_t index;
	DlgCrlProperties * Dlg1;
	DlgCertProperties * Dlg2;

	SelectedItem = m_listCtrl->GetNextItem(-1, wxLIST_NEXT_ALL, wxLIST_STATE_SELECTED);
	if(SelectedItem == -1) return;

	index = (size_t)m_listCtrl->GetItemData(SelectedItem);

	switch(EntryType)
	{
		case CA_OBJECTS_VIEW_CRL:
			Dlg1 = new DlgCrlProperties(this, CrlsList[index]);
			if(Dlg1) delete Dlg1;
			break;
		case CA_OBJECTS_VIEW_CERT:
			Dlg2 = new DlgCertProperties(this, CertsList[index].get_cert());
			if(Dlg2) delete Dlg2;
			break;
		default:
			return;
			break;
	}
}

void DlgCaAdmin::OnGenerateCRL(wxCommandEvent &event)
{
	if(DisplayMessage(this, _("Are you sure, you want to generate a new CRL ?"), wxYES_NO) == wxNO) return;

	m_DlgMsg->wShow(_("Generating CRL..."));
	if(!m_ClientPki->GenerateCRL())
	{
		m_DlgMsg->wHide();
		HandleError(m_ClientPki->GetError(), this);
		return;
	}
	m_DlgMsg->wHide();
	DisplayCRL(((wxSpinCtrl*)FindWindow(IDC_NUM_ENTRIES))->GetValue());
}


void DlgCaAdmin::OnViewCaP7b(wxCommandEvent &event)
{
	DlgP7bProperties Dlg(this, m_CaP7b);
}

void DlgCaAdmin::OnShowCertExts(wxCommandEvent &event)
{
	DlgExtension Dlg(this, m_EntityConf.get_body().get_caConf().CA_CONF_PTR.get_exts());
}

void DlgCaAdmin::OnShowCrlExts(wxCommandEvent &event)
{
	DlgExtension Dlg(this, m_EntityConf.get_body().get_caConf().CA_CONF_PTR.get_crlexts());
}

