/* This file is part of Netwib.
*/

/*-------------------------------------------------------------*/
#include "unix/devices_ioctl.c"

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_conf_binnetstatpn(netwib_priv_confwork *pcw)
{
  netwib_io *piofile, *pioline;
  netwib_priv_confwork_arpcache *pca;
  netwib_buf buf;
  netwib_bool pcaset;
  netwib_err ret;

  netwib_er(netwib_buf_init_ext_text("/bin/netstat -pn", &buf));
  ret = netwib_io_init_exec(&buf, NETWIB_IO_WAYTYPE_READ, NETWIB_FALSE,
                            NULL, NULL, &piofile);
  if (ret != NETWIB_ERR_OK) {
    if (ret == NETWIB_ERR_NOTFOUND) {
      ret = NETWIB_ERR_LONOTSUPPORTED;
    }
    return(ret);
  }
  netwib_er(netwib_io_init_data_line(&pioline));
  netwib_er(netwib_io_plug_read(pioline, piofile));

  netwib_er(netwib_buf_init_mallocdefault(&buf));

  /* prepare for error handling */
  pcaset = NETWIB_FALSE;
  ret = NETWIB_ERR_OK;

  /* read each line */
  while (NETWIB_TRUE) {
    netwib__buf_reinit(&buf);
    ret = netwib_io_read(pioline, &buf);
    if (ret != NETWIB_ERR_OK) {
      if (ret == NETWIB_ERR_DATAEND) ret = NETWIB_ERR_OK;
      break;
    }
    netwib_eg(netwib_priv_confwork_arpcache_init(&pca));
    pcaset = NETWIB_TRUE;
    ret = netwib_buf_decode_fmt(&buf, "%{buf} %{ip} %{*;ip} %{*;buf:glob} %{eth}", &pca->device, &pca->ip, &pca->eth);
    if (ret == NETWIB_ERR_NOTCONVERTED) {
      netwib__buf_reinit(&pca->device);
      ret = netwib_buf_decode_fmt(&buf, "%{buf} %{eth} %{*;buf} %{*;buf} %{ip}", &pca->device, &pca->eth, &pca->ip);
    }
    if (ret == NETWIB_ERR_NOTCONVERTED) {
      /* ignore unrecognized lines */
      netwib_eg(netwib_priv_confwork_arpcache_close(&pca));
      pcaset = NETWIB_FALSE;
      continue;
    } else if (ret != NETWIB_ERR_OK) {
      break;
    }
    /* ignore 224.x and eth=33:33:x */
    if ((pca->ip.iptype == NETWIB_IPTYPE_IP4 &&
         (pca->ip.ipvalue.ip4&0xFF000000u) == 0xE0000000u) ||
        (pca->ip.iptype == NETWIB_IPTYPE_IP6 &&
         pca->eth.b[0] == 0x33 && pca->eth.b[1] == 0x33)) {
      netwib_eg(netwib_priv_confwork_arpcache_close(&pca));
      pcaset = NETWIB_FALSE;
      continue;
    }
    netwib_eg(netwib_priv_confwork_arpcache_add(pcw, pca));
    pcaset = NETWIB_FALSE;
  }

  /* clean and leave */
 netwib_gotolabel:
  netwib_er(netwib_buf_close(&buf));
  netwib_er(netwib_io_close(&pioline));
  if (pcaset) {
    netwib_er(netwib_priv_confwork_arpcache_close(&pca));
  }
  return(ret);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_conf_decodedst(netwib_constbuf *pbufdst,
                                             netwib_bool ipv6wastried,
                                             netwib_ip *pdst,
                                             netwib_uint32 *pprefix)
{
  netwib_buf buf;
  netwib_cmp cmp;
  netwib_err ret;

  /* IP4 : 1.2.3.4 or default
     IP6 : 1::2, 1::/prefix or default
  */

  netwib_er(netwib_buf_init_ext_text("default", &buf));
  netwib_er(netwib_buf_cmp(pbufdst, &buf, &cmp));
  if (cmp == NETWIB_CMP_EQ) {
    if (ipv6wastried) {
      netwib_er(netwib_ip_init_ip6_fields(0,0,0,0, pdst));
      *pprefix = 0;
    } else {
      netwib_er(netwib_ip_init_ip4_fields(0,0,0,0, pdst));
    }
    return(NETWIB_ERR_OK);
  }

  if (ipv6wastried) {
    ret = netwib_buf_decode_fmt(pbufdst, "%{ip}/%{uint32}", pdst, pprefix);
    if (ret != NETWIB_ERR_NOTCONVERTED) {
      return(ret);
    }
  }

  netwib_er(netwib_ip_init_buf_best(pbufdst, pdst));
  if (ipv6wastried) {
    *pprefix = 128;
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_conf_binnetstatrvn(netwib_priv_confwork *pcw)
{
  netwib_io *piofile, *pioline;
  netwib_priv_confwork_routes *pcr;
  netwib_buf buf, bufdst, bufflags;
  netwib_string strflags, pc;
  netwib_bool pcrset, ipv6wastried;
  netwib_err ret;

  netwib_er(netwib_buf_init_ext_text("/bin/netstat -rvn", &buf));
  ret = netwib_io_init_exec(&buf, NETWIB_IO_WAYTYPE_READ, NETWIB_FALSE,
                            NULL, NULL, &piofile);
  if (ret != NETWIB_ERR_OK) {
    if (ret == NETWIB_ERR_NOTFOUND) {
      ret = NETWIB_ERR_LONOTSUPPORTED;
    }
    return(ret);
  }
  netwib_er(netwib_io_init_data_line(&pioline));
  netwib_er(netwib_io_plug_read(pioline, piofile));

  netwib_er(netwib_buf_init_mallocdefault(&buf));
  netwib_er(netwib_buf_init_mallocdefault(&bufdst));
  netwib_er(netwib_buf_init_mallocdefault(&bufflags));

  /* prepare for error handling */
  pcrset = NETWIB_FALSE;
  ret = NETWIB_ERR_OK;

  /* read each line */
  while (NETWIB_TRUE) {
    netwib__buf_reinit(&buf);
    ret = netwib_io_read(pioline, &buf);
    if (ret != NETWIB_ERR_OK) {
      if (ret == NETWIB_ERR_DATAEND) ret = NETWIB_ERR_OK;
      break;
    }
    netwib__buf_reinit(&bufdst);
    netwib__buf_reinit(&bufflags);
    netwib_eg(netwib_priv_confwork_routes_init(&pcr));
    pcrset = NETWIB_TRUE;
    ret = netwib_buf_decode_fmt(&buf, "%{buf} %{ip} %{ip} %{buf} %{*;uint32}* %{*;uint32} %{*;uint32} %{buf} %{*;uint32} %{*;uint32}",
                                &bufdst, &pcr->mask, &pcr->gw, &pcr->device, &bufflags);
    if (ret == NETWIB_ERR_NOTCONVERTED) {
      netwib__buf_reinit(&bufdst);
      netwib__buf_reinit(&bufflags);
      netwib__buf_reinit(&pcr->device);
      /* try without device */
      ret = netwib_buf_decode_fmt(&buf, "%{buf} %{ip} %{ip} %{*;uint32}* %{*;uint32} %{*;uint32} %{buf} %{*;uint32} %{*;uint32}",
                                  &bufdst, &pcr->mask, &pcr->gw, &bufflags);
    }
    ipv6wastried = NETWIB_FALSE;
    if (ret == NETWIB_ERR_NOTCONVERTED) {
      ipv6wastried = NETWIB_TRUE;
      netwib__buf_reinit(&bufdst);
      netwib__buf_reinit(&bufflags);
      netwib__buf_reinit(&pcr->device);
      ret = netwib_buf_decode_fmt(&buf, "%{buf} %{ip} %{buf} %{*;uint32}* %{*;uint32} %{*;uint32} %{buf} %{*;uint32} %{*;uint32}",
                                  &bufdst, &pcr->gw, &pcr->device, &bufflags);
    }
    if (ret == NETWIB_ERR_NOTCONVERTED) {
      netwib__buf_reinit(&bufdst);
      netwib__buf_reinit(&bufflags);
      netwib__buf_reinit(&pcr->device);
      ret = netwib_buf_decode_fmt(&buf, "%{buf} %{ip} %{*;uint32}* %{*;uint32} %{*;uint32} %{buf} %{*;uint32} %{*;uint32}",
                                  &bufdst, &pcr->gw, &bufflags);
    }
    if (ret == NETWIB_ERR_NOTCONVERTED) {
      ret = NETWIB_ERR_OK;
      goto trynextentry;
    } else if (ret != NETWIB_ERR_OK) {
      break;
    }
    /* check if flags contains 'G' for Gateway */
    netwib_eg(netwib_buf_ref_string(&bufflags, &strflags));
    pc = netwib_c_strchr(strflags, 'G');
    if (pc != NULL) {
      pcr->gwset = NETWIB_TRUE;
    } else {
      pcr->src = pcr->gw;
      pcr->srcset = NETWIB_TRUE;
    }
    /* convert bufdst */
    ret = netwib_priv_conf_decodedst(&bufdst, ipv6wastried,
                                     &pcr->dst, &pcr->prefix);
    if (ret == NETWIB_ERR_NOTCONVERTED) {
      ret = NETWIB_ERR_OK;
      goto trynextentry;
    } else if (ret != NETWIB_ERR_OK) {
      break;
    }
    /* there is no metric : every route has 1 */
    pcr->metric = 1;
    /* ignore 224.x and ff00:: */
    if ((pcr->dst.iptype == NETWIB_IPTYPE_IP4 &&
         (pcr->dst.ipvalue.ip4&0xFF000000u) == 0xE0000000u) ||
        (pcr->dst.iptype == NETWIB_IPTYPE_IP6 &&
         pcr->dst.ipvalue.ip6.b[0] == 0xFF &&
         pcr->dst.ipvalue.ip6.b[1] == 0x00 &&
         pcr->dst.ipvalue.ip6.b[2] == 0x00 /* stop here */)) {
      goto trynextentry;
    }
    /* ignore default route with fe80 source */
    if (pcr->dst.iptype == NETWIB_IPTYPE_IP6 &&
        pcr->prefix == 0 &&
        pcr->srcset && !pcr->gwset &&
        pcr->src.ipvalue.ip6.b[0] == 0xFE &&
        pcr->src.ipvalue.ip6.b[1] == 0x80 &&
         pcr->dst.ipvalue.ip6.b[2] == 0x00 /* stop here */) {
      goto trynextentry;
    }
    /* add it */
    netwib_eg(netwib_priv_confwork_routes_add(pcw, pcr));
    pcrset = NETWIB_FALSE;
  trynextentry:
    if (pcrset) {
      netwib_eg(netwib_priv_confwork_routes_close(&pcr));
      pcrset = NETWIB_FALSE;
    }
  }

  /* clean and leave */
 netwib_gotolabel:
  netwib_er(netwib_buf_close(&bufflags));
  netwib_er(netwib_buf_close(&bufdst));
  netwib_er(netwib_buf_close(&buf));
  netwib_er(netwib_io_close(&pioline));
  if (pcrset) {
    netwib_er(netwib_priv_confwork_routes_close(&pcr));
  }
  return(ret);
}

/*-------------------------------------------------------------*/
static netwib_err netwib_priv_confwork_obtain_sys(netwib_priv_confwork *pcw)
{
  netwib_bool ip6supported;

  netwib_er(netwib_priv_ip_ip6_supported(&ip6supported));

  netwib_er(netwib_priv_conf_devices_ioctl(pcw));
  netwib_er(netwib_priv_confwork_debug(pcw, "After devices_ioctl"));

  if (ip6supported) {
    /* this function only adds IPv6 information */
    netwib_er(netwib_priv_conf_devices_ioctl6(pcw));
    netwib_er(netwib_priv_confwork_debug(pcw, "After devices_ioctl6"));
  }

  /* arp table */
  netwib_er(netwib_priv_conf_binnetstatpn(pcw));
  netwib_er(netwib_priv_confwork_debug(pcw, "After binnetstatpn"));

  /* routes */
  netwib_er(netwib_priv_conf_binnetstatrvn(pcw));
  netwib_er(netwib_priv_confwork_debug(pcw, "After binnetstatrvn"));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_priv_confwork_obtain_arpcache(netwib_priv_confwork *pcw)
{

  netwib_er(netwib_priv_conf_binnetstatpn(pcw));

  return(NETWIB_ERR_OK);
}
