/*
 * File: listen.c
 * Author: Brent Hendricks
 * Project: NetSpades
 * Date: 5/22/98
 *
 * This file contains the code for the portion of NetSpades which
 * listens to the port.  First it forks off the taunt server process.
 * Then, each time four players have registered, it forks off a process
 * to play a game.
 *
 * Copyright (C) 1998 Brent Hendricks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <config.h>     /* Site specific config */

#include <stdlib.h>
#include <errno.h>
#include <unistd.h>   
#include <stdio.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <sys/wait.h>
#include <netinet/in.h>

#include <engine.h>
#include <taunt.h>
#include <socketfunc.h>

#define basename(path) (strrchr(path,'/')==NULL) ? path : strrchr(path, '/')+1

/* Global game information variable */
gameInfo_t gameInfo;

/* Arry of game pids indexed by game number*/
int gamePids[MAX_GAME];
int tauntPid;

/* Game Availability flag */
sig_atomic_t gamesAvail = MAX_GAME;

/* Spades server socket and pipe to TauntServer(tm) */
int sock, pipe_to_taunt[2]; 

/* Log output flag*/
int log = 0;

/* Signal Handlers*/
void KillServer( int );
void GameOver( int );
void BrokenPipe( int );

int ReadOptions( void );

int main( int argc, char *argv[] ) {

  int i, j;                      /* counters */
  int spadePort = SPADEPORT;
  int tauntPort = TAUNTPORT;
  int tmpsock;
  int option;
  int status;
  struct sockaddr_in clients[4], tmpclient;
  size_t size=sizeof(clients[0]);

  
  /* Parse cmd-line options */
  opterr=0;
  while( (option=getopt( argc, argv, "Vlp:t:")) != -1 ) {
    switch( option ) {
    case 'l': log=1; break;
    case 'p': spadePort = atoi(optarg); break;
    case 't': tauntPort = atoi(optarg); break;
    case 'V':
      printf("NetSpades Server Version %s\n",VERSION);
      return(0);
      break;
    default:
      fprintf( stderr, "Usage: %s [-V] [-l] [-p main_port] [-t taunt_port]\n", basename(argv[0]));
      return(-1);
      break;
    }
  }

  
  /* Set up signal handlers */
  signal( SIGINT, KillServer );
  signal( SIGPIPE, BrokenPipe );
  signal( SIGCHLD, GameOver );
  

  /* Create pipe to TauntServer and fork its process*/
  pipe(pipe_to_taunt);
  if ( (tauntPid = fork()) == 0) {
    tauntServer( pipe_to_taunt, tauntPort );
    exit(0);
  }


  /* Initialize pids and sockets to non-value */
  for( i=0; i < MAX_GAME; i++) {
    gamePids[i] = PID_INVALID;
    gameInfo.playerSock[i] = SOCK_INVALID;
  }
  
  /* Initialize first game info */
  gameInfo.gameNum = gameInfo.gamePid = PID_INVALID;
  gameInfo.opt.bitOpt = MSK_NILS;
  gameInfo.opt.endGame = END_GAME;
  gameInfo.opt.minBid = MIN_BID;
  
  /* Create SERVER socket on spadePort */
  sock = makesocket( spadePort, SERVER, NULL );

  /* Check for socket errors */
  if( sock < 0 ) {
    fprintf( stderr, "[%d]: %s\n", getpid(), errorMsg[ sock+ERR_OFFSET ] );
    exit(1);
  }

  
  /* Start accepting connections */
  if (listen (sock, 4) < 0) {
    fprintf( stderr, "[%d]: Error listening to socket\n", getpid() );
    close( sock );
    exit (1);
  }

  
  /* Main port-listener loop */
  while( 1 ) {

    while(!gamesAvail) {
      /* If there is an incoming connection */
      if( input_timeout(sock, 1 ) == 1 ) {
	tmpsock = accept( sock, (struct sockaddr*)&tmpclient, &size);
	if( tmpsock == -1 ) {
	  perror("Spades server accepting: ");
	}
	else if( tmpsock > 0 ) {
	  if( writeint( tmpsock, -1 ) < 0 ) {
	    fprintf( stderr, "[%d]: Error writing to socket\n", getpid() );
	    close( tmpsock );
	  }
	  else if( log ) {
	    printf( "[%d]: No free games.  Couldn't register from %s\n",
		   getpid(), inet_ntoa( tmpclient.sin_addr ));
	  }
	}
      } /* if( input_timeout ) */
    } /* while(!gamesAvail) */

    /* Find new gameNum */
    while( gamePids[ ++gameInfo.gameNum ] > 0 ) {}
    
    /* Accept 4 connections and then start a game*/
    i = 0;
    while( i < 4 ) {

      if( gameInfo.playerSock[i] == SOCK_COMP ) { 
	if( log ) {
	  printf( "[%d]: Computer player %d registered \n", getpid(), i );
	}
	i++;
	continue;
      }
      gameInfo.playerSock[i] = accept( sock, (struct sockaddr*)&(clients[i]),
				      &size);
      
      /* Check for errors */
      if( gameInfo.playerSock[i] < 0 )
	  fprintf( stderr, "[%d]: Error accepting on socket\n", getpid() );
      /* Send player number */
      else if( writeint(gameInfo.playerSock[i], i ) < 0 ) {
	fprintf( stderr, "[%d]: Error writing to socket\n", getpid() );
	close( gameInfo.playerSock[i] );
      }
      /* Read player name */
      else if( readstring( gameInfo.playerSock[i], &(gameInfo.players[i])) <= 0 ) {
	fprintf( stderr, "[%d]: Error reading name from socket\n", getpid() );
	close( gameInfo.playerSock[i] );
      }
      /* Read client pid */
      else if( readint( gameInfo.playerSock[i], &(gameInfo.clientPids[i]) ) <= 0 ) {
	fprintf( stderr, "[%d]: Error reading pid from socket\n", getpid() );
	close( gameInfo.playerSock[i] );
      }
      /* Read options from first player */
      else if( (i == 0) && (gameInfo.clientPids[0] == PID_OPTION) && (ReadOptions() < 0) ) {
	fprintf( stderr, "[%d]: Error reading options\n", getpid() );
	close( gameInfo.playerSock[i] );
      }
      else {
	if( log ) {
	  printf( "[%d]: Registered %s on %s with pid %d as player %d\n",
		 getpid(),
		 gameInfo.players[i],
		 inet_ntoa( clients[i].sin_addr  ),
		 gameInfo.clientPids[i], i );
	}
	i++;
      }
    } /* while( i < 4 ) */
    
    /* One less available game */
    gamesAvail--;
      
    /*Fork a process to run game */ 
    if ( ( gameInfo.gamePid = fork() ) == 0) {
      gameInfo.gamePid = getpid();
      
      /* Send game data to TauntServer */
      writeint( pipe_to_taunt[1], gameInfo.gameNum );
      writeint( pipe_to_taunt[1], gameInfo.gamePid);
      for( j=0; j<4; j++) {
	writeint( pipe_to_taunt[1], gameInfo.clientPids[j] );
	writestring( pipe_to_taunt[1], gameInfo.players[j] );
      }
      engine();
      exit(0);
    }
    else { /* We're still the listener process */
      
      if( log )
	  printf( "[%d]: Forked new game %d with pid %d\n", getpid(),
		 gameInfo.gameNum, gameInfo.gamePid);
      
      gamePids[ gameInfo.gameNum ] = gameInfo.gamePid;
      
      /* Reset key gameInfo struct members */
      gameInfo.gameNum = -1;
      gameInfo.gamePid = PID_INVALID;
      gameInfo.opt.bitOpt = MSK_NILS;
      gameInfo.opt.endGame = END_GAME;
      gameInfo.opt.minBid = MIN_BID;
      for( j=0; j<4; j++) {
	if( gameInfo.playerSock[j] != SOCK_COMP ) {
	  free( gameInfo.players[j] );
	  close( gameInfo.playerSock[j] );
	}
	gameInfo.playerSock[j] = SOCK_INVALID;
      }
    } /* else we're parent proces */
  } /* while( 1 ) */

  close( sock );
}


int ReadOptions( void ) {
  
  int status = -1;
  
  if( log )
      printf( "[%d]: Reading options from player 0\n", getpid() );

  if( readint(gameInfo.playerSock[0], &(gameInfo.opt.bitOpt) ) < 0 ) {
    fprintf( stderr, "[%d]: Error reading bitOpt\n", getpid() );
  } else if( readint(gameInfo.playerSock[0], &(gameInfo.opt.endGame) ) < 0 ) { 
    fprintf( stderr, "[%d]: Error reading endGame\n", getpid() );
  } else if( readint(gameInfo.playerSock[0], &(gameInfo.opt.minBid) ) < 0 ) { 
    fprintf( stderr, "[%d]: Error reading minBid\n", getpid() );
  } else if( readint( gameInfo.playerSock[0], &(gameInfo.clientPids[0])) < 0) {
    fprintf( stderr, "[%d]: Error reading pid from socket\n", getpid() );
  }
  else {
    status = 0;
    if( gameInfo.opt.bitOpt & MSK_COMP_1 ) {
      gameInfo.playerSock[1] = SOCK_COMP;
      gameInfo.players[1] = "Moe";
    }
    if( gameInfo.opt.bitOpt & MSK_COMP_2 ) {
      gameInfo.playerSock[2] = SOCK_COMP;
      gameInfo.players[2] = "Larry";
    }
    if( gameInfo.opt.bitOpt & MSK_COMP_3 ) {
      gameInfo.playerSock[3] = SOCK_COMP;
      gameInfo.players[3] = "Curly";
    }
#ifdef DEBUG
    printf( "[%d]: bitOpt: %d, endGame: %d, minBid: %d\n", getpid(), 
	   gameInfo.opt.bitOpt, gameInfo.opt.endGame, gameInfo.opt.minBid);
#endif
    if( log ) {
      switch( gameInfo.opt.bitOpt & MSK_GAME ) {
      case GAME_SPADES: 
	printf( "[%d]: Game choice is spades\n", getpid() );
	printf( "[%d]: Game ends at %d points\n", getpid(), gameInfo.opt.endGame);
	printf( "[%d]: Minimum bid of %d\n", getpid(), gameInfo.opt.minBid);
	if( (gameInfo.opt.bitOpt & MSK_NILS) ) {
	  printf( "[%d]: Nil bids allowed\n", getpid() );
	}
	else {
	  printf( "[%d]: Nil bids prohibited\n", getpid() );
	}
	if( (gameInfo.opt.bitOpt & MSK_BAGS) ) {
	  printf( "[%d]: Penalty imposed for 10 bags\n", getpid() );
	}
	else {
	  printf( "[%d]: No penalty for bags\n", getpid() );
	}
	break;
      case GAME_HEARTS: 
	printf( "[%d]: Game choice is hearts\n", getpid() );
	break;
      }
      if( gameInfo.opt.bitOpt & MSK_COMP_1 ) {
	printf( "[%d]: Player 1 is a computer player\n", getpid() );
      }
      if( gameInfo.opt.bitOpt & MSK_COMP_2 ) {
	printf( "[%d]: Player 2 is a computer player\n", getpid() );
      }
      if( gameInfo.opt.bitOpt & MSK_COMP_3 ) {
	printf( "[%d]: Player 3 is a computer player\n", getpid() );
      }
    }
  }
  
  return status;
}
  

RETSIGTYPE KillServer( int signum ) {
  signal (signum, SIG_DFL);
  close(sock);
  raise(signum);
}

RETSIGTYPE BrokenPipe( int signum ) {
  signal (signum, SIG_DFL);
  close(sock);
  raise(signum);
}

RETSIGTYPE GameOver( int signum ) {
  int i, pid, status;
  
  while( 1 ) {
    pid = waitpid (WAIT_ANY, &status, WNOHANG);
    if( pid < 0 ) {
#if 0
      perror ("Waitpid");
#endif
      break;
    }
    if( pid == 0 ) {
	break;
    }
    if( pid == tauntPid ) {
      fprintf(stderr, "[%d]: Taunt Server died.\n", getpid() );
      raise(SIGINT);
    }
    else {
      for( i=0; i<MAX_GAME; i++ ) {
	if( gamePids[i] == pid ) {
	  gamePids[i] = -1;
	  gamesAvail++;
	  if( log )
	      printf( "[%d]: Game %d over\n", getpid(), i );
	  writeint( pipe_to_taunt[1], i );
	  writeint( pipe_to_taunt[1], -1 );
	}
      }
    }
  }
}


/* This function accepts a file descriptor and a time in seconds.  If there is
 * data to read after the specified time, it returns 1.  If there is no data, 
 * it returns 0 */
int input_timeout (int filedes, unsigned int seconds) {

  fd_set set;
  struct timeval timeout;
     
  /* Initialize the file descriptor set. */
  FD_ZERO (&set);
  FD_SET (filedes, &set);
     
  /* Initialize the timeout data structure. */
  timeout.tv_sec = seconds;
  timeout.tv_usec = 0;
  /* `select' returns 0 if timeout, 1 if input available, -1 if error. */
  return (select (FD_SETSIZE, &set, NULL, NULL, &timeout));
}
