using Nemerle.Collections;
using Nemerle.Compiler;
using Nemerle.Compiler.Parsetree;
using Nemerle.Text;
using Nemerle.Utility;

[assembly: System.Reflection.AssemblyTitle("Nemerle Evaluation Library")]
[assembly: System.Reflection.AssemblyDescription("Nemerle (http://nemerle.org) Runtime evaluation library")]
[assembly: System.Reflection.AssemblyCompany("University of Wroclaw")]
[assembly: System.Reflection.AssemblyProduct("Nemerle Evaluation Library")]
[assembly: System.Reflection.AssemblyCopyright("Copyright @ University of Wroclaw 2005-2006")]

[assembly: System.Reflection.AssemblyVersion("0.9.3")]
[assembly: System.Runtime.InteropServices.ComVisible (false)]

namespace Nemerle.Evaluation
{
  /**
   * Allows dynamic evaluation of code, with persistent computation 
   * history (retaining computed variables, functions, etc.) between
   * subsequent calls to Evaluator ().Eval ().
   *
   * <seealso cref="Nemerle.Compiler.NemerleCodeProvider">
   *   NemerleCodeProvider
   * </seealso>
   */
  public class Evaluator
  {
    // Evaluated code.
    static mutable code : string;
    // Whether the code compiled successfully.
    mutable static compiled : bool;
    // [(is_mutable, name, type)]
    static mutable locals : list [bool * string * PExpr] = [];
    // The new values returned after evaluation. This 
    // was made a field because it's in a try block.
    mutable static newvals : list [object] = [];
    // A list of namespace aliases and namespaces that are to be opened.
    // [(Some (shortname), longname) or (None (), longname)]
    public static mutable ns : list [option [string] * list [string]] = [];
    // A list of assemblies to reference.
    public static mutable refr : list [string] = [];
    // [value]
    public static mutable vals : list [object] = [];

    public this () {
      this (true);
    }
    
    public this (persistent_mode : bool) {
      Message.InitOutput (System.Console.Out);         

      Options.PersistentLibraries = persistent_mode;
      Options.CompileToMemory = true;
      Options.IgnoreConfusion = true;
      Options.ProgressBar = false;
      Options.Sources = [""];
  
      Passes.ParsingPipeline = fun (_) { [null] };
      Passes.ScanningPipeline = DoTheStuff;

      when (persistent_mode)
        MacroRegistry.register_macro (StagedMacro ());
    }
    
    public static EvaluateExpression ([Nemerle.Assertions.NotNull] code : string) : object 
    {
      def eval = Evaluator (false);
      match (eval.Eval (code)) {
        | [] => null
        | returned => returned.Last [3];
      }
    }
    
    /**
     * Evaluates supplied code in memory and returns computation results.
     * Persistent computation history is maintained between subsequent calls.
     * 
     * <param name="code">
     *   The code to evaluate.
     * </param>
     * <returns>
     *   A sorted list of computed variables and functions with additional 
     *   descriptive information. The tuple members from the list are:
     *   #1 - true if the variable was introduced during the last Eval,
     *        false if it was on the list before.
     *   #2 - true if the variable is mutable, false if not.
     *   #3 - name of the variable.
     *   #4 - value of the variable.
     *   #5 - type of the variable.
     *
     *   The last computed value - the return value of the evaluated code
     *   is returned as a special variable named "it".
     * </returns>
     */
    public Eval (code: string) : list [bool * bool * string * object * PExpr] {
      Evaluator.code = code;

      Passes.LexingPipeline = fun (_) { null };
      Passes.ParsingPipeline = fun (_) { [null] };
      Passes.ScanningPipeline = DoTheStuff;

//    Options.DumpTypedTree = true;
//    Options.AdditionalDebug = true;
      
      // Link ourselves.
      Options.ReferencedLibraries = System.Reflection.Assembly.GetAssembly 
                                    (this.GetType ()).Location :: 
                                    Evaluator.refr;
      
      // unused variable
      WarningOptions.Disable (168);   
      // ignore computed value
      WarningOptions.Disable (10005); 

      // Locals and vals from previous calls to Eval.
      def prevlocals = Evaluator.locals;
      def prevvals = Evaluator.vals;

      try {
        Passes.Run ();

        def ass = Passes.GeneratedAssembly;
        def meth = ass.GetTypes () [0].GetMethod ("Run");
        // And here are the new values (along with the old ones).
        newvals = meth.Invoke (null, null) :> list [object];
        Evaluator.compiled = true
      }
      catch {
        | e => match (e) {
                 | _ is AssertionException
                 | _ is Recovery 
                 | _ is System.ApplicationException => 
                  if (e.Source == "Nemerle.Compiler" || 
                     e.Source == "Nemerle.Evaluation") {
                    Evaluator.locals = prevlocals;
                    Evaluator.newvals = List.Rev (prevvals);
                    Evaluator.compiled = false;
                    unless (e.Message == "Nothing to parse.")
                      System.Console.WriteLine ($"Warning: $e");
                  }
                  else throw;

                 | _ => throw
               }
      }
      
      // Check which variables are new to this call and set the first field
      // in the return tuple - true for variable "it", new ones and those 
      // whose values and/or type has changed; false - for the rest.
      def loop (l1, l2, l3, l4, acc) {
        match ((l1, l2, l3, l4)) {
          | ([], [], [], []) => acc
          | ((mut, name, ty) :: t1, [], h3 :: t3, []) => 
            loop (t1, [], t3, [], (true, mut, name, h3, ty) :: acc)
          | ((mut, name, ty) :: t1, (prevmut, prevname, prevty) :: t2, 
            h3 :: t3, h4 :: t4) =>
            if (name == prevname)
              if ((name == "it" && Evaluator.compiled == true) ||
                 (h3 != null && !(h3.Equals (h4))) ||
                 (h3 == null && h4 != null) ||                
                 mut != prevmut)
                loop (t1, t2, t3, t4, (true, mut, name, h3, ty) :: acc)
              else
                loop (t1, t2, t3, t4, (false, mut, name, h3, ty) :: acc)
            else
              loop (t1, (prevmut, prevname, prevty) :: t2, t3, h4 :: t4,
                   (true, mut, name, h3, ty) :: acc)
          // This should match only when no code has been successfully
          // evaluated yet and we have _no_ variables at all.
          | _ => acc
        }
      }
  
      Evaluator.vals = List.Rev (newvals);
  
      loop (Evaluator.locals, prevlocals, newvals, List.Rev (prevvals), [])
    }
    
    // Does the same that Eval does, but instead of evaluation the result
    // provides a list of possible completion types and members
    mutable static IsCompletionMode : bool = false;
    
    public Complete (code: string) : Nemerle.Completion.CompletionInfo {
      IsCompletionMode = true;
      Evaluator.code = code;
      Message.InitOutput (System.IO.StreamWriter (System.IO.MemoryStream ()));
      mutable completionList : Nemerle.Completion.CompletionInfo = null;
      
      Nemerle.Completion.Engine.LesserInit();
      def lexer = LexerString (code, Location (0, 1, 1));
      
      Passes.Hierarchy = TypesManager();
      Passes.ParsingPipeline = MainParser.Parse;
      Passes.ScanningPipeline = ScanTypeHierarchy.ProcessDeclaration;
      
      def decls = Passes.ParsingPipeline (lexer);
      List.Iter (decls, Passes.ScanningPipeline);
      
      // Link ourselves.
      Options.ReferencedLibraries = System.Reflection.Assembly.GetAssembly 
                                    (this.GetType ()).Location :: 
                                    Evaluator.refr;
      
      // Locals and vals from previous calls to Eval.
      def prevlocals = Evaluator.locals;
      def prevvals = Evaluator.vals;
      
      Passes.Hierarchy.Run ();
      
      try {
        mutable my_method = null;
        
        Passes.Hierarchy.infos.Iter (fun (ti) {
          def members = ti.GetMethods();
          members.Iter ( fun (member : IMethod) {
            when (member.Name == "Main") { 
              my_method = member :> MethodBuilder;
            }
          } );
        } );

        def lexer = LexerCompletion (Evaluator.code + " ", Evaluator.code.Length);
        my_method.GetHeader ().body =
          FunBody.Parsed (MainParser.ParseExpr (my_method.DeclaringType.GlobalEnv, lexer));
        my_method.RunBodyTyper ();
      
        Evaluator.compiled = false
      }
      catch {
        | e => match (e) {
                 | cr is Nemerle.Compiler.CompletionResult => completionList = Nemerle.Completion.Engine.translate_ovpossibility_to_info (cr.Overloads);
                 | _ is AssertionException
                 | _ is Recovery 
                 | _ is System.ApplicationException => 
                  if (e.Source == "Nemerle.Compiler" || 
                     e.Source == "Nemerle.Evaluation") {
                    Evaluator.locals = prevlocals;
                    Evaluator.newvals = List.Rev (prevvals);
                    Evaluator.compiled = false;
                    //System.Console.WriteLine (e);
                  }
                  else throw;

                 | _ => throw
               }
      }
      
      Message.InitOutput (System.Console.Out);
      Passes.ParsingPipeline = fun (_) { [null] };
      Passes.ScanningPipeline = DoTheStuff;

      IsCompletionMode = false;
      completionList
    }
  
    internal static DoTheStuff (_tdecl : Parsetree.TopDeclaration) : void 
    {
      unless (Options.PersistentLibraries)
        MacroRegistry.register_macro (StagedMacro ());
    
      def n = (None (), ["Nemerle", "Core"]) :: 
              (None (), ["Nemerle", "Core", "option"]) :: Evaluator.ns;
      // Open namespaces and set aliases.
      def env = List.FoldLeft (n, GlobalEnv.Empty, 
                              fun (x : option [string] * list [string], 
                                  acc : GlobalEnv) { 
                  match (x [0]) {
                    | None => acc.AddOpenNamespace (x [1], Location.Default)
                    | Some (sname) => acc.AddNamespaceAlias (sname, x [1], 
                                                             Location.Default)
                  }
                });
  
      MacroColorizer.PushNewColor (-1, env);
  
      // Set the class in which we're going to put the evaluated code.
      def cname = Macros.NewSymbol ();
      def tb = GlobalEnv.Core.Define (<[ decl: public class $(cname : name) {
                                               } ]>);
      mutable body = null;
      if (IsCompletionMode) {
        def lexer = LexerCompletion (Evaluator.code, Evaluator.code.Length);
        body = MainParser.ParseExpr (env, lexer);
      }
      else {
        body = MainParser.ParseExpr (env, Evaluator.code, allow_empty = true); 
      }
      match (body) {
        | null
        | <[ ]> => throw System.ApplicationException ("Nothing to parse.")
        | _ => ()
      }
      
      def make_last (last) {
        | <[ def $_ = $_ ]> 
        | <[ mutable $_ = $_ ]> 
        | <[ def .. $_ ]> =>
          <[ $last; $("stagedmacro" : usesite) (()) ]>
        | _ => <[ $("stagedmacro" : usesite) ($last) ]>
      }
      
      // If the code ends with an assignment, append `()'.
      // Put a call to stagedmacro at the end of the evaluated code
      // with the last expression moved to the stagedmacro argument.
      def whole = match (body) {
        | <[ {..$seq } ]> => 
          def (beg, last) = List.DivideLast (seq);
          def last = make_last (last);
          <[ {..$ (beg + [last]) } ]>
          
        | _ => make_last (body)
      }
  
      // Recreate variables defined in previous calls to Eval.
      def inits = List.FoldLeft (Evaluator.locals, [], fun (x, acc) {
        def (mut, name, ty) = x;
        match ((mut, ty)) {
          | (false, <[ System.Object ]>) => 
            <[ def $(name : usesite) = List.Hd (Evaluator.vals) : $ty ]> :: 
            <[ Evaluator.vals = List.Tl (Evaluator.vals) ]> :: acc
          | (false, _) => 
            <[ def $(name : usesite) = List.Hd (Evaluator.vals) :> $ty ]> ::
            <[ Evaluator.vals = List.Tl (Evaluator.vals) ]> :: acc
          | (_, <[ System.Object ]>) => 
            <[ mutable $(name : usesite) = 
            List.Hd (Evaluator.vals) : $ty ]> ::
            <[ Evaluator.vals = List.Tl (Evaluator.vals) ]> :: acc; 
          | _ => 
            <[ mutable $(name : usesite) = 
            List.Hd (Evaluator.vals) :> $ty ]> ::
            <[ Evaluator.vals = List.Tl (Evaluator.vals) ]> :: acc; 
        }
      });
      
      def w = <[ {.. $(inits + [whole]) } ]>;
  
      // PrettyPrint.PrintExpr (None (), w);
      
      tb.Define (<[ decl: public static Run () : list [System.Object] {
                          $w } ]>);
    
      tb.Compile ();
                          
      MacroColorizer.PopColor ();
    }
  
    internal class StagedMacro : IMacro 
    {
      public Run (ctx : Typer, val : list [SyntaxElement]) : PExpr {
        match (val) {
          // Fish out variables/functions and store them in our fields.
          | [SyntaxElement.Expression (expr)] =>
            Evaluator.locals = [];
            def l = ctx.GetLocals ().GetLocals ();
            mutable values = l.Fold ([], fun (n : Name, loc : LocalValue, acc) {
              match (n.ToString ()) {
                | "it" => acc
                | _ when loc.ValKind is LocalValue.Kind.BlockReturn => acc
                | _ => 
                  Evaluator.locals = (loc.IsMutable, loc.Name, 
                                      PrettyPrint.TyVarToParseTree 
                                      (loc.Type)) :: Evaluator.locals;
                  <[ ($(n : name) : System.Object) ]> :: acc
              }
            });
            def exprty = ctx.TypeExpr (expr).MType;
            match (exprty) {
              | MType.Void => ()
              | _ =>
                Evaluator.locals = (false, "it", PrettyPrint.TyVarToParseTree (exprty)) :: Evaluator.locals;
                values ::= <[ $expr : System.Object ]>;
            }
            <[ [..$values] ]>
          | _ => Util.ice ()
        }
      }
  
      // This below is only to satisfy the interface requirements.
      public CallTransform (l : list [PExpr]) : list [SyntaxElement]
      { List.Map (l, fun (x) { SyntaxElement.Expression (x) }) }
      public GetName () : string { "stagedmacro" }
      public GetNamespace () : string { "" }
      public IsInherited : bool { get { false } }
      public IsTailRecursionTransparent : bool { get { false } }
      public Keywords : list [string] { get { [] } }
      public SyntaxExtension () : GrammarElement * (list [SyntaxElement] -> 
                                                    list [SyntaxElement])
      { (null, fun (_) { [] }) }
      public Usage : Nemerle.MacroUsageAttribute { get { null } }
    }
  }
}
