/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Sioux;
using Nemerle.Collections;
using Nemerle.Xml;
using System.Xml;
using System.Xml.Xsl;
using System;

namespace Sioux.Docs
{
    public class Docs : Application
    {
        /* -- PUBLIC METHODS ----------------------------------------------------- */
        
        public ServePage (page_handler : option [XmlTemplate -> XmlTemplate],
                      xml_file_name : option [string], 
                      xsl_file_name : option [string]) : void
        {
            
            // load the page XML document, if any
            def generator =
            match (xml_file_name)
            {
                | Some (xml_file_name) =>
                  XmlGenerator(xml_file_name)
                  
                | _ => null
                  
            };
            
            mutable transformers =
            match (page_handler)
            {
                | Some (page_handler) =>
                  [XmlTransformer.Handler(page_handler)]
                | _ =>
                  []
            };
        
            match (xsl_file_name)
            {
                | Some (xsl_file_name) =>
                  def xsl = XslTransform ();
                  xsl.Load (xsl_file_name, null);
        
                   transformers =  XmlTransformer.Xsl(xsl) :: transformers;
                | _ => ()
            };
            
            transformers = List.Rev(  XmlTransformer.Xsl(docs_xsl) :: transformers);
            
            FormTemplate = Some (XmlPipeline.Process(generator,transformers));
        }

        private add_section_link_handler (t : XmlTemplate) : XmlTemplate
        {
            def doc = t.GetDocument();
            def root = doc.SelectSingleNode("page");
            mutable page_header = doc.SelectSingleNode("page/page-header");
            when(page_header == null)
            {
                page_header = doc.CreateNode(XmlNodeType.Element,"page-header","");
                ignore(root.AppendChild(page_header))
            }

            def add_links ( node_names : list [string] , from_node : XmlNode , to_node : XmlNode )
            {
                match(node_names)
                {
                    | [] => ()
                    | head :: tail =>
                         def nodes = from_node.SelectNodes(head);

                         def lst  = doc.CreateElement("ul");
                         def  attr = doc.CreateAttribute("class");
                         attr.Value = "toc";

                         ignore(lst.SetAttributeNode(attr));
                         
                         foreach(n is XmlElement in nodes)
                         {
                             def title = n.GetAttribute("title");
                             def anchor = n.GetAttribute("anchor");
                             
                             when(title != "" && anchor != "")
                             {
                                 def link = (doc.CreateNode(XmlNodeType.Element,"a","") :> XmlElement);
                                 link.SetAttribute("href","#" + anchor);
                                 link.InnerText = title;

                                 def position   = doc.CreateNode(XmlNodeType.Element,"li","");
                                 
                                 ignore(position.AppendChild(link));
                                 ignore(lst.AppendChild(position));
                                 add_links ( tail , n , position );
                                 
                                 //ignore(to_node.AppendChild(doc.CreateNode(XmlNodeType.Element,"br","")))
                             }
                         }
                         when(nodes.Count != 0)
                             ignore(to_node.AppendChild(lst));
                 }
            }
            add_links( ["section","subsection"],root,page_header);
            XmlTemplate(doc)
        }

        private show_sources() : void
        {
            match(GetVars.Get("app"))
            {
                | None => report_error("Error in request url - no \"app\" argument")
                | Some (app) =>
                  match(app)
                  {
                      | "Sessions" => response.WriteRedirect("/docs/sessions/sessions.n")
                      | "RequestParameters" => response.WriteRedirect("/docs/request_parameters/request_parameters.n")
                      | "RequestHeaders" => response.WriteRedirect("/docs/request_headers/request_headers.n")
                      | "RequestInfo" => response.WriteRedirect("/docs/request_info/request_info.n")
                      | "Cookies" => response.WriteRedirect("/docs/cookies/cookies.n")
                      | "Hello" => response.WriteRedirect("/docs/hello/hello.n")
                      | "Upload" => response.WriteRedirect("/docs/upload/upload.n")
                      | "Dirgenerator" => response.WriteRedirect("/docs/directory_generator/directory_generator.n")
                      | ap => report_error (ap + " - unknown application") 
                  }
            }
        }

        private report_error(msg : string) : void
        {
            def report_error_handler(t : XmlTemplate) : XmlTemplate
            {
                t.SetText("msg",msg);
                t
            }
            ServePage(Some (report_error_handler) , Some  ("docs/error.xml") , None ())
        }
        
        override protected DoGet () : void
        {

            def serve_static (xml_file_name : string) : void
            {
                ServePage (None (), Some (xml_file_name), None ())
            }

            def serve_with_section_links(xml_file_name : string) : void
            {
                ServePage(Some (add_section_link_handler),Some(xml_file_name),None())
            }
            
            match (PageName)
            {
                | "/source.xml" => show_sources();
                | "/reference.xml" => serve_with_section_links("docs/reference.xml");
                | "/examples.xml" => serve_static ("docs/examples.xml");
                | "/tutorial.xml" => serve_with_section_links ("docs/tutorial.xml");
                | _ => serve_static ("docs/index.xml")
            }
        }
        
        /* -- CONSTRUCTOR ------------------------------------------------- */
        
        public this()
        {
            docs_xsl = XslTransform ();
            docs_xsl.Load ("docs/docs.xsl", null);
        }

                 
        /* -- PRIVATE FIELDS ---------------------------------------------- */

        private mutable static docs_xsl : XslTransform;
    }
}
