//
// CodeCompletionEngine.n
//
// Author:
//  Alejandro Serrano (trupill at yahoo dot es)
//

/*
 * Copyright (c) 2005 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
 
using System;
using Nemerle.Compiler;
using Nemerle.Utility;
using Nemerle.Collections;
using Nemerle.Compiler.Parsetree;

using Typed = Nemerle.Compiler.Typedtree;
using SR = System.Reflection;

#define DEBUG

namespace Nemerle.Completion
{
    public class CodeCompletionEngineException : System.Exception
    {
        public mutable ParsingException : System.Exception;
        
        public this (innerException : System.Exception)
        {
            ParsingException = innerException;
        }
    }
    
    internal variant InternalReference
    {
    | Library { path : string }
    | Assembly { assembly : System.Reflection.Assembly }
    }
    
    public class DefineCollection : System.Collections.Generic.IEnumerable [string]
    {
        internal this ()
        {
            defines = [];
        }
        
        internal mutable defines : list[string];
        
        public Add (Define : string) : void
        {
            defines ::= Define;
            Engine.set_unparsed_state ();
        }
        
        public Remove (Define : string) : void
        {
            _ = defines.Remove (Define);
            Engine.set_unparsed_state ();
        }
        
        public Contains (Define : string) : bool
        {
            defines.Contains (Define)
        }
        
        public Clear () : void
        {
            defines = [];
            Engine.set_unparsed_state();
        }
        
        public GetEnumerator () : Nemerle.Collections.IEnumerator[string]
        {
            Nemerle.Collections.ListEnumerator (defines)
        }
    }
    
    public class ReferenceCollection
    {
        internal this ()
        {
            references = Hashtable ();
        }
        
        internal mutable references : Hashtable[string, InternalReference];
        
        public Add (Key : string, Path : string) : void
        {
            references.Add (Key, InternalReference.Library (Path));
            Engine.set_unparsed_state ();
        }
        
        public Add (Key : string, LoadedAssembly : System.Reflection.Assembly) : void
        {
            references.Add (Key, InternalReference.Assembly (LoadedAssembly));
            Engine.set_unparsed_state ();
        }
        
        public Remove (Key : string) : void
        {
            references.Remove (Key);
        }
        
        public Clear () : void
        {
            references = Hashtable ();
            Engine.set_unparsed_state();
        }
    }
    
    internal variant ParsedFile
    {
        | NotParsed { code : string }
        | Parsed { decls : list [TopDeclaration]; code : string; }
    }
    
    public class SourceCollection
    {
        internal this ()
        {
            sources = Hashtable ();
        }
        
        internal mutable sources : Hashtable[string, ParsedFile];

        public Add (File : string, Contents : string) : void
        {
            sources.Add (File, ParsedFile.NotParsed (Contents));
        }
        
        public Remove (File : string) : void
        {
            sources.Remove (File);
            set_unparsed_state ();
        }
        
        public Clear () : void
        {
            sources = Hashtable ();
            set_unparsed_state();
        }       
        
        internal set_unparsed_state () : void
        {
            mutable fileList : list[string * string] = [];
            foreach (file in sources) {
                match (file.Value) {
                | Parsed as p => fileList ::= (file.Key, p.code);
                | _ => ();
                }
            }
            foreach (item in fileList) {
                sources [item [0]] = ParsedFile.NotParsed (item [1]);
            }
        }
    }
    
    public module Engine
    {
        public mutable IsInCompletionMode : bool;
        internal mutable textWriter : System.IO.TextWriter;
        
        public mutable Defines : DefineCollection;
        public mutable References : ReferenceCollection;
        public Sources : SourceCollection;
        
        syncObject : object;

        internal set_unparsed_state () : void
        {
            Sources.set_unparsed_state ();
        }
        
        public static this()
        {
            IsInCompletionMode = false;
            syncObject = object();
            Defines = DefineCollection ();
            References = ReferenceCollection ();
            Sources = SourceCollection ();
            Passes.ParsingPipeline = MainParser.Parse;
            Passes.ScanningPipeline = ScanTypeHierarchy.ProcessDeclaration;
            textWriter = null;
            Message.MessageOccured += Message.MessageEventHandler (process_error_message);
        }
        
        public Init() : void
        {
            Location.Init();
            if (textWriter == null)
                Message.InitOutput (System.Console.Out);
            else
                Message.InitOutput (textWriter);
            MacroColorizer.Clear ();
            MacroRegistry.Init ();
            LibraryReferenceManager.Init ();     
            NamespaceTree.Init ();
            NamespaceTree.namespace_tree.Clear ();
            Util.Init ();
            AttributeCompiler.Init ();
            PreParser.Init ();
            Passes.Solver = Solver ();
      
            Options.Clear();
            Options.GreedyReferences = true;
            Options.ColorMessages = false;
            Options.IgnoreConfusion = true;
            
            LibraryReferenceManager.AddLibrary ("mscorlib");
            LibraryReferenceManager.AddLibrary ("System");
            LibraryReferenceManager.AddLibrary ("Nemerle");
            SystemType.Init ();                 
            InternalType.InitSystemTypes ();
            InternalType.InitNemerleTypes ();
            LibraryReferenceManager.LoadMacrosFrom ("Nemerle.Macros");
            Sources.set_unparsed_state ();
            
            GlobalEnv.Init();
            
            listMessages = [];
        }
        
        public LesserInit() : void
        {
            LibraryReferenceManager.Init ();     
            NamespaceTree.Init ();
            Util.Init ();
            AttributeCompiler.Init ();
            PreParser.Init ();
            Passes.Solver = Solver ();
            
            GlobalEnv.Init();
        }
        
        // If you want to recover the messages done by the parser/typer
        public Output : System.IO.TextWriter
        {
            get { textWriter }
            set { textWriter = value }
        }
        
        public CompilerMessages : array[CompilerMessage]
        {
            get
            {
                if (listMessages == null)
                    array(0)
                else
                    listMessages.ToArray ()
            }
        }
        
        public RunCompletionEngine (memberInfo : NemerleMemberInfo, contents : string) : CompletionInfo
        {
            lock (syncObject)
            {
                // Tell the methods we are in completion mode
                IsInCompletionMode = true;
                def theMember = memberInfo._member;
                def observed_method = theMember :> MethodBuilder;
                
                mutable completionList  = null;
                when (observed_method != null) {
                    def env = observed_method.DeclaringType.GlobalEnv;
                    def my_body = contents.Trim();
    
                    def lexer = LexerCompletion (my_body + " ", my_body.Length);
                    observed_method.GetHeader ().body =
                        FunBody.Parsed (MainParser.ParseExpr (env, lexer));
                
                    try
                    {
                        observed_method.RunBodyTyper ();
                    }
                    catch
                    {
                    | e is CompletionResult => completionList = translate_ovpossibility_to_info (e.Overloads);
                    | e => System.Console.WriteLine (e.Message);
                    }
                }
                IsInCompletionMode = false;
                completionList
            }
        }
        
        mutable listMessages : list[CompilerMessage];
        process_error_message (location : Location, message : string) : void
        {
            mutable error = CompilerMessage();
            error.Location = CodeLocation(location.File, location.Line, location.Column,
                location.EndLine, location.EndColumn);
            if (message.IndexOf ("error: ") != -1)
            {
                error.Message = message.Substring (message.IndexOf ("error: ")).Replace ("error: ", "");
                error.MessageKind = MessageKind.Error;
                listMessages ::= error;
            }
            else if (message.IndexOf ("warning: ") != -1)
            {
                error.Message = message.Substring (message.IndexOf ("warning: ")).Replace ("warning: ", "");
                error.MessageKind = MessageKind.Warning;
                listMessages ::= error;
            }
            else when (message.IndexOf ("hint: ") != -1)
            {
                error.Message = message.Substring (message.IndexOf ("hint: ")).Replace ("hint: ", "");
                error.MessageKind = MessageKind.Hint;
                listMessages ::= error;
            }
        }
        
        # region Code completion
        
        public translate_ovpossibility_to_info (ovpos : list[OverloadPossibility]) : CompletionInfo
        {
            mutable info : CompletionInfo = null;
            if (ovpos.Length == 0)
                info = null;
            else
            {
                mutable complete_types = false;
                def first = ovpos.Head;
                when (first.Member.Name == ".ctor" || first.Member.Name == ".cctor" ||
                    first.Member is TypeInfo) {
                   complete_types = true;
                }

                if (complete_types)
                {
                    def cinfo = CompletionTypes ();
                    
                    def node = first.from.tycon.FullName;
                    def ns_node = NamespaceTree.namespace_tree.Path (node).Parent;
                    mutable already_added = [];
                    foreach (child in ns_node.children)
                    {
                        match (child.Value.Value) {
                        | Cached (tycon) => def type_name = child.Key.TrimEnd('*').TrimEnd ('1', '2', '3', '4').TrimEnd('`');
                            when (!already_added.Contains (type_name))
                            {
                                cinfo.AddType (complete_type (tycon));
                                already_added ::= type_name;
                            }
                        //| NotLoaded (tycon) => System.Console.WriteLine ("notloaded " + child.Key);
                        | NamespaceReference => cinfo.AddNamespace (child.Key);
                        | _ => ();
                        }
                    }
                    
                    info = cinfo;
                }
                else
                {
                    def cinfo = CompletionMembers ();
                    try {
                    foreach (possibility in ovpos) {
                      match (possibility.Member) {
                      | y is IField => cinfo.AddMember (complete_field (y));
                      | y is IMethod => cinfo.AddMember (complete_method (y));
                      | y is IProperty => cinfo.AddMember (complete_property (y));
                      | y is IEvent => cinfo.AddMember (complete_event (y));
                      // | y is TypeInfo => cinfo.AddMember (complete_type (y));
                      | _ => ();
                      }
                    }
                    info = cinfo;
                    }
                    catch {
                    | e => System.Console.WriteLine (e.Message + "\n" + e.StackTrace);
                    }
                 }
            }
            info
        }
        
        complete_common_member_props (member : IMember, info : NemerleMemberInfo) : void
        {
            info.Name = member.Name;
            
            info.IsStatic = member.IsStatic;
            info.IsPublic = member.Attributes %&& NemerleAttributes.Public;
            info.IsPrivate = member.Attributes %&& NemerleAttributes.Private;
            info.IsProtected = member.Attributes %&& NemerleAttributes.Protected;
            info.IsInternal = member.Attributes %&& NemerleAttributes.Internal;
        }

        
        complete_field (field : IField) : FieldInfo
        {
            mutable returnField = FieldInfo();
            complete_common_member_props(field, returnField);
            
            if (field is FieldBuilder)
                returnField.Type = construct_type ((field :> FieldBuilder).GetMemType());
            else
                returnField.Type = ConstructedTypeInfo.Class ( 
                    ReferencedTypeInfo (field.GetFieldInfo ().FieldType), array (0));
 
            returnField.IsMutable = field.IsMutable;
            returnField.IsVolatile = field.IsVolatile;
            returnField.IsLiteral = field.IsLiteral;
            
            if (field.IsLiteral) {
                def value = field.GetValue();
                match (value) {
                    | Void
                    | Null => returnField.Value = null;
                    | String as s => returnField.Value = s.val;
                    | Float as f => returnField.Value = f.val;
                    | Double as d => returnField.Value = d.val;
                    | Decimal as dec => returnField.Value = dec.val;
                    | Char as ch => returnField.Value = ch.val;
                    | Bool as b => returnField.Value = b.val;
                    | Integer as i => if (i.is_negative) returnField.Value = - (i.val :> long);
                                      else returnField.Value = i.val;
                    | Enum as l => if (l.val.is_negative) returnField.Value = - (l.val.val :> long);
                                   else returnField.Value = l.val.val;
                }
            }
            else
                returnField.Value = null;
            
            returnField
        }

        complete_method (method : IMethod) : MethodInfo
        {
            mutable returnMethod = MethodInfo();
            complete_common_member_props(method, returnMethod);
            
            def attributes = method.Attributes;
            returnMethod.IsAbstract = attributes %&& NemerleAttributes.Abstract;
            returnMethod.IsFinal = attributes %&& NemerleAttributes.Sealed;
            returnMethod.IsOverride = attributes %&& NemerleAttributes.Override;
            returnMethod.IsVirtual = attributes %&& NemerleAttributes.Virtual;    
            returnMethod.IsNew = attributes %&& NemerleAttributes.New;
            returnMethod.IsExtern = attributes %&& NemerleAttributes.Extern;
            
            returnMethod.IsConstructor = false;
            returnMethod.IsStaticConstructor = false;
            match (method.GetFunKind()){
                | FunKind.Constructor => returnMethod.IsConstructor = true;
                | FunKind.StaticConstructor => returnMethod.IsStaticConstructor = true;
                | _ => ();
            }
            
            if (method.ReturnType == null)
                returnMethod.ReturnType = null;
            else
                returnMethod.ReturnType = construct_type (method.ReturnType :> MType);
            
            def param = method.GetParameters();
            def listParams = List.Map (param, get_parm);
            returnMethod.Parameters = listParams.ToArray ();
            def typarms = method.GetHeader().typarms;
            def listTyparms = List.Map (typarms, get_typarm);
            returnMethod.TypeParameters = listTyparms.ToArray ();
            
            returnMethod
        }
        
        complete_property (property : IProperty) : PropertyInfo
        {
            mutable returnProperty = PropertyInfo();
            complete_common_member_props (property, returnProperty);
            if (property is PropertyBuilder)
                returnProperty.Type = construct_type ((property :> PropertyBuilder).GetMemType());
            else
                returnProperty.Type = ConstructedTypeInfo.Class ( 
                    ReferencedTypeInfo (property.GetPropertyInfo().PropertyType), array(0));
            returnProperty.IsMutable = property.IsMutable;
            returnProperty.IsIndexer = property.IsIndexer;
            
            def attributes = property.Attributes;
            returnProperty.IsAbstract = attributes %&& NemerleAttributes.Abstract;
            returnProperty.IsFinal = attributes %&& NemerleAttributes.Sealed;
            returnProperty.IsOverride = attributes %&& NemerleAttributes.Override;
            returnProperty.IsVirtual = attributes %&& NemerleAttributes.Virtual;    
            returnProperty.IsNew = attributes %&& NemerleAttributes.New;
            returnProperty.IsExtern = attributes %&& NemerleAttributes.Extern;

            if (property.GetGetter() == null)
                returnProperty.Getter = null;
            else
                returnProperty.Getter = complete_method (property.GetGetter());
           
            if (property.GetSetter() == null)
                returnProperty.Setter = null;
            else
                returnProperty.Setter = complete_method (property.GetSetter());
                
            returnProperty
        }

        complete_event (ev : IEvent) : EventInfo
        {
            mutable returnEvent = EventInfo();
            complete_common_member_props (ev, returnEvent);
            if (ev is EventBuilder)
                returnEvent.Type = construct_type ((ev :> EventBuilder).GetMemType());
            else
                returnEvent.Type = ConstructedTypeInfo.Class ( 
                    ReferencedTypeInfo (ev.GetEventInfo().EventHandlerType), array(0));

            if (ev.GetAdder() == null)
                returnEvent.Adder = null;
            else
                returnEvent.Adder = complete_method (ev.GetAdder());
             
            if (ev.GetRemover() == null)
                returnEvent.Remover = null;
            else
                returnEvent.Remover = complete_method (ev.GetRemover());
             
            returnEvent
        }
        
        complete_type (t : TypeInfo) : NemerleTypeInfo
        {
        | x is TypeBuilder => get_type (x)
        | _ => ReferencedTypeInfo (t.SystemType)
        }
        
        # endregion
        
        # region Type Tree building
        /** This section of code is used to build the Type Tree
            All errors are handled doing nothing. We just try to
            have the most information we could take, but if there's
            some error in the code, the engine must continue */
        mutable listTypes : list[DeclaredTypeInfo];
        mutable uniqueTypesTable : Hashtable[string, DeclaredTypeInfo];
        mutable tree : TypeTree;
        
        public GetTypeTree () : TypeTree
        {
            lock (syncObject)
            {
                tree = TypeTree ();
                Init ();
            
                foreach (define in Defines.defines)
                    Options.DefineConstant (define);
            
                foreach (references in References.references.Values)
                {
                | Library as l => LibraryReferenceManager.AddLibrary (l.path);
                | Assembly as a => LibraryReferenceManager.AddAssembly (a.assembly);
                }
    
                System.Console.WriteLine ("just before lexing");
                mutable trees = [];
                try
                {
                    def filenames = Sources.sources.Fold ([], fun (k, _, acc) { k :: acc });
                  
                    foreach (filename in filenames)
                    {
                        match (Sources.sources [filename])
                        {
                        | NotParsed as np =>
                            def contents = np.code;
                            def lexer = LexerString (contents, Location (Location.GetFileIndex (filename), 1, 1));
                            def decls = Passes.ParsingPipeline (lexer);
                            Sources.sources[filename] = ParsedFile.Parsed (decls, contents);
                            trees ::= decls;
                        | Parsed as p => trees ::= p.decls;
                        }
                    }
                }
                catch
                {
                    | _e => 
                      #if DEBUG
                      System.Console.WriteLine (_e);
                      #endif
                      {}
                }
                
                System.Console.WriteLine ("just before TypesManager");
                Passes.Hierarchy = TypesManager ();
    
                // create N.C.TypeBuilders for all parsed types and add them to namespace hierarchy
                System.Console.WriteLine ("just before parsing");
                try
                {
                    foreach (group in trees) {
                        List.Iter (group, Passes.ScanningPipeline);
                    }
                }
                catch
                {
                    | _e => 
                      #if DEBUG
                      System.Console.WriteLine (_e);
                      #endif
                      {}
                }
                
                System.Console.WriteLine ("just before building the hierarchy");
                try
                {
                    Passes.Hierarchy.Run();
                }
                catch
                {
                    | _e => 
                      #if DEBUG
                      System.Console.WriteLine (_e);
                      #endif
                      {}
                }
                
                listTypes = [];
                uniqueTypesTable = Hashtable ();
                
                loop (NamespaceTree.namespace_tree);
                tree.Types = listTypes.ToArray ();
                
                tree
             }
         }
             
         loop (x : NamespaceTree.Node) : void
         {
             foreach ( pair in x.children )
             {
                 def node = pair.Value;
                 match (node.Value) {
                 | NamespaceReference => loop(node);
                 | Cached as c =>
                     match (c.tycon) {
                       | tb is TypeBuilder =>
                         listTypes ::= get_type (tb);
                       | _ => ();
                     }
                 | _ => ()
                 }
             }
         }
         
         get_type (t : TypeBuilder) : DeclaredTypeInfo
         {
             mutable uniqueName = t.FrameworkTypeName;
             
             if (uniqueTypesTable.Contains(uniqueName))
                 uniqueTypesTable[uniqueName]
             else 
             {
                 mutable returnType = DeclaredTypeInfo();
                 uniqueTypesTable.Add(uniqueName, returnType);
                 try
                 {
                     returnType.Name = t.Name.TrimEnd('*').TrimEnd ('1', '2', '3', '4').TrimEnd('`');
                     
                     if (t.DeclaringType == null) {
                         returnType.Namespace = t.NamespaceNode.Parent.Name.ToString(".");
                         returnType.IsNested = false;
                     }
                     else {
                         returnType.Namespace = (t.DeclaringType :> TypeBuilder).NamespaceNode.Parent.Name.ToString(".");
                         returnType.IsNested = true;
                     }
                     
                     returnType.Location = CodeLocation (t.Location.File, t.Location.Line,
                         t.Location.Column, t.Location.EndLine, t.Location.EndColumn);
                     returnType.IsSealed = t.IsSealed;
                     returnType.IsAbstract = t.IsAbstract;
                     
                     returnType.IsInterface = t.IsInterface;
                     returnType.IsDelegate = t.IsDelegate;
                     returnType.IsEnum = t.IsEnum;
                     returnType.IsValueType = t.IsValueType;
                     returnType.IsModule = t.IsModule;
                     returnType.IsAlias = t.IsAlias;
                     returnType.IsStruct = t.IsStruct;
                     returnType.IsVariant = (t.GetVariantOptions().Length > 0);
                     
                     returnType.IsPublic = t.IsPublic;
                     returnType.IsInternal = t.IsInternal;
                     returnType.IsProtected = t.IsProtected;
                     returnType.IsPrivate = t.IsPrivate;
                     
                     match (t.DefaultIndexerName) {
                         | None => returnType.DefaultIndexerName = null;
                         | Some (x) => returnType.DefaultIndexerName = x;
                     }
                     if (t.DeclaringType == null)
                         returnType.DeclaringType = null;
                     else
                         returnType.DeclaringType = get_type(t.DeclaringType :> TypeBuilder);
                     
                     mutable listFields = [];
                     mutable listMethods = [];
                     mutable listProperties = [];
                     mutable listEvents = [];
                     mutable listNestedTypes = [];
                     
                     // Add all members to type information
                     def members = t.GetDirectMembers();
                     foreach (member in members){
                         | x is FieldBuilder => listFields ::= get_field (x);
                         | x is MethodBuilder => listMethods ::= get_method (x);
                         | x is PropertyBuilder => listProperties ::= get_property (x);
                         | x is EventBuilder => listEvents ::= get_event (x);
                         | x is TypeBuilder => listNestedTypes ::= get_type (x);
                         | x => throw System.Exception (x.ToString ())
                     }
                     
                     returnType.Fields = listFields.ToArray ();
                     returnType.Methods = listMethods.ToArray ();
                     returnType.Properties = listProperties.ToArray ();
                     returnType.Events = listEvents.ToArray ();
                     returnType.NestedTypes = listNestedTypes.ToArray ();
                     
                     def listVarOpts = t.GetVariantOptions();
                     def listVarOptsReal = listVarOpts.Map (get_type);
                     returnType.VariantOptions = listVarOptsReal.ToArray ();
                     
                     returnType.BaseType = get_type_info (t.BaseType);
                     def listInterf = t.InterfacesToImplement ();
                     def listInterfaces = listInterf.Map (get_type_info);
                     returnType.Interfaces = listInterfaces.ToArray ();
                     
                     def typarms = t.GetTyparms ();
                     def listTyparms = typarms.Map (get_typarm);
                     returnType.TypeParameters = listTyparms.ToArray ();
                 }
                 catch
                 {
                     | _ => ();
                 }
                 
                 returnType
             }
         }
         
         get_type_info (x : TypeInfo) : NemerleTypeInfo
         {
             if (x == null)
                 null
             else {
                 match (x) {
                   | x is TypeBuilder => get_type (x)
                   | _ => ReferencedTypeInfo (x.SystemType)
                 }
             }
         }
         
         fill_common_member_props (member : MemberBuilder, info : NemerleMemberInfo) : void
         {
             info.Name = member.Name;
             info.Location = CodeLocation (member.BodyLocation.File, member.BodyLocation.Line, 
                 member.BodyLocation.Column, member.BodyLocation.EndLine, member.BodyLocation.EndColumn);
             
             info.IsStatic = member.IsStatic;
             info.IsPublic = member.IsPublic;
             info.IsPrivate = member.IsPrivate;
             info.IsProtected = member.IsProtected;
             info.IsInternal = member.IsInternal;
             info._member = member;
         }
         
         construct_type (t : MType) : ConstructedTypeInfo
         {
            | Class as c =>
                 def returnClass = ConstructedTypeInfo.Class();
                 returnClass.Type = match (c.tycon) {
                   | x is TypeBuilder => get_type (x)
                   | x => ReferencedTypeInfo (x.SystemType)
                 }
                 def listArgs = List.Map(c.args, fun(x) {
                   | x is MType => construct_type(x)
                   | _ => null
                 });
                 returnClass.SubstitutedArguments = listArgs.ToArray ();
                 returnClass
                 
             | TyVarRef as r =>
                 def returnStaTyVar = ConstructedTypeInfo.GenericSpecifier();
                 returnStaTyVar.Name = r.tyvar.Name;
                 def listTypes = List.Map(r.tyvar.Constraints, construct_type);
                 returnStaTyVar.TypeConstraints = listTypes.ToArray ();
                 returnStaTyVar.SpecialConstraints = (r.tyvar.SpecialConstraints :> int) 
                     :> Nemerle.Completion.Constraint;
                 returnStaTyVar
            
             | Fun as f =>
                 def returnFun = ConstructedTypeInfo.Function();
                 returnFun.From = construct_type (f.from :> MType);
                 returnFun.To = construct_type (f.to :> MType);
                 returnFun
             
             | Tuple as tuple =>
                 def returnTuple = ConstructedTypeInfo.Tuple();
                 def listTypes = List.Map (tuple.args, fun(x) {construct_type (x :> MType)});
                 returnTuple.Types = listTypes.ToArray ();
                 returnTuple
                 
             | Array as a =>
                 def returnArray = ConstructedTypeInfo.Array();
                 returnArray.Type = construct_type (a.t :> MType);
                 returnArray.Rank = a.rank;
                 returnArray
            
             | Void => ConstructedTypeInfo.Void()
             | _ => null // Ref and out are only available in parameters,
                         // so they should never appear here
         }
         
         get_field (field : FieldBuilder) : FieldInfo
         {
             mutable returnField = FieldInfo();
             fill_common_member_props (field, returnField);
             
             returnField.Type = construct_type (field.GetMemType());
             returnField.IsMutable = field.IsMutable;
             returnField.IsVolatile = field.IsVolatile;
             returnField.IsLiteral = field.IsLiteral;
             
             if (field.IsLiteral) {
                 def value = field.GetValue();
                 match (value) {
                     | Void
                     | Null => returnField.Value = null;
                     | String as s => returnField.Value = s.val;
                     | Float as f => returnField.Value = f.val;
                     | Double as d => returnField.Value = d.val;
                     | Decimal as dec => returnField.Value = dec.val;
                     | Char as ch => returnField.Value = ch.val;
                     | Bool as b => returnField.Value = b.val;
                     | Integer as i => if (i.is_negative) returnField.Value = - (i.val :> long);
                                       else returnField.Value = i.val;
                     | Enum as l => if (l.val.is_negative) returnField.Value = - (l.val.val :> long);
                                    else returnField.Value = l.val.val;
                 }
             }
             else
                 returnField.Value = null;
             
             returnField
         }
         
         get_method (method : MethodBuilder) : MethodInfo
         {
             mutable returnMethod = MethodInfo();
             fill_common_member_props(method, returnMethod);
             
             returnMethod.IsVarArgs = method.IsVarArgs;
             def attributes = method.Attributes;
             returnMethod.IsAbstract = attributes %&& NemerleAttributes.Abstract;
             returnMethod.IsFinal = attributes %&& NemerleAttributes.Sealed;
             returnMethod.IsOverride = attributes %&& NemerleAttributes.Override;
             returnMethod.IsVirtual = attributes %&& NemerleAttributes.Virtual;    
             returnMethod.IsNew = attributes %&& NemerleAttributes.New;
             returnMethod.IsExtern = attributes %&& NemerleAttributes.Extern;
             
             returnMethod.IsConstructor = false;
             returnMethod.IsStaticConstructor = false;
             match (method.GetFunKind()){
                 | FunKind.Constructor => returnMethod.IsConstructor = true;
                 | FunKind.StaticConstructor => returnMethod.IsStaticConstructor = true;
                 | _ => ();
             }
             
             if (method.ReturnType == null)
                 returnMethod.ReturnType = null;
             else
                 returnMethod.ReturnType = construct_type (method.ReturnType :> MType);
             
             def param = method.GetParameters();
             def listParams = List.Map (param, get_parm);
             returnMethod.Parameters = listParams.ToArray ();
             def typarms = method.GetHeader().typarms;
             def listTyparms = List.Map (typarms, get_typarm);
             returnMethod.TypeParameters = listTyparms.ToArray ();
             
             returnMethod
         }
         
         get_parm (parm : Typed.Fun_parm) : ParameterInfo
         {
             mutable returnParam = ParameterInfo();
             returnParam.Name = parm.name;
             
             returnParam.IsRefParameter = false;
             returnParam.IsOutParameter = false;
             match (parm.ty :> MType) {
                 | Ref (x) => returnParam.IsRefParameter = true;
                              returnParam.Type = construct_type (x :> MType);
                 | Out (x) => returnParam.IsOutParameter = true;
                              returnParam.Type = construct_type (x :> MType);
                 | _ as x => returnParam.Type = construct_type (x);
             }
             
             match (parm.default_value) {
                 | None => returnParam.HasDefaultValue = false;
                           returnParam.DefaultValue = null;
                 | Some (value) => returnParam.HasDefaultValue = true;
                     match (value) {
                         | Void
                         | Null => returnParam.DefaultValue = null;
                         | String as s => returnParam.DefaultValue = s.val;
                         | Float as f => returnParam.DefaultValue = f.val;
                         | Double as d => returnParam.DefaultValue = d.val;
                         | Decimal as dec => returnParam.DefaultValue = dec.val;
                         | Char as ch => returnParam.DefaultValue = ch.val;
                         | Bool as b => returnParam.DefaultValue = b.val;
                         | Integer as i => if (i.is_negative) returnParam.DefaultValue = - (i.val :> long);
                                           else returnParam.DefaultValue = i.val;
                         | Enum as l => if (l.val.is_negative) returnParam.DefaultValue = - (l.val.val :> long);
                                        else returnParam.DefaultValue = l.val.val;
                     }
             }
             
             returnParam
         }
         
         get_typarm (typarm : StaticTyVar) : TypeParameterInfo
         {
             mutable returnTyparm = TypeParameterInfo();
             returnTyparm.Name = typarm.Name;
             def listTypes = List.Map (typarm.Constraints, construct_type);
             returnTyparm.TypeConstraints = listTypes.ToArray ();
             returnTyparm.SpecialConstraints = (typarm.SpecialConstraints :> int) 
                 :> Nemerle.Completion.Constraint;
             returnTyparm
         }
         
         get_property (property : PropertyBuilder) : PropertyInfo
         {
             mutable returnProperty = PropertyInfo();
             fill_common_member_props (property, returnProperty);
             returnProperty.Type = construct_type (property.GetMemType());

             returnProperty.IsMutable = property.IsMutable;
             returnProperty.IsIndexer = property.IsIndexer;
             def listIndexParams = List.Map (property.parms, construct_type);
             returnProperty.IndexerParameters = listIndexParams.ToArray ();
             
             def attributes = property.Attributes;
             returnProperty.IsAbstract = attributes %&& NemerleAttributes.Abstract;
             returnProperty.IsFinal = attributes %&& NemerleAttributes.Sealed;
             returnProperty.IsOverride = attributes %&& NemerleAttributes.Override;
             returnProperty.IsVirtual = attributes %&& NemerleAttributes.Virtual;    
             returnProperty.IsNew = attributes %&& NemerleAttributes.New;
             returnProperty.IsExtern = attributes %&& NemerleAttributes.Extern;

             if (property.GetGetter() == null)
                 returnProperty.Getter = null;
             else
                 returnProperty.Getter = get_method (property.GetGetter() :> MethodBuilder);
            
             if (property.GetSetter() == null)
                 returnProperty.Setter = null;
             else
                 returnProperty.Setter = get_method (property.GetSetter() :> MethodBuilder);
                 
             returnProperty
         }
         
         get_event (ev : EventBuilder) : EventInfo
         {
             mutable returnEvent = EventInfo();
             fill_common_member_props (ev, returnEvent);
             returnEvent.Type = construct_type (ev.GetMemType());

             if (ev.GetAdder() == null)
                 returnEvent.Adder = null;
             else
                 returnEvent.Adder = get_method (ev.GetAdder());
             
             if (ev.GetRemover() == null)
                 returnEvent.Remover = null;
             else
                 returnEvent.Remover = get_method (ev.GetRemover());
             
             returnEvent
         }
         # endregion
         
    } // end class CodeCompletionEngine
} // end namespace

