/*
 * Copyright (c) 2003, 2004 The University of Wroclaw.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *    1. Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *    2. Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *    3. The name of the University may not be used to endorse or promote
 *       products derived from this software without specific prior
 *       written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE UNIVERSITY BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

using Nemerle.Compiler;
using Nemerle.Collections;
using System.Text.RegularExpressions;

namespace Nemerle.Data 
{
  public module Tools 
  {
    /** Extracts parameters after $, perform typing of expressions containing
        variables with names of those parameters.
    
        Returns:
        - final SQL provider secific query
        - collection of pairs of name and typed expression containing variable with this
          name
        - list of expressions initializing [querycmd.Parameters] to value of
          above variables
    */
    public ExtractParameters (ctx : Typer, query : string, param_prefix : char)
    : string * Hashtable [string, Typedtree.TExpr] * list [Parsetree.PExpr]
    {
      // take names after $
      def namesreg = Regex (@"(([^\$]*)\$(\w+))|(.+)", RegexOptions.Singleline);
      mutable m = namesreg.Match (query);
      def fquery = System.Text.StringBuilder (query);
      def pars = Hashtable ();

      // extract names and prefix them with Sql parameter delimiter `:'
      while (m.Success) {
        when (m.Groups[1].Success) {
          pars.Set (m.Groups[3].ToString (), null);
          fquery[m.Groups[3].Index - 1] = param_prefix;
        };
        m = m.NextMatch ();
      };

      /// initializers of sql command parameters
      def tpars = Hashtable ();
      /// expressions initializing [querycmd.Parameters] with their values    
      mutable pars_init = [];
      // itreate through parameters to type them and create initializers
      pars.Iter (fun (x : string, _) {
        def tvar = ctx.TypeExpr (<[ $(x : usesite) ]>);
        tpars.Set (x, tvar);

        /// add parameter initializer
        pars_init = <[
          _ = querycmd.Parameters.Add ($(x : string), $(tvar : typed))
        ]> :: pars_init;
      });
      (fquery.ToString (), tpars, pars_init)
    }
  }
}
