/* Input line handling.

	Copyright (C) 1993-1998 Sebastiano Vigna 
	Copyright (C) 1999-2001 Todd M. Lewis and Sebastiano Vigna

	This file is part of ne, the nice editor.

	This program is free software; you can redistribute it and/or modify it
	under the terms of the GNU General Public License as published by the
	Free Software Foundation; either version 2, or (at your option) any
	later version.
	
	This program is distributed in the hope that it will be useful, but
	WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
	General Public License for more details.
	
	You should have received a copy of the GNU General Public License along
	with this program; see the file COPYING.  If not, write to the Free
	Software Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
	02111-1307, USA.  */


#include "ne.h"
#include "termchar.h"


/* This is the maximum number of characters which can be typed on the input line. */

#define MAX_INPUT_LINE_LEN		1024



/* This function prints an input prompt in the input line. A colon is
postpended to the prompt. The position of the first character to use for
input is returned. Moreover, the status bar is reset, so that it will be
updated. */


static unsigned int print_prompt(const char *prompt) {

	assert(prompt != NULL);

	move_cursor(ne_lines-1, 0);

	clear_to_eol();

	standout_on();

	output_string(prompt);
	output_string(":");

	standout_off();

	output_chars(NULL, 1);

	reset_status_bar();

	return(strlen(prompt)+2);
}



/* This function prompts the user for a yes/no answer. default_value has to
be TRUE or FALSE. TRUE is returned if 'y' was typed, FALSE in any other
case. Escaping is not allowed. RETURN returns the default value. */


int request_response(buffer *b, const char *prompt, int default_value) {

	char response = request_char(b, prompt, default_value ? 'y' : 'n');

	if (response == 'Y') return(TRUE);
	return(FALSE);
}



/* This function prompts the user for a single character answer.
default_value has to be a character which is used for the default answer.
The character typed by the user (upper cased) is returned. The default
character is used if the user types RETURN. Note that the cursor is moved
back to its current position. This offers a clear distinction between
immediate and long inputs, and allows for interactive search and replace. */


char request_char(buffer *b, const char *prompt, const char default_value) {

	int c;
	input_class ic;

	print_prompt(prompt);

	if (default_value) output_chars(&default_value, 1);
	move_cursor(b->cur_y, b->cur_x);

	while(TRUE) {
		while((ic = char_class[c = get_key_code()]) == IGNORE);

		switch(ic) {
			case ALPHA:
				return((char)up_case[(unsigned char)c]);

			case RETURN:
				return((char)up_case[(unsigned char)default_value]);

			default:
				break;
		}
	}
}



/* This function requests a number, with a given prompt and default value.
Only nonnegative integers can be entered. The effects of a negative
default_value are mysterious, and not worth an investigation. The returned
value is nonnegative if something was entered, negative on escaping or on
entering the empty string. */


int request_number(const char *prompt, int default_value) {

	char t[MAX_INT_LEN], *result, *p;
	int n;

	if (default_value >= 0) sprintf(t, "%d", default_value);

	if (!(result = request(prompt, default_value >= 0 ? t : NULL, FALSE, 0)) || !*result) return(ERROR);

	n = strtol(result, &p, 0);

	return *p || n<0 ? ERROR : n;
}



/* This function requests a string. The returned string is never longer than
MAX_INPUT_LINE_LEN, and has been malloc()ed, so you can use it and then
free() it. NULL is returned on escaping, or if entering an empty string
(unless accept_null_string is TRUE, in which case the empty string is
duplicated and returned). completion_allowed work as in request(). */

char *request_string(const char *prompt, const char *default_string, int accept_null_string, int completion_allowed) {

	char *result = request(prompt, default_string, TRUE, completion_allowed);

	if (result && (*result || accept_null_string)) return(str_dup(result));

	return(NULL);
}

static buffer *history_buff = NULL;

static void init_history( void )
	{
		if ( !history_buff )
				{
					history_buff = alloc_buffer(NULL);
					if (history_buff)
						{
							clear_buffer(history_buff);
							history_buff->opt.do_undo = 0;
							history_buff->opt.auto_prefs = 0;
							load_file_in_buffer(history_buff, "~/.ne/.history" );
							change_filename( history_buff, str_dup("~/.ne/.history") );
							assert_buffer(history_buff);
						}
				}
		if ( history_buff )
				{
					move_to_bof(history_buff);
					move_to_sol(history_buff);
				}
	}

void close_history( void )
	{
		if ( history_buff )
			{
				if (history_buff->buffer_is_modified)
					{
						while( history_buff->line_num > 500 )
							{
								move_to_sof(history_buff);
								delete_lin(history_buff,
								           history_buff->cur_line_desc,
								           history_buff->cur_line );
								assert_buffer(history_buff);
							}
						save_buffer_to_file(history_buff,NULL);
					}
				free_buffer(history_buff);
				history_buff = NULL;
			}
	}	  

static void add_to_history( char *str )
	{
		if ( !history_buff || !str || !*str ) return;

		move_to_bof(history_buff);

		if ( history_buff->cur_line_desc->line && history_buff->cur_line_desc->line_len ) {
			move_to_eol(history_buff );
			insert_lin(	history_buff,
							history_buff->cur_line_desc,
							history_buff->cur_line,
							history_buff->cur_line_desc->line_len);
			move_to_sol(history_buff);
			line_down(history_buff);
		}
			
		insert_stream( history_buff,
							history_buff->cur_line_desc,
							history_buff->cur_line,
							history_buff->cur_line_desc->line_len,
							str,
							strlen(str));
		assert_buffer( history_buff);
	}

/* This is the main function which serves request_number() and
request_string(). Given a prompt, a default string and a boolean flag which
establish the possibility of any alphabetical input (as opposed to digits
only), the user can edit a string of at most MAX_INPUT_LINE_LEN characters.
Many useful commands can be used here. The string edited by the user is
returned, or NULL if the input was escaped, or the empty string was entered.
Note that presently this function always returns a pointer to a static
buffer, but this could change in the future.

completion_allowed has three possible values: 0 means no completion, 1 complete
as a filename, 2 complete as a command followed by a filename.

The function relies on a number of auxiliary functions and static data. As
always, we would really need nested functions, but, alas, C can't cope with
them. */

static char input_buffer[MAX_INPUT_LINE_LEN+1];

/* start_x  is the first usable x position for editing;
	len      is the current length of the input buffer;
	x        is the x position of the cursor;
	pos      is the position of the cursor inside the input buffer;
	offset   is always pos-(x-start_x). */

static int start_x, len, pos, x, offset;


/* The following functions are rather intuitive. They perform basic editing
actions on the input line. */


static void input_refresh(void) {
	int i;

	move_cursor(ne_lines-1, start_x);
	for(i=0; i<ne_columns-1-start_x && i+offset<len; i++)
		output_chars(&input_buffer[i+offset], 1);
	clear_to_eol();
}


static void input_move_left(void) {
	if (pos>0) {

		pos--;
		x--;

		if (x < start_x) {
			x++;
			offset--;
			if (char_ins_del_ok) {
				move_cursor(ne_lines-1, ne_columns-2);
				delete_chars(1);
				move_cursor(ne_lines-1, start_x);
				insert_chars(&input_buffer[pos], 1);
			}
			else input_refresh();
		}
	}
}


static void input_move_right(void) {
	if (pos<len) {
		pos++;
		x++;

		if (x == ne_columns-1) {
			x--;
			offset++;

			if (char_ins_del_ok) {
				move_cursor(ne_lines-1, start_x);
				delete_chars(1);
				move_cursor(ne_lines-1, ne_columns-2);
				if (pos < len) output_chars(&input_buffer[pos], 1);
			}
			else input_refresh();
		}
	}
}


static void input_move_to_sol(void) {

	if (offset == 0) {
		x = start_x;
		pos = 0;
		return;
	}

	x = start_x;
	offset = pos = 0;
	input_refresh();
}


static void input_move_to_eol(void) {

	if (x+(len-pos) < ne_columns-1) {
		x += len-pos;
		pos = len;
		return;
	}

	x = ne_columns-2;
	pos = len;
	offset = len-(ne_columns-2-start_x);
	input_refresh();
}


static void input_next_word(void) {

	int space_skipped = FALSE;

	while(pos<len) {
		if (isspace((unsigned char)input_buffer[pos]) || ispunct((unsigned char)input_buffer[pos])) space_skipped = TRUE;
		else if (space_skipped) break;
		input_move_right();
	}
}


static void input_prev_word(void) {

	int word_skipped = FALSE;

	while(pos > 0) {
		if (!(isspace((unsigned char)input_buffer[pos-1]) || ispunct((unsigned char)input_buffer[pos-1]))) word_skipped = TRUE;
		else if (word_skipped) break;
		input_move_left();
	}
}


static void input_paste(void) {

	clip_desc *cd = get_nth_clip(cur_buffer->opt.cur_clip);

	if (cd) {
		strncpy(input_buffer, cd->cs->stream, MAX_INPUT_LINE_LEN);

		len = strlen(input_buffer);
		pos = offset = 0;
		x = start_x;
		input_refresh();
	}
}


char *request(const char *prompt, const char *default_string, int alpha_allowed, int completion_allowed) {

	action a;
	input_class ic;
	int c, first_char_typed = TRUE, last_char_completion = FALSE, selection = FALSE, quoted;
	char *completion, *prefix, *p;

	pos = len = offset = 0;
	x = start_x = print_prompt(prompt);

	init_history();
	
	if (default_string) {
		strncpy(input_buffer, default_string, MAX_INPUT_LINE_LEN);
		len = strlen(input_buffer);
		input_refresh();
	}

	while(TRUE) {

		move_cursor(ne_lines-1, x);

		while((ic = char_class[c = get_key_code()]) == IGNORE);

		if (ic != TAB) last_char_completion = FALSE;
		if (ic == TAB && !completion_allowed) ic = ALPHA;

		switch(ic) {
			case ALPHA:

				if (first_char_typed) {
					len = 0;
					clear_to_eol();
				}

				if (len<MAX_INPUT_LINE_LEN && (alpha_allowed || isxdigit(c) || c=='X' || c=='x')) {

					memmove(&input_buffer[pos+1], &input_buffer[pos], len-pos);
					input_buffer[pos] = c;
					len++;

					move_cursor(ne_lines-1, x);

					if (x < ne_columns-2) {
						if (pos == len-1) output_chars(input_buffer+pos, 1);
						else if (char_ins_del_ok) insert_chars(input_buffer+pos, 1);
						else input_refresh();
					}
					else output_chars(input_buffer+pos, 1);

					input_move_right();
				}
				break;

			case RETURN:
				selection = TRUE;
				break;


			case TAB:
				if (completion_allowed) {
					input_buffer[len] = 0;
					quoted = FALSE;
					if (len && input_buffer[len-1] == '"') {
						input_buffer[len-1] = 0;
						if (prefix = strrchr(input_buffer, '"')) {
							quoted = TRUE;
							prefix++;
						}
						else input_buffer[len-1] = '"';
					}

					if (!quoted) {
						prefix = strrchr(input_buffer, ' ');
						if (prefix) prefix++;
						else prefix = input_buffer;
					}

					if (last_char_completion) {
						completion = p = request_files(prefix, TRUE);
						reset_window();
						if (completion) {
							if (*completion) selection = TRUE;
							else completion++;
						}
					}
					else {
						completion = p = complete(prefix);
						last_char_completion = TRUE;
						if (!completion) ring_bell();
					}

					if (completion && (prefix-input_buffer)+strlen(completion)+1 < MAX_INPUT_LINE_LEN) {
						strcpy(prefix, completion);
						if (quoted) strcat(prefix, "\"");
						len = strlen(input_buffer);
						pos = offset = 0;
						x = start_x;
						input_move_to_eol();
						if (quoted) input_move_left();
						input_refresh();
					}
					free(p);
				}
				break;
					

			case COMMAND:
				if ((a = parse_command_line(key_binding[c], NULL, NULL, FALSE))>=0) {
					switch(a) {

						case LINEUP_A:
						case LINEDOWN_A:
							if ( history_buff )
								{
									(a==LINEUP_A) ? line_up( history_buff ) : line_down( history_buff );
									if ( history_buff->cur_line_desc->line ) {
											strncpy(input_buffer,
											        history_buff->cur_line_desc->line,
											        min(history_buff->cur_line_desc->line_len,MAX_INPUT_LINE_LEN));
											input_buffer[min(history_buff->cur_line_desc->line_len,MAX_INPUT_LINE_LEN)]='\0';
											}
										else
											strncpy(input_buffer, "", MAX_INPUT_LINE_LEN);
									len    = strlen(input_buffer);
									x      = start_x;
									pos    = 0;
									offset = pos-(x-start_x);
									input_refresh();
								}	
							break;

						case MOVELEFT_A:
							input_move_left();
							break;

						case MOVERIGHT_A:
							input_move_right();
							break;

						case BACKSPACE_A:
							if (pos == 0) break;
							input_move_left();

						case DELETECHAR_A:
							if (len>0 && pos<len) {

								memmove(&input_buffer[pos], &input_buffer[pos+1], len-pos-1);
								len--;

								if (char_ins_del_ok) {
									move_cursor(ne_lines-1, x);
									delete_chars(1);

									if (pos+(ne_columns-2-x) < len) {
										move_cursor(ne_lines-1, ne_columns-2);
										output_chars(&input_buffer[pos+(ne_columns-2-x)], 1);
									}
								}
								else input_refresh();
							}
							break;

						case DELETELINE_A:
							move_cursor(ne_lines-1, start_x);
							clear_to_eol();
							len = pos = offset = 0;
							x = start_x;
							break;

						case DELETEEOL_A:
							len = pos;
							clear_to_eol();
							break;

						case MOVESOL_A:
						case MOVESOF_A:
						case MOVEINCUP_A:
							input_move_to_sol();
							break;

						case MOVEEOL_A:
						case MOVEEOF_A:
						case MOVEINCDOWN_A:
							input_move_to_eol();
							break;

						case TOGGLESEOL_A:
						case TOGGLESEOF_A:
							if (pos != 0) input_move_to_sol();
							else input_move_to_eol();
							break;

						case PREVWORD_A:
							input_prev_word();
							break;

						case NEXTWORD_A:
							input_next_word();
							break;

						case REFRESH_A:
							input_refresh();
							break;

						case PASTE_A:
							input_paste();
							break;

						case ESCAPE_A:
							return(NULL);

					default:
					    	break;
					}
				}
				break;

			default:
				break;
		}

		if (selection) {
			input_buffer[len] = 0;
			add_to_history(input_buffer);
			return input_buffer;
		}

		first_char_typed = FALSE;
	}
}
