# -*-ruby-*-
#
# Copyright (c) 2004 SASADA Koichi <ko1 at atdot.net>
#
# This program is free software with ABSOLUTELY NO WARRANTY.
# You can re-distribute and/or modify this program under
# the same terms of the Ruby's lisence.
#
#
# $Id: googlebot.nb 101 2004-08-20 09:55:07Z ko1 $
#

=begin

== Usage with irc client

  google> keyword
    -> search keyword by google with default search langage

  google:[lang]> keyword
    -> search keyword by google with [lang] langage

  googlec> k1 k2 k3 k4 k5(max 5 words)
    -> search and show each hit count
   
  googlec> k1 k2 k3 k4 k5(max 5 words)
    -> search and show each hit count with default count language

  googlec:[lang]> k1 k2 k3 k4 k5(max 5 words)
    -> search and show each hit count with [lang] langage


== Configuration:

BotConfig = {
  :GoogleBot => {
    :ch => /.*/,                     # available channel
    
    :key     => 'xxxx',              # 
    :keypath => '/x/y/z/google_key', # specify one
                                     # or put google key file on ~/.google_key
    #
    :googlec_maxwords => 5,
    :search_default_lang => 'lang_ja',
    :count_default_lang  => '',
  }
}

=end


require 'soap/wsdlDriver'
require 'iconv'
require 'kconv'
require 'shellwords'
require 'cgi'

class GoogleBot < Nadoka::NDK_Bot
  def bot_initialize
    @available_channel   = @bot_config[:ch] || /.*/
    @google_key          = google_key
    @search_default_lang = @bot_config[:search_default_lang] || 'lang_ja'
    @googlec_maxwords    = @bot_config[:googlec_maxwords]    || 5
    @count_default_lang  = @bot_config[:count_default_lang]  || ''
    google_wsdl = 'http://api.google.com/GoogleSearch.wsdl'
    @google = SOAP::WSDLDriverFactory.new(google_wsdl).create_driver
  end
  
  def on_privmsg prefix, ch, msg
    if @available_channel === ch
      if response = dispatch_command(msg)
        send_notice(ch, response)
      end
    end
  end

  def dispatch_command msg
    begin
      case msg
      when /^goo(o*)gle(:.*)?>\s*(.+)/, /^gu(u*)guru(:.+)?>\s*(.+)/
        "goo#{$1}gle bot#{$2}: #{search($1.length, $2, $3)}"
        
      when /^googlec(:.*)?>\s*(.+)/
        "googlec bot(#{$2}): #{googlec($1, $2)}"
      end
    rescue Exception => e
      "google bot: #{e.class}(#{e.message})"
    end
  end
  
  def search cnt, lang, word
    lang = lang_check(lang)
    word = search_char_code(word)
    
    result = @google.doGoogleSearch(
      @google_key, word, cnt, 1, false, "", false, lang, 'utf-8', 'utf-8' )

    count = result.estimatedTotalResultsCount
    if count > 0
      count = count.to_s.gsub(/(\d)(?=\d{3}+$)/, '\\1,')
      url   =  title = ''
    
      result.resultElements.each_with_index{|e, i|
        url   = e.URL
        title = show_char_code_and_erace_tag(e.title)
      }
      
      "#{title} - #{url} (and #{count} hits)"
    else
      "no match"
    end
  end

  def googlec lang, word
    lang = lang_check(lang)
    words = Shellwords.shellwords(word)
    return 'too many options' if words.size > @googlec_maxwords

    words.map{|rw|
      w = search_char_code(rw)
      result = @google.doGoogleSearch(
        @google_key, "'#{w}'", 0, 1, false, "", false, lang, 'utf-8', 'utf-8')
      "#{rw}(#{result.estimatedTotalResultsCount.to_s.gsub(/(\d)(?=\d{3}+$)/, '\\1,')})"
    }.join(', ')
  end
  
  def google_key
    @bot_config[:key] ||
    if @bot_config[:key_path]
      File.open(@bot_config[:key_path]).read.chomp
    else
      File.open(File.expand_path("~/.google_key")).read.chomp
    end
  end
  
  def erace_tag str
    CGI.unescapeHTML(str.gsub(/\<.+?\>/, ''))
  end

  def lang_check lang
    if !lang
      @search_default_lang
    else
      lang = lang[1..-1]
      if lang.empty? || /^lang_/ =~ lang
        lang
      else
        'lang_' + lang
      end
    end
  end
  
  def show_char_code_and_erace_tag str
    case $KCODE
    when 'EUC', 'SJIS'
      CGI.unescapeHTML(str.gsub(/\<.+?\>/, '')).tojis
    when 'NONE', 'UTF-8'
      begin
        str = Iconv.conv("EUC-JP", "UTF-8", str)
        CGI.unescapeHTML(str.gsub(/\<.+?\>/, '')).tojis
      rescue => e
        "(char code problem: #{e.class}[#{e.message.dump}])"
      end
    else
      str
    end
  end
  
  def search_char_code str
    case $KCODE
    when 'EUC', 'SJIS'
      str.toeuc
    when 'NONE'
      begin
        Iconv.conv("UTF-8", "EUC-JP", str.toeuc)
      rescue => e
        "(char code problem: #{e.class})"
      end
    when 'UTF-8'
      str
    else
      raise
    end
  end
end

