unit AdminRestore;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, StdCtrls, ComCtrls,
  InstanceSections, AdditionalClasses,
  ImgList, Buttons, Contnrs, Menus, ApplicationDataModule, AuxFuncs,
  myx_public_interface, myx_admin_public_interface, SchemataTreeView,
  PNGImage, AuxAdminBackupRestore, MySQLConnection, MyxError,
  TntExtCtrls, TntMenus, TntComCtrls, TntStdCtrls, TntDialogs,gnugettext,
  AdminRestoreProgress;

type
  TRestoreOperationSet = set of TRestoreOperation;

  TAdminRestoreForm = class;


  TRestoreThread = class(TThread)
  private
   BackupError: MYX_BACKUP_ERROR;
   BackupContent: TMYX_BACKUP_CONTENT;
   AdminRestoreForm: TAdminRestoreForm;
   AdminRestoreProgressForm: TAdminRestoreProgressForm;
   PMysql: Pointer;
   restoreType: TRestoreOperationSet;
   internalErrorMsg: WideString;
   interrupted: Boolean;
   currOp: TRestoreOperation;
   procedure ShowLibError();
   procedure ShowInterrupted();
   procedure ShowInternalError();
   procedure ShowSuccess();
   procedure UpdateForm();
  protected
    procedure Execute; override;
  public
    constructor Create(  AdminRestoreForm: TAdminRestoreForm;
                         AdminRestoreProgressForm: TAdminRestoreProgressForm;
                         restoreType: TRestoreOperationSet);
    destructor Destroy;  override;
    procedure BuildContentTree();

    procedure RestoreWarning(msg: WideString);
    function RestoreProgress(bytes_read: Int64; bytes_total: Int64):Integer;
        
  published
    procedure Ending(Sender: TObject);
  end;

  TAdminRestoreForm = class(TInstanceSectionForm)
    ServerBackupRestorePnl: TTntPanel;
    BackupRestorePageControl: TTntPageControl;
    ContentSheet: TTabSheet;
    ItemSelectImageList: TImageList;
    Panel1: TTntPanel;
    RestoreBevel: TTntBevel;
    Label3: TTntLabel;
    RestoreImg: TTntImage;
    RestoreTreeView: TTntTreeView;
    RestoreTreeViewHeaderControl: THeaderControl;
    BackupTreeViewPopupMenu: TTntPopupMenu;
    RemoveBackupNodeMI: TTntMenuItem;
    SubTreeSearchPopupMenu: TTntPopupMenu;
    AllcatalogsMI: TTntMenuItem;
    system1: TTntMenuItem;
    dev1: TTntMenuItem;
    world1: TTntMenuItem;
    GeneralSheet: TTabSheet;
    Panel6: TTntPanel;
    BackupBevel: TTntBevel;
    Label6: TTntLabel;
    Restore2Img: TTntImage;
    Label7: TTntLabel;
    Label1: TTntLabel;
    GroupBox1: TTntGroupBox;
    Label9: TTntLabel;
    Label10: TTntLabel;
    Label11: TTntLabel;
    Label12: TTntLabel;
    Label13: TTntLabel;
    Label14: TTntLabel;
    FileToRestoreEd: TTntEdit;
    BackupFileTypesLU: TTntComboBox;
    TargetSchemaCBox: TTntComboBox;
    Panel7: TTntPanel;
    Panel8: TTntPanel;
    StartRestoreBtn: TTntButton;
    OpenBackupFileBtn: TTntButton;
    GroupBox2: TTntGroupBox;
    Label5: TTntLabel;
    Label8: TTntLabel;
    CharsetCBox: TTntComboBox;
    AutoDetectCharsetBtn: TTntButton;
    AutoDetectCharsetLbl: TTntLabel;
    AnalyzeBackupContentBtn: TTntButton;
    TntGroupBox1: TTntGroupBox;
    ForceCheckbox: TTntCheckBox;
    CreateDBsCheckbox: TTntCheckBox;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);

    procedure RefreshSchemaTree(searchStr: WideString = '');
    procedure ServerBackupRestorePnlResize(Sender: TObject);
    procedure BackupTreeViewAdvancedCustomDrawItem(Sender: TCustomTreeView;
      Node: TTreeNode; State: TCustomDrawState; Stage: TCustomDrawStage;
      var PaintImages, DefaultDraw: Boolean);
    procedure BackupTreeViewMouseDown(Sender: TObject;
      Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure SetChildrenSelectState(Node: TTntTreeNode);
    procedure CheckParentSelectState(Node: TTntTreeNode);
    function AllChildNodeSelected(Node: TTntTreeNode): TBackupCheckType;
    procedure RestoreTreeViewHeaderControlSectionResize(
      HeaderControl: THeaderControl; Section: THeaderSection);

    procedure SetNodeSelectState(Node: TTntTreeNode; Selected: TBackupCheckType);
    procedure SchemaTreeViewExpanding(Sender: TObject; Node: TTntTreeNode;
      var AllowExpansion: Boolean);

    procedure RemoveTreeNode(Node: TTntTreeNode);
    procedure SchemaTreeViewCollapsing(Sender: TObject; Node: TTntTreeNode;
      var AllowCollapse: Boolean);
    procedure SubTreeSeachEdMagnifyGlassImgClick(Sender: TObject);


    procedure OpenBackupFileBtnClick(Sender: TObject);
    procedure StartRestoreBtnClick(Sender: TObject);
    procedure ClearControls;

    procedure Disconnected(var Message: TMessage); message WM_Disconnected;
    procedure Reconnected(var Message: TMessage); message WM_Reconnected;
    procedure RestoreTreeViewChanging(Sender: TObject; Node: TTreeNode;
      var AllowChange: Boolean);
    procedure AnalyzeBackupContentBtnClick(Sender: TObject);
    procedure AutoDetectCharsetBtnClick(Sender: TObject);
    procedure TargetSchemaCBoxChange(Sender: TObject);
  private
    { Private declarations }
    //MousePos: TPoint;
    CatalogList: TMYX_CATALOGS;
    RestorePNGImg: TPNGObject;
    AdminRestoreProgressForm: TAdminRestoreProgressForm;
    procedure GetAllSchemas();
  public
    { Public declarations }
    CurrentCatalog: TMYX_CATALOG;
    CurrentSchema: TMYX_SCHEMA;

    //User must not expand SchemaTree manually
    LockSchemaTreeExpansion: Boolean;
    function getRestoreOptions(): Integer;
    procedure GetCatalogList(Sender: TObject);
    procedure RefreshCatalogList(Sender: TObject);
  end;

  function AnalyzeProgress(bytes_read: Int64; bytes_total: Int64; user_data: Pointer):Integer cdecl;
  function RestoreProgress(bytes_read: Int64; bytes_total: Int64; user_data: Pointer):Integer cdecl;
  procedure RestoreWarning(msg: PChar; user_data: Pointer) cdecl;

var
  AdminRestoreForm: TAdminRestoreForm;

implementation

{$R *.dfm}

uses
  Main, AdminBackup, NameEditor, PNGTools;

procedure TAdminRestoreForm.FormCreate(Sender: TObject);
begin
  InitForm(self);
  
  DockedPanel:=ServerBackupRestorePnl;

  BackupRestorePageControl.ActivePageIndex:=0;

  CurrentCatalog:=nil;
  CurrentSchema:=nil;

  LockSchemaTreeExpansion:=True;

  RestorePNGImg:=LoadPNGImageFromResource('restore', RestoreImg);
  Restore2Img.Picture.Assign(RestorePNGImg);

  DisableEnableControls(GeneralSheet, False);
  DisableEnableControls(ContentSheet, False);

  if(Not(MySQLConn.Connected))then
    OpenBackupFileBtn.Enabled:=False;

  BackupFileTypesLU.ItemIndex:=0;

  //No freeing necessary since the form should be automatically freeed when
  //this instance of TAdminRestoreForms is destroyed
  AdminRestoreProgressForm:= TAdminRestoreProgressForm.Create(self);
end;

procedure TAdminRestoreForm.FormDestroy(Sender: TObject);
begin
  RestorePNGImg.Free;
end;

procedure TAdminRestoreForm.FormClose(Sender: TObject;
  var Action: TCloseAction);
begin
  //
end;

procedure TAdminRestoreForm.RefreshSchemaTree(searchStr: WideString = '');
begin
  //
end;

procedure TAdminRestoreForm.ServerBackupRestorePnlResize(
  Sender: TObject);
begin
  RestoreTreeViewHeaderControl.Width:=RestoreTreeView.Width-2-18;
end;

procedure TAdminRestoreForm.BackupTreeViewAdvancedCustomDrawItem(
  Sender: TCustomTreeView; Node: TTreeNode; State: TCustomDrawState;
  Stage: TCustomDrawStage; var PaintImages, DefaultDraw: Boolean);
var NodeRect: TRect;
  XPos: integer;
  theHeaderControl: THeaderControl;
  i: integer;
begin
  if(Stage=cdPrePaint)then
  begin
    Sender.Canvas.Pen.Style:=psSolid;
    if(cdsFocused in State)then
      Sender.Canvas.Pen.Color:=clGray
    else
      Sender.Canvas.Pen.Color:=clSilver;

    NodeRect:=Node.DisplayRect(False);

    if(cdsSelected in State)then
    begin
      Sender.Canvas.Brush.Color:=clSilver;
      Sender.Canvas.Rectangle(NodeRect.Left+19+Node.Level*19+5-2, NodeRect.Top,
        Sender.Width-4, NodeRect.Bottom);
    end
    else
    begin
      Sender.Canvas.Brush.Color:=clWhite;
      Sender.Canvas.FillRect(Rect(NodeRect.Left+19+Node.Level*19+5-2, NodeRect.Top,
        Sender.Width-4, NodeRect.Bottom+1));
    end;

    DefaultDraw:=True;
  end
  else if(Stage=cdPostPaint)then
  begin
    if(Node.Data<>nil)then
    begin
      NodeRect:=Node.DisplayRect(False);

      if(cdsFocused in State)then
      begin
        Sender.Canvas.Pen.Color:=clGray;
        Sender.Canvas.Pen.Style:=psSolid;
        Sender.Canvas.Brush.Style:=bsClear;
        Sender.Canvas.Rectangle(NodeRect.Left+19+Node.Level*19+5-2, NodeRect.Top,
          Sender.Width-4, NodeRect.Bottom);

        Sender.Canvas.Brush.Color:=clSilver;
        Sender.Canvas.Brush.Style:=bsSolid;
        Sender.Canvas.FillRect(Rect(NodeRect.Left+19+Node.Level*19+5+
          Sender.Canvas.TextWidth(Node.Text)+1, NodeRect.Top+1,
          NodeRect.Left+19+Node.Level*19+5+
          Sender.Canvas.TextWidth(Node.Text)+1+5, NodeRect.Bottom-1));
      end;

      ItemSelectImageList.Draw(Sender.Canvas,
        NodeRect.Left+19+Node.Level*19+5, NodeRect.Top+1,
        Ord(TBackupNode(Node.Data).GetSelectState));

      if(TBackupNode(Node.Data).GetObjType=BOTTable)and
        (TBackupNode(Node.Data).Substrings<>nil)then
      begin
        {if(Sender.Name='BackupTreeView')then
          theHeaderControl:=BackupTreeViewHeaderControl
        else}
          theHeaderControl:=RestoreTreeViewHeaderControl;

        XPos:=0;
        for i:=0 to 3 do
        begin
          XPos:=XPos+theHeaderControl.Sections[i].Width;
          Sender.Canvas.TextOut(XPos+2, NodeRect.Top+2,
            TBackupNode(Node.Data).Substrings[i]);
        end;
      end;
    end;
  end;
end;

procedure TAdminRestoreForm.SetNodeSelectState(Node: TTntTreeNode; Selected: TBackupCheckType);
begin
  TBackupNode(Node.Data).SetSelectState(Selected);

  //Check/Uncheck all childs as well
  SetChildrenSelectState(Node);

  //See, if the current select state of the parent is valid
  CheckParentSelectState(Node);

  Node.TreeView.Invalidate;
end;

procedure TAdminRestoreForm.BackupTreeViewMouseDown(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var Node: TTntTreeNode;
  NodeRect: TRect;
  Selected: TBackupCheckType;
begin
  Node:=TTntTreeView(Sender).GetNodeAt(x, y);
  if(Assigned(Node))then
  begin
    NodeRect:=Node.DisplayRect(False);

    if(X>NodeRect.Left+19+Node.Level*19+5)and(X<NodeRect.Left+19+Node.Level*19+5+14)and
      (Node.Data<>nil)then
    begin
      if(TBackupNode(Node.Data).GetSelectState<>BCTAll)then
        Selected:=BCTAll
      else
        Selected:=BCTNone;

      SetNodeSelectState(Node, Selected);
    end;
  end;
end;

procedure TAdminRestoreForm.SetChildrenSelectState(Node: TTntTreeNode);
var ChildNode: TTntTreeNode;
begin
  //Set Select state for all childs
  ChildNode:=Node.getFirstChild;
  while(ChildNode<>nil)do
  begin
    if(ChildNode.Data<>nil)then
      TBackupNode(ChildNode.Data).SetSelectState(TBackupNode(Node.Data).GetSelectState);

    //Call recursive if the child has another child
    if(ChildNode.getFirstChild<>nil)then
      SetChildrenSelectState(ChildNode);

    ChildNode:=Node.GetNextChild(ChildNode);
  end;
end;

procedure TAdminRestoreForm.CheckParentSelectState(Node: TTntTreeNode);
begin
  if(Node.Parent<>nil)then
    if(Node.Parent.Data<>nil)then
    begin
      //Check, if all nodes under the parent are selected. If so, select the parent as well
      TBackupNode(Node.Parent.Data).SetSelectState(AllChildNodeSelected(Node.Parent));

      if(Node.Parent.Parent<>nil)then
        CheckParentSelectState(Node.Parent);
    end;
end;

function TAdminRestoreForm.AllChildNodeSelected(Node: TTntTreeNode): TBackupCheckType;
var ChildNode: TTntTreeNode;
  AllSelected, AtLeastOneSelected: Boolean;
  SubChildsSelectState: TBackupCheckType;
begin
  AllSelected:=True;
  AtLeastOneSelected:=False;

  //check if all child nodes are selected
  ChildNode:=Node.getFirstChild;
  while(ChildNode<>nil)do
  begin
    if(ChildNode.Data<>nil)then
    begin
      AllSelected:=AllSelected and
        (TBackupNode(ChildNode.Data).GetSelectState=BCTAll);

      AtLeastOneSelected:=AtLeastOneSelected or
        (TBackupNode(ChildNode.Data).GetSelectState=BCTAll) or
        (TBackupNode(ChildNode.Data).GetSelectState=BCTSome);
    end;

    //Call recursive if the child has another child
    if(ChildNode.getFirstChild<>nil)then
    begin
      SubChildsSelectState:=AllChildNodeSelected(ChildNode);
      AllSelected:=AllSelected and (SubChildsSelectState=BCTAll);
      AtLeastOneSelected:=AtLeastOneSelected or
        (SubChildsSelectState=BCTAll) or
        (SubChildsSelectState=BCTSome);
    end;

    ChildNode:=Node.GetNextChild(ChildNode);
  end;

  if(AllSelected)then
    AllChildNodeSelected:=BCTAll
  else if(AtLeastOneSelected)then
    AllChildNodeSelected:=BCTSome
  else
    AllChildNodeSelected:=BCTNone;
end;

procedure TAdminRestoreForm.RestoreTreeViewHeaderControlSectionResize(
  HeaderControl: THeaderControl; Section: THeaderSection);
begin
  RestoreTreeView.Invalidate;
end;

procedure TAdminRestoreForm.SchemaTreeViewExpanding(Sender: TObject;
  Node: TTntTreeNode; var AllowExpansion: Boolean);
begin
  //Expansion not allowed in Backup/Restore
  AllowExpansion:=Not(LockSchemaTreeExpansion);
end;

procedure TAdminRestoreForm.RemoveTreeNode(Node: TTntTreeNode);
begin
  //Remove Child nodes first
  while(Node.Count>0)do
    RemoveTreeNode(Node.Item[0]);

  TBackupNode(Node.Data).Free;

  TTntTreeView(Node.TreeView).Items.Delete(Node);
end;

procedure TAdminRestoreForm.SchemaTreeViewCollapsing(Sender: TObject;
  Node: TTntTreeNode; var AllowCollapse: Boolean);
begin
  if(Node.Data<>nil)then
    if(TObject(Node.Data) is TMYX_CATALOG)then
      AllowCollapse:=False;
end;

procedure TAdminRestoreForm.SubTreeSeachEdMagnifyGlassImgClick(
  Sender: TObject);
begin
  SubTreeSearchPopupMenu.Popup(Mouse.CursorPos.X, Mouse.CursorPos.Y);
end;

function Test: integer;
begin
  ShowMessage('Test');

  Result:=0;
end;


procedure TAdminRestoreForm.OpenBackupFileBtnClick(Sender: TObject);
var OpenDialog: TTntOpenDialog;
begin
  BackupRestorePageControl.ActivePage:=GeneralSheet;

  OpenDialog:=TTntOpenDialog.Create(self);
  try
    OpenDialog.Filter:=_('SQL Files')+'|*.sql|'+_('All Files')+'|*.*';

    if(OpenDialog.Execute)then
    begin
      ClearTreeView(RestoreTreeView, myx_ndt_tobject);

      DisableEnablePages(BackupRestorePageControl, True);
     { AutoDetectCharsetBtn.Enabled:=False;
      AutoDetectCharsetLbl.Enabled:=False;   }
      RestoreTreeViewHeaderControl.Enabled:=False;
      RestoreTreeView.Enabled:=False;

      FileToRestoreEd.Text:=OpenDialog.FileName;

      StartRestoreBtn.Enabled:=True;
      GetAllSchemas();
    end;
  finally
    OpenDialog.Free;
  end;
end;

procedure TAdminRestoreForm.GetAllSchemas();
begin
  if(CatalogList<>nil)then
  begin
    CatalogList.Free;
    CatalogList:=nil;
  end;

  //(Re-)load CatalogList
  //AllowDuplicatedKindOfData
  MySQLConn.FetchData(KindOfData_CatalogSchema,
    GetCatalogList, RefreshCatalogList,
    nil, _('Fetching Catalogs/Schemata data ...'),
    False, True);

end;

procedure TAdminRestoreForm.GetCatalogList(Sender: TObject);
var catalogs: PMYX_CATALOGS;
begin
  catalogs:=myx_get_catalogs(TFetchDataThread(Sender).Connection.MySQL);
  if(catalogs=nil)then
    raise EMyxSQLError.Create(_('Could not fetch Catalogs/Schemata data.'),
      myx_mysql_errno(TFetchDataThread(Sender).Connection.MySQL),
      myx_mysql_error(TFetchDataThread(Sender).Connection.MySQL));

  try
    CatalogList:=TMYX_CATALOGS.Create(catalogs);
  finally
    myx_free_catalogs(catalogs);
  end;
end;


procedure TAdminRestoreForm.RefreshCatalogList(Sender: TObject);
var i,j: integer;
begin
  TargetSchemaCBox.Clear;
  TargetSchemaCBox.AddItem(_('Original Schema'), nil);
  TargetSchemaCBox.AddItem(_('New Schema...'), nil);
  TargetSchemaCBox.ItemIndex:= 0;

  for i:=0 to CatalogList.Catalogs.Count-1 do
  begin
    for j:=0 to CatalogList.Catalogs[i].schemata.Count-1 do
    begin
      TargetSchemaCBox.AddItem(CatalogList.Catalogs[i].schemata[j].schema_name, nil);
    end;
  end;
end;



function AnalyzeProgress(bytes_read: Int64; bytes_total: Int64; user_data: Pointer):Integer;
var PSender: ^TRestoreThread;
begin
  PSender:=user_data;
  Result:= PSender.RestoreProgress(bytes_read, bytes_total);
end;


function RestoreProgress(bytes_read: Int64; bytes_total: Int64; user_data: Pointer):Integer;
var PSender: ^TRestoreThread;
begin
  PSender:=user_data;
  Result:= PSender.RestoreProgress(bytes_read, bytes_total);
end;

procedure RestoreWarning(msg: PChar; user_data: Pointer);
var PSender: ^TRestoreThread;
begin
  PSender:=user_data;
  PSender.RestoreWarning(UTF8Decode(msg));
end;






procedure TAdminRestoreForm.StartRestoreBtnClick(Sender: TObject);
var
  RestoreThread: TRestoreThread;
begin

  if (RestoreTreeView.Items.Count = 0) then
    RestoreThread:= TRestoreThread.Create(self, AdminRestoreProgressForm,
                                          [TR_ANALYZE, TR_RESTORE])
  else
    RestoreThread:=TRestoreThread.Create(self, AdminRestoreProgressForm,
                                         [TR_RESTORE]);

  RestoreThread.FreeOnTerminate:=True;
  //start the thread
  RestoreThread.Resume;
end;

procedure TAdminRestoreForm.ClearControls;
begin
  FileToRestoreEd.Text:='';
  TargetSchemaCBox.ItemIndex:=0;
  BackupFileTypesLU.ItemIndex:=0;
  ClearTreeView(RestoreTreeView, myx_ndt_tobject);
end;

procedure TAdminRestoreForm.Disconnected(var Message: TMessage);
begin
  DisableEnablePages(BackupRestorePageControl, False);
  ClearControls;

  OpenBackupFileBtn.Enabled:=False;
end;

procedure TAdminRestoreForm.Reconnected(var Message: TMessage);
begin
  OpenBackupFileBtn.Enabled:=True;
end;

procedure TAdminRestoreForm.RestoreTreeViewChanging(Sender: TObject;
  Node: TTreeNode; var AllowChange: Boolean);
begin
  if(RestoreTreeView.Items.Count>0)then
    if(Node=RestoreTreeView.Items[0])then
      AllowChange:=False;
end;

procedure TAdminRestoreForm.AnalyzeBackupContentBtnClick(Sender: TObject);
var
  thread: TRestoreThread;
begin
  //create a new thread where the file is analyzed
  if(RestoreTreeView.Items.Count=0)and
    (FileToRestoreEd.Text<>'')then
  begin
    thread:= TRestoreThread.Create(self, AdminRestoreProgressForm,[TR_ANALYZE]);
    thread.FreeOnTerminate:= true;
    thread.Resume;
  end;
end;

procedure TAdminRestoreForm.AutoDetectCharsetBtnClick(Sender: TObject);
var charset: WideString;
    itemInd: Integer;
begin
  charset:= myx_detect_charset_from_sql_file(FileToRestoreEd.Text);

  itemInd:= self.CharsetCBox.Items.IndexOf(charset);
  if itemInd= -1 then
  begin
    ShowModalDialog(_('Unsuccessful'),
      _('Could not determine the character-set of this file.'),
      myx_mtConfirmation, _('OK'));
  end
  else
  begin
    CharsetCBox.ItemIndex:= itemInd;
    ShowModalDialog(_('Character-set successfully determined'),
      format(_('This file is probably encoded in %s.'),[charset]),
      myx_mtInformation, _('OK') );
  end;
end;

function TAdminRestoreForm.getRestoreOptions(): Integer;
var options: Integer;
begin
  options:= 0;
  if self.ForceCheckbox.Checked then
    options:= options or Integer(MYX_RBS_FORCE);
  if self.CreateDBsCheckbox.Checked then
    options:= options or Integer(MYX_RBS_DONT_CREATE_TARGETS);
    
  Result:= options;
end;





//************************
//New Restore-Thread code
//************************

//Simple wrapper around BuildcontentTreeFromRestoreContent that can
//be used together with Synchronize()   
procedure TRestoreThread.BuildContentTree();
begin
  BuildContentTreeFromRestoreContent(AdminRestoreForm.RestoreTreeView,
                                     BackupContent);
  AdminRestoreForm.RestoreTreeViewHeaderControl.Enabled:=True;
  AdminRestoreForm.RestoreTreeView.Enabled:=True;
end;

//updates the ProgressForm
function TRestoreThread.RestoreProgress(bytes_read: Int64; bytes_total: Int64):Integer;
begin

  AdminRestoreProgressForm.UpdateProgress(bytes_read, bytes_total);
  if (AdminRestoreProgressForm.Stopping) then
  begin
    Result:= 1;
    interrupted:= true;
  end
  else
    Result:= 0;
end;

procedure TRestoreThread.RestoreWarning(msg: WideString);
begin

  AdminRestoreProgressForm.AddWarning(msg);
end;


//Simple wrapper that can be used together with Synchronize
procedure TRestoreThread.UpdateForm();
begin
  AdminRestoreProgressForm.UpdateForm(currOp,
                                      AdminRestoreForm.FileToRestoreEd.Text);
end;

procedure TRestoreThread.Execute();
var
  target_schema, target_catalog: WideString;
  BackupType: MYX_BACKUP_TYPE;
  PBackupContent: PMYX_BACKUP_CONTENT;
begin

  //Get Backup FileType
  case AdminRestoreForm.BackupFileTypesLU.ItemIndex of
    0:
      BackupType:=MYX_BT_SQL_SCRIPT;
  else
    BackupType:=MYX_BT_SQL_SCRIPT;
  end;

  Synchronize(AdminRestoreProgressForm.Show);

  if (TR_ANALYZE in restoreType) then
  begin
    currOp:= TR_ANALYZE;
    Synchronize(UpdateForm);

    //Get tables from backup file
    PBackupContent:=myx_get_backup_content(
      AdminRestoreForm.FileToRestoreEd.Text,
      AdminRestoreForm.CharsetCBox.Text, BackupType, 128,
      AnalyzeProgress, Addr(self), @BackupError);

    if((BackupError<>MYX_BACKUP_NO_ERROR) or (PBackupContent = nil))then
    begin
      exit;
    end;

    self.BackupContent:=TMYX_BACKUP_CONTENT.create(PBackupContent);
    myx_free_backup_content(PBackupContent);

    Synchronize(self.BuildContentTree);
  end;

  BackupContent.Free;

  if (TR_RESTORE in restoreType) then
  begin
    BackupContent:= GetBackupContent(AdminRestoreForm.RestoreTreeView);
    currOp:= TR_RESTORE;
    Synchronize(UpdateForm);

    target_catalog:='';

    if(AdminRestoreForm.TargetSchemaCBox.Text<>'Original Schema')then
      target_schema:=AdminRestoreForm.TargetSchemaCBox.Text
    else
      target_schema:='';

    BackupError:=myx_restore_backup(PMySQL,
      AdminRestoreForm.FileToRestoreEd.Text,
      AdminRestoreForm.CharsetCBox.Text,
      BackupContent.get_record_pointer,
      target_catalog, target_schema, BackupType,
      AdminRestoreForm.getRestoreOptions(), 512, Adminrestore.RestoreProgress,
      Addr(self), Adminrestore.RestoreWarning, Addr(self));

    if(BackupError<>MYX_BACKUP_NO_ERROR)then
    begin
      exit;
    end;      
  end;

  //Synchronize(AdminRestoreProgressForm.Hide);
end;

procedure TRestoreThread.ShowLibError();
var ErrorTxt: WideString;
begin
  ErrorTxt:=GetBackupErrorMsg(BackupError);
  case BackupError of
    MYX_BACKUP_MYSQL_ERROR:
      ErrorTxt:=Format(ErrorTxt, [myx_mysql_errno(PMySQL),
        myx_mysql_error(PMySQL)]);
    MYX_BACKUP_CANT_OPEN_FILE:
      ErrorTxt:=Format(ErrorTxt, [AdminRestoreForm.FileToRestoreEd.Text]);
  end;

  AdminRestoreProgressForm.OperationFinished(ErrorTxt);
end;

procedure TRestoreThread.ShowInterrupted();
begin
  if (currOp= TR_ANALYZE) then
    AdminRestoreProgressForm.OperationFinished( _('The analyze operation was stopped.'))
  else
    AdminRestoreProgressForm.OperationFinished( _('The restore operation was stopped.'));
end;

procedure TRestoreThread.ShowInternalError();
begin
  if (currOp= TR_ANALYZE) then
    AdminRestoreProgressForm.OperationFinished(_('The analyze operation was not finished successfully.')+internalErrorMsg)
  else
    AdminRestoreProgressForm.OperationFinished(_('The restore operation was not finished successfully.')+internalErrorMsg);
end;

procedure TRestoreThread.ShowSuccess();
begin
  if (currOp= TR_ANALYZE) then
    AdminRestoreProgressForm.OperationFinished(_('The analyze operation was finished successfully.'))
  else
    AdminRestoreProgressForm.OperationFinished(_('The restore operation was finished successfully.'));
end;

procedure TRestoreThread.Ending(Sender: TObject);
begin
  if (BackupError <> MYX_BACKUP_NO_ERROR) then
  begin
    Synchronize(ShowLibError);
  end
  else if (internalErrorMsg <> '') then
  begin
    Synchronize(ShowInternalError);
  end
  else if (interrupted) then
    Synchronize(ShowInterrupted)
  else
    Synchronize(ShowSuccess);

  AdminRestoreForm.MySQLConn.SchemaListChanged;
end;


constructor TRestoreThread.Create(
  AdminRestoreForm: TAdminRestoreForm;
  AdminRestoreProgressForm: TAdminRestoreProgressForm;
  restoreType: TRestoreOperationSet);
begin
  inherited Create(True);

  self.restoreType:= restoreType;
  self.AdminRestoreForm:= AdminRestoreForm;
  self.AdminRestoreProgressForm:= Adminrestoreprogressform;
  self.OnTerminate:= Ending;

  InternalErrorMsg:= '';
  interrupted:= false;
  
  PMySQL:=myx_mysql_init();
  if(PMySQL=nil)then
    InternalErrorMsg:= _('Error while allocating memory for MySQL Struct in the Restore Thread.');

  
  if(myx_connect_to_instance(
    AdminRestoreForm.MySQLConn.User_Connection.get_record_pointer,
    PMySQL)<>0)then
    InternalErrorMsg:= (_('Restore Thread cannot connect to MySQL'));
end;

destructor TRestoreThread.Destroy;
begin
  myx_mysql_close(PMySQL);

  inherited Destroy;
end;




procedure TAdminRestoreForm.TargetSchemaCBoxChange(Sender: TObject);
var newSchema: String;
begin
  if (TargetSchemaCBox.Items[TargetSchemaCBox.ItemIndex] = _('New Schema...')) then
  begin
    //show dialog-box
    NameEditorForm:= TNameEditorForm.Create(nil);
    try
      if (NameEditorForm.ShowModal() = mrOk) then
      begin
        newSchema:= NameEditorForm.NameEd.Text;
        TargetSchemaCBox.AddItem(newSchema, nil);
        TargetSchemaCBox.ItemIndex:= TargetSchemaCBox.Items.Count-1;
      end;
    finally
      NameEditorForm.Release();
    end;
  end;
end;

end.
