/*
 * Implementation of the fancy tooltip for displaying song information.
 *
 * Music Applet
 * Copyright (C) 2004-2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 *
 */

#include <config.h>

#include "ma-song-tooltip.h"
#include "ma-util.h"

#include <glib/gi18n.h>
#include <gtk/gtklabel.h>
#include <gtk/gtktooltips.h>
#include <gtk/gtkvbox.h>


#define GET_PRIVATE(o) 			(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_SONG_TOOLTIP, MaSongTooltipPrivate))


typedef struct _MaSongTooltipPrivate	MaSongTooltipPrivate;


struct _MaSongTooltipPrivate
{
	MaProxy *proxy;

	GtkTooltips *tooltips;
	GtkWidget *box;
	GtkWidget *title;
	GtkWidget *whence;
	GtkWidget *time;
};

static GObjectClass *parent_class;


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void ma_song_tooltip_class_init (MaSongTooltipClass *klass);
static void ma_song_tooltip_init (MaSongTooltip *tip);

static void ma_song_tooltip_dispose (GObject *object);
static void ma_song_tooltip_finalize (GObject *object);

static void update_visibility (MaSongTooltip *tip,
			       MaProxy *proxy);

static void update_time (MaSongTooltip *tip,
			 MaProxy *proxy);

static void title_changed_cb (MaProxy *proxy,
			      GParamSpec *pspec,
			      MaSongTooltip *tip);

static void whence_changed_cb (MaProxy *proxy,
			       GParamSpec *pspec,
			       MaSongTooltip *tip);

static void duration_changed_cb (MaProxy *proxy,
				 GParamSpec *pspec,
				 MaSongTooltip *tip);

static void elapsed_changed_cb (MaProxy *proxy,
				GParamSpec *pspec,
				MaSongTooltip *tip);

/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_song_tooltip_get_type (void)
{
	static GType this_type = 0;

	if (!this_type)
	{
		static const GTypeInfo this_info = {
			sizeof (MaSongTooltipClass),
			NULL,
			NULL,
			(GClassInitFunc) ma_song_tooltip_class_init,
			NULL,
			NULL,
			sizeof (MaSongTooltip),
			0,
			(GInstanceInitFunc) ma_song_tooltip_init,
			NULL
		};
		this_type = g_type_register_static (G_TYPE_OBJECT,
						    "MaSongTooltip",
						    &this_info,
						    0);
	}

	return this_type;
}

static void
ma_song_tooltip_class_init (MaSongTooltipClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;

	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_song_tooltip_dispose;
	object_class->finalize = ma_song_tooltip_finalize;

	g_type_class_add_private (klass, sizeof (MaSongTooltipPrivate));
}

static void
ma_song_tooltip_init (MaSongTooltip *tip)
{
	MaSongTooltipPrivate *priv = GET_PRIVATE (tip);

	priv->proxy = NULL;

	priv->tooltips = gtk_tooltips_new ();
	g_object_ref (priv->tooltips);
	gtk_object_sink (GTK_OBJECT (priv->tooltips));
	gtk_tooltips_force_window (priv->tooltips);

	priv->box = gtk_vbox_new (FALSE, 2);

	priv->title = gtk_label_new ("");
	gtk_misc_set_alignment (GTK_MISC (priv->title), 0.0, 0.5);
	gtk_label_set_justify (GTK_LABEL (priv->title), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start_defaults (GTK_BOX (priv->box), priv->title);

	priv->whence = gtk_label_new ("");
	gtk_misc_set_alignment (GTK_MISC (priv->whence), 0.0, 0.5);
	gtk_label_set_justify (GTK_LABEL (priv->whence), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start_defaults (GTK_BOX (priv->box), priv->whence);

	priv->time = gtk_label_new ("");
	gtk_misc_set_alignment (GTK_MISC (priv->time), 0.0, 0.5);
	gtk_label_set_justify (GTK_LABEL (priv->time), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start_defaults (GTK_BOX (priv->box), priv->time);

	/* Such a horrible abuse of GtkTooltips... */
	g_object_ref (priv->tooltips->tip_label);
	gtk_container_remove (GTK_CONTAINER (priv->tooltips->tip_window),
			      priv->tooltips->tip_label);
	gtk_container_add (GTK_CONTAINER (priv->tooltips->tip_window),
			   priv->box);

	gtk_widget_show (priv->title);
	gtk_widget_show (priv->box);
}

static void
ma_song_tooltip_dispose (GObject *object)
{
	MaSongTooltipPrivate *priv = GET_PRIVATE (object);

	g_signal_handlers_disconnect_by_func (priv->proxy, title_changed_cb, object);
	g_signal_handlers_disconnect_by_func (priv->proxy, whence_changed_cb, object);
	g_signal_handlers_disconnect_by_func (priv->proxy, duration_changed_cb, object);
	g_signal_handlers_disconnect_by_func (priv->proxy, elapsed_changed_cb, object);

	g_object_unref (priv->tooltips->tip_label);
	g_object_unref (priv->tooltips);
	g_object_unref (priv->proxy);

	parent_class->dispose (object);
}

static void
ma_song_tooltip_finalize (GObject *object)
{
	parent_class->finalize (object);
}

MaSongTooltip *
ma_song_tooltip_new (MaProxy *proxy)
{
	MaSongTooltip *tip;
	MaSongTooltipPrivate *priv;

	g_return_val_if_fail (proxy != NULL, NULL);

	tip = g_object_new (MA_TYPE_SONG_TOOLTIP, NULL);
	priv = GET_PRIVATE (tip);

	priv->proxy = proxy;
	g_object_ref (proxy);

	g_signal_connect (proxy, "notify::title", G_CALLBACK (title_changed_cb), tip);
	g_signal_connect (proxy, "notify::artist", G_CALLBACK (whence_changed_cb), tip);
	g_signal_connect (proxy, "notify::album", G_CALLBACK (whence_changed_cb), tip);
	g_signal_connect (proxy, "notify::duration", G_CALLBACK (duration_changed_cb), tip);
	g_signal_connect (proxy, "notify::elapsed", G_CALLBACK (elapsed_changed_cb), tip);

	/* Abuse the event handlers to initialize the display. */
	title_changed_cb (priv->proxy, NULL, tip);
	whence_changed_cb (priv->proxy, NULL, tip);
	duration_changed_cb (priv->proxy, NULL, tip);
	elapsed_changed_cb (priv->proxy, NULL, tip);

	return tip;
}


/*********************************************************************
 *
 * Public interface
 *
 *********************************************************************/

void
ma_song_tooltip_attach (MaSongTooltip *tip, GtkWidget *owner)
{
	g_return_if_fail (tip != NULL);
	g_return_if_fail (MA_IS_SONG_TOOLTIP (tip));
	g_return_if_fail (owner != NULL);

	gtk_tooltips_set_tip (GET_PRIVATE (tip)->tooltips, owner, "", "");
}


/*********************************************************************
 *
 * Internal functions
 *
 *********************************************************************/

static void
update_visibility (MaSongTooltip *tip, MaProxy *proxy)
{
	MaSongTooltipPrivate *priv = GET_PRIVATE (tip);

	if (ma_proxy_get_title (proxy) != NULL)
	{
		/*
		 * Internet radio streams don't provide album and artist information
		 * as fields separate from the title, so don't bother showing it.
		 */

		if (ma_proxy_get_duration (proxy) > 0)
			gtk_widget_show (priv->whence);
		else
			gtk_widget_hide (priv->whence);

		gtk_widget_show (priv->time);
	}
	else
	{
		gtk_widget_hide (priv->whence);
		gtk_widget_hide (priv->time);
	}
}

static void
update_time (MaSongTooltip *tip, MaProxy *proxy)
{
	MaSongTooltipPrivate *priv = GET_PRIVATE (tip);
	gint elapsed = ma_proxy_get_elapsed (proxy);
	gint duration = (gint) ma_proxy_get_duration (proxy);

	gchar *text;

	if (duration > 0)
	{
		/* Playing an ordinary file. */
		text = ma_util_format_elapsed_duration (elapsed, duration);
	}
	else
	{
		/* Playing an Internet radio stream. */
		text = ma_util_format_elapsed (elapsed);
	}

	gtk_label_set_text (GTK_LABEL (priv->time), text);
	g_free (text);
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static void
title_changed_cb (MaProxy *proxy,
		  GParamSpec *pspec,
		  MaSongTooltip *tip)
{
	MaSongTooltipPrivate *priv = GET_PRIVATE (tip);
	const gchar *title = ma_proxy_get_title (proxy);

	update_visibility (tip, proxy);

	if (title != NULL)
	{
		gchar *text = g_markup_printf_escaped ("<big><b>%s</b></big>", title);
		gtk_label_set_markup (GTK_LABEL (priv->title), text);
		g_free (text);
	}
	else
		gtk_label_set_text (GTK_LABEL (priv->title), _("Not playing"));
}

static void
whence_changed_cb (MaProxy *proxy,
		   GParamSpec *pspec,
		   MaSongTooltip *tip)
{
	MaSongTooltipPrivate *priv = GET_PRIVATE (tip);
	const gchar *album = ma_proxy_get_album (proxy);
	const gchar *artist = ma_proxy_get_artist (proxy);

	gchar *text;

	if (album == NULL)
		album = _("Unknown");
	if (artist == NULL)
		artist = _("Unknown");

	/* To translator: from <i>ALBUM</i> by <i>ARTIST</i> */
	text = g_markup_printf_escaped (_("from <i>%s</i> by <i>%s</i>"), album, artist);
	gtk_label_set_markup (GTK_LABEL (priv->whence), text);
	g_free (text);
}

static void
duration_changed_cb (MaProxy *proxy,
		     GParamSpec *pspec,
		     MaSongTooltip *tip)
{
	update_visibility (tip, proxy);
	update_time (tip, proxy);
}

static void
elapsed_changed_cb (MaProxy *proxy,
		    GParamSpec *pspec,
		    MaSongTooltip *tip)
{
	update_time (tip, proxy);
}
