/*
 * Implementation of the proxy base class for communicating with a music player.
 *
 * Music Applet
 * Copyright (C) 2004-2006 Paul Kuliniewicz <paul.kuliniewicz@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02111-1301, USA.
 *
 */

#include <config.h>

#include "ma-proxy.h"

#include "ma-marshal.h"

#include <glib/gprintf.h>


#define GET_PRIVATE(o) 			(G_TYPE_INSTANCE_GET_PRIVATE ((o), MA_TYPE_PROXY, MaProxyPrivate))
#define ENUM_ENTRY(name, desc) 		{ name, #name, desc }

#define DISCONNECTED_INTERVAL		2000 	/* milliseconds */
#define CONNECTED_INTERVAL		500 	/* milliseconds */
#define STREAM_REFRESH_INTERVAL		3 	/* seconds */


typedef struct _MaProxyPrivate		MaProxyPrivate;


struct _MaProxyPrivate
{
	gchar *title;
	gchar *artist;
	gchar *album;
	glong duration;

	gint elapsed;
	gdouble rating;

	MaProxyState state;
	gboolean playing;

	gchar *name;
	gchar *icon_name;

	guint disconnected_source;
	guint connected_source;

	MaConf *conf;
};

typedef enum
{
	PROP_NONE,
	PROP_TITLE,
	PROP_ARTIST,
	PROP_ALBUM,
	PROP_DURATION,
	PROP_ELAPSED,
	PROP_RATING,
	PROP_STATE,
	PROP_PLAYING,
	PROP_NAME,
	PROP_ICON_NAME,
	PROP_CONF,
} MaProxyProperty;

typedef enum
{
	ERROR_REPORTED,
	LAST_SIGNAL,
} MaProxySignal;

static GObjectClass *parent_class;

static guint ma_proxy_signals[LAST_SIGNAL] = { 0 };


/*********************************************************************
 *
 * Function declarations
 *
 *********************************************************************/

static void	ma_proxy_class_init (MaProxyClass *klass);
static void	ma_proxy_init (MaProxy *proxy);

static void	ma_proxy_dispose (GObject *object);
static void	ma_proxy_finalize (GObject *object);

static void	ma_proxy_get_property (GObject *object,
				       guint prop_id,
				       GValue *value,
				       GParamSpec *pspec);

static void	ma_proxy_set_property (GObject *object,
				       guint prop_id,
				       const GValue *value,
				       GParamSpec *pspec);

static void	transition_state (MaProxy *proxy,
				  MaProxyState new_state);

static gboolean	poll_disconnected_cb (gpointer data);
static gboolean	poll_connected_cb (gpointer data);


/*********************************************************************
 *
 * GType stuff
 *
 *********************************************************************/

GType
ma_proxy_state_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GEnumValue values[] = {
			ENUM_ENTRY (MA_PROXY_STATE_DISABLED, "Disabled"),
			ENUM_ENTRY (MA_PROXY_STATE_DISCONNECTED, "Disconnected from the player"),
			ENUM_ENTRY (MA_PROXY_STATE_CONNECTED, "Connected to the player"),
			{ 0, NULL, NULL }
		};

		type = g_enum_register_static ("MaProxyStateType", values);
	}

	return type;
}

GType
ma_proxy_get_type (void)
{
	static GType type = 0;

	if (type == 0)
	{
		static const GTypeInfo info = {
			sizeof (MaProxyClass),				/* class_size */
			NULL,						/* base_init */
			NULL,						/* base_finalize */
			(GClassInitFunc) ma_proxy_class_init,		/* class_init */
			NULL,						/* class_finalize */
			NULL,						/* class_data */
			sizeof (MaProxy),				/* instance_size */
			0,						/* n_preallocs */
			(GInstanceInitFunc) ma_proxy_init,		/* instance_init */
			NULL
		};

		type = g_type_register_static (G_TYPE_OBJECT, "MaProxy", &info, G_TYPE_FLAG_ABSTRACT);
	}

	return type;
}

static void
ma_proxy_class_init (MaProxyClass *klass)
{
	GObjectClass *object_class = (GObjectClass *) klass;
	parent_class = g_type_class_peek_parent (klass);

	object_class->dispose = ma_proxy_dispose;
	object_class->finalize = ma_proxy_finalize;

	object_class->get_property = ma_proxy_get_property;
	object_class->set_property = ma_proxy_set_property;

	g_object_class_install_property (object_class,
					 PROP_TITLE,
					 g_param_spec_string ("title",
						 	      "title",
							      "Title of the current song",
							      NULL,
							      G_PARAM_READABLE));

	g_object_class_install_property (object_class,
					 PROP_ARTIST,
					 g_param_spec_string ("artist",
						 	      "artist",
							      "Artist who made the current song",
							      NULL,
							      G_PARAM_READABLE));

	g_object_class_install_property (object_class,
					 PROP_ALBUM,
					 g_param_spec_string ("album",
						 	      "album",
							      "Album the current song is from",
							      NULL,
							      G_PARAM_READABLE));

	g_object_class_install_property (object_class,
					 PROP_DURATION,
					 g_param_spec_long ("duration",
						 	    "duration",
							    "Duration in seconds of the current song",
							    0, G_MAXLONG,
							    0,
							    G_PARAM_READABLE));

	g_object_class_install_property (object_class,
					 PROP_ELAPSED,
					 g_param_spec_int ("elapsed",
							   "elapsed",
							   "Elapsed time, in seconds",
							   -1, G_MAXINT,
							   -1,
							   G_PARAM_READABLE));

	g_object_class_install_property (object_class,
					 PROP_RATING,
					 g_param_spec_double ("rating",
							      "rating",
							      "Rating of current song",
							      0.0, 5.0,
							      0.0,
							      G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_STATE,
					 g_param_spec_enum ("state",
							    "state",
							    "Connection state",
							    MA_TYPE_PROXY_STATE,
							    MA_PROXY_STATE_DISABLED,
							    G_PARAM_READABLE));

	g_object_class_install_property (object_class,
					 PROP_PLAYING,
					 g_param_spec_boolean ("playing",
							       "playing",
							       "Whether the current song is playing",
							       FALSE,
							       G_PARAM_READABLE));

	g_object_class_install_property (object_class,
					 PROP_NAME,
					 g_param_spec_string ("name",
							      "name",
							      "What the proxy connects to",
							      NULL,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class,
					 PROP_ICON_NAME,
					 g_param_spec_string ("icon-name",
						 	      "icon-name",
							      "Name of icon of what the proxy connects to",
							      NULL,
							      G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY));

	g_object_class_install_property (object_class,
					 PROP_CONF,
					 g_param_spec_object ("conf",
						 	      "conf",
							      "Storage for configuration information",
							      MA_TYPE_CONF,
							      G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));

	ma_proxy_signals[ERROR_REPORTED] =
		g_signal_new ("error-reported",
			      MA_TYPE_PROXY,
			      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
			      G_STRUCT_OFFSET (MaProxyClass, error_reported),
			      NULL, NULL,
			      ma_marshal_VOID__STRING_STRING_BOOLEAN,
			      G_TYPE_NONE,
			      3,
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN);

	g_type_class_add_private (klass, sizeof (MaProxyPrivate));
}

static void
ma_proxy_init (MaProxy *proxy)
{
	MaProxyPrivate *priv = GET_PRIVATE (proxy);

	priv->title = NULL;
	priv->artist = NULL;
	priv->album = NULL;
	priv->duration = 0;

	priv->elapsed = -1;
	priv->rating = 0.0;

	priv->state = MA_PROXY_STATE_DISABLED;
	priv->playing = FALSE;

	priv->name = NULL;
	priv->icon_name = NULL;

	priv->disconnected_source = 0;
	priv->connected_source = 0;

	priv->conf = NULL;
}


/*********************************************************************
 *
 * Public interface
 *
 *********************************************************************/

const gchar *
ma_proxy_get_title (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, NULL);
	g_return_val_if_fail (MA_IS_PROXY (proxy), NULL);

	return GET_PRIVATE (proxy)->title;
}

const gchar *
ma_proxy_get_artist (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, NULL);
	g_return_val_if_fail (MA_IS_PROXY (proxy), NULL);

	return GET_PRIVATE (proxy)->artist;
}

const gchar *
ma_proxy_get_album (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, NULL);
	g_return_val_if_fail (MA_IS_PROXY (proxy), NULL);

	return GET_PRIVATE (proxy)->album;
}

glong
ma_proxy_get_duration (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, 0);
	g_return_val_if_fail (MA_IS_PROXY (proxy), 0);

	return GET_PRIVATE (proxy)->duration;
}

gint
ma_proxy_get_elapsed (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, -1);
	g_return_val_if_fail (MA_IS_PROXY (proxy), -1);

	return GET_PRIVATE (proxy)->elapsed;

#if 0
	return (priv->song != NULL) ? priv->elapsed : -1;
#endif
}

gdouble
ma_proxy_get_rating (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, 0.0);
	g_return_val_if_fail (MA_IS_PROXY (proxy), 0.0);

	return GET_PRIVATE (proxy)->rating;
}

void
ma_proxy_set_rating (MaProxy *proxy, gdouble rating)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (rating >= 0.0);
	g_return_if_fail (rating <= 5.0);

	GET_PRIVATE (proxy)->rating = rating;
	MA_PROXY_GET_CLASS (proxy)->rate_song (proxy, rating);

	g_object_notify (G_OBJECT (proxy), "rating");
}

MaProxyState
ma_proxy_get_state (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, MA_PROXY_STATE_DISABLED);
	g_return_val_if_fail (MA_IS_PROXY (proxy), MA_PROXY_STATE_DISABLED);

	return GET_PRIVATE (proxy)->state;
}

gboolean
ma_proxy_get_playing (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, FALSE);
	g_return_val_if_fail (MA_IS_PROXY (proxy), FALSE);

	return GET_PRIVATE (proxy)->playing;
}

const gchar *
ma_proxy_get_name (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, NULL);
	g_return_val_if_fail (MA_IS_PROXY (proxy), NULL);

	return GET_PRIVATE (proxy)->name;
}

const gchar *
ma_proxy_get_icon_name (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, NULL);
	g_return_val_if_fail (MA_IS_PROXY (proxy), NULL);

	return GET_PRIVATE (proxy)->icon_name;
}

void
ma_proxy_toggle_playback (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (GET_PRIVATE (proxy)->state == MA_PROXY_STATE_CONNECTED);

	MA_PROXY_GET_CLASS (proxy)->toggle_playback (proxy);
}

void
ma_proxy_previous (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (GET_PRIVATE (proxy)->state == MA_PROXY_STATE_CONNECTED);

	MA_PROXY_GET_CLASS (proxy)->previous (proxy);
}

void
ma_proxy_next (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (GET_PRIVATE (proxy)->state == MA_PROXY_STATE_CONNECTED);

	MA_PROXY_GET_CLASS (proxy)->next (proxy);
}

void
ma_proxy_prepare_prefs (MaProxy *proxy, GladeXML *xml)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (xml != NULL);

	MA_PROXY_GET_CLASS (proxy)->prepare_prefs (proxy, xml);
}

void
ma_proxy_enable (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));

	if (GET_PRIVATE (proxy)->state == MA_PROXY_STATE_DISABLED)
	{
		transition_state (proxy, MA_PROXY_STATE_DISCONNECTED);
		MA_PROXY_GET_CLASS (proxy)->enable (proxy);
	}
}

void
ma_proxy_disable (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));

	if (GET_PRIVATE (proxy)->state != MA_PROXY_STATE_DISABLED)
	{
		transition_state (proxy, MA_PROXY_STATE_DISABLED);
		MA_PROXY_GET_CLASS (proxy)->disable (proxy);
	}
}

void
ma_proxy_launch (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (GET_PRIVATE (proxy)->state == MA_PROXY_STATE_DISCONNECTED);

	MA_PROXY_GET_CLASS (proxy)->launch (proxy);
}


/*********************************************************************
 *
 * Protected interface
 *
 *********************************************************************/

void
_ma_proxy_set_connected (MaProxy *proxy, gboolean connected)
{
	MaProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	/* It's no error for a (newly) disabled proxy to report it's not connected. */

	g_return_if_fail (priv->state != MA_PROXY_STATE_DISABLED || !connected);

	if (priv->state != MA_PROXY_STATE_DISABLED)
		transition_state (proxy, connected ? MA_PROXY_STATE_CONNECTED : MA_PROXY_STATE_DISCONNECTED);
}

void
_ma_proxy_set_no_song (MaProxy *proxy)
{
	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));

	_ma_proxy_set_title (proxy, NULL);
	_ma_proxy_set_artist (proxy, NULL);
	_ma_proxy_set_album (proxy, NULL);
	_ma_proxy_set_duration (proxy, 0);
	_ma_proxy_set_elapsed (proxy, -1);
	_ma_proxy_set_rating (proxy, 0.0);
	_ma_proxy_set_playing (proxy, FALSE);
}

void
_ma_proxy_set_title (MaProxy *proxy, const gchar *title)
{
	MaProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	g_free (priv->title);
	priv->title = g_strdup (title);
	g_object_notify (G_OBJECT (proxy), "title");
}

void
_ma_proxy_set_artist (MaProxy *proxy, const gchar *artist)
{
	MaProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	g_free (priv->artist);
	priv->artist = g_strdup (artist);
	g_object_notify (G_OBJECT (proxy), "artist");
}

void
_ma_proxy_set_album (MaProxy *proxy, const gchar *album)
{
	MaProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	g_free (priv->album);
	priv->album = g_strdup (album);
	g_object_notify (G_OBJECT (proxy), "album");
}

void
_ma_proxy_set_duration (MaProxy *proxy, glong duration)
{
	MaProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (duration >= 0);

	priv = GET_PRIVATE (proxy);

	if (priv->duration != duration)
	{
		GET_PRIVATE (proxy)->duration = duration;
		g_object_notify (G_OBJECT (proxy), "duration");
	}
}

void
_ma_proxy_set_rating (MaProxy *proxy, gdouble rating)
{
	MaProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (rating >= 0.0);
	g_return_if_fail (rating <= 5.0);

	priv = GET_PRIVATE (proxy);

	if (priv->rating != rating)
	{
		priv->rating = rating;
		g_object_notify (G_OBJECT (proxy), "rating");
	}
}

void
_ma_proxy_set_playing (MaProxy *proxy, gboolean playing)
{
	MaProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	if (priv->playing != playing)
	{
		priv->playing = playing;
		g_object_notify (G_OBJECT (proxy), "playing");
	}
}

void
_ma_proxy_set_elapsed (MaProxy *proxy, gint elapsed)
{
	MaProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (elapsed >= -1);

	priv = GET_PRIVATE (proxy);

	if (priv->elapsed != elapsed)
	{
		priv->elapsed = elapsed;
		g_object_notify (G_OBJECT (proxy), "elapsed");
	}
}

MaConf *
_ma_proxy_get_conf (MaProxy *proxy)
{
	g_return_val_if_fail (proxy != NULL, NULL);
	g_return_val_if_fail (MA_IS_PROXY (proxy), NULL);

	return GET_PRIVATE (proxy)->conf;
}

void
_ma_proxy_set_icon_name (MaProxy *proxy, const gchar *icon_name)
{
	MaProxyPrivate *priv;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));

	priv = GET_PRIVATE (proxy);

	g_free (priv->icon_name);
	priv->icon_name = g_strdup (icon_name);
	g_object_notify (G_OBJECT (proxy), "icon-name");
}

void
_ma_proxy_launch_command (MaProxy *proxy, const gchar *key)
{
	MaProxyPrivate *priv;
	gchar *command;
	GError *error = NULL;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (key != NULL);

	priv = GET_PRIVATE (proxy);

	command = ma_conf_get_string (priv->conf, key);
	g_spawn_command_line_async (command, &error);

	if (error != NULL)
	{
		gchar *primary = g_strdup_printf (_("Failed to launch %s"), priv->name);
		_ma_proxy_report_error (proxy, primary, error->message, TRUE);
		g_free (primary);
		g_error_free (error);
	}

	g_free (command);
}

gboolean
_ma_proxy_should_refresh_stream_metadata (MaProxy *proxy)
{
	MaProxyPrivate *priv;

	g_return_val_if_fail (proxy != NULL, FALSE);
	g_return_val_if_fail (MA_IS_PROXY (proxy), FALSE);

	priv = GET_PRIVATE (proxy);

	/* Once every STREAM_REFRESH_INTERVAL, but not immediately. */

	return (priv->playing && (priv->elapsed + 1) % STREAM_REFRESH_INTERVAL == 0);
}

void
_ma_proxy_report_error (MaProxy *proxy,
			const gchar *primary,
			const gchar *secondary,
			gboolean important)
{
	const gchar *name;

	g_return_if_fail (proxy != NULL);
	g_return_if_fail (MA_IS_PROXY (proxy));
	g_return_if_fail (primary != NULL);
	
	name = (important) ? "error-reported::important" : "error-reported::normal";

	if (secondary != NULL)
		g_printf("%s (%s)\n", primary, secondary);
	else
		g_printf("%s\n", primary);

	g_signal_emit_by_name (proxy, name, primary, secondary, important);
}


/*********************************************************************
 *
 * GObject overrides
 *
 *********************************************************************/

static void
ma_proxy_dispose (GObject *object)
{
	MaProxy *proxy = MA_PROXY (object);
	MaProxyPrivate *priv = GET_PRIVATE (proxy);

	if (priv->disconnected_source != 0)
	{
		g_source_remove (priv->disconnected_source);
		priv->disconnected_source = 0;
	}

	if (priv->connected_source != 0)
	{
		g_source_remove (priv->connected_source);
		priv->connected_source = 0;
	}

	if (priv->conf != NULL)
	{
		g_object_unref (priv->conf);
		priv->conf = NULL;
	}

	parent_class->dispose (object);
}

static void
ma_proxy_finalize (GObject *object)
{
	MaProxy *proxy = MA_PROXY (object);
	MaProxyPrivate *priv = GET_PRIVATE (proxy);

	g_free (priv->title);
	g_free (priv->artist);
	g_free (priv->album);

	g_free (priv->name);
	g_free (priv->icon_name);

	parent_class->finalize (object);
}

static void
ma_proxy_get_property (GObject *object,
		       guint prop_id,
		       GValue *value,
		       GParamSpec *pspec)
{
	MaProxy *proxy = MA_PROXY (object);

	switch (prop_id)
	{
	case PROP_TITLE:
		g_value_set_string (value, ma_proxy_get_title (proxy));
		break;

	case PROP_ARTIST:
		g_value_set_string (value, ma_proxy_get_artist (proxy));
		break;

	case PROP_ALBUM:
		g_value_set_string (value, ma_proxy_get_album (proxy));
		break;

	case PROP_DURATION:
		g_value_set_long (value, ma_proxy_get_duration (proxy));
		break;

	case PROP_ELAPSED:
		g_value_set_int (value, ma_proxy_get_elapsed (proxy));
		break;

	case PROP_RATING:
		g_value_set_double (value, ma_proxy_get_rating (proxy));
		break;

	case PROP_STATE:
		g_value_set_enum (value, ma_proxy_get_state (proxy));
		break;

	case PROP_PLAYING:
		g_value_set_boolean (value, ma_proxy_get_playing (proxy));
		break;

	case PROP_NAME:
		g_value_set_string (value, ma_proxy_get_name (proxy));
		break;

	case PROP_ICON_NAME:
		g_value_set_string (value, ma_proxy_get_icon_name (proxy));
		break;

	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}

static void
ma_proxy_set_property (GObject *object,
		       guint prop_id,
		       const GValue *value,
		       GParamSpec *pspec)
{
	MaProxy *proxy = MA_PROXY (object);
	MaProxyPrivate *priv = GET_PRIVATE (proxy);

	switch (prop_id)
	{
	case PROP_RATING:
		ma_proxy_set_rating (proxy, g_value_get_double (value));
		break;

	case PROP_NAME:
		g_free (priv->name);
		priv->name = g_value_dup_string (value);
		break;

	case PROP_ICON_NAME:
		_ma_proxy_set_icon_name (proxy, g_value_get_string (value));
		break;

	case PROP_CONF:
		if (priv->conf != NULL)
			g_object_unref (priv->conf);
		priv->conf = MA_CONF (g_value_dup_object (value));
		break;

	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


/*********************************************************************
 *
 * Internal functions
 *
 *********************************************************************/

static void
transition_state (MaProxy *proxy, MaProxyState new_state)
{
	MaProxyPrivate *priv = GET_PRIVATE (proxy);

	if (priv->state == new_state)
		return;

	/* Leave old state */

	switch (priv->state)
	{
	case MA_PROXY_STATE_DISABLED:
		break;

	case MA_PROXY_STATE_DISCONNECTED:
		if (priv->disconnected_source != 0)
		{
			g_source_remove (priv->disconnected_source);
			priv->disconnected_source = 0;
		}
		break;

	case MA_PROXY_STATE_CONNECTED:
		if (priv->connected_source != 0)
		{
			g_source_remove (priv->connected_source);
			priv->connected_source = 0;
		}

		if (priv->title != NULL)
		{
			g_free (priv->title);
			priv->title = NULL;
			g_object_notify (G_OBJECT (proxy), "title");
		}

		if (priv->artist != NULL)
		{
			g_free (priv->artist);
			priv->artist = NULL;
			g_object_notify (G_OBJECT (proxy), "artist");
		}

		if (priv->album != NULL)
		{
			g_free (priv->album);
			priv->album = NULL;
			g_object_notify (G_OBJECT (proxy), "album");
		}

		if (priv->rating > 0.0)
		{
			priv->rating = 0.0;
			g_object_notify (G_OBJECT (proxy), "rating");
		}

		if (priv->playing)
		{
			priv->playing = FALSE;
			g_object_notify (G_OBJECT (proxy), "playing");
		}

		if (priv->elapsed != -1)
		{
			priv->elapsed = -1;
			g_object_notify (G_OBJECT (proxy), "elapsed");
		}

		break;
	}

	/* Enter new state */

	priv->state = new_state;

	switch (priv->state)
	{
	case MA_PROXY_STATE_DISABLED:
		break;

	case MA_PROXY_STATE_DISCONNECTED:
		if ((MA_PROXY_GET_CLASS (proxy))->poll_disconnected != NULL)
		{
			priv->disconnected_source = g_timeout_add (DISCONNECTED_INTERVAL,
								   poll_disconnected_cb, proxy);
		}
		break;

	case MA_PROXY_STATE_CONNECTED:
		if ((MA_PROXY_GET_CLASS (proxy))->poll_connected != NULL)
		{
			priv->connected_source = g_timeout_add (CONNECTED_INTERVAL,
								poll_connected_cb, proxy);
		}
		break;
	}

	g_object_notify (G_OBJECT (proxy), "state");
}


/*********************************************************************
 *
 * Callbacks
 *
 *********************************************************************/

static gboolean
poll_disconnected_cb (gpointer data)
{
	MaProxy *proxy = MA_PROXY (data);

	MA_PROXY_GET_CLASS (proxy)->poll_disconnected (proxy);

	return TRUE;
}

static gboolean
poll_connected_cb (gpointer data)
{
	MaProxy *proxy = MA_PROXY (data);

	MA_PROXY_GET_CLASS (proxy)->poll_connected (proxy);

	return TRUE;
}
