//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: sndfile.h,v 1.3 2002/02/13 11:42:56 muse Exp $
//
//  (C) Copyright 2001 Werner Schweer (ws@seh.de)
//  parts based on libsndfile:
//  Copyright (C) 1999-2000 Erik de Castro Lopo <erikd@zip.com.au>
//=========================================================

#ifndef SNDFILE_H
#define SNDFILE_H

#include <list>
#include <stdio.h>
#include <stdlib.h>

#include <sys/types.h>
#include <qstring.h>
#include <qfileinfo.h>

const int cacheMag = 128;

#define SF_BUFFER_LEN	 4096
#define SF_FILENAME_LEN	 256
#define SF_HEADER_LEN	 2048

#define BITWIDTH2BYTES(x) (((x) + 7) / 8)

enum {	
      SF_MODE_READ = 1,
	SF_MODE_WRITE = 2,
	SF_MODE_RW = 3
      };

enum {	
      SF_ENDIAN_LITTLE = 100,
	SF_ENDIAN_BIG = 200
      };

enum {
      SFE_NO_ERROR = 0,
	SFE_BAD_FILE,
	SFE_OPEN_FAILED,
	SFE_BAD_OPEN_FORMAT,
	SFE_BAD_SNDFILE_PTR,
	SFE_BAD_SF_INFO_PTR,
	SFE_BAD_INT_FD,
	SFE_BAD_INT_PTR,
	SFE_MALLOC_FAILED,
	SFE_BAD_SEEK,
	SFE_NOT_SEEKABLE,
	SFE_UNIMPLEMENTED,
	SFE_BAD_READ_ALIGN,
	SFE_BAD_WRITE_ALIGN,
	SFE_UNKNOWN_FORMAT,
	SFE_NOT_READMODE,
	SFE_NOT_WRITEMODE,
	SFE_BAD_MODE_RW,
	SFE_BAD_SF_INFO,
	SFE_SHORT_READ,
	SFE_SHORT_WRITE,
	SFE_INTERNAL,
	
	SFE_WAV_NO_RIFF,
	SFE_WAV_NO_WAVE,
	SFE_WAV_NO_FMT,
	SFE_WAV_FMT_SHORT,
	SFE_WAV_FMT_TOO_BIG,
	SFE_WAV_BAD_FORMAT,
	SFE_WAV_BAD_BLOCKALIGN,
	SFE_WAV_NO_DATA,
	SFE_WAV_ADPCM_NOT4BIT,
	SFE_WAV_ADPCM_CHANNELS,
	SFE_WAV_GSM610_FORMAT,
	SFE_WAV_UNKNOWN_CHUNK,

	SFE_AIFF_NO_FORM,
	SFE_AIFF_UNKNOWN_CHUNK,
	SFE_COMM_CHUNK_SIZE,
	SFE_AIFF_NO_SSND,
	SFE_AIFF_NO_DATA,

	SFE_AU_UNKNOWN_FORMAT,
	SFE_AU_NO_DOTSND,
	
	SFE_RAW_READ_BAD_SPEC,
	SFE_RAW_BAD_BITWIDTH,
	
	SFE_PAF_NO_MARKER,
	SFE_PAF_VERSION,
	SFE_PAF_UNKNOWN_FORMAT,
	SFE_PAF_SHORT_HEADER,
	
	SFE_SVX_NO_FORM,
	SFE_SVX_NO_BODY,
	SFE_SVX_NO_DATA,
	SFE_SVX_BAD_COMP, 	

	SFE_NIST_BAD_HEADER,
	SFE_NIST_BAD_ENCODING,

	SFE_MAX_ERROR   // This must be last in list.
      };

void	endswap_short_array(short *ptr, int len);
void	endswap_int_array(int *ptr, int len);

/* The following file types can be read and written.
** A file type would consist of a major type (ie SF_FORMAT_WAV) bitwise
** ORed with a minor type (ie SF_FORMAT_PCM). SF_FORMAT_TYPEMASK and
** SF_FORMAT_SUBMASK can be used to separate the major and minor file
** types.
*/

enum {	
      SF_FORMAT_WAV       = 0x10000,  // Microsoft WAV format (big endian).
	SF_FORMAT_AIFF	  = 0x20000,  // Apple/SGI AIFF format (little endian).
	SF_FORMAT_AU	  = 0x30000,  // Sun/NeXT AU format (big endian).
	SF_FORMAT_AULE	  = 0x40000,  // DEC AU format (little endian).
	SF_FORMAT_RAW	  = 0x50000,  // RAW PCM data.
	SF_FORMAT_PAF	  = 0x60000,  // Ensoniq PARIS file format.
	SF_FORMAT_SVX	  = 0x70000,  // Amiga IFF / SVX8 / SV16 format.
	SF_FORMAT_NIST	  = 0x80000,  // Sphere NIST format.
	
	SF_FORMAT_PCM	  = 0x0001,	  // PCM data in 8, 16, 24 or 32 bits.
	SF_FORMAT_FLOAT	  = 0x0002,	  // 32 bit floats.
	SF_FORMAT_ULAW	  = 0x0003,	  // U-Law encoded.
	SF_FORMAT_ALAW	  = 0x0004,	  // A-Law encoded.
	SF_FORMAT_IMA_ADPCM = 0x0005,	  // IMA ADPCM.
	SF_FORMAT_MS_ADPCM  = 0x0006,	  // Microsoft ADPCM.

	SF_FORMAT_PCM_BE	  = 0x0007,	  // Big endian PCM data.
	SF_FORMAT_PCM_LE    = 0x0008,	  // Little endian PCM data.
	SF_FORMAT_PCM_S8	  = 0x0009,	  // Signed 8 bit PCM.
	SF_FORMAT_PCM_U8    = 0x000A,	  // Unsigned 8 bit PCM.
	
	SF_FORMAT_SVX_FIB	  = 0x000B,   // SVX Fibonacci Delta encoding.
	SF_FORMAT_SVX_EXP	  = 0x000C,   // SVX Exponential Delta encoding.

	SF_FORMAT_GSM610	  = 0x000D,   // GSM 6.10 encoding.

	SF_FORMAT_G721_32	  = 0x000E,   // 32kbs G721 ADPCM encoding.
	SF_FORMAT_G723_24	  = 0x000F,   // 24kbs G723 ADPCM encoding.

	SF_FORMAT_SUBMASK	  = 0xFFFF,		
	SF_FORMAT_TYPEMASK  = 0x7FFF0000
      };

//---------------------------------------------------------
//   SndFileFormat
//---------------------------------------------------------

class SndFile;

class SndFileFormat {
      size_t unimpl() const;

   protected:
      SndFile* sfile;
      float buffer[SF_BUFFER_LEN];

   public:
      SndFileFormat(SndFile* s)             { sfile = s; }
      virtual off_t seek(off_t, int)        { return unimpl(); }
      virtual size_t read(float**, size_t)  { return unimpl(); }
      virtual size_t write(float**, size_t) { return unimpl(); }
      virtual void close()                  {}
      };

//---------------------------------------------------------
//   SampleV
//    peak file value
//---------------------------------------------------------

struct SampleV {
      unsigned char peak;
      unsigned char rms;
      };

union WAV_FMT;

//---------------------------------------------------------
//   SndFile
//---------------------------------------------------------

class SndFile {
      QFileInfo* finfo;
      int refs;         // reference count to file

	int endian;
	
      SndFileFormat* fmt;
	unsigned long current;

	unsigned int _blockwidth;     // Size in bytes of one set of interleaved samples.
	unsigned long _dataoffset;    // Offset in number of bytes from beginning of file.

      SampleV** cache;
      int csize;                    // frames in cache

	unsigned _samples;
      unsigned _samplerate;
	unsigned _channels;
	unsigned _format;
	unsigned _sections;
	unsigned _pcmbitwidth;
	bool _seekable;

	FILE* _file;
	unsigned int _bytewidth;	 // Size in bytes of one sample (one channel).

	unsigned long _filelength;
	unsigned long _datalength;     // Length in bytes of the audio data.
	int _mode;

      int wav_open_read();
      int wav_open_write();
      bool validate();
      bool validateSfinfo();
      void readCache(const QString& path);
      void writeCache(const QString& path);
      int read_fmt_chunk(WAV_FMT* wav_fmt);
      int write_header(WAV_FMT *wav_fmt, unsigned int size, int do_fact);

   public:
      SndFile(const QString path);
      ~SndFile();

      int _errno;

      FILE* file() const { return _file; }
      bool openRead();
      bool openWrite();
      void close();

      unsigned samples() const      { return _samples; }
      unsigned channels() const     { return _channels; }
      unsigned samplerate() const   { return _samplerate; }
	unsigned format() const       { return _format; }
      int sampleBits() const        { return _pcmbitwidth; }
      void setFormat(int fmt, int ch, int rate, int bits);

      size_t read(float**, size_t);
      size_t write(float**, size_t);
      off_t	seek(off_t frames, int whence);
      void read(SampleV* s, int mag, unsigned pos);

      void setErrno(int n) { _errno = n; }

      QString path() const                { return finfo->filePath(); }
      QString basename() const            { return finfo->baseName(); }
      QString name() const                { return finfo->fileName(); }

      int references()                    { return refs; }
      int incRef();
      int decRef();

      unsigned dataoffset() const         { return _dataoffset; }
	unsigned long filelength() const    { return _filelength; }
	void setFilelength(unsigned long v) { _filelength = v; }
	unsigned long datalength()          { return _datalength; }
	void setDatalength(unsigned long v) { _datalength = v; }
	int mode() const                    { return _mode; }
	unsigned int bytewidth() const      { return _bytewidth; }
      void setSamples(unsigned v)         { _samples = v; }
	unsigned blockwidth() const         { return _blockwidth; }
	void setBlockwidth(unsigned v)      { _blockwidth = v;    }
      };

//---------------------------------------------------------
//   SndFileList
//---------------------------------------------------------

class SndFileList : public std::list<SndFile*> {
   public:
      SndFile* search(const QString& name);
      };

typedef SndFileList::iterator iSndFile;

#endif
