//=========================================================
//  MusE
//  Linux Music Editor
//  $Id: midiport.cpp,v 1.6 2002/02/27 09:52:46 muse Exp $
//
//  (C) Copyright 2000 Werner Schweer (ws@seh.de)
//=========================================================

#include <unistd.h>           // wg. usleep()
#include <qpopupmenu.h>
#include "driver/mididev.h"
#include "midiport.h"
#include "midictrl.h"
#include "midi.h"
#include "minstrument.h"
#include "xml.h"
#include "globals.h"
#include "midithread.h"
#include "event.h"

MidiPort midiPorts[MIDI_PORTS];

//---------------------------------------------------------
//   initMidiPorts
//---------------------------------------------------------

void initMidiPorts()
      {
      for (int i = 0; i < MIDI_PORTS; ++i) {
            MidiPort* port = &midiPorts[i];
            port->setInstrument(genericMidiInstrument);
            }
      }

//---------------------------------------------------------
//   MidiPort
//---------------------------------------------------------

MidiPort::MidiPort()
   : _state("not configured")
      {
      _masterVol  = 16383;
      _device     = 0;
      _instrument = 0;
      for (int i = 0; i < MIDI_CHANNELS; ++i)
            resetIstate(i, false);
      }

//---------------------------------------------------------
//   guiVisible
//---------------------------------------------------------

bool MidiPort::guiVisible() const
      {
      return _instrument ? _instrument->guiVisible() : false;
      }

//---------------------------------------------------------
//   hasGui
//---------------------------------------------------------

bool MidiPort::hasGui() const
      {
      return _instrument ? _instrument->hasGui() : false;
      }

//---------------------------------------------------------
//   showGui
//---------------------------------------------------------

void MidiPort::showGui(bool f)
      {
      if (_instrument)
            midiThread->msgShowInstrumentGui(_instrument, f);
      }

//---------------------------------------------------------
//   setDevice
//---------------------------------------------------------

void MidiPort::setMidiDevice(MidiDevice* dev)
      {
      if (_device) {
            _device->close();
            _device->setPort(-1);
            }
      _device = dev;
      if (_device) {
            // restrict flags to allowed
            _rwFlags &= _device->rwFlags();
            for (int i = 0; i < MIDI_PORTS; ++i) {
                  if (&midiPorts[i] == this) {
                        _device->setPort(i);
                        break;
                        }
                  }
            for (int i = 0; i < MIDI_CHANNELS; ++i)
                  resetCstate(i, true);
            _state = _device->open(_rwFlags);
            _device->setPort(portno());
            }
      else {
            _rwFlags = 0;
            _state = "not configured";
            }
      }

//---------------------------------------------------------
//   portno
//---------------------------------------------------------

int MidiPort::portno() const
      {
      for (int i = 0; i < MIDI_PORTS; ++i) {
            if (&midiPorts[i] == this)
                  return i;
            }
      return -1;
      }

//---------------------------------------------------------
//   midiPortsPopup
//---------------------------------------------------------

QPopupMenu* midiPortsPopup(QWidget* parent)
      {
      QPopupMenu* p = new QPopupMenu(parent);
      for (int i = 0; i < MIDI_PORTS; ++i) {
            MidiPort* port = &midiPorts[i];
            MidiDevice* dev = port->device();
            QString name;
            name.sprintf("%d(%s)", port->portno()+1,
                     dev ? dev->name().latin1() : "none");
            p->insertItem(name, i);
            }
      return p;
      }

//---------------------------------------------------------
//   portname
//---------------------------------------------------------

const QString& MidiPort::portname() const
      {
      static const QString none("none");
      if (_device)
            return _device->name();
      else
            return none;
      }

//---------------------------------------------------------
//   resetCstate
//---------------------------------------------------------

void MidiPort::resetCstate(int channel, bool)
      {
      _cState[channel].pitch   = -1;
      _cState[channel].program = -1;
      for (int k = 0; k < 128; ++k)
            _cState[channel].controller[k] = -1;
      }

//---------------------------------------------------------
//   resetIstate
//---------------------------------------------------------

void MidiPort::resetIstate(int channel, bool)
      {
      _iState[channel].pitch   = -1;
      _iState[channel].program = -1;
      for (int k = 0; k < 128; ++k)
            _iState[channel].controller[k] = -1;
      resetCstate(channel, false);
      }

//---------------------------------------------------------
//   gmOn
//---------------------------------------------------------

void MidiPort::gmOn()
      {
      sysex(gmOnMsg, gmOnMsgLen);
      for (int i = 0; i < MIDI_CHANNELS; ++i) {
            resetCstate(i, false);
            _cState[i].program = 0;
            _cState[i].pitch   = 0;
            _cState[i].controller[CTRL_VOLUME] = 100;
            _cState[i].controller[CTRL_PANPOT] = 64;
            _cState[i].controller[CTRL_REVERB_SEND] = 40;
            _cState[i].controller[CTRL_CHORUS_SEND] = 0;
            }
      }

//---------------------------------------------------------
//   gsOn
//---------------------------------------------------------

void MidiPort::gsOn()
      {
      static unsigned char data2[] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x01, 0x33, 0x50, 0x3c };
      static unsigned char data3[] = { 0x41, 0x10, 0x42, 0x12, 0x40, 0x01, 0x34, 0x50, 0x3b };

      sysex(gsOnMsg, gsOnMsgLen);
      usleep(10000);   // wait 10ms
      sysex(data2, sizeof(data2));
      usleep(10000);   // wait 10ms
      sysex(data3, sizeof(data3));

      for (int i = 0; i < MIDI_CHANNELS; ++i) {
            resetCstate(i, false);
            _cState[i].program = 0;
            _cState[i].pitch   = 0;
            _cState[i].controller[CTRL_VOLUME] = 100;
            _cState[i].controller[CTRL_PANPOT] = 64;
            _cState[i].controller[CTRL_REVERB_SEND] = 40;
            _cState[i].controller[CTRL_CHORUS_SEND] = 0;
            }
      }

//---------------------------------------------------------
//   xgOn
//---------------------------------------------------------

void MidiPort::xgOn()
      {
      gmOn();
      usleep(200000);   // wait 200ms
      sysex(xgOnMsg, xgOnMsgLen);
      for (int i = 0; i < MIDI_CHANNELS; ++i) {
            _cState[i].controller[0] = 0;
            _cState[i].controller[1] = 0;
            _cState[i].controller[5] = 0;
            _cState[i].controller[7] = 0x64;
            _cState[i].controller[10] = 0x40;
            _cState[i].controller[11] = 0x7f;
            _cState[i].controller[32] = 0;
            _cState[i].controller[64] = 0;
            _cState[i].controller[65] = 0;
            _cState[i].controller[66] = 0;
            _cState[i].controller[67] = 0;
            _cState[i].controller[71] = 0x40;
            _cState[i].controller[72] = 0x40;
            _cState[i].controller[73] = 0x40;
            _cState[i].controller[74] = 0x40;
            _cState[i].controller[91] = 0x28;
            _cState[i].controller[93] = 0;
            _cState[i].controller[94] = 0;
            }
      }

//---------------------------------------------------------
//   programChange
//---------------------------------------------------------

void MidiPort::programChange(int chn, int hbank, int lbank, int prg)
      {
      if (_iState[chn].controller[CTRL_HBANK] != hbank) {
            _cState[chn].program = -1;
            _cState[chn].controller[CTRL_LBANK] = -1;
            }
      if (_iState[chn].controller[CTRL_LBANK] != lbank)
            _cState[chn].program = -1;

      _iState[chn].program = prg;
      _iState[chn].controller[CTRL_HBANK] = hbank;
      _iState[chn].controller[CTRL_LBANK] = lbank;
      if (hbank != -1)
            setCtrl(chn, CTRL_HBANK, hbank);
      if (lbank != -1)
            setCtrl(chn, CTRL_LBANK, lbank);
      programChange(chn, prg);
      }

//---------------------------------------------------------
//   programChange
//---------------------------------------------------------

void MidiPort::programChange(int chn, int prg)
      {
      if (_cState[chn].program != prg) {
            _cState[chn].program = prg;
            if (_device) {
                  MidiEvent event(portno(), chn, 0, MidiEvent::Program,
                     prg, 0, 0, 0);
                  _device->putEvent(&event);
                  }
            }
      }

//---------------------------------------------------------
//   bender
//---------------------------------------------------------

void MidiPort::bender(int chn, int val)
      {
      if (val != _cState[chn].pitch) {
            _cState[chn].pitch = val;
            if (_device) {
                  MidiEvent event(portno(), chn, 0, MidiEvent::Pitch,
                     val, 0, 0, 0);
                  _device->putEvent(&event);
                  }
            }
      }

//---------------------------------------------------------
//   sysex
//---------------------------------------------------------

void MidiPort::sysex(const unsigned char* p, int n)
      {
      if (_device) {
            MidiEvent event(portno(), 0, 0, MidiEvent::Sysex, n, p);
            _device->putEvent(&event);
            }
      }

//---------------------------------------------------------
//   nrpn
//    non registered parameter
//---------------------------------------------------------

void MidiPort::nrpn(int chan, int ctrl, int val)
      {
      int ctrlH = (ctrl >> 7) & 0x7f;
      int ctrlL = ctrl & 0x7f;
      int dataH = (val >> 7) & 0x7f;
      int dataL = val & 0x7f;
      setCtrl(chan, CTRL_LDATA, dataL);
      setCtrl(chan, CTRL_HDATA, dataH);
      setCtrl(chan, CTRL_LNRPN, ctrlL);
      setCtrl(chan, CTRL_HNRPN, ctrlH);
      // invalidate any rpn setting:
      _cState[chan].controller[CTRL_LRPN] = -1;
      _cState[chan].controller[CTRL_HRPN] = -1;
      }

//---------------------------------------------------------
//   rpn
//    registered parameter
//---------------------------------------------------------

void MidiPort::rpn(int chan, int ctrl, int val)
      {
      int ctrlH = (ctrl >> 7) & 0x7f;
      int ctrlL = ctrl & 0x7f;
      int dataH = (val >> 7) & 0x7f;
      int dataL = val & 0x7f;
      setCtrl(chan, CTRL_LDATA, dataL);
      setCtrl(chan, CTRL_HDATA, dataH);
      setCtrl(chan, CTRL_LRPN, ctrlL);
      setCtrl(chan, CTRL_HRPN, ctrlH);
      // invalidate any nrpn setting:
      _cState[chan].controller[CTRL_HNRPN] = -1;
      _cState[chan].controller[CTRL_LNRPN] = -1;
      }

//---------------------------------------------------------
//   setMasterVol
//---------------------------------------------------------

void MidiPort::setMasterVol(int val)
      {
      _masterVol = val;
      unsigned char data[6] = { 0x7f, 0x7f, 0x04, 0x01, 0x00, 0x00 };
      data[4] = val & 0x7f;
      data[5] = (val >> 7) & 0x7f;
      sysex(data, 6);
      }

//---------------------------------------------------------
//   resetRunstate
//---------------------------------------------------------

void MidiPort::resetRunstate()
      {
//      if (_device)
//            _device->resetRunstate();
      }

//---------------------------------------------------------
//   clock
//---------------------------------------------------------

void MidiPort::clock() const
      {
      if (_device)
            _device->putClock();
      }

//---------------------------------------------------------
//   start
//---------------------------------------------------------

void MidiPort::putStart() const
      {
      if (_device)
            _device->putStart();
      }

//---------------------------------------------------------
//   putContinue
//---------------------------------------------------------

void MidiPort::putContinue() const
      {
      if (_device)
            _device->putContinue();
      }

//---------------------------------------------------------
//   setrwFlags
//---------------------------------------------------------

void MidiPort::setrwFlags(int val)
      {
      int mask = 0x3;
      if (_device)            // mask with device capabilities
            mask = _device->rwFlags();
// printf("MidiPort: setFlags %d mask %d\n", val, mask);
      _rwFlags = val & mask;
      }

//---------------------------------------------------------
//   setCtrl
//---------------------------------------------------------

void MidiPort::setCtrl(int chn, int ctrl, int val)
      {
      if (_cState[chn].controller[ctrl] != val) {
            _cState[chn].controller[ctrl] = val;
            if (_device) {
                  MidiEvent event(portno(), chn, 0, MidiEvent::Ctrl7,
                     ctrl, val, 0, 0);
                  _device->putEvent(&event);
                  }
            }
      }

//---------------------------------------------------------
//   putEvent
//---------------------------------------------------------

void MidiPort::putEvent(const MidiEvent* ev)
      {
      if (!_device)
            return;
      int chn = ev->channel();
      int a = ev->dataA();
      int b = ev->dataB();
      switch(ev->type()) {
            case MidiEvent::Ctrl14:
                  if (_cState[chn].controller[a & 0x7f] == b & 0x7f) {
                        a >>= 7;
                        b >>= 7;
                        if (_cState[chn].controller[a] == b)
                              return;
                        MidiEvent e(0, chn, 0, MidiEvent::Ctrl7, a, b, 0, 0);
                        _device->putEvent(&e);
                        _cState[chn].controller[a] = b;
                        return;
                        }
                  break;
            case MidiEvent::Ctrl7:
                  if (_cState[chn].controller[a] == b)
                        return;
                  _cState[chn].controller[a] = b;
                  break;
            default:
                  // TODO
                  break;
            }
      _device->putEvent(ev);
      }

