//=========================================================
//  MusE
//  Linux Music Editor
//    $Id: midirawdev.cpp,v 1.2 2002/02/07 08:14:48 muse Exp $
//  (C) Copyright 1999 Werner Schweer (ws@seh.de)
//=========================================================

#include "midirawdev.h"
#include "globals.h"
#include "event.h"
#include "midictrl.h"

#include <stdio.h>
#include <errno.h>

//---------------------------------------------------------
//   MidiRawDevice
//---------------------------------------------------------

MidiRawDevice::MidiRawDevice(const QString& name)
   : MidiDevice(name)
      {
      runState      = 0;
      rp = wp = iobuffer;
      }

//---------------------------------------------------------
//   put
//---------------------------------------------------------

void MidiRawDevice::put(unsigned char c)
      {
      unsigned char* p = wp+1;
      if (p >= iobuffer+IO_BUFFERSIZE)
            p = iobuffer;
      while (p == rp)   // buffer full?
            flush();
      *wp = c;
      wp  = p;
      }

//---------------------------------------------------------
//   flush
//---------------------------------------------------------

void MidiRawDevice::flush()
      {
      if (wp == rp)
            return;
      if (wp > rp) {
            int n = wp - rp;
            int nn = write(rp, n);
            if (nn < 0) {
                  fprintf(stderr, "midi write error: %s\n", strerror(errno));
                  exit(-1);
                  }
            rp += nn;
            }
      else {
            int n = iobuffer + IO_BUFFERSIZE - rp;
            int nn = write(rp, n);
            if (nn < 0) {
                  fprintf(stderr, "midi write error: %s\n", strerror(errno));
                  exit(-1);
                  }
            rp += nn;
            if (nn == n) {
                  int n  = wp - iobuffer;
                  int nn = write(iobuffer, n);
                  if (nn < 0) {
                        fprintf(stderr, "midi write error: %s\n", strerror(errno));
                        exit(-1);
                        }
                  rp = iobuffer + nn;
                  }
            }
      }

//---------------------------------------------------------
//   chnEvent1
//---------------------------------------------------------

void MidiRawDevice::chnEvent1(int chn, int event, int p1)
      {
      unsigned int stat = event | chn;
      if (runState != stat) {
            put(stat);
            runState = stat;
            }
      put(p1);
      flush();
      }

//---------------------------------------------------------
//   chnEvent2
//---------------------------------------------------------

void MidiRawDevice::chnEvent2(int chn, int event, int p1, int p2)
      {
      unsigned int stat = event | chn;
      if (runState != stat) {
            put(stat);
            runState = stat;
            }
      put(p1);
      put(p2);
      flush();
      }

//---------------------------------------------------------
//   putClock
//---------------------------------------------------------

void MidiRawDevice::putClock()
      {
      put(0xf8);        // no lock necessary
      flush();
      }

//---------------------------------------------------------
//   putStart
//---------------------------------------------------------

void MidiRawDevice::putStart()
      {
      put(0xfa);        // no lock necessary
      flush();
      }

//---------------------------------------------------------
//   putStop
//---------------------------------------------------------

void MidiRawDevice::putStop()
      {
      put(0xfc);        // no lock necessary
      flush();
      }

//---------------------------------------------------------
//   putContinue
//---------------------------------------------------------

void MidiRawDevice::putContinue()
      {
      put(0xfb);        // no lock necessary
      flush();
      }

//---------------------------------------------------------
//   putSongpos
//---------------------------------------------------------

void MidiRawDevice::putSongpos(int beat)
      {
      put(0xf2);
      put(beat & 0x7f);
      put((beat >> 7) & 0x7f);
      flush();
      }

//---------------------------------------------------------
//   putEvent
//---------------------------------------------------------

void MidiRawDevice::putEvent(const MidiEvent* e)
      {
      int chn = e->channel();
      int a = e->dataA();
      int b = e->dataB();
      int c = e->dataC();
      switch(e->type()) {
            case MidiEvent::Note:
                  chnEvent2(chn, 0x90, a, b);
                  break;
            case MidiEvent::NoteOff:
                  chnEvent2(chn, 0x80, a, b);
                  break;
            case MidiEvent::Program:
                  chnEvent1(chn, 0xc0, a);
                  break;
            case MidiEvent::Ctrl7:
                  chnEvent2(chn, 0xb0, a, b);
                  break;
            case MidiEvent::Ctrl14:
                  chnEvent2(chn, 0xb0, a, b & 0x7f);
                  chnEvent2(chn, 0xb0, a >> 7, b >> 7);
                  break;
            case MidiEvent::RPN:
                  chnEvent2(chn, 0xb0, CTRL_LDATA, b & 0x7f);
                  chnEvent2(chn, 0xb0, CTRL_HDATA, b >> 7);
                  chnEvent2(chn, 0xb0, CTRL_LRPN, a & 0x7f);
                  chnEvent2(chn, 0xb0, CTRL_HRPN, a >> 7);
                  break;
            case MidiEvent::NRPN:
                  chnEvent2(chn, 0xb0, CTRL_LDATA, b & 0x7f);
                  chnEvent2(chn, 0xb0, CTRL_HDATA, b >> 7);
                  chnEvent2(chn, 0xb0, CTRL_LNRPN, a & 0x7f);
                  chnEvent2(chn, 0xb0, CTRL_HNRPN, a >> 7);
                  break;
            case MidiEvent::Pitch:
                  {
                  int lval = (a - 8192) & 0x7f;
                  int hval = ((a - 8192) >> 7) & 0x7f;
                  chnEvent2(chn, 0xe0, lval, hval);
                  }
                  break;
            case MidiEvent::PAfter:
                  chnEvent2(chn, 0xa0, a, b);
                  break;
            case MidiEvent::CAfter:
                  chnEvent1(chn, 0xd0, a);
                  break;
            case MidiEvent::Sysex:
                  {
                  const unsigned char* data = e->data();
                  int n = e->dataLen();
                  runState = 0;
                  put(0xf0);
                  while (n--)
                        put(*data++);
                  put(0xf7);
                  flush();
                  }
                  break;
            default:
                  printf("Midi Raw Device: illegal event type %d\n",
                     e->type());
                  break;
            }
      }

//---------------------------------------------------------
//   processInput
//---------------------------------------------------------

void MidiRawDevice::processInput()
      {
      unsigned char buffer[64];
      int n = read(buffer, 64);
      if (n < 0) {
            if (errno == EAGAIN)
                  return;
            printf("MidiRawDevice: read() failed(%d): %s\n",
               errno, strerror(errno));
            exit(-1);
            }
      unsigned char* p = buffer;

      if (midiInputTrace) {
            printf("MidiInput(%d Bytes from %s): ", n, name().latin1());
            for (int i = 0; i < n; ++i)
                  printf("%02x ", p[i]);
            printf("\n");
            }
      if (midiInputPorts & (1<< port()))
            return;
      dataInput(p, n);
      }

void MidiRawDevice::realtimeSystemInput(int data)
      {
      midiThread->realtimeSystemInput(port(), data);
      }

void MidiRawDevice::mtcInputQuarter(int data)
      {
      midiThread->mtcInputQuarter(port(), data);
      }

void MidiRawDevice::setSongPosition(int data)
      {
      midiThread->setSongPosition(port(), data);
      }

void MidiRawDevice::sysexReceived(const unsigned char* p, int n)
      {
      MidiEvent* event = new MidiEvent(port(), 0, 0, MidiEvent::Sysex, n, p);
      midiThread->eventReceived(event);
      }

void MidiRawDevice::eventReceived(int ch, int a, int b)
      {
      midiThread->eventReceived(port(), ch, a, b);
      }

