#ifndef RSA_H
#define RSA_H

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <openssl/evp.h>
#include <musclecard.h>
#include <openssl/x509.h>

#define MSC_KEY_SIZE 128
#define RANDOM_DEVICE "/dev/urandom"
#define PUBMSC_KEY_FILE   "/etc/musclepam/pubkey"

/**
 * Generates a random number of size len.
 * In fact it reads the numbers from /dev/(u)random.
 *
 * @param buf - buffer of at least len bytes to write the number into
 * @param len - length of the random number
 * @returns 0 on success, -1 otherwise
 */
int getRandom(char *buf, int len);

/**
 * Reads a certificate stored in DER-format from a file.
 *
 * @param file - filename of the file where the certificate are stored
 * @param cert - a X509 certificate to store the certificate in
 * @returns 0 on success, -1 otherwise
 */
int getFileCert(const char *file, X509 **cert);

int getFileCertPEM(const char *file, X509 **cert);

/**
 * Reads a certificate stored in DER-format from a smartcard.
 *
 * @param file - connection to the card
 * @param cert - a X509 certificate to store the certificate in
 * @returns 0 on success, -1 otherwise
 */
int getCardCert(MSCTokenConnection pConnection, X509 **cert);

/**
 * Checks the validity of a certificate.
 * This includes checking the expiration dates and the revokation list.
 *
 * @param cert - a X509 certificate to be checked
 * @returns 0 on success, -1 otherwise
 */
int checkCert(X509 *cert);

/**
 * Extract the public key of a given certificate.
 *
 * @param cert    - certificate that stores the public key
 * @param pub_key - public key object to write key into
 * @returns 0 on success, -1 otherwise
 */
int getPublicKey(X509 *cert, EVP_PKEY** pub_key);

#endif

/*** end of file ***********************************************************/
