/* $Id: muh.c,v 1.29 2000/06/19 16:52:23 zap Exp $
 * -------------------------------------------------------
 * Copyright (c) 1998-2000 Sebastian Kienzl <zap@riot.org>
 * -------------------------------------------------------
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

/* #define DEBUG */

#include "muh.h"
#include "irc.h"
#include "tools.h"
#include "messages.h"
#include "ignore.h"
#include "channels.h"
#include "perm.h"
#include "ascii.h"

FILE *messagelog = NULL;

status_type status;
cfg_type cfg = {
    0, /* listenport */
    1, /* logging */
    1, /* leave */
    1, /* getnick */
    0, /* antiidle */
    0, /* nevergiveup */
    0, /* jumprestricted */
    0, /* rejoin */
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL /* string-pointers */
};

permlist_type hostlist;
permlist_type peoplelist;
serverlist_type servers;
timer_type timers;
channellist_type channels;

/*****verbindungs-bezogene variablen*************/
server_info i_server;
client_info i_client;
client_info i_newclient;

connection_type c_server;
connection_type c_client;
connection_type c_newclient;

int listensocket = 0;  	/* listensocket */
char *forwardmsg = NULL;

void escape()
{
    rawsock_close( listensocket );
    sock_close( &c_server );
    sock_close( &c_client );
    sock_close( &c_newclient );
    if( messagelog ) fclose( messagelog );
    unlink( FILE_PID );
    error( MUH_ERREXIT );
    exit( 1 );
}

void read_cfg()
{
    lineno = 1;
    report( MUH_PARSING );
    if( !( yyin = fopen( MUHRC, "r" ) ) ) {
        error( MUH_ERRCFG, cfg.home );
        escape();
    }
    yyrestart( yyin );
    if( yyparse() ) {
        fclose( yyin );
        escape();
    }
    fclose( yyin );
}

void sig_term()
{
    error( MUH_SIGTERM );
    escape();
}

void drop_client( char *reason )
{
    if( i_client.connected ) {
        if( reason ) {
            /* set back to blocking */
            sock_setblock( c_server.socket );
            if( reason[ 0 ] == ':' )
                irc_write( &c_client, "ERROR %s", reason );
            else
                irc_write( &c_client, "ERROR :Closing Link: %s", reason );
        }
        sock_close( &c_client );
        i_client.connected = 0;
    }
}

void drop_newclient( char *reason )
{
    if( i_newclient.connected ) {
        if( reason ) {
            sock_setblock( c_server.socket );
            irc_write( &c_newclient, "ERROR :Closing Link: %s", reason );
        }
        sock_close( &c_newclient );
        i_newclient.connected = 0;
        status.init = 0;
    }
}

void drop_server( char *reason )
{
    if( c_server.socket ) { /* die abfrage _SO_ lassen */
        if( reason ) {
            sock_setblock( c_server.socket );
            irc_write( &c_server, "QUIT :%s", reason );
        }
        sock_close( &c_server );
    }
    i_server.connected = 0;
    if( !cfg.rejoin ) drop_channels(); /* we don't want to rejoin */
}

void drop_all( char *reason )
{
    drop_server( reason );
    drop_client( reason );
    drop_newclient( NULL ); /* this ain't newclients' business */
}

void server_next( int disablecurrent )
{
    int i = i_server.current;
    int x;

    if( disablecurrent ) servers.data[ i_server.current ]->working = 0;
    
    do {
        i_server.current++;
        if( i_server.current >= servers.amount ) i_server.current = 0;
    } while( !servers.data[ i_server.current ]->working && ( i_server.current != i ) );

    if( !servers.data[ i_server.current ]->working ) {
        if( cfg.nevergiveup ) {
            report( MUH_OUTOFSERVERSNEVER );
            for( x = 0; x < servers.amount; x++ )
                servers.data[ x ]->working = 1;
            i_server.current = 0;
        }
        else {
            error( MUH_OUTOFSERVERS );
            escape();
        }
    }

    if( i == i_server.current ) {
        report( SOCK_RECONNECT, servers.data[ i_server.current ]->name );
        sleep( 5 );
    }
}

int proceed_timer( int *timer, int warn, int exceed )
{
    (*timer)++;
    if( *timer == warn ) return 1;
    if( *timer >= exceed ) {
        *timer = 0;
        return 2;
    }
    return 0;
}

void create_listen()
{
    if( listensocket ) rawsock_close( listensocket );
    
    /* creating listener */
    if( ( listensocket = sock_open() ) < 0 ) {
        error( SOCK_ERROPEN );
        escape();
    }

    if( !sock_bind( listensocket, cfg.bind, cfg.listenport ) ) {
        if( cfg.bind )
            error( SOCK_ERRBINDHOST, cfg.bind, cfg.listenport );
        else
            error( SOCK_ERRBIND, cfg.listenport );
        escape();
    }
    
    if( !sock_listen( listensocket ) ) {
        error( SOCK_ERRLISTEN );
        escape();
    }

    if( cfg.bind )
        report( SOCK_LISTENOKHOST, cfg.bind, cfg.listenport );
    else
        report( SOCK_LISTENOK, cfg.listenport );
    /* .. end */
}

void rehash()
{
    char *oldnick;
    char *oldbind;
    char *oldrealname;
    char *oldusername;
    int oldlistenport;

    report( MUH_REHASH"\n" );

    oldnick = strdup( cfg.nickname );
    oldusername = strdup( cfg.username );
    oldrealname = strdup( cfg.realname );
    oldlistenport = cfg.listenport;    
    if( cfg.bind )
        oldbind = strdup( cfg.bind );
    else
        oldbind = NULL;

    read_cfg();

    if( strcmp( oldnick, cfg.nickname ) ) status.got_nick = 0;

    if( ( oldlistenport != cfg.listenport ) ||
        ( ( oldbind == NULL ?
            ( cfg.bind == NULL ? 0 : 1 ) : ( cfg.bind == NULL ?
                                             1 : strcmp( oldbind, cfg.bind ) ) ) ) )
        create_listen();

    if( !cfg.logging && messagelog ) {
        fclose( messagelog );
        messagelog = NULL;
    }

    if( cfg.logging && !messagelog ) {
        if( !( messagelog = fopen( FILE_MESSAGES, "a+" ) ) ) {
            report( MUH_ERRMSGFILE );
        }
    }

    if( strcmp( oldrealname, cfg.realname ) || strcmp( oldusername, cfg.username ) ) {
        drop_all( MUH_RECONNECT );
        report( MUH_RECONNECT"\n" );
        i_server.current--;
        sleep( 5 );
        server_next( 0 );
    }

    FREESTRING( oldnick );
    FREESTRING( oldrealname );
    FREESTRING( oldusername );
    FREESTRING( oldbind );
}

void client_left()
{
    char *chns;

    if( ( chns = list_channels() ) ) {
        if( cfg.leavemsg ) irc_write( &c_server, "PRIVMSG %s :\1ACTION %s\1", chns, cfg.leavemsg );
        if( cfg.leave ) {
            report( MUH_LEAVING );
            irc_write( &c_server, "PART %s", chns );
        }
        FREESTRING( chns );
    }
}

void check_timers()
{
    FILE *fwd;
    
    if( i_client.connected ) {
        switch( proceed_timer( &c_client.timer, 60, 120 ) ) {
        case 0:
            break;
        case 1:
            irc_write( &c_client, "PING :%s", i_server.realname );
            break;
        case 2:
            drop_client( "Ping timeout" );
            error( CLNT_STONED );
            client_left();
            break;
        }
    }
    else {
        if( cfg.antiidle && ( proceed_timer( &timers.antiidle, 0, 600 ) == 2 ) ) irc_write( &c_server, "PRIVMSG" );
    }

    if( i_server.connected == 2 ) {
        switch( proceed_timer( &c_server.timer, 90, 150 ) ) {
        case 0:
            break;
        case 1:
            irc_write( &c_server, "PING :%s", i_server.realname );
            break;
        case 2:
            drop_all( SERV_STONED );
            error( SERV_STONED );
            server_next( 0 );
            break;
        }
    }

    if( i_newclient.connected ) {
        switch( proceed_timer( &c_newclient.timer, 0, 20 ) ) {
        case 0:
        case 1:
            break;
        case 2:
            if( status.init ) {
                error( CLNT_AUTHFAIL );
                irc_write( &c_newclient, ":%s 464 %s :Password Incorrect", i_server.realname, cfg.nickname );
                drop_newclient( "Bad Password" );
            }
            else {
            	error( CLNT_AUTHTO );
                drop_newclient( "Ping timeout" );
            }
            if( i_client.connected ) irc_notice( &c_client, status.nickname, CLNT_AUTHFAILNOTICE, i_newclient.hostname );
            FREESTRING( i_newclient.nickname );
            FREESTRING( i_newclient.username );
            FREESTRING( i_newclient.hostname );
            /* deallocate would also happen on next newclient-connect, but let's do it here */
            break;
        }
    }

    if( proceed_timer( &timers.listen, 0, 15 ) == 2 ) status.allowconnect = 1;

    if( proceed_timer( &timers.reply, 0, 4 ) == 2 ) status.allowreply = 1;
    
    if( cfg.getnick && !status.got_nick && !i_client.connected && ( proceed_timer( &timers.nickname, 0, 3 ) == 2 ) )
        irc_write( &c_server, "NICK %s", cfg.nickname );

    if( cfg.forwardmsg ) {
        switch( proceed_timer( &timers.forward, 0, 180 ) ) {
        case 0:
        case 1:
            break;
        case 2:
            if( forwardmsg && ( fwd = popen( cfg.forwardmsg, "w" ) ) ) {
                fprintf( fwd, "%s", forwardmsg );
                pclose( fwd );
                FREESTRING( forwardmsg );
            }
            break;
        }
    }
}


void server_reply_num( int command, char *origin, char *param1, char *param2, int *pass )
{
    char *chns;
    char *channel;
    int chop, chvoice;

    switch( command ) {
    case RPL_WELCOME:
        /* reply 001 - server-welcome */
        i_server.connected++;

        FREESTRING( i_server.realname );
        i_server.realname = strdup( origin );

        FREESTRING( status.nickname );
        status.nickname = strdup( param1 );

        FREESTRING( i_server.greeting[ 0 ] );
        i_server.greeting[ 0 ] = strdup( param2 );

        FREESTRING( status.idhostname );
        if( strchr( param2, '!' ) )
            status.idhostname = strdup( strchr( param2, '!' ) + 1 );
        else
            status.idhostname = NULL;

        i_server.greeting[ 0 ][ lastpos( i_server.greeting[ 0 ], ' ' ) ] = 0;
        report( IRC_CONNECTED, i_server.realname );
        irc_write( &c_server, "MODE %s :+i", status.nickname );

        if( cfg.rejoin && ( chns = list_channels() ) ) {
            report( MUH_REJOINING, chns );
            irc_write( &c_server, "JOIN %s", chns );
            FREESTRING( chns );
            drop_channels(); /* list will rebuild itself */
        }
        break;

    case RPL_YOURHOST:
    case RPL_SERVERIS:
    case RPL_SERVERVER:
        /* your-host, your-server-is, server-version replies */
        FREESTRING( i_server.greeting[ command - 1 ] );
        i_server.greeting[ command - 1 ] = strdup( param2 );
        break;

    case RPL_RESTRICTED:
        if( cfg.jumprestricted ) {
            drop_all( CLNT_RESTRICTED );
            report( SERV_RESTRICTED );
            server_next( 1 );
        }
        break;

    default:
        /* let's ignore other replies for now */
        break;
    }

    if( !i_client.connected ) { /* falls kein client da */
        switch( command ) {
        case ERR_ERRONEUSNICKNAME:
            error( IRC_ERRIN, cfg.nickname );
            escape();
            break;

        case ERR_NICKUNAVAILABLE:
        case ERR_NICKNAMEINUSE:
            if( status.got_nick || ( i_server.connected != 2 ) ) {
                if( status.nickname && strcmp( status.nickname, cfg.altnickname ) == 0 ) { /* altnick doesn't work as well */
                    FREESTRING( status.nickname );
                    status.nickname = ( char * )malloc( 9 );
                    randname( status.nickname, 8 );
                }
                else {
                    FREESTRING( status.nickname );
                    status.nickname = strdup( cfg.altnickname );
                }

                if( command == ERR_NICKNAMEINUSE )
                    report( IRC_NICKINUSE, cfg.nickname, status.nickname );
                else
                    report( IRC_NICKUNAVAIL, cfg.nickname, status.nickname );
                irc_write( &c_server, "NICK %s", status.nickname );
            }
            status.got_nick = 0;
            break;

        default:
            break;
        }
    }

    if( status.suppress ) {
        switch( command ) {
        case RPL_WHOISUSER:
        case RPL_WHOISSERVER:
        case RPL_WHOISOPERATOR:
        case RPL_WHOISIDLE:
            *pass = 0;
            break;

        case RPL_WHOISCHANNELS:
            *pass = 0;
            if( strncasecmp( status.nickname, param2, pos( param2, ' ' ) ) == 0 ) { /* whois-reply of our nick? */
                while( ( param2 = nextword( param2 ) ) != 0 && strlen( param2 ) ) {
                    if( param2[ 0 ] == ':' ) param2++;
                    channel = ( char * )malloc( strlen( param2 ) + 1 );
                    strncpy( channel, param2, pos( param2, ' ' ) );
                    channel[ pos( param2, ' ' ) ] = 0;

                    chop = 0; chvoice = 0;
                    if( ismodechan( channel ) ) {
                        if( channel[ 0 ] == '@' ) chop = 1;
                        if( channel[ 0 ] == '+' ) chvoice = 1;
                        strcpy( channel, channel + 1 );
                    }

                    if( !cfg.leave && i_client.connected ) {
                        irc_write( &c_client, ":%s!%s JOIN :%s", status.nickname, status.idhostname, channel );
                        if( chop ) irc_write( &c_client, ":%s!%s MODE %s +o %s", status.nickname, status.idhostname, channel, status.nickname );
                        if( chvoice ) irc_write( &c_client, ":%s!%s MODE %s +v %s", status.nickname, status.idhostname, channel, status.nickname );
                        if( channel[ 0 ] != '+' ) irc_write( &c_server, "TOPIC %s", channel );
                        irc_write( &c_server, "NAMES %s", channel );
                    }
                    free( channel );
                }
            }
            break;

        case RPL_ENDOFWHOIS:
            *pass = 0;
            status.suppress = 0;
            break;
        }
    }
}

void server_reply_str( char *command, char *origin, char *param1, char *param2, int *pass )
{
    char *nick = 0, *hostname = 0;
    int i, l;

    if( ( i = pos( origin, '!' ) ) > 0 ) {
        nick = ( char * )malloc( i + 1 );
        strncpy( nick, origin, i );
        nick[ i ] = 0;
        hostname = strdup( origin + i + 1 );
    }
    else {
        nick = strdup( origin );
        hostname = strdup( origin );
    }

    upcase( command );
    
    if( strcmp( command, "NICK" ) == 0 ) { /* nickchange */
        if( strcasecmp( status.nickname, nick ) == 0 ) {
            /* is that us who changed nick? */
            FREESTRING( status.nickname );
            status.nickname = strdup( param1 + 1 );
            if( strcasecmp( status.nickname, cfg.nickname ) == 0 ) {
                status.got_nick = 1;
                report( MUH_GOTNICK, status.nickname );
            }
            else status.got_nick = 0;
        }
    }

    if( strcmp( command, "PONG" ) == 0 ) *pass = 0;

    /* channel-track-stuff */
    if( ( strcmp( command, "KICK" ) == 0 ) &&
        ( strncmp( status.nickname, param2, pos( param2, ' ' ) - 1 ) == 0 ) &&
        ( strlen( status.nickname ) == ( pos( param2, ' ' ) ) ) ) {
        if( !i_client.connected ) report( CLNT_KICK, origin, param1, nextword( param2 ) + 1 );
        rem_channel( param1 );
    }

    if( ( strcmp( command, "JOIN" ) == 0 ) &&
        ( strcasecmp( status.nickname, nick ) == 0 ) ) {
        add_channel( param1 + 1 );
    }

    if( ( strcmp( command, "PART" ) == 0 ) &&
        ( strcasecmp( status.nickname, nick ) == 0 ) ) {
        rem_channel( param1 );
    }
    /* ...end */

    if( ( strcmp( command, "KILL" ) == 0 ) &&
        ( strcasecmp( status.nickname, param1 ) == 0 ) ) {
        error( IRC_KILL, nick );
    }

    if( !i_client.connected &&
        status.nickname && !strcasecmp( param1, status.nickname ) &&
        ( !strcmp( command, "PRIVMSG" ) || !strcmp( command, "NOTICE" ) ) &&
        is_perm( &peoplelist, origin ) ) { /* privmsg || notice to us? filtered by peoplelist? */
        if( param2[ 1 ] == '\1' ) { /* CTCP */
            upcase( param2 );
            if( !is_ignore( hostname, IGNORE_CTCP ) && status.allowreply ) {
                report( CLNT_CTCP, param2 + 1, origin );
                if( strcmp( param2 + 2, "VERSION\1" ) == 0 )
                    irc_notice( &c_server, nick, VERSIONREPLY );
                if( strcmp( param2 + 2, "TIME\1" ) == 0 )
                    irc_notice( &c_server, nick, TIMEREPLY );
                if( strcmp( param2 + 2, "USERINFO\1" ) == 0 )
                    irc_notice( &c_server, nick, USERINFOREPLY );
                if( strncmp( param2 + 2, "PING", 4 ) == 0 ) {
                    if( strlen( param2 + 1 ) > 6 ) irc_notice( &c_server, nick, param2 + 1 );
                }
                if( strcmp( param2 + 2, "CLIENTINFO\1" ) == 0 )
                    irc_notice( &c_server, nick, CLIENTINFOREPLY );
                if( strcmp( param2 + 2, "FINGER\1" ) == 0 )
                    irc_notice( &c_server, nick, FINGERREPLY );
                add_ignore( hostname, 6, IGNORE_CTCP );
                status.allowreply = 0;
                timers.reply = 0;
            }
            else {
                report( CLNT_CTCPNOREPLY, param2 + 1, origin );
            }
        }
        else { /* normale message/notice */
            if( !is_ignore( hostname, IGNORE_MESSAGE ) && status.allowreply ) {
                if( cfg.awaynotice ) irc_notice( &c_server, nick, cfg.awaynotice );
                add_ignore( hostname, 120, IGNORE_MESSAGE );
                status.allowreply = 0;
                timers.reply = 0;
            }
            if( cfg.logging && messagelog ) {
                fprintf( messagelog, "[%s](%s) %s\n", gettimestamp(), origin, param2 + 1 );
                fflush( messagelog );
            }
            if( cfg.forwardmsg ) {
                timers.forward = 0;
                l = ( forwardmsg ? strlen( forwardmsg ) : 0 );
                i = l + strlen( origin ) + strlen( param2 + 1 ) + 5;
                forwardmsg = ( char * )realloc( forwardmsg, i );
                sprintf( forwardmsg + l, "(%s) %s\n", origin, param2 + 1 );
            }
        }
    } /* kein client da */

    FREESTRING( nick );
    FREESTRING( hostname );
}

void server_commands( char *command, char *param, int *pass )
{
    upcase( command );

    if( strcmp( command, "PING" ) == 0 ) {
        *pass = 0;
        if( param && param[ 0 ] == ':' ) param++;
        /* don't make this global (see ERROR) */
        if( param ) irc_write( &c_server, "PONG :%s", param );
    }
    
    if( strcmp( command, "ERROR" ) == 0 ) {
        *pass = 0;

        drop_server( NULL );
        drop_newclient( NULL );
        if( param )
            drop_client( param );
        else
            drop_client( SERV_ERR );

        if( param )
            error( IRC_SERVERERROR, param );
        else
            error( IRC_SERVERERROR, "unknown" );
        server_next( ( i_server.connected == 1 ) );
        /* disables current server if tcp-connection only */
    }
}


int read_server()
{
    char *backup = 0;
    char *origin, *command, *param1, *param2;
    int rstate;
    int pass = 0;

    rstate = irc_read( &c_server );
    if( rstate > 0 ) { /* new data...go for it! */
        rstate = 0;
        if( strlen( c_server.buffer ) ) {
            pass = 1;

            backup = strdup( c_server.buffer );

            if( c_server.buffer[ 0 ] == ':' ) { /* reply */
                origin = strtok( c_server.buffer + 1, " " );
                command = strtok( NULL, " " );
                param1 = strtok( NULL, " " );
                param2 = strtok( NULL, "\0" );
#ifdef DEBUG
                printf( "[%s] [%s] [%s] [%s]\n", origin, command, param1, param2 );
#endif
                if( command ) {
                    server_reply_num( atoi( command ), origin, param1, param2, &pass );
                    server_reply_str( command, origin, param1, param2, &pass );
                }
            }
            else { /* command */
                command = strtok( c_server.buffer, " " );
                param1 = strtok( NULL, "\0" );

                if( command )
                    server_commands( command, param1, &pass );
            }

            if( i_client.connected && pass ) irc_write( &c_client, "%s", backup );
            FREESTRING( backup );
        }
    }
    return rstate;
}


void fakeconnect()
{
    int i;
    time_t t;
    struct tm *tt;
    int pic = 0;

    irc_write( &c_client, ":%s 001 %s %s %s!~%s@%s", i_server.realname, status.nickname, i_server.greeting[ 0 ], status.nickname, i_client.username, i_client.hostname );
    for( i = 1; i < 4; i++ )
        irc_write( &c_client, ":%s %03d %s %s", i_server.realname, i + 1, status.nickname, i_server.greeting[ i ] );

    irc_write( &c_server, "LUSERS" );

    if( !cfg.leave ) report( MUH_REINTRODUCE );

    status.suppress = 1;
    irc_write( &c_server, "WHOIS %s", status.nickname );
    /* das kann man deswegen immer senden, da (falls muh brav aus allen channels gegangen ist) es eh keinen channel
     * reply im whois gibt (mach ich wegen rehashs - damit alles klappt falls cfg.leave umgestellt wird)
     */

    irc_write( &c_client, ":%s 375 %s :- muh version "VERSION" -", i_server.realname, status.nickname );

    /* don't rely on extern int daylight */
    time( &t );
    tt = localtime( &t );
    
    pic = ( tt->tm_hour > 6 && tt->tm_hour < 19 ) ? 0 : 1;
    /* cows get up at six and go to sleep at seven :) */

    if( tt->tm_mday == 24 && tt->tm_mon == 11 ) pic = 2; /* xmas! (tm_mon is from 0..11) */

    for( i = 0; i < PIC_Y; i++ )
        irc_write( &c_client, ":%s 372 %s :%s", i_server.realname, status.nickname, pics[ pic ][ i ] );

    irc_write( &c_client, ":%s 372 %s :- running on server %s with nickname %s", i_server.realname, status.nickname, servers.data[ i_server.current ]->name, status.nickname  );

    if( messagelog && ftell( messagelog ) )
        irc_write( &c_client, ":%s 372 %s :- "CLNT_MESSAGES, i_server.realname, status.nickname );
    else
        irc_write( &c_client, ":%s 372 %s :- "CLNT_NOMESSAGES, i_server.realname, status.nickname );

    irc_write( &c_client, ":%s 376 %s :End of /MOTD command.", i_server.realname, status.nickname );

    if( strcmp( i_client.nickname, status.nickname ) != 0 ) {
        irc_write( &c_client, ":%s NICK :%s", i_client.nickname, status.nickname );
        FREESTRING( i_client.nickname );
        i_client.nickname = strdup( status.nickname );
    }
}

int read_newclient()
{
    char *command, *param;
    int stat = irc_read( &c_newclient );

    if( stat > 0 ) {
        stat = 0;
        c_newclient.buffer[ 30 ] = 0;
        command = strtok( c_newclient.buffer, " " );
        param = strtok( NULL, "\0" );

        if( command && param ) {
            upcase( command );
	    if( ( strcmp( command, "PASS" ) == 0 ) && !( status.init & 1 ) ) {
		/* only accept password once */
		if( *param == ':' ) param++;
                if( strlen( cfg.password ) == 13 ) { /* assume it's crypted */
                    if( strcmp( crypt( param, cfg.password ), cfg.password ) == 0 )
                        status.passok = 1;
                }
                else {
                    if( strcmp( param, cfg.password ) == 0 )
                        status.passok = 1;
                }
                status.init = status.init | 1;
            }
            if( strcmp( command, "NICK" ) == 0 ) {
		status.init = status.init | 2;
		FREESTRING( i_newclient.nickname );
		i_newclient.nickname = strdup( strtok( param, " " ) );
            }

            if( strcmp( command, "USER" ) == 0 ) {
		status.init = status.init | 4;
		FREESTRING( i_newclient.username );
		i_newclient.username = strdup( strtok( param, " " ) );
            }

            if( status.init == 7 && status.passok ) {  /* client ist da! */
                report( CLNT_AUTHOK );

                if( i_client.connected ) {
                    drop_client( CLNT_NEWCLIENT );
                    report( CLNT_DROP );
                }

                FREESTRING( i_client.nickname );
                FREESTRING( i_client.username );
                FREESTRING( i_client.hostname );

                i_client.nickname = i_newclient.nickname;
                i_client.username = i_newclient.username;
                i_client.hostname = i_newclient.hostname;

                i_newclient.nickname = NULL;
                i_newclient.username = NULL;
                i_newclient.hostname = NULL;

                i_client.connected = 1;
                i_newclient.connected = 0;

                c_client.socket = c_newclient.socket;
                c_newclient.socket = 0;

                status.passok = 0;
                status.init = 0;
                fakeconnect();
            }
        }
        /* falls pass nicht ok wird einfach ausgetimed */
    }
    return stat;
}

void muh_commands( char *command, char *param )
{
    int corr = 0;
    int i;
    char *s;

    if( !command ) {
        irc_notice( &c_client, status.nickname, CLNT_COMANDS );
        return;
    }
    
    upcase( command );

    if( strcmp( command, "READ" ) == 0 ) {
        corr++;
        if( messagelog && ftell( messagelog ) ) {
            fflush( messagelog );
            rewind( messagelog );

            irc_notice( &c_client, status.nickname, CLNT_MSGLOGSTART );

            s = ( char * )malloc( 1024 );
            while( fgets( s, 1023, messagelog ) ) {
                if( s[ strlen( s ) - 1 ] == '\n' ) s[ strlen( s ) - 1 ] = 0;
                irc_notice( &c_client, status.nickname, s );
            }
            FREESTRING( s );
            
            irc_notice( &c_client, status.nickname, CLNT_MSGLOGEND );
            fseek( messagelog, 0, SEEK_END );
        }
        else irc_notice( &c_client, status.nickname, CLNT_HAVENOMSGS );
    }

    if( strncmp( command, "DEL", 3 ) == 0 ) {
        corr++;
        if( messagelog ) {
            fclose( messagelog );
        }
        unlink( FILE_MESSAGES );
        if( cfg.logging )
            messagelog = fopen( FILE_MESSAGES, "w+" );

        irc_notice( &c_client, status.nickname, CLNT_KILLEDMSGS );
    }

    if( strcmp( command, "REHASH" ) == 0 ) {
        corr++;
        irc_notice( &c_client, status.nickname, MUH_REHASH );
        rehash();
    }

    if( strcmp( command, "JUMP" ) == 0 ) {
        corr++;
        drop_all( MUH_JUMP );
        report( MUH_JUMP"\n" );
        if( param ) {
            i = atoi( param );
            i--;
            if( i < 0 ) i = 0;
            if( i >= servers.amount ) i = servers.amount - 1;
            servers.data[ i ]->working = 1;
            i_server.current = i - 1;
        }
        server_next( 0 );
    }

    if( strcmp( command, "DIE" ) == 0 ) {
        corr++;
        drop_all( MUH_DIE_CL );
        report( MUH_DIE );
        escape();
    }

    if( strcmp( command, "PRINT" ) == 0 ) {
        corr++;
        irc_notice( &c_client, status.nickname, CLNT_SLIST );
        for( i = 0; i < servers.amount; i++ ) {
            irc_notice( &c_client, status.nickname, "%c[%d] %s:%d%s",
                        ( servers.data[ i ]->working ? '+' : '-' ),
                        i + 1,
                        servers.data[ i ]->name,
                        servers.data[ i ]->port,
                        ( servers.data[ i ]->password ? ":*" : "" ) );
        }
        irc_notice( &c_client, status.nickname, CLNT_CURRENT, i_server.current + 1 );
    }

    if( !corr ) irc_notice( &c_client, status.nickname, CLNT_COMANDS );
}

int read_client()
{
    int stat;
    char *backup, *command, *param1, *param2;
    int pass = 1;

    stat = irc_read( &c_client );
    if( stat > 0 ) { /* new data...go for it!! */
        stat = 0;

        if( strlen( c_client.buffer ) ) {
            backup = strdup( c_client.buffer );

            command = strtok( c_client.buffer, " " );
            param1 = strtok( NULL, " " );
            param2 = strtok( NULL, "\0" );
            if( param1 && param1[ 0 ] == ':' ) param1++;

            if( command ) {
                upcase( command );

                if( strcmp( command, "MUH" ) == 0 ) {
                    muh_commands( param1, param2 );
                    pass = 0;
                }

                if( strcmp( command, "QUIT" ) == 0 ) {
                    drop_client( "muh!" );
                    client_left();
                    report( CLNT_LEFT );
                    pass = 0;
                }

                if( strcmp( command, "PONG" ) == 0 ) {
                    /* burp */
                    pass = 0;
                }

                if( strcmp( command, "PING" ) == 0 ) {
                    irc_write( &c_client, ":%s PONG %s :%s", i_server.realname, i_server.realname, status.nickname );
                    pass = 0;
                }
            }
            if( pass ) irc_write( &c_server, "%s", backup );
            FREESTRING( backup );
        }
    }
    return stat;
}

void run()
{
    fd_set rfds;
    struct timeval tv;
    int selret;

    while( 1 ) {
        if( !i_server.connected ) { /*schon wohin connected?*/
            report( SERV_TRYING, servers.data[ i_server.current ]->name, servers.data[ i_server.current ]->port );
            switch( irc_connect( &c_server, servers.data[ i_server.current ], cfg.nickname, cfg.username, cfg.realname, cfg.bind ) ) {
            case 0:
                report( SOCK_CONNECT, servers.data[ i_server.current ]->name );
                i_server.connected = 1;
                break;
            case 1:
                error( SOCK_ERROPEN );
                escape();
                break;
            case 2:
                error( SOCK_ERRRESOLVE, servers.data[ i_server.current ]->name );
                drop_server( NULL );
                server_next( 1 );
                break;
            case 3:
                if( cfg.bind )
                    error( SOCK_ERRBINDHOST, cfg.bind, cfg.listenport );
                else
                    error( SOCK_ERRBIND, cfg.listenport );
                escape();
                break;
            case 4:
                error( SOCK_ERRCONNECT, servers.data[ i_server.current ]->name, strerror( errno ) );
                drop_server( NULL );
                server_next( 1 );
                break;
            case 5:
                error( SOCK_ERRWRITE, servers.data[ i_server.current ]->name );
                drop_server( NULL );
                server_next( 0 );
                break;
            case 6:
                error( SOCK_GENERROR, net_errstr );
                escape();
                break;
            default:
                break;
            }
        }

        tv.tv_usec = 0;
        tv.tv_sec = 1;

        FD_ZERO( &rfds );

        if( ( i_server.connected == 2 ) && status.allowconnect && ( !i_newclient.connected ) ) FD_SET( listensocket, &rfds );
        if( i_server.connected ) FD_SET( c_server.socket, &rfds );
        if( i_client.connected ) FD_SET( c_client.socket, &rfds );
        if( i_newclient.connected ) FD_SET( c_newclient.socket, &rfds );
#ifdef DEBUG
        if( FD_ISSET( c_server.socket, &rfds ) ) printf( "subscribed server-read (highest is %d)\n", highest_socket );
#endif

        selret = select( highest_socket + 1, &rfds, NULL, NULL, &tv );

        if( selret > 0 ) {
            if( FD_ISSET( c_server.socket, &rfds ) ) { /* der server will was */
                if( read_server() < 0 ) {
                    drop_all( SERV_DROPPED );
                    error( SERV_DROPPED );
                    server_next( 0 );
                }
            }

            if( FD_ISSET( listensocket, &rfds ) ) { /* ein neuer client! */
                status.allowconnect = 0;
                FREESTRING( i_newclient.hostname );
                if( ( c_newclient.socket = sock_accept( listensocket, &i_newclient.hostname ) ) > 0 ) {
                    c_newclient.offset = 0;
                    i_newclient.connected = 1;
                    if( !i_client.connected )
                        report( CLNT_CAUGHT, i_newclient.hostname );
                    else {
                        report( CLNT_RECONNECT, i_newclient.hostname );
                    }
                }
                else {
                    if( c_newclient.socket ) /* denied */
                        report( CLNT_DENIED, i_newclient.hostname );
                    else /* error */
                        error( SOCK_ERRACCEPT, i_newclient.hostname );
                    FREESTRING( i_newclient.hostname );
                    c_newclient.socket = 0;
                }
            }
        
            if( FD_ISSET( c_client.socket, &rfds ) ) { /* der client redet */
                if( read_client() < 0 ) {
                    /* client hat connection gedroppt */
                    drop_client( NULL );
                    client_left();
                    report( CLNT_DROPPED );
                }
            }

            if( FD_ISSET( c_newclient.socket, &rfds ) ) { /* der neue client sagt was */
                if( read_newclient() < 0 ) {
                    /* newclient hat connection gedroppt */
                    drop_newclient( NULL );
                    report( CLNT_DROPPEDUNAUTH );
                }
            }
        }

        if( !selret ) { /* it was a timeout then (*every* second) */
            check_timers();
            process_ignores();
        }
    }
}

void init()
{
    struct sigaction sv;

    sigemptyset( &sv.sa_mask );
    sv.sa_flags = 0;
    sv.sa_handler = sig_term;
    sigaction( SIGTERM, &sv, NULL );
    sigaction( SIGINT, &sv, NULL );
    sigaction( SIGKILL, &sv, NULL );

    sv.sa_handler = rehash;
    sigaction( SIGHUP, &sv, NULL );

    sv.sa_handler = SIG_IGN;
    sigaction( SIGUSR1, &sv, NULL );
    sigaction( SIGPIPE, &sv, NULL );
    /* signalhandler hooken */
    
    umask( ~S_IRUSR & ~S_IWUSR );
    /* fuer logfile(s) */

    status.got_nick = 1; /* assume everythings fine */
    status.allowconnect = 1;
    i_server.current = 0; /* start with first server */
    
    srand( time( NULL ) );

    create_listen();
    
    if( !( messagelog = fopen( FILE_MESSAGES, "a+" ) ) ) {
        report( MUH_ERRMSGFILE );
    }
}

int checkconfig()
{
    int err = 0;
#define REPORTERROR(x) { error( PARSE_MK, x ); err++; }
    if( !cfg.listenport ) REPORTERROR( "listenport" );
    if( !cfg.password ) REPORTERROR( "password" );
    if( !cfg.nickname ) REPORTERROR( "nickname" );
    if( !cfg.altnickname ) REPORTERROR( "altnickname" );
    if( !cfg.username ) REPORTERROR( "username" );
    if( !cfg.realname ) REPORTERROR( "realname" );
    if( !servers.amount ) { error( PARSE_NOSERV ); err++; };
    return ( err == 0 );
}

void setup_home( char *s )
{
    if( s ) {
        cfg.home = strdup( s );
        if( cfg.home[ strlen( cfg.home ) - 1 ] != '/' ) {
            cfg.home = realloc( cfg.home, strlen( cfg.home ) + 2 );
            strcat( cfg.home, "/" );
        }
    }
    else {
        if( !( s = getenv( "HOME" ) ) ) {
            error( MUH_ERRNOHOME );
            escape();
        }

        cfg.home = malloc( strlen( s ) + strlen( MUHDIR ) + 2 );
        strcpy( cfg.home, s );

        if( cfg.home[ strlen( cfg.home ) - 1 ] != '/' ) {
            strcat( cfg.home, "/" );
        }
        strcat( cfg.home, MUHDIR );
    }

    if( chdir( cfg.home ) < 0 ) {
        error( MUH_ERRCHDIR, cfg.home );
        escape();
    }
}

int main( int paramc, char *params[] )
{
    int pid = 0;
    FILE *pidfile;
    char *muhdir = 0;
    int dofork = 1;
    int c;
    char salt[ 3 ];

    fprintf( stdout, BANNER );

    opterr = 0;

    while( ( c = getopt( paramc, params, ":cfd:" ) ) > 0 ) {
        switch( c ) {
        case 'c':
            srand( time( NULL ) );
            randname( salt, 2 );
            printf( MUH_THISPASS, crypt( getpass( MUH_ENTERPASS ), salt ) );
            return 0;
            break;
        case 'd':
            muhdir = optarg;
            break;
        case 'f':
            dofork = 0;
            break;
        case ':':
            error( MUH_ERRNEEDARG, optopt );
            escape();
            break;
        default:
            printf( SYNTAX, params[ 0 ] );
            return 1;
            break;
        }
    }

    setup_home( muhdir );
    
    read_cfg();
    if( !checkconfig() ) escape();

    init();
    
    if( dofork ) {
        pid = fork();
        if( pid < 0 ) {
            error( MUH_ERRFORK );
            escape();
        }
        if( pid == 0 ) {
            if( !freopen( FILE_LOG, "a", stdout ) ) {
                error( MUH_ERRFILE, cfg.home );
                escape();
            }
#ifndef SETVBUF_REVERSED
            setvbuf( stdout, NULL, _IONBF, 0 );
#else
            setvbuf( stdout, _IONBF, NULL, 0 );
#endif
            printf( "\n" );
            report( MUH_NEWSESSION );
            report( MUH_STARTINGLOG );
            if( cfg.bind )
                report( SOCK_LISTENOKHOST, cfg.bind, cfg.listenport );
            else
                report( SOCK_LISTENOK, cfg.listenport );
            report( MUH_NICK, cfg.nickname );

        }
        if( pid ) {
            if( !( pidfile = fopen( FILE_PID, "w" ) ) ) {
                error( MUH_ERRFILE, cfg.home );
                kill( pid, SIGTERM );
                escape();
            }
            fprintf( pidfile, "%d\n", pid );
            fclose( pidfile );
            report( MUH_NICK, cfg.nickname );
            report( MUH_FORKED, pid );
            exit( 0 );
        }
    }
    run();
    return 0;
}
