/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include "MRI.h"
#include "vmedian.h"
#include <lcms.h>

struct MRIVMedianFilterData {
	struct link *next;
	int	width;
	int	y;
	int	height;
	int	useLab;
	double	vtol;
	int	freedata;
	struct MRI_ScanLine  *fp[3];
};

static inline fpixel
FP (struct MRI_ScanLine *sl, int x)
{
	fpixel res;

	res.r = ((double *)(sl->sl_Channel[0]))[x];
	res.g = ((double *)(sl->sl_Channel[1]))[x];
	res.b = ((double *)(sl->sl_Channel[2]))[x];
	return res;
}

void
MRIVMedianFilterStart (void *private, int width, int height, int freedata)
{
	struct MRIVMedianFilterData *wd = private;
	int i;

	(*wd->next->start) (wd->next->private, width, height, TRUE);
	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	wd->y = 0;
	for (i = 0; i < 3; i++) {
		wd->fp[i] = (struct MRI_ScanLine *)0;
	}
}

static void
DoVMFRow (struct MRIVMedianFilterData *wd)
{
	int x;
	struct MRI_ScanLine *out;
	double *R, *G, *B;

	out = MRI_NewScanLine (LINETYPE_DOUBLE, wd->width);
	R = out->sl_Channel[0];
	G = out->sl_Channel[1];
	B = out->sl_Channel[2];
	for (x = 0; x < wd->width; x++) {
		fpixel filter[9], res;
		int i, n;
		for (n = i = 0; i < 3; i++)
			if (wd->fp[i] != (struct MRI_ScanLine *)0) {
				if (x > 0) filter[n++] = FP (wd->fp[i], x-1);
				filter[n++] = FP (wd->fp[i], x);
				if (x < wd->width-1) filter[n++] = FP (wd->fp[i], x+1);
			}
		res = VectorMedian (n, filter, wd->vtol);
		R[x] = res.r;
		G[x] = res.g;
		B[x] = res.b;
	}

	(*wd->next->row) (wd->next->private, out);
}

static void
MRIVMedianFilterRow (void *private, void *data)
{
	struct MRIVMedianFilterData *wd = private;

	wd->fp[2] = (struct MRI_ScanLine *)data;

	if (wd->fp[1] != (struct MRI_ScanLine *)0) {
		DoVMFRow (wd);
	}
	wd->y++;
	if (wd->fp[0] != (struct MRI_ScanLine *)0 && wd->freedata)
		MRI_FreeScanLine (wd->fp[0]);
	wd->fp[0] = wd->fp[1];
	wd->fp[1] = wd->fp[2];
}

static void
MRIVMedianFilterClose (void *private)
{
	struct MRIVMedianFilterData *wd = private;

	DoVMFRow (wd);
	(*wd->next->close) (wd->next->private);
	if (wd->freedata) {
		MRI_FreeScanLine (wd->fp[0]);
		MRI_FreeScanLine (wd->fp[1]);
	}
	free (wd->next);
	free (wd);
}

struct link *
GenVMedianFilter (struct link *next, int useLab, double vtol)
{
	struct MRIVMedianFilterData *wd = malloc (sizeof (struct MRIVMedianFilterData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct MRIVMedianFilterData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = MRIVMedianFilterStart;
	ep->row = MRIVMedianFilterRow;
	ep->close = MRIVMedianFilterClose;
	ep->private = wd;
	wd->next = next;

	wd->useLab = useLab;
	wd->vtol = vtol;
	return ep;
}


struct MRIVMedianCondenserData {
	struct link *next;
	int	iwidth, owidth;
	int	iheight, oheight;
	int	y;
	double	vtol;
	int	freedata;
	struct MRI_ScanLine  *fp[4];
};

static void
MRIVMedianCondenserStart (void *private, int width, int height, int freedata)
{
	struct MRIVMedianCondenserData *wd = private;
	int i;

	wd->iwidth = width;
	wd->iheight = height;
	wd->owidth = width & 1 ? width+1 : width;
	wd->oheight = (height/2)+1;
	wd->freedata = freedata;
	(*wd->next->start) (wd->next->private, wd->owidth, wd->oheight, TRUE);
	wd->y = 0;
	for (i = 0; i < 4; i++) {
		wd->fp[i] = (struct MRI_ScanLine *)0;
	}
}

static void
DoVMCRow (struct MRIVMedianCondenserData *wd)
{
	int x;
	struct MRI_ScanLine *out;
	double *R, *G, *B;

	out = MRI_NewScanLine (LINETYPE_DOUBLE, wd->owidth);
	R = out->sl_Channel[0];
	G = out->sl_Channel[1];
	B = out->sl_Channel[2];
	for (x = 0; x < wd->owidth; x++) {
		fpixel filter[8], res;
		int i, n;
		for (n = i = 0; i < 4; i++)
			if (wd->fp[i] != (struct MRI_ScanLine *)0) {
				if (x > 0)
				    filter[n++] = FP (wd->fp[i], x-1);
				if (x < wd->iwidth)
				    filter[n++] = FP (wd->fp[i], x);
			}
		res = VectorMedian (n, filter, wd->vtol);
		R[x] = res.r;
		G[x] = res.g;
		B[x] = res.b;
	}

	(*wd->next->row) (wd->next->private, out);
}

static void
MRIVMedianCondenserRow (void *private, void *data)
{
	struct MRIVMedianCondenserData *wd = private;

	if (wd->y & 1) {
		wd->fp[3] = (struct MRI_ScanLine *)data;
		DoVMCRow (wd);
		if (wd->freedata) {
			MRI_FreeScanLine (wd->fp[0]);
			MRI_FreeScanLine (wd->fp[1]);
		}
		wd->fp[0] = wd->fp[2];
		wd->fp[1] = wd->fp[3];
	}
	else {
		wd->fp[2] = (struct MRI_ScanLine *)data;
	}
	wd->y++;
}

static void
MRIVMedianCondenserClose (void *private)
{
	struct MRIVMedianCondenserData *wd = private;

	wd->fp[2] = wd->fp[3] = (struct MRI_ScanLine *)0;
	DoVMCRow (wd);
	if (wd->freedata) {
		MRI_FreeScanLine (wd->fp[0]);
		MRI_FreeScanLine (wd->fp[1]);
	}
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

struct link *
GenVMedianCondenser (double vtol, struct link *next)
{
	struct MRIVMedianCondenserData *wd = malloc (sizeof (struct MRIVMedianCondenserData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct MRIVMedianCondenserData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = MRIVMedianCondenserStart;
	ep->row = MRIVMedianCondenserRow;
	ep->close = MRIVMedianCondenserClose;
	ep->private = wd;
	wd->next = next;

	wd->vtol = vtol;
	return ep;
}
