/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001      Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <time.h>
#include <string.h>
#include <bonobo.h>
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "util/time-utils.h"
#include "util/id-map.h"
#include "util/marshallers.h"
#include "task-model.h"
#include "task-manager.h"

#define DEBUG 0
#include "util/debug.h"

typedef struct {
	TaskModel *task;
	GNode     *g_node;
	gboolean   visited;
} GraphNode;

typedef struct {
	TaskManager *manager;
	GraphNode   *root;
	IdMap       *node_map;
} Graph;


static void task_manager_init          (TaskManager         *manager);
static void task_manager_class_init    (TaskManagerClass    *klass);
static void bubble_times               (TaskManager         *manager,
					TaskModel           *task);
static void task_destroyed             (TaskModel           *task, 
					TaskManager         *manager);
static void task_model_changed_cb      (TaskModel           *task_model,
					TaskChangeMask       mask,
					TaskManager         *task_manager);
static gboolean task_check_constraints (TaskManager         *manager,
					TaskModel           *task_model,
					GM_Time              start,
					GM_Time              end,
					GM_Time             *c_start,
					GM_Time             *c_end);
static Graph *dfs_build_graph          (TaskManager         *manager);
static GSList *dfs_sort                (Graph               *graph);
static void tm_set_task_type           (TaskManager         *manager,
					TaskModel           *task_model,
					GM_TaskType          type);


#define PARENT_TYPE BONOBO_X_OBJECT_TYPE
static GtkObjectClass *parent_class;

/* Private members. */
struct _TaskManagerPriv {
	/* Used to prevent recursive bubbling. */
	TaskModel       *root_task_model;
	time_t		 first_time;
	time_t           last_time;

	GList           *bubble_queue;
	
	gboolean         block_bubble : 1;
	gint             bubble_idle_id;

	GM_ProjectState  state;
};

/* Object arguments. */
enum {
	ARG_0,
	ARG_START,
	ARG_FIRST_TIME,
	ARG_LAST_TIME,
	ARG_STATE
};

static void
impl_TaskManager__set_state (PortableServer_Servant  servant,
			     const GM_ProjectState   state,
			     CORBA_Environment      *ev)
{
	TaskManager          *manager;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	manager->priv->state = state;
}

static GM_ProjectState
impl_TaskManager__get_state (PortableServer_Servant  servant,
			     CORBA_Environment      *ev)
{
	TaskManager          *manager;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	return manager->priv->state;
}

static GM_Task *
impl_TaskManager_createTask (PortableServer_Servant  servant,
			     CORBA_Environment      *ev)
{
	TaskManagerClass     *klass;
	TaskManager          *manager;
	GM_Task *task;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	/* Create a task and set default values. */
	task                  = GNOME_MrProject_Task__alloc ();
	task->taskId          = -1;
	task->parentId        = -1;
	task->name            = CORBA_string_dup ("");
	task->start           = -1;
	task->end             = -1;
	task->percentComplete = 0;
	task->type            = GNOME_MrProject_TASK_NORMAL;
	
	return task;
}

static GM_Id
tm_insert_full (TaskManager       *manager,
		const GM_Task     *task,
		GM_Id              parentId,
		GM_Id              siblingId,
		GM_TaskOrderType   type,
		CORBA_Environment *ev)
{		
	TaskManagerClass     *klass;
	TaskManagerPriv      *priv;
	GM_EventTaskInserted *event;
	GM_Task		     *task_copy;
	TaskModel            *task_model, *parent_task_model, *sibling_task_model;
	CORBA_any            *any;
	CORBA_Environment     e;
	GM_TaskManager        tm_co;
	Bonobo_PropertyBag    pb_co;
	glong                 start;
	glong                 finish;
	
	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	priv = manager->priv;

	parent_task_model = task_manager_get_task_model (manager, parentId);
	if (!parent_task_model) {
		corba_util_set_exception (ev,
					  GNOME_MrProject_TaskManager_NoSuchParent,
					  GNOME_MrProject_ERROR_INFO,
					  "Cannot insert a task under a non-existant parent task.");
		return -1;
	}

	if (siblingId != -1) {
		sibling_task_model = task_manager_get_task_model (manager, siblingId);
		if (!sibling_task_model) {
			corba_util_set_exception (ev,
						  GNOME_MrProject_TaskManager_NoSuchParent,
						  GNOME_MrProject_ERROR_INFO,
						  "Cannot insert a task next to a non-existant sibling task.");
			return -1;
		}
	}
	else {
		sibling_task_model = NULL;
	}
	
	task_model = klass->insert_task (manager,
					 task,
					 parent_task_model,
					 sibling_task_model,
					 type,
					 ev);
	if (task_model == NULL) {
		return -1;
	}

	task_copy = task_model_get_task (task_model);
	task_copy->parentId = parentId;
	
	/* Bubble up times in the tree. */
	bubble_times (manager, task_model);

	gtk_signal_connect (GTK_OBJECT (task_model), 
			    "changed", 
			    task_model_changed_cb, 
			    manager);

	gtk_signal_connect (GTK_OBJECT (task_model), 
			    "destroy", 
			    task_destroyed, 
			    manager);

	CORBA_exception_init (&e);

	tm_co = BONOBO_OBJREF (manager);
	pb_co = Bonobo_Unknown_queryInterface (tm_co,
					       "IDL:Bonobo/PropertyBag:1.0",
					       &e);
	
	if (BONOBO_EX (&e) || pb_co == CORBA_OBJECT_NIL) {
		g_warning ("Couldn't get PropertyBag");
		CORBA_exception_free (&e);
		return -1;
	}

	start = bonobo_property_bag_client_get_value_glong (pb_co, 
							    "Start",
							    &e);
	
	finish = bonobo_property_bag_client_get_value_glong (pb_co, 
							     "Finish",
							     &e);
	
	start  = MIN (start, task->start);
	finish = MAX (finish, task->end);
						    
	bonobo_property_bag_client_set_value_glong (pb_co, 
						    "Start",
						    start,
						    &e);
	
	bonobo_property_bag_client_set_value_glong (pb_co, 
						    "Finish",
						    finish,
						    &e);
	
	bonobo_object_release_unref (pb_co, NULL);

	CORBA_exception_free (&e);
						    

	manager->priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	event = GNOME_MrProject_EventTaskInserted__alloc ();
	corba_util_task_copy (&event->theTask, task_copy);
	event->siblingId = siblingId;
	event->type = type;
	
	any = CORBA_any__alloc ();
	any->_type = TC_GNOME_MrProject_EventTaskInserted;
	any->_value = event;

	CORBA_any_set_release (any, TRUE);

	if (manager->event_source) {
		CORBA_exception_init (&e);
		bonobo_event_source_notify_listeners (manager->event_source, 
						      "GNOME/MrProject:task:inserted",
						      any,
						      &e);
		CORBA_exception_free (&e);
	}

	CORBA_free (any);

	tm_set_task_type (manager, parent_task_model, GNOME_MrProject_TASK_SUMMARY);

	manager->priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	return task_copy->taskId;
}

static GM_Id
impl_TaskManager_insertTask (PortableServer_Servant  servant, 
			     const GM_Task          *task, 
			     GM_Id                   parentId,
			     CORBA_Environment      *ev)
{
	TaskManager *manager;

	manager = TASK_MANAGER (bonobo_x_object (servant));

	return tm_insert_full (manager, task, parentId, -1, 0, ev);
}

static GM_Id
impl_TaskManager_insertTaskFull (PortableServer_Servant  servant, 
				 const GM_Task          *task, 
				 GM_Id                   parentId,
				 GM_Id                   siblingId,
				 GM_TaskOrderType        type,
				 CORBA_Environment      *ev)
{
	TaskManager *manager;

	manager = TASK_MANAGER (bonobo_x_object (servant));

	return tm_insert_full (manager, task, parentId, siblingId, type, ev);
}

static void
impl_TaskManager_repositionTask (PortableServer_Servant  servant, 
				 GM_Id                   taskId, 
				 GM_Id                   siblingId,
				 GM_TaskOrderType        type,
				 CORBA_Environment      *ev)
{
	TaskManagerClass                      *klass;
	TaskManager                           *manager;
	TaskManagerPriv                       *priv;
	TaskModel                             *task_model, *sibling_task_model;
	GNOME_MrProject_EventTaskRepositioned *data;
	CORBA_any                             *any;
	CORBA_Environment                      e;
	
	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);
	priv = manager->priv;

	task_model = task_manager_get_task_model (manager, taskId);
	if (!task_model) {
		corba_util_set_exception (ev,
					  GNOME_MrProject_TaskManager_NoSuchTask,
					  GNOME_MrProject_ERROR_INFO,
					  "Cannot reposition non-existant task.");
		return;
	}

	if (siblingId != -1) {
		sibling_task_model = task_manager_get_task_model (manager, siblingId);
		if (!sibling_task_model) {
			corba_util_set_exception (ev,
						  GNOME_MrProject_TaskManager_NoSuchSibling,
						  GNOME_MrProject_ERROR_INFO,
						  "Cannot move task next to a non-existant task.");
			return;
		}
	} else {
		sibling_task_model = NULL;
	}

	/* Do it. */
	klass->reposition_task (manager, task_model, sibling_task_model, type, ev);
	if (BONOBO_EX (ev)) {
		return;
	}
	
	data            = GNOME_MrProject_EventTaskRepositioned__alloc ();
	data->taskId    = taskId;
	data->siblingId = siblingId;
	data->type      = type;
	
	any = CORBA_any__alloc ();
	any->_type = TC_GNOME_MrProject_EventTaskRepositioned;
	any->_value = (GNOME_MrProject_EventTaskRepositioned *) data;
	CORBA_any_set_release (any, TRUE);
	
	if (manager->event_source) {
		CORBA_exception_init (&e);
		bonobo_event_source_notify_listeners (manager->event_source, 
						      "GNOME/MrProject:task:repositioned",
						      any,
						      &e);
		CORBA_exception_free (&e);
	}
	
	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	CORBA_free (any);
}

static void
impl_TaskManager_updateTask (PortableServer_Servant  servant, 
			     GM_Id                   taskId,
			     const GM_Task          *task,
			     CORBA_Environment      *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;
	TaskManagerPriv  *priv;
	TaskModel        *old_task_model;
	GM_Task          *old_task;
	GM_Time           start, end, duration;
	TaskChangeMask    mask;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	priv = manager->priv;

	old_task_model = task_manager_get_task_model (manager, taskId);
	g_return_if_fail (IS_TASK_MODEL (old_task_model));
	old_task = task_model_get_task (old_task_model);
	g_return_if_fail (old_task_model != NULL);
	
	/* Check constraints. */
	start    = task->start;
	end      = task->end;
	duration = end - start;
	
	mask = TASK_CHANGE_ALL;
	if (!task_check_constraints (manager, old_task_model, start, end, &start, &end)) {
		g_print ("** Engine: update not allowed\n");
		return;
		
		mask = corba_util_task_update (old_task, task, mask);
		old_task->start = start;
		old_task->end = end;
		task_model_changed (old_task_model, mask | TASK_CHANGE_START | TASK_CHANGE_END);
	} else {
 		mask = corba_util_task_update (old_task, task, mask);
		task_model_changed (old_task_model, mask);
 	}

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	if (klass->update_task)
		klass->update_task (manager, taskId, old_task, ev);
}

static void
impl_TaskManager_moveTask (PortableServer_Servant  servant,
			   GM_Id                   taskId,
			   GM_Time                 start,
			   GM_TaskEdge             edge,
			   CORBA_Environment      *ev)
{
	TaskManager      *manager;
	TaskManagerPriv  *priv;
	TaskManagerClass *klass;
	TaskModel        *task_model;
	GM_Task          *task;
	GM_Time           end, duration;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	priv = manager->priv;

	task_model = task_manager_get_task_model (manager, taskId);
	task = task_model_get_task (task_model);
	
	duration = task->end - task->start;
	
	if (!task_check_constraints (manager, task_model, start, end, &start, &end)) {
		g_print ("** Engine: moveTask not allowed\n");
	}

	if (edge == GNOME_MrProject_TASK_START) {
		task->start = start;
		task->end = start + duration;
	} else {
		task->end = end;
		task->start = end - duration;
	}
	
	task_model_changed (task_model, TASK_CHANGE_START | TASK_CHANGE_END);

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	if (klass->update_task) {
		klass->update_task (manager, taskId, task, ev);
	}
}

static void
impl_TaskManager_setTaskDuration (PortableServer_Servant  servant,
				  GM_Id                   taskId,
				  GM_Time                 duration,
				  GM_TaskEdge             relativeEdge,
				  CORBA_Environment      *ev)
{
	TaskManager      *manager;
	TaskManagerPriv  *priv;
	TaskManagerClass *klass;
	TaskModel        *task_model;
	GM_Task          *task;
	GM_Time           start, end;

	g_return_if_fail (duration >= 0);
	
	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	priv = manager->priv;

	task_model = task_manager_get_task_model (manager, taskId);
	task = task_model_get_task (task_model);

	if (relativeEdge == GNOME_MrProject_TASK_START) {
		start = task->start;
		end = start + duration;
	} else {
		end = task->end;
		start = end - duration;
	}
	
	if (!task_check_constraints (manager, task_model, start, end, &start, &end)) {
		g_print ("** Engine: setStartDuration not allowed\n");
	}

	if (relativeEdge == GNOME_MrProject_TASK_START) {
		task->end = end;
	} else {
		task->start = start;
	}
	
	task_model_changed (task_model, TASK_CHANGE_START | TASK_CHANGE_END);

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	if (klass->update_task) {
		klass->update_task (manager, taskId, task, ev);
	}
}

static void
impl_TaskManager_removeTasks (PortableServer_Servant  servant, 
			      const GM_IdSeq         *ids,
			      CORBA_Environment      *ev)
{
	TaskManagerClass  *klass;
	TaskManager       *manager;
	CORBA_Environment  e;
	GM_Task           *task;
	guint		   i;
	GHashTable        *removee_hash;
	GSList            *task_list, *l;
	GSList		  *trimmed_task_list, *trimmed_id_list;
	GSList            *parent_task_model_list;
	
	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	d(puts (__FUNCTION__));
	
	/* We have to filter out all tasks that are children of
	 * any other of the tasks that are to be removed, since we must
	 * only emit events for each removed subtree _root_.
	 */

	removee_hash = g_hash_table_new (g_int_hash, g_int_equal);
	task_list = NULL;
	for (i = 0; i < ids->_length; i++) {
		gint id, *key;

		id = ids->_buffer[i];
		task = klass->get_task (manager, id, ev);

		if (!task || id == 0) {
			corba_util_set_exception (ev,
						  GNOME_MrProject_TaskManager_NoSuchTask,
						  GNOME_MrProject_ERROR_INFO,
						  "Could not find tasks to remove.");
			g_hash_table_foreach (removee_hash, (GHFunc) g_free, NULL);
			g_hash_table_destroy (removee_hash);
			g_slist_free (task_list);
			return;
		}
		
		key = g_new (gint, 1);
		*key = id;
		g_hash_table_insert (removee_hash, key, task);
		task_list = g_slist_prepend (task_list, task);
	}

	/* Perform the trimming. */
	l = task_list;
	trimmed_task_list = NULL;
	trimmed_id_list = NULL;
	parent_task_model_list = NULL;
	for (i = 0; i < ids->_length; i++) {
		TaskModel *task_model, *parent_task_model;
		gint       id, parent_id;

		task = l->data;
		id = task->taskId;
		parent_id = task->parentId;
		if (!g_hash_table_lookup (removee_hash, &parent_id)) {
			trimmed_task_list = g_slist_prepend (trimmed_task_list, task);
			trimmed_id_list = g_slist_prepend (trimmed_id_list, GINT_TO_POINTER (id));

			/* Also keep the parents of the tasks we remove so that
			 * we can update their summary/normal status.
			 */
			task_model = task_manager_get_task_model (manager, id);
			parent_task_model = task_manager_get_parent_task_model (manager, task_model);
			if (parent_task_model) {
				parent_task_model_list =
					g_slist_prepend (parent_task_model_list, parent_task_model);
			}
		}
		
		l = l->next;
	}

	/* Notify clients. */
	if (!BONOBO_EX (ev)) {
		BonoboArg *arg;
		gchar     *id_str;
		
		arg = bonobo_arg_new (BONOBO_ARG_STRING);
		id_str = corba_util_id_string_from_list (trimmed_id_list);
		
		BONOBO_ARG_SET_STRING (arg, id_str);
		
		if (manager->event_source) {
			CORBA_exception_init (&e);
			bonobo_event_source_notify_listeners (manager->event_source, 
							      "GNOME/MrProject:task:removed_seq",
							      arg,
							      &e);
			CORBA_exception_free (&e);
		}

		CORBA_free (id_str);
		bonobo_arg_release (arg);
	}

	/* Remove predecessor/successor relations. */
	// DO IT!
	
	if (klass->remove_tasks) {
		klass->remove_tasks (manager, trimmed_task_list, ev);
	}

	/* Update the summary/normal status for all the removed tasks' parents. */
	for (l = parent_task_model_list; l; l = l->next) {
		TaskModel *parent_task_model;

		parent_task_model = l->data;

		if (task_model_get_num_children (parent_task_model) == 0) {
			tm_set_task_type (manager, parent_task_model, GNOME_MrProject_TASK_NORMAL);
		}
	}
	
	g_slist_free (task_list);
	g_slist_free (trimmed_task_list);
	g_slist_free (trimmed_id_list);
	g_hash_table_foreach (removee_hash, (GHFunc) g_free, NULL);
	g_hash_table_destroy (removee_hash);
	
	manager->priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
}

static void
impl_TaskManager_reparentTask (PortableServer_Servant  servant, 
			       GM_Id                   id,
			       GM_Id                   parent_id,
			       CORBA_Environment      *ev)
{
	TaskManagerClass                    *klass;
	TaskManager                         *manager;
	TaskManagerPriv                     *priv;
	TaskModel                           *task_model, *parent_task_model;
	TaskModel                           *old_sibling_task_model;
	gint                                 old_parent_id;
	GNOME_MrProject_EventTaskReparented *data;
	GM_Task                             *task, *parent_task;
	CORBA_any                           *any;
	CORBA_Environment                    e;
	
	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	priv = manager->priv;

	task_model = task_manager_get_task_model (manager, id);
	
	if (!task_model) {
		corba_util_set_exception (ev,
					  GNOME_MrProject_TaskManager_NoSuchTask,
					  GNOME_MrProject_ERROR_INFO,
					  "Could not find task to reparent.");
		return;
	}
	
	parent_task_model = task_manager_get_parent_task_model (manager, task_model);
	if (parent_task_model != NULL) {
		GM_Task *parent_task;
		parent_task = task_model_get_task (parent_task_model);
		old_parent_id = parent_task->taskId;
	} else {
		old_parent_id = 0;
	}

	if (old_parent_id == parent_id) {
		return;
	}

	/* Get the task's sibling so that we can rebubble it after the
	 * reparenting.
	 */
	old_sibling_task_model = task_manager_get_first_sibling (manager, task_model);
	if (old_sibling_task_model == task_model) {
		old_sibling_task_model = task_manager_get_next_sibling (manager, task_model);
	}
	
	parent_task_model = task_manager_get_task_model (manager, parent_id);
	
	if (!parent_task_model) {
			corba_util_set_exception (ev,
						  GNOME_MrProject_TaskManager_NoSuchParent,
						  GNOME_MrProject_ERROR_INFO,
						  "Could not find new parent task.");
		return;
	}

	/* Make sure we don't have a dependency between the task and it's new
	 *  parent.
	 */
	if (task_model_has_predecessor (task_model, parent_id, NULL)) {	
		/* FIXME: Raise exception. */
		return;
	}

	klass->reparent_task (manager, task_model, parent_task_model, ev);

	task = task_model_get_task (task_model);
	parent_task = task_model_get_task (parent_task_model);
	task->parentId = parent_task->taskId;

	/* Update summary/normal status. */
	tm_set_task_type (manager, parent_task_model, GNOME_MrProject_TASK_SUMMARY);

	if (old_parent_id) {
		TaskModel *old_parent_task_model;

		old_parent_task_model = task_manager_get_task_model (manager, old_parent_id);
		
		if (task_model_get_num_children (old_parent_task_model) > 0) {
			tm_set_task_type (manager, old_parent_task_model, GNOME_MrProject_TASK_SUMMARY);
		} else {
			tm_set_task_type (manager, old_parent_task_model, GNOME_MrProject_TASK_NORMAL);
		}
	}
	
	/* Bubble the new tree. */
	bubble_times (manager, task_model);
	
	/* And bubble the old tree. */
	if (old_sibling_task_model) {
		bubble_times (manager, old_sibling_task_model);
	}

	/* Notify clients. */
	data              = GNOME_MrProject_EventTaskReparented__alloc ();
	data->taskId      = id;
	data->newParentId = parent_id;
	
	any         = CORBA_any__alloc ();
	any->_type  = TC_GNOME_MrProject_EventTaskReparented;
	any->_value = data;
	CORBA_any_set_release (any, TRUE);
					
	if (manager->event_source) {
		CORBA_exception_init (&e);
		bonobo_event_source_notify_listeners (manager->event_source, 
						      "GNOME/MrProject:task:reparented",
						      any, &e);
		CORBA_exception_free (&e);
	}

	CORBA_free (any);

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
}

static GM_Task *
impl_TaskManager_getTask (PortableServer_Servant  servant, 
			  GM_Id                   id,
			  CORBA_Environment      *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;
	GM_Task          *task;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	task = klass->get_task (manager, id, ev);
	if (BONOBO_EX (ev)) {
		return CORBA_OBJECT_NIL;
	}
	
	if (task == CORBA_OBJECT_NIL) {
		gchar *msg = g_strdup_printf ("Could not find task with Id %d.",
					      (int) id);
		corba_util_set_exception (ev,
					  GNOME_MrProject_TaskManager_NoSuchTask,
					  GNOME_MrProject_ERROR_INFO,
					  msg);
		g_free (msg);
		return CORBA_OBJECT_NIL;
	}

	return corba_util_task_duplicate (task);
}

static GM_TaskSeq *
impl_TaskManager_getAllTasks (PortableServer_Servant  servant, 
			      CORBA_Environment      *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;
	GSList           *list;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	list = klass->get_all_tasks (manager, ev);
	if (BONOBO_EX (ev)) {
		return CORBA_OBJECT_NIL;
	}

	return corba_util_task_seq_from_list (list);
}

static GM_Id
impl_TaskManager_linkTasks (PortableServer_Servant  servant,
			    GM_Id                   taskId,
			    GM_Id                   predecessorId,
			    GM_DependencyType       type,
			    CORBA_Environment      *ev)
{
	TaskManagerClass  *klass;
	TaskManager       *manager;
	TaskManagerPriv   *priv;
	TaskModel         *task_model, *predecessor_task_model;
	GM_Task           *task, *predecessor_task;
	GM_Dependency     *dependency;
	CORBA_Environment  e;
	CORBA_any         *any;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);
	priv    = manager->priv;
	
	task_model = klass->get_task_model (manager, taskId);
	predecessor_task_model = klass->get_task_model (manager, predecessorId);

	if (!task_model || !predecessor_task_model) {
		corba_util_set_exception (ev,
					  GNOME_MrProject_TaskManager_NoSuchTask,
					  GNOME_MrProject_ERROR_INFO,
					  "Could not find tasks to link.");
		return -1;
	}

	task = task_model_get_task (task_model);
	predecessor_task = task_model_get_task (predecessor_task_model);

	/* Check that we don't try to link a task to it's ancestor. */
	if (task_model_has_ancestor (task_model, predecessor_task_model)) {
		/* FIXME: Raise exception. */
		g_warning ("Can't add dependency between a task and its ancestor.");
		return -1;
	}

	/* FIXME: what the **** is this?? */
#if 0
	/* Check to see if they're reversed. */
	if (task_model_has_predecessor (predecessor_task_model, taskId, NULL) &&
	    task_model_has_successor (task_model, predecessorId, NULL)) {
		/* Reverse them. */
		TaskModel *tmp;
		GM_Id tmpid;

		tmp = predecessor_task_model;
		predecessor_task_model = task_model;
		task_model = tmp;

		tmpid = predecessorId;
		predecessorId = taskId;
		taskId = tmpid;
	}
#endif
	
	if (task_model_has_predecessor (task_model, predecessorId, NULL) ||
	    task_model_has_successor (predecessor_task_model, taskId, NULL)) {
		g_warning ("Tasks already linked.");
		return -1;
	}

	dependency = GNOME_MrProject_Dependency__alloc ();
	dependency->taskId = taskId;
	dependency->predecessorId = predecessorId;
	dependency->type = type;
	dependency->empty = CORBA_string_dup ("");
 
	dependency->depId = klass->link_tasks (manager, dependency, ev);
	if (dependency->depId == -1) {
		g_warning ("Could not insert dependency.\n");
		g_free (dependency);
		return -1;
	}

	/* Add the dependency as predecessor for the "main" task, and as a
	 * successor for the predecessor task.
	 */
	task_model_add_predecessor (task_model, dependency);
	task_model_add_successor (predecessor_task_model, dependency);

	/* Hardcode to FS for now, and no lag time. */

	/* Update successors. FIXME: Don't cut'n'paste it like this, just a fix for M2. */
	/* Hardcode to FS for now, and no lag time. */
	if (predecessor_task->end > task->start) {
		gint diff;
		
		diff = predecessor_task->end - task->start;
		task->start += diff;
		task->end += diff;
		task_model_changed (task_model, TASK_CHANGE_START | TASK_CHANGE_END);
	}
	
	any = CORBA_any__alloc ();
 	any->_type = TC_GNOME_MrProject_Dependency;
	any->_value = (GM_Dependency *) dependency;
	CORBA_any_set_release (any, FALSE);
	
	CORBA_exception_init (&e);
	bonobo_event_source_notify_listeners (manager->event_source, 
					      "GNOME/MrProject:task:linked",
					      any,
					      &e);
	CORBA_exception_free (&e);
	CORBA_free (any);

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	return dependency->depId;
}

static void
impl_TaskManager_unlinkTasks (PortableServer_Servant  servant,
			      GM_Id                   taskId,
			      GM_Id                   predecessorId,
			      CORBA_Environment      *ev)
{
	TaskManager       *manager;
	TaskManagerPriv   *priv;
	TaskManagerClass  *klass;
	GM_Id              dependency_id;
	GM_Dependency     *dependency;
	TaskModel         *task, *predecessor_task;
	CORBA_Environment  e;
	CORBA_any	  *any;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);
	priv    = manager->priv;

	task = task_manager_get_task_model (manager, taskId);
	if (!task) {
		/* FIXME: raise ex. */
		return;
	}

	predecessor_task = task_manager_get_task_model (manager, predecessorId);
	if (!predecessor_task) {
		/* FIXME: raise ex. */
		return;
	}

	/* check to see if they're reversed */
	if (task_model_has_predecessor (predecessor_task, taskId, NULL) &&
	    task_model_has_successor (task, predecessorId, NULL)) {
		/* reverse them */
		TaskModel *tmp;
		GM_Id tmpid;

		tmp = predecessor_task;
		predecessor_task = task;
		task = tmp;

		tmpid = predecessorId;
		predecessorId = taskId;
		taskId = tmpid;
	}
		

	/* Try to find a dependency from the task to the predecessorId. */
	if (!task_model_has_predecessor (task, predecessorId, &dependency_id) ||
	    !task_model_has_successor (predecessor_task, taskId, NULL)) {
		/* FIXME: raise ex. */
		g_warning ("Dependency between task %d and %d does not exist.",
			   taskId, predecessorId);
		return;
	}

	/* Remove the dependency. */
	task_model_remove_predecessor (task, predecessorId);
	task_model_remove_successor (predecessor_task, taskId);

	dependency = klass->get_dependency (manager, dependency_id, ev);
	klass->unlink_tasks (manager, dependency->depId, ev);

	any = CORBA_any__alloc ();
	any->_type = TC_GNOME_MrProject_Dependency;
	any->_value = (GM_Dependency *) dependency;
	CORBA_any_set_release (any, FALSE);

	if (manager->event_source) {
		CORBA_exception_init (&e);
		bonobo_event_source_notify_listeners (
			manager->event_source, 
			"GNOME/MrProject:task:unlinked",
			any,
			&e);
		CORBA_exception_free (&e);
	}

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;

	CORBA_free (dependency);
	CORBA_free (any);
}
					     
static void
impl_TaskManager_removeDependency (PortableServer_Servant  servant,
				   GM_Id      dependencyId,
				   CORBA_Environment      *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	klass->remove_dependency (manager, dependencyId, ev);
}

static GM_Dependency *
impl_TaskManager_getDependency (PortableServer_Servant  servant,
				const GM_Id             dependencyId,
				CORBA_Environment      *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	/* FIXME: dup here! */
	return klass->get_dependency (manager, dependencyId, ev);
}

static GM_DependencySeq *
impl_TaskManager_getPredecessors (PortableServer_Servant  servant,
				  GM_Id                   taskId,
				  CORBA_Environment      *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;
	TaskModel        *task_model;
	
	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	task_model = task_manager_get_task_model (manager, taskId);
	if (!task_model) {
		corba_util_set_exception (ev,
					  GNOME_MrProject_TaskManager_NoSuchTask,
					  GNOME_MrProject_ERROR_INFO,
					  "Could not find task.");
		return CORBA_OBJECT_NIL;
	}
	
	return task_model_get_predecessors_seq (task_model);
}

static GM_DependencySeq *
impl_TaskManager_getSuccessors (PortableServer_Servant  servant,
				const GM_Id             taskId,
				CORBA_Environment      *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;
	TaskModel        *task_model;
	
	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	task_model = task_manager_get_task_model (manager, taskId);
	if (!task_model) {
		corba_util_set_exception (ev,
					  GNOME_MrProject_TaskManager_NoSuchTask,
					  GNOME_MrProject_ERROR_INFO,
					  "Could not find task.");
		return CORBA_OBJECT_NIL;
	}
	
	return task_model_get_successors_seq (task_model);
}

static void
impl_TaskManager_dumpTree (PortableServer_Servant        servant, 
			   CORBA_Environment            *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;

	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	klass->dump_tree (manager, ev);
}

static GNOME_MrProject_IdSeq *
impl_TaskManager_getAllTaskIdsSorted (PortableServer_Servant  servant, 
				      CORBA_Environment      *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;
	Graph            *graph;
		
	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	graph = dfs_build_graph (manager);

	return corba_util_id_seq_from_list (dfs_sort (graph));
}

static CORBA_char *
impl_TaskManager_getNote (PortableServer_Servant  servant,
			  const GM_Id             taskId,
			  CORBA_Environment      *ev)
{
	TaskManagerClass *klass;
	TaskManager      *manager;
	gchar            *note;
	
	manager = TASK_MANAGER (bonobo_x_object (servant));
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	note = klass->get_note (manager, taskId, ev);
	
	if (!note) {
		note = "";
	}
	
	return CORBA_string_dup (note);
}

static void
impl_TaskManager_setNote (PortableServer_Servant  servant,
			  const GM_Id             taskId,
			  const CORBA_char       *note,
			  CORBA_Environment      *ev)
{
	TaskManagerPriv                  *priv;
	TaskManagerClass                 *klass;
	TaskManager                      *manager;
	CORBA_any                        *any;
	GNOME_MrProject_EventNoteChanged *event;
	CORBA_Environment                 e;
	
	manager = TASK_MANAGER (bonobo_x_object (servant));
	priv    = manager->priv;
	klass   = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	klass->set_note (manager, taskId, note, ev);

	event         = GNOME_MrProject_EventNoteChanged__alloc ();
	event->taskId = taskId;
	event->note   = CORBA_string_dup (note);
	
	any = CORBA_any__alloc ();
	any->_type = TC_GNOME_MrProject_EventNoteChanged;
	any->_value = event;
	CORBA_any_set_release (any, TRUE);

	if (manager->event_source) {
		CORBA_exception_init (&e);
		bonobo_event_source_notify_listeners (
			manager->event_source, 
			"GNOME/MrProject:task:note_changed",
			any,
			&e);
		CORBA_exception_free (&e);
	}

	priv->state = GNOME_MrProject_PROJECT_STATE_DIRTY;
	
	CORBA_free (any);
}

/* --------------- */


static void
task_model_changed_cb (TaskModel      *task_model,
		       TaskChangeMask  mask,
		       TaskManager    *task_manager)
{
	const GM_Task *task;
	CORBA_any                  *any;
	CORBA_Environment           e;

	task = task_model_get_task (task_model);

	/* Bubble up times in the tree if neccessary. */
	if ((mask & TASK_CHANGE_START) || (mask & TASK_CHANGE_END)) {
		bubble_times (task_manager, task_model);
	}

	/* Update successors. */
	{
		GSList *list;

		list = task_model_get_successors (task_model);
		for (; list; list = list->next) {
			TaskModel                  *successor_task_model;
			GM_Task       *successor_task;
			GM_Dependency *dependency;

			dependency = list->data;
			successor_task_model = task_manager_get_task_model (task_manager, dependency->taskId);
			g_assert (successor_task_model);

			successor_task = task_model_get_task (successor_task_model);
			
			/* Hardcode to FS for now, and no lag time. */
			if (task->end > successor_task->start) {
				gint diff;

				diff = task->end - successor_task->start;
				successor_task->start += diff;
				successor_task->end += diff;
				task_model_changed (successor_task_model, TASK_CHANGE_START | TASK_CHANGE_END);
			}
		}
	}
	
	any         = CORBA_any__alloc ();
	any->_type  = TC_GNOME_MrProject_Task;
	any->_value = (GM_Task *) task;
	CORBA_any_set_release (any, FALSE);
	
	if (task_manager->event_source) {
		CORBA_exception_init (&e);
		bonobo_event_source_notify_listeners (task_manager->event_source, 
						      "GNOME/MrProject:task:updated",
						      any,
						      &e);
		CORBA_exception_free (&e);
	}

	CORBA_free (any);	
}

static void
task_manager_destroy (GtkObject *object)
{
	TaskManager     *manager;
	TaskManagerPriv *priv;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_TASK_MANAGER (object));

	manager = TASK_MANAGER (object);
	priv = manager->priv;

	if (manager->event_source) {
		manager->event_source = NULL;
	}	
	
	g_free (priv);
	manager->priv = NULL;

	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
task_manager_init (TaskManager *manager)
{
	TaskManagerPriv *priv;
	
	priv = g_new0 (TaskManagerPriv, 1);
	manager->priv = priv;

	priv->state = GNOME_MrProject_PROJECT_STATE_EMPTY;

	/* For debugging: */
	priv->first_time = time_from_day (2001, 5, 1);
	priv->last_time = time_from_day (2001, 8, 1);
}

void
task_manager_construct (TaskManager       *manager,
			TaskModel         *root_task_model,
			BonoboEventSource *event_source)
{
	TaskManagerPriv *priv;
	
	g_return_if_fail (manager->event_source == NULL);

	priv = manager->priv;

	priv->root_task_model = root_task_model;
	
	manager->event_source = event_source;
}

static void
task_manager_set_arg (GtkObject *object, 
		      GtkArg    *arg, 
		      guint      arg_id)
{
	TaskManager     *manager;	
	TaskManagerPriv *priv;

	manager = (TaskManager *) object;
	priv = manager->priv;

	switch (arg_id) {
	case ARG_FIRST_TIME:
		priv->first_time = (time_t) GTK_VALUE_LONG (*arg);
		break;

	case ARG_LAST_TIME:
		priv->last_time = (time_t) GTK_VALUE_LONG (*arg);
		break;

	case ARG_START:
		break;

	case ARG_STATE:
		priv->state = GTK_VALUE_INT (*arg);
		break;
	default:
		return;
		break;
	}
}

static void
task_manager_get_arg (GtkObject *object, 
		      GtkArg    *arg, 
		      guint      arg_id)
{
	TaskManager     *manager;
	TaskManagerPriv *priv;

	manager = TASK_MANAGER (object);
	priv = manager->priv;

	switch (arg_id) {
	case ARG_FIRST_TIME:
		GTK_VALUE_LONG (*arg) = priv->first_time;
		break;

	case ARG_LAST_TIME:
		GTK_VALUE_LONG (*arg) = priv->last_time;
		break;

	case ARG_START:
		/*GTK_VALUE_LONG (*arg) = priv->start;*/
		break;

	case ARG_STATE:
		GTK_VALUE_INT (*arg) = priv->state;
		break;

	default:
		arg->type = GTK_TYPE_INVALID;
		break;
	}
}

static void
task_manager_class_init (TaskManagerClass *klass)
{
	POA_GNOME_MrProject_TaskManager__epv *epv;
	GtkObjectClass                       *object_class;
	
	epv          = &klass->epv;
	object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class (PARENT_TYPE);

	epv->_set_state          = impl_TaskManager__set_state;
	epv->_get_state          = impl_TaskManager__get_state;
	epv->createTask          = impl_TaskManager_createTask;
	epv->insertTask          = impl_TaskManager_insertTask;
	epv->insertTaskFull      = impl_TaskManager_insertTaskFull;
	epv->updateTask          = impl_TaskManager_updateTask;
	epv->moveTask            = impl_TaskManager_moveTask;
	epv->setTaskDuration     = impl_TaskManager_setTaskDuration;
	epv->repositionTask      = impl_TaskManager_repositionTask;
	epv->removeTasks         = impl_TaskManager_removeTasks;
	epv->reparentTask        = impl_TaskManager_reparentTask;
	epv->getTask             = impl_TaskManager_getTask;
	epv->getAllTasks         = impl_TaskManager_getAllTasks;
	epv->linkTasks           = impl_TaskManager_linkTasks;
	epv->unlinkTasks         = impl_TaskManager_unlinkTasks;
	epv->removeDependency    = impl_TaskManager_removeDependency;
	epv->getDependency       = impl_TaskManager_getDependency;
	epv->getPredecessors     = impl_TaskManager_getPredecessors;
	epv->getSuccessors       = impl_TaskManager_getSuccessors;
	epv->getNote             = impl_TaskManager_getNote;
	epv->setNote             = impl_TaskManager_setNote;
	epv->getAllTaskIdsSorted = impl_TaskManager_getAllTaskIdsSorted;
	epv->dumpTree            = impl_TaskManager_dumpTree;
	
	/* Add arguments. */
	gtk_object_add_arg_type ("TaskManager::start", 
				 GTK_TYPE_LONG, 
				 GTK_ARG_READWRITE, 
				 ARG_START);

	gtk_object_add_arg_type ("TaskManager::first_time", 
				 GTK_TYPE_LONG, 
				 GTK_ARG_READWRITE, 
				 ARG_FIRST_TIME);

	gtk_object_add_arg_type ("TaskManager::last_time", 
				 GTK_TYPE_LONG, 
				 GTK_ARG_READWRITE, 
				 ARG_LAST_TIME);

	gtk_object_add_arg_type ("TaskManager::state", 
				 GTK_TYPE_INT, 
				 GTK_ARG_READWRITE, 
				 ARG_STATE);

	object_class->destroy = task_manager_destroy;
	object_class->set_arg = task_manager_set_arg;
	object_class->get_arg = task_manager_get_arg;
}

static void
remove_dependencies (TaskManager *manager, TaskModel *task_model)
{
	GSList        *list, *l;
	GM_Dependency *dep;
	TaskModel     *dep_task;
	
	TRACE ();

	/* First, go through the predecessors of the task and remove
	 * this task from the predecessors's list of successors.
	 */
	list = task_model_get_predecessors (task_model);
	for (l = list; l; l = l->next) {
		dep = l->data;

		dep_task = task_manager_get_task_model (manager, dep->predecessorId);
		
		task_model_remove_successor (dep_task, dep->taskId);
	}

	g_slist_free (list);
	
	/* Then, go through the successors of the task and remove
	 * this task from the successors's list of predecessors.
	 */
	list = task_model_get_successors (task_model);
	for (l = list; l; l = l->next) {
		dep = l->data;

		dep_task = task_manager_get_task_model (manager, dep->taskId);

		task_model_remove_predecessor (dep_task, dep->predecessorId);
	}

	g_slist_free (list);
}

static gboolean
remove_subtree_traverse_func (TaskManager *manager,
			      TaskModel   *task_model,
			      gpointer     data)
{
	TaskManagerClass *klass;
	GM_Task          *task, *queued_task;
	GList            *l;
	
	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	task = task_model_get_task (task_model);

	/* Make sure we don't have this task in the bubble
	 * idle.
	 */
	for (l = manager->priv->bubble_queue; l; l = l->next) {
		queued_task = task_model_get_task (l->data);
		if (task->taskId == queued_task->taskId) {
			manager->priv->bubble_queue = g_list_remove_link (
				manager->priv->bubble_queue, l);
			g_list_free (l);
			d(g_print ("-----removed destroyed task from queue\n"));
			break;
		}
	}
	
	klass->remove_task_model (manager, task_model);

	gtk_object_destroy (GTK_OBJECT (task_model));
	return FALSE;
}

/* Prepare to destroy the task. */
static gboolean
pre_destroy_traverse_func (TaskManager *manager,
			   TaskModel   *task_model,
			   gpointer     data)
{
	gtk_signal_disconnect_by_func (GTK_OBJECT (task_model),
				       task_destroyed,
				       manager);

	remove_dependencies (manager, task_model);
	
	return FALSE;
}

static void
task_destroyed (TaskModel   *task, 
		TaskManager *manager)
{
	TaskModel *sibling_task;

	/* Get the first sibling that is not the node itself, to
	 * re-bubble up the start and end times for the group.
	 */
	sibling_task = task_manager_get_first_sibling (manager, task);
	if (sibling_task == task) {
		sibling_task = task_manager_get_next_sibling (manager, task);
		/* FIXME: is this needed or will we always return NULL if
		 * there is no more siblings?
		 */
		if (sibling_task == task) {
			sibling_task = NULL;
		}
	}

	/* Disconnect the destroy signal from the whole subtree,
	 * and update predecessors/successors (tell them that we
	 * went away).
	 */
	task_manager_traverse (manager, 
			       task,
			       G_POST_ORDER, 
			       pre_destroy_traverse_func,
			       NULL);
	
	/* Remove the node and its children. */
	task_manager_traverse (manager, 
			       task,
			       G_POST_ORDER, 
			       remove_subtree_traverse_func,
			       NULL);

	/* Re-bubble. */
	if (sibling_task) {
		bubble_times (manager, sibling_task);
	}
}

TaskModel *
task_manager_get_parent_task_model (TaskManager   *manager, 
				    TaskModel     *task)
{
	TaskManagerClass     *klass;

	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);
	return klass->get_parent_task_model (manager, task);
}

#if 0
gint
task_manager_get_position (TaskManager   *manager, 
			   TaskModel     *task)
{
	GNode   *parent_node, *node;

	g_return_val_if_fail (manager != NULL, 0);
	g_return_val_if_fail (IS_TASK_MANAGER (manager), 0);
	g_return_val_if_fail (task != NULL, 0);
	g_return_val_if_fail (IS_TASK_MODEL (task), 0);

	node = task_model_get_node (task);
	g_assert (node != NULL);

	parent_node = node->parent;
	g_assert (parent_node != NULL);

	return g_node_child_position (parent_node, node);
}
#endif

#if 0
gint
task_manager_get_n_children (TaskManager   *manager, 
			     TaskModel     *task)
{
	GNode   *node;

	g_return_val_if_fail (manager != NULL, 0);
	g_return_val_if_fail (IS_TASK_MANAGER (manager), 0);
	g_return_val_if_fail (task != NULL, 0);
	g_return_val_if_fail (IS_TASK_MODEL (task), 0);

	node = task_model_get_node (task);
	g_assert (node != NULL);
	
	return g_node_n_children (node);
}
#endif

static gboolean
bubble_idle_cb (gpointer user_data)
{
	TaskManager    *manager;
	TaskModel      *task_model;
	TaskModel      *child_task_model, *parent_task_model, *temp_task_model;
	time_t          start, end;
	time_t          parent_start, parent_end;
	time_t          next_start, next_end;
	gboolean        start_changed, end_changed;
	TaskChangeMask  mask;
	GList          *l;
	d(gint            i = 1);
	
	manager = user_data;
	
	if (manager->priv->block_bubble) {
 		g_print ("*** BLOCK\n");
		manager->priv->bubble_idle_id = 0;
 		return FALSE;
 	}

	manager->priv->block_bubble = TRUE;

	for (l = manager->priv->bubble_queue; l; l = l->next) {
		task_model = l->data;

		{
			d(GM_Task *task = task_model_get_task (task_model));
			d(g_print ("* %d: bubbling task %d.\n", i++, task->taskId));
		}

		
		parent_task_model = task_manager_get_parent_task_model (manager, task_model);
		
		child_task_model = task_model;
		while (parent_task_model) {
			GM_Task *parent_task;
			
			start_changed = end_changed = FALSE;
			
			parent_task = task_model_get_task (parent_task_model);
			parent_start = parent_task->start;
			parent_end = parent_task->end;
			if (parent_start == -1 || parent_end == -1) {
				goto skip;
			}

			/*
			 * Find the sibling with the earliest start time and the one
			 * with the last end time and assign those times to the
			 * parent.
			 */
			child_task_model = task_manager_get_first_sibling (manager, child_task_model);
		
			next_start = -1;
			next_end = -1;

			while (child_task_model) {
				GM_Task *child_task;

				child_task = task_model_get_task (child_task_model);
			
				start = child_task->start;
				end = child_task->end;

				if (start == -1 || end == -1) {
					goto skip_child;
				}

				if (next_start == -1)
					next_start = start;
				else
					next_start = MIN (start, next_start);

				if (next_end == -1)
					next_end = end;
				else
					next_end = MAX (end, next_end);

			skip_child:
				child_task_model = task_manager_get_next_sibling (manager, child_task_model);
			} /* for each sibling */

			mask = TASK_CHANGE_NONE;
			if (parent_start != next_start) {
				mask |= TASK_CHANGE_START;
			}

			if (parent_end != next_end) {
				mask |= TASK_CHANGE_END;
			}

			if (mask & (TASK_CHANGE_START | TASK_CHANGE_END)) {
				if (task_check_constraints (manager, parent_task_model, next_start, next_end, NULL, NULL)) {
					parent_task->start = next_start;
					parent_task->end = next_end;
					task_model_changed (parent_task_model, mask);
				}
			}

		skip:
			temp_task_model = parent_task_model;
			parent_task_model = task_manager_get_parent_task_model (manager, parent_task_model);
			child_task_model = temp_task_model;

			if (parent_task_model == NULL || parent_task_model == manager->priv->root_task_model) {
				break;
			} 
		} /* for each parent */

	} /* for each task in the queue */ 

	g_list_free (manager->priv->bubble_queue);
	manager->priv->bubble_queue = NULL;
	manager->priv->block_bubble = FALSE;
	manager->priv->bubble_idle_id = 0;

	return FALSE;
}


/*
 * Enqueue a bubbling of times, which propagates the earliest
 * start time and the latest end time through the task tree,
 * from the task (and its siblings) to its ancestors.
 */
static void
bubble_times (TaskManager *manager, TaskModel *task_model)
{
	TaskManagerPriv *priv;
	GList           *l;
	TaskModel       *queued_task_model;

	priv = manager->priv;

	/* See if the task is already in the queue, or
	 * if it has children (or grandchildren etc...)
	 * in the queue.
	 */
	for (l = priv->bubble_queue; l; l = l->next) {
		queued_task_model = l->data;

		if (queued_task_model == task_model) {
			return;
		}

		if (task_model_has_ancestor (queued_task_model, task_model)) {
			return;
		}
	}

	/* Remove all parents to this task, since they
	 * will be bubbled anyway when this one is.
	 */
/*	l = g_list_copy (priv->bubble_queue);
	for (; l; l = l->next) {
		queued_task_model = l->data;

		if (task_model_has_ancestor (task_model, queued_task_model)) {
			priv->bubble_queue = g_list_remove_link (
				priv->bubble_queue, l);
			g_list_free (l);
		}
	}
*/
	{
		d(GM_Task *task = task_model_get_task (task_model));
		d(g_print ("adding task %d to queue.\n", task->taskId));
	}
	
	priv->bubble_queue = g_list_prepend (priv->bubble_queue, task_model);
	
	if (manager->priv->bubble_idle_id) {
		return;
	}

	manager->priv->bubble_idle_id = gtk_idle_add (bubble_idle_cb, manager);
}

BONOBO_X_TYPE_FUNC_FULL (TaskManager, 
			 GNOME_MrProject_TaskManager,
			 PARENT_TYPE,
			 task_manager);


/* Test. */

static gboolean
task_check_constraints (TaskManager *manager,
			TaskModel   *task_model,
			GM_Time      start,
			GM_Time      end,
			GM_Time     *c_start,
			GM_Time     *c_end)
{
	GM_Dependency *dependency;
	GM_Task       *task, *predecessor_task;
	GSList        *list;
	TaskModel     *predecessor_task_model;
	TaskModel     *parent_task_model;
	gboolean       retval = TRUE;
	
	g_return_val_if_fail (manager != NULL, FALSE);
	g_return_val_if_fail (IS_TASK_MANAGER (manager), FALSE);
	g_return_val_if_fail (task_model != NULL, FALSE);
	g_return_val_if_fail (IS_TASK_MODEL (task_model), FALSE);

	task = task_model_get_task (task_model);

	parent_task_model = task_manager_get_parent_task_model (manager,
								task_model);
#if 0
	if (parent_task_model) {
		if (!task_check_constraints (manager,
					     parent_task_model,
					     start,
					     end,
					     c_start,
					     c_end)) {
			
			if (c_start)
				start = *c_start;
			if (c_end)
				end = *c_end;
			retval = FALSE;
		}
	}
#endif	
	/* Check if the task's predecessors allow the change. */
	
	list = task_model_get_predecessors (task_model);
	for (; list; list = list->next) {
		dependency = list->data;
		
		predecessor_task_model = task_manager_get_task_model (manager, dependency->predecessorId);
		g_assert (predecessor_task_model);

		predecessor_task = task_model_get_task (predecessor_task_model);
		
		if (start != -1) {
			/* Hardcoded to FS for now. */
			if (predecessor_task->end < start) {
				continue;
			} else {
				if (c_start && c_end) {
					gint old_duration, duration;
					
					/* FIXME: Quick hack for M2. :/ */
					
					if (start < predecessor_task->end) {
						old_duration = task->end - task->start;
						*c_start = predecessor_task->end;
						*c_end = *c_start + old_duration;
					} else {
						duration = end - start;
						*c_start = predecessor_task->end;
						*c_end = *c_start + duration;
					}
				}
				return FALSE;
			}
		}
	}

	return retval;
}

void
task_manager_traverse (TaskManager               *manager, 
		       TaskModel                 *task_model,
		       GTraverseType		  type,
		       TaskManagerTraverseFunc    func, 
		       gpointer                   data)
{
	TaskManagerClass     *klass;

	g_return_if_fail (task_model != NULL);
	g_return_if_fail (IS_TASK_MODEL (task_model));

	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	klass->traverse (manager, task_model, type, func, data);
}

GM_Task *
task_manager_create_root_task (TaskManager *manager)
{
	GM_Task *root_task;

	g_return_val_if_fail (manager != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER (manager), NULL);

	root_task           = GNOME_MrProject_Task__alloc ();
	root_task->name     = CORBA_string_dup ("Root");
	root_task->taskId   = 0;
	root_task->parentId = -1;
	root_task->start    = G_MININT; 
	root_task->end      = G_MAXINT;
	
	return root_task;
}


TaskModel *
task_manager_get_task_model (TaskManager *manager, 
			     GM_Id        id)
{
	TaskManagerClass *klass;

	g_return_val_if_fail (manager != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER (manager), NULL);
	g_return_val_if_fail (id >= 0, NULL);

	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);
	return klass->get_task_model (manager, id);
}

TaskModel *
task_manager_get_first_sibling (TaskManager *manager, 
				TaskModel   *task_model)
{
	TaskManagerClass *klass;

	g_return_val_if_fail (manager != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER (manager), NULL);
	g_return_val_if_fail (task_model != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MODEL (task_model), NULL);

	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);
	return klass->get_first_sibling (manager, task_model);
}

TaskModel *
task_manager_get_next_sibling (TaskManager *manager, 
			       TaskModel   *task_model)
{
	TaskManagerClass *klass;

	g_return_val_if_fail (manager != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER (manager), NULL);
	g_return_val_if_fail (task_model != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MODEL (task_model), NULL);

	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);
	return klass->get_next_sibling (manager, task_model);
}

TaskModel *
task_manager_get_root_task_model (TaskManager *manager)
{
	g_return_val_if_fail (manager != NULL, NULL);
	g_return_val_if_fail (IS_TASK_MANAGER (manager), NULL);

	return manager->priv->root_task_model;
}


/* Return successor task models instead of dep structs. */
static GSList *
dfs_task_model_get_successors (TaskManager *manager, TaskModel *task)
{
	GSList        *successors;
	GSList        *list, *l;
	GM_Dependency *dep;
	TaskModel     *dep_task;
	GM_Id          id;

	successors = task_model_get_successors (task);

	id = task_model_get_task (task)->taskId;
	
	list = NULL;
	
	for (l = successors; l; l = l->next) {
		dep = l->data;

		dep_task = task_manager_get_task_model (manager, dep->taskId);

		list = g_slist_prepend (list, dep_task);
	}
	
	return list;
}

static GraphNode *
graph_node_new (TaskModel *task)
{
	GraphNode *node;

	node = g_new0 (GraphNode, 1);
	node->g_node = g_node_new (node);
	node->task = task;

	return node;
}

static GSList *
task_manager_get_children (TaskManager *manager, TaskModel *model)
{
	TaskManagerClass *klass;
	GSList           *children;

	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);
	
	children = klass->get_children (manager, model);

	return children;
}

static gint
task_manager_get_task_depth (TaskManager *manager, TaskModel *model)
{
	TaskManagerClass *klass;

	klass = TASK_MANAGER_CLASS (GTK_OBJECT (manager)->klass);

	/* The root is at depth 1. */
	if (model == NULL) {
		return 1;
	}
	
	return klass->get_task_depth (manager, model);
}

static TaskModel *
get_ancestor_at_depth (TaskManager *manager, TaskModel *task, gint depth)
{
	gint my_depth, i;

	my_depth = task_manager_get_task_depth (manager, task);

	if (my_depth < depth) {
		return NULL;
	}
	else if (my_depth == depth) {
		return task;
	}
	
	for (i = my_depth; i > depth; i--) {
		task = task_manager_get_parent_task_model (manager, task);
	}

	return task;
}

static void
dfs_insert_tasks (GraphNode *parent, GSList *task_models, Graph *graph);

static void
dfs_insert_tasks (GraphNode *parent, GSList *task_models, Graph *graph)
{
	GSList    *l;
	GSList    *successors, *children;
	TaskModel *task;
	GM_Id      id;
	GraphNode *node;

	for (l = task_models; l; l = l->next) {
		task = l->data;

		/* Sanity check: if we have somehow stored a NULL task here,
		 * just exit because something is majorly wrong.
		 */
		if (!task) {
			g_warning ("dfs_insert_tasks got NULL task.");
			return;
		}
		
		id = task_model_get_task (task)->taskId;
		node = id_map_lookup (graph->node_map, id);
		if (!node) {
			/*g_print ("Can't find %d\n", id);*/
			node = graph_node_new (task);
			id_map_insert_id (graph->node_map, id, node);
			g_node_append (parent->g_node, node->g_node);
		}

		/* Add children as successors so that we get parents before children. */
		children = task_manager_get_children (graph->manager, task);
		dfs_insert_tasks (node, children, graph);
		g_slist_free (children);

		/* Also add the ancestor of the task that is at the same depth
		 * as the parent. This way we make sure that predecessors are listed
		 * before the containing tasks of the successors.
		 */
		{
			gint parent_depth;
			TaskModel *ancestor;
			
			parent_depth = g_node_depth (parent->g_node);
			parent_depth = task_manager_get_task_depth (graph->manager, parent->task);
			/*g_print ("Parent depth: %d\n", parent_depth);*/
			
			ancestor = get_ancestor_at_depth (graph->manager, task, parent_depth);
			if (ancestor && ancestor != parent->task && ancestor != task) {
				GSList *foo = NULL;

				/*g_print ("Foo: parent id %d, task id %d, ancestor id %d\n",
					 parent && parent->task ? task_model_get_task (parent->task)->taskId : -1,
					 task ? task_model_get_task (task)->taskId : -1,
					 ancestor ? task_model_get_task (ancestor)->taskId : -1);*/
				
				foo = g_slist_prepend (foo, ancestor);
				dfs_insert_tasks (parent, foo, graph);
				g_slist_free (foo);
			}
		}
		
		/* Successors. */
		successors = dfs_task_model_get_successors (graph->manager, task);
		dfs_insert_tasks (node, successors, graph);
		g_slist_free (successors);
	}
}


/* If we just were sane enough to keep this structure
 * in a tree from the beginning, this wouldn't be needed.
 */
static Graph *
dfs_build_graph (TaskManager *manager)
{
	GSList *children;
	Graph  *graph;

	d(puts (__FUNCTION__));
	
	graph = g_new0 (Graph, 1);
	
	children = task_manager_get_children (
		manager, task_manager_get_root_task_model (manager));

	graph->root = graph_node_new (NULL);
	graph->node_map = id_map_new (0);
	graph->manager = manager;

	dfs_insert_tasks (graph->root, children, graph);

	g_slist_free (children);
	
	return graph;
}

static void
graph_traverse (GraphNode *node, gpointer user_data)
{
	GNode   *g_node, *g_next_node;
	GSList **list;

	if (node->visited) {
		return;
	}

	node->visited = TRUE;

	g_node = node->g_node;

	g_next_node = g_node_first_child (g_node);
	while (g_next_node) {
		graph_traverse (g_next_node->data, user_data);

		g_next_node = g_node_next_sibling (g_next_node);
	}

	if (node->task != NULL) {
		GM_Id id;

		id = task_model_get_task (node->task)->taskId;
		if (id != 0) {
			/*g_print ("Adding: %d\n", id);*/
		
			list = user_data;
			*list = g_slist_prepend (*list, GINT_TO_POINTER (id));
		}
	}
}

static void
free_graph_node (gpointer key, gpointer value, gpointer user_data)
{
	g_free (value);
}

static GSList *
dfs_sort (Graph *graph)
{
	GSList *list = NULL;
	
	graph_traverse (graph->root, &list);
	
	id_map_foreach (graph->node_map, free_graph_node, NULL);

	gtk_object_unref (GTK_OBJECT (graph->node_map));	
	g_node_destroy (graph->root->g_node);
	g_free (graph);

	return list;
}

static void
tm_set_task_type (TaskManager *manager,
		  TaskModel   *task_model,
		  GM_TaskType  type)
{
	GM_Task        *task, updated_task;
	TaskChangeMask  mask;

	task = task_model_get_task (task_model);

	updated_task.type = type;
	updated_task.taskId = task->taskId;

	mask = corba_util_task_update (task, &updated_task, TASK_CHANGE_TYPE);
	if (mask == TASK_CHANGE_TYPE) {
		task_model_changed (task_model, TASK_CHANGE_TYPE);
	}
}
