/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2001      Mikael Hallendal <micke@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libgnomevfs/gnome-vfs.h>
#include <libgnome/gnome-defs.h>
#include <bonobo/bonobo-main.h>
#include <bonobo/bonobo-context.h>
#include <bonobo/bonobo-generic-factory.h>
#include <liboaf/liboaf.h>
#include <orb/orbit.h>

#include "util/eel-debug.h"
#include "project.h"
#include "engine.h"
#include "backends/file-project.h"

#define FACTORY_OAFIID "OAFIID:GNOME_MrProject_ProjectFactory"
#define ENGINE_OAFIID "OAFIID:GNOME_MrProject_Engine"
#define PROJECT_OAFIID "OAFIID:GNOME_MrProject_Project"

static Engine *engine = NULL;

static gboolean
register_engine (Engine *engine)
{
	OAF_RegistrationResult result;

	result = oaf_active_server_register (ENGINE_OAFIID, BONOBO_OBJREF (engine));

	switch (result) {
	case OAF_REG_SUCCESS:
		g_message ("Yay! Registered engine");
		return TRUE;

	case OAF_REG_NOT_LISTED:
		g_message ("register_engine (): Cannot register the engine: "
			   "not listed");
		break;

	case OAF_REG_ALREADY_ACTIVE:
		g_message ("register_engine (): Cannot register the engine: "
			   "already active");
		break;

	case OAF_REG_ERROR:
	default:
		g_message ("register_engine (): Cannot register the engine: "
			   "generic error");
		break;
	}

	return FALSE;
}

static void
engine_destroyed (Engine *engine, gpointer data)
{
	g_message ("Engine destroyed!");
	gtk_main_quit ();
}

static BonoboObject *
project_factory_fn (BonoboGenericFactory *factory,
		    const char           *object_id,
		    gpointer              user_data)
{
	Project *project;
	
	g_return_val_if_fail (object_id != NULL, NULL);

	/* Only support file backend for now... */

	if (!strcmp (object_id, PROJECT_OAFIID)) {
		if (!engine) {
			engine = engine_new ();
			register_engine (engine);

			gtk_signal_connect (GTK_OBJECT (engine),
					    "destroy",
					    engine_destroyed,
					    NULL);
		}

		project = engine_create_project (engine);

		return BONOBO_OBJECT (project);	
	}
	else if (!strcmp (object_id, ENGINE_OAFIID)) {
		engine = engine_new ();
		register_engine (engine);

		gtk_signal_connect (GTK_OBJECT (engine),
				    "destroy",
				    engine_destroyed,
				    NULL);

		return BONOBO_OBJECT (engine);	
	}
	
	g_warning ("Failing to manufacture a '%s'", object_id);
	return NULL;
}

int
main (int argc, char *argv [])
{
	CORBA_ORB             orb;
	BonoboGenericFactory *factory;
	
	bindtextdomain (PACKAGE, GNOMELOCALEDIR);
	textdomain (PACKAGE);

	gnome_init_with_popt_table ("mrproject-engine",
				    VERSION,
				    argc, argv,
				    oaf_popt_options,
				    0,
				    NULL);
      
	orb = oaf_init (argc, argv);
	gnome_vfs_init ();

	if (!bonobo_init (orb, NULL, NULL)) {
		g_error ("The project engine could not initialize Bonobo.");
	}

	if (g_getenv ("MRPROJECT_DEBUG") != NULL) {
		eel_make_warnings_and_criticals_stop_in_debugger
			(G_LOG_DOMAIN, g_log_domain_glib,
			 "MrProject",
			 "Util",
			 "Backend",
			 NULL);
	}

	factory = bonobo_generic_factory_new_multi (FACTORY_OAFIID,
						    project_factory_fn,
						    NULL);
	bonobo_running_context_auto_exit_unref (BONOBO_OBJECT (factory));

	g_message ("Engine started.");

	bonobo_main ();     

	g_message ("Engine shutdown.");
	
	return 0;
}

