/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <bonobo.h>
#include <bonobo/bonobo-listener.h>
#include <liboaf/liboaf.h>
#include "util/type-utils.h"
#include "shell-component.h"

static void  shell_component_init       (ShellComponent        *component);
static void  shell_component_class_init (ShellComponentClass   *klass);
static void  add_shell_listener         (ShellComponent        *component,
					 GNOME_MrProject_Shell  shell,
					 const gchar           *mask);

static void  remove_shell_listener      (ShellComponent        *component);

#define d(x)

#define PARENT_TYPE BONOBO_X_OBJECT_TYPE
static GtkObjectClass *parent_class;

enum {
	ADD_TO_SHELL,
	REMOVE_FROM_SHELL,
	SET_DEBUG_FLAGS,
	SHELL_EVENT,
	LAST_SIGNAL
};

static gint signals[LAST_SIGNAL] = { 0 };

struct _ShellComponentPriv {
	BonoboListener                *shell_listener;
	Bonobo_EventSource_ListenerId  shell_listener_id;
	gchar                         *mask;
};

static CORBA_boolean
impl_addToShell (PortableServer_Servant  servant, 
		 GNOME_MrProject_Shell   shell, 
		 CORBA_Environment      *ev)
{
	ShellComponent *component;
	gboolean        retval;
	
	d(puts (__FUNCTION__));
		
	component = SHELL_COMPONENT (bonobo_x_object (servant));

	if (component->shell != CORBA_OBJECT_NIL) {
		g_warning ("Can't set shell, already got one.");
		return CORBA_FALSE;
	}

	component->shell = CORBA_Object_duplicate (shell, ev);

	if (component->priv->mask) {
		add_shell_listener (component,
				    component->shell,
				    component->priv->mask);
	}

	retval = FALSE;
	gtk_signal_emit (GTK_OBJECT (component),
			 signals[ADD_TO_SHELL],
			 shell,
			 &retval);

	return retval;
}

static void
impl_removeFromShell (PortableServer_Servant servant, 
		      CORBA_Environment      *ev)
{
	ShellComponent     *component;

	d(puts (__FUNCTION__));
	
	component = SHELL_COMPONENT (bonobo_x_object (servant));

	gtk_signal_emit (GTK_OBJECT (component), signals[REMOVE_FROM_SHELL], NULL);
	
	if (component->shell == CORBA_OBJECT_NIL) {
		/* FIXME: raise exception. */
		return;
	}

	remove_shell_listener (component);

	CORBA_Object_release (component->shell, ev);
	component->shell = CORBA_OBJECT_NIL;
}

static void
impl_setDebugFlags (PortableServer_Servant  servant,
		    const CORBA_long        flags,
		    CORBA_Environment      *ev)
{
	ShellComponent *component;
	
	d(puts (__FUNCTION__));
	
	component = SHELL_COMPONENT (bonobo_x_object (servant));

	gtk_signal_emit (GTK_OBJECT (component), signals[SET_DEBUG_FLAGS], flags);
}

static void
shell_listener_callback (BonoboListener    *listener,
			 char              *event_name, 
			 CORBA_any         *any,
			 CORBA_Environment *ev,
			 gpointer           user_data)
{
	ShellComponent     *component;
	
	component = SHELL_COMPONENT (user_data);

	gtk_signal_emit (GTK_OBJECT (component),
			 signals[SHELL_EVENT],
			 event_name,
			 any);
}

static void
add_shell_listener (ShellComponent        *component,
		    GNOME_MrProject_Shell  shell,
		    const gchar           *mask)
{
	ShellComponentPriv *priv;
	CORBA_Environment   ev;
	CORBA_Object        event_src;

	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_SHELL_COMPONENT (component));
	g_return_if_fail (mask != NULL);

	priv = component->priv;
	
	CORBA_exception_init (&ev);
	
	event_src = GNOME_MrProject_Shell_getProxyEventSource (shell, &ev);
	if (BONOBO_EX (&ev) || event_src == CORBA_OBJECT_NIL) {
		g_warning ("Could not find shell event source.");
		CORBA_exception_free (&ev);
		return;
	}
	
	priv->shell_listener = bonobo_listener_new (shell_listener_callback,
						    component);

	priv->shell_listener_id = Bonobo_EventSource_addListenerWithMask (
		event_src,
		BONOBO_OBJREF (priv->shell_listener),
		mask,
		&ev);
	
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not add listener: '%s'", bonobo_exception_get_text (&ev));
		bonobo_object_unref (BONOBO_OBJECT (priv->shell_listener));
		priv->shell_listener = NULL;
	}

	bonobo_object_release_unref (event_src, NULL);
	CORBA_exception_free (&ev);
}

static void
shell_component_destroy (GtkObject *object)
{
	ShellComponent      *component;
	CORBA_Environment    ev;

	d(puts (__FUNCTION__));

	component = SHELL_COMPONENT (object);

	CORBA_exception_init (&ev);

	remove_shell_listener (component);

	if (component->priv->shell_listener) {
		bonobo_object_unref (BONOBO_OBJECT (component->priv->shell_listener));
		component->priv->shell_listener = NULL;
	}

	if (component->shell) {
		CORBA_Object_release (component->shell, &ev);
		if (BONOBO_EX (&ev)) {
			CORBA_exception_free (&ev);
		}
		component->shell = CORBA_OBJECT_NIL;
	}

	CORBA_exception_free (&ev);

	g_free (component->priv->mask);
	component->priv->mask = NULL;
	
	if (component->priv) {
		g_free (component->priv);
		component->priv = NULL;
	}
	
	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
remove_shell_listener (ShellComponent *component)
{
	Bonobo_EventSource  es;
	CORBA_Environment   ev;
	ShellComponentPriv *priv;

	CORBA_exception_init (&ev);

	priv = component->priv;
	
	/* Disconnect the shell listener. */
	if (priv->shell_listener_id != 0) {
		es = Bonobo_Unknown_queryInterface (component->shell, 
						    "IDL:Bonobo/EventSource:1.0",
						    &ev);
		if (!BONOBO_EX (&ev) && es != CORBA_OBJECT_NIL) {
			Bonobo_EventSource_removeListener (es, priv->shell_listener_id, &ev);
			if (BONOBO_EX (&ev)) {
				CORBA_exception_free (&ev);
			}
			priv->shell_listener_id = 0;
			bonobo_object_release_unref (es, &ev);
			if (BONOBO_EX (&ev)) {
				CORBA_exception_free (&ev);
			}
		}
	}
}

static void
shell_component_class_init (ShellComponentClass *klass)
{
	POA_GNOME_MrProject_ShellComponent__epv *epv;
	GtkObjectClass                          *object_class;


	epv          = &klass->epv;
	object_class = (GtkObjectClass *) klass;
	parent_class = gtk_type_class (PARENT_TYPE);

	object_class->destroy = shell_component_destroy;

	epv->addToShell       = impl_addToShell;
	epv->removeFromShell  = impl_removeFromShell;
	epv->setDebugFlags    = impl_setDebugFlags;

	signals[ADD_TO_SHELL] = 
		gtk_signal_new ("add_to_shell",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ShellComponentClass,
						   add_to_shell),
				gtk_marshal_BOOL__POINTER,
				GTK_TYPE_BOOL,
				1,
				GTK_TYPE_POINTER);

	signals[REMOVE_FROM_SHELL] = 
		gtk_signal_new ("remove_from_shell",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ShellComponentClass,
						   remove_from_shell),
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE,
				0);

	signals[SET_DEBUG_FLAGS] =
		gtk_signal_new ("set_debug_flags",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ShellComponentClass,
						   set_debug_flags),
				gtk_marshal_NONE__INT,
				GTK_TYPE_NONE,
				1,
				GTK_TYPE_INT);

	signals[SHELL_EVENT] =
		gtk_signal_new ("shell_event",
				GTK_RUN_LAST,
				object_class->type,
				GTK_SIGNAL_OFFSET (ShellComponentClass,
						   shell_event),
				gtk_marshal_NONE__POINTER_POINTER,
				GTK_TYPE_NONE,
				2,
				GTK_TYPE_POINTER,
				GTK_TYPE_POINTER);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);	
}

static void 
shell_component_init (ShellComponent *component)
{
	component->shell = CORBA_OBJECT_NIL;
	component->priv = g_new0 (ShellComponentPriv, 1);
}

/* If mask is "", we listen to all events. */
ShellComponent *
shell_component_construct (ShellComponent        *component, 
			   const gchar           *mask)
{
	/* Keep the mask so we have it when we add the listener. */
	component->priv->mask = g_strdup (mask);

	return component;
}


BONOBO_X_TYPE_FUNC_FULL (ShellComponent, 
			 GNOME_MrProject_ShellComponent,
			 PARENT_TYPE,
			 shell_component);
