/*
 * Copyright (C) 2001  CodeFactory AB
 * Copyright (C) 2001  Richard Hult
 * Copyright (C  2001  Mikael Hallendal
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult <rhult@codefactory.se>
 *         Mikael Hallendal <micke@codefactory.se>
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <gtk/gtk.h>
#include <libgnomeui/gnome-canvas.h>
#include <libgnomeui/gnome-canvas-util.h>
#include <libgnomeui/gnome-canvas-line.h>
#include <libgnomeui/gnome-canvas-rect-ellipse.h>

#include "util/type-utils.h"
#include "network-item.h"

#define DEBUG 0
#include "util/debug.h"

enum {
	ARG_0,
	ARG_TASK
};

enum {
	MOVED,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

static void network_item_init	    (NetworkItem      *network_item);
static void network_item_class_init (NetworkItemClass *klass);

GNOME_CLASS_BOILERPLATE (NetworkItem, network_item,
			 GnomeCanvasGroup, gnome_canvas_group);


static void
ni_set_arg (GtkObject *object, GtkArg *arg, guint arg_id)
{
	GnomeCanvasItem  *item;
	NetworkItem      *ni;
	NetworkItemClass *ni_class;
	GM_Task          *task_copy;

	item     = GNOME_CANVAS_ITEM (object);
	ni       = NETWORK_ITEM (object);
	ni_class = NETWORK_ITEM_CLASS (object->klass);
	
	switch (arg_id){
	case ARG_TASK:
		task_copy = corba_util_task_duplicate (GTK_VALUE_POINTER (*arg)); 
		if (ni->task) {
			CORBA_free (ni->task);
		}
		ni->task = task_copy;
		ni_class->set_task (ni);
		break;		

	default:
		break;
	}
}

static void
ni_get_arg (GtkObject *o, GtkArg *arg, guint arg_id)
{
	GnomeCanvasItem *item;
	NetworkItem     *ni;

	item = GNOME_CANVAS_ITEM (o);
	ni = NETWORK_ITEM (o);

	switch (arg_id){
	case ARG_TASK:
		GTK_VALUE_POINTER (*arg) = ni->task;
		break;
		
	default:
		arg->type = GTK_TYPE_INVALID;
	}
}

static void
ni_destroy (GtkObject *object)
{
	NetworkItem *ni;

	d(puts (__FUNCTION__));
	
	ni = NETWORK_ITEM (object);
	
	CORBA_free (ni->task);
	ni->task = NULL;
	
	GNOME_CALL_PARENT_HANDLER (GTK_OBJECT_CLASS, destroy, (object));
}

static void
network_item_class_init (NetworkItemClass *klass)
{
	GtkObjectClass *object_class;

        klass->set_task    = NULL;
        klass->update_task = NULL;
	
	object_class = (GtkObjectClass*) klass;

	object_class->destroy = ni_destroy;
	object_class->get_arg = ni_get_arg;
	object_class->set_arg = ni_set_arg;

	gtk_object_add_arg_type ("NetworkItem::task", 
                                 GTK_TYPE_POINTER, 
                                 GTK_ARG_READWRITE, 
                                 ARG_TASK);

	signals[MOVED] = 
		gtk_signal_new ("moved",
				GTK_RUN_FIRST,
				object_class->type,
				0,
				gtk_marshal_NONE__NONE,
				GTK_TYPE_NONE, 0);
	
	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);	

}

static void
network_item_init (NetworkItem *item)
{
        item->task = NULL;
}

void
network_item_update_task (NetworkItem *item, GM_Task *task)
{
	NetworkItemClass *ni_class;

        g_return_if_fail (item != NULL);
        g_return_if_fail (IS_NETWORK_ITEM (item));
        g_return_if_fail (task != NULL);
        
        d(puts(__FUNCTION__));

	ni_class = NETWORK_ITEM_CLASS (GTK_OBJECT (item)->klass);

        corba_util_task_update (item->task, task, TASK_CHANGE_ALL);
        
        if (ni_class->update_task) {
                ni_class->update_task (item);
        }
}

static void
predecessor_destroyed (GtkObject *predecessor, NetworkItem *item)
{
	item->predecessors = g_slist_remove (item->predecessors, predecessor);
}

void
network_item_link (NetworkItem *item, NetworkItem *predecessor)
{
        g_return_if_fail (item != NULL);
        g_return_if_fail (IS_NETWORK_ITEM (item));
        g_return_if_fail (predecessor != NULL);
        g_return_if_fail (IS_NETWORK_ITEM (predecessor));

	gtk_signal_connect_while_alive (GTK_OBJECT (predecessor),
					"destroy",
					predecessor_destroyed,
					item, GTK_OBJECT (item));

	item->predecessors = g_slist_prepend (item->predecessors, predecessor);
}

void
network_item_unlink (NetworkItem *item, NetworkItem *predecessor)
{
        g_return_if_fail (item != NULL);
        g_return_if_fail (IS_NETWORK_ITEM (item));
        g_return_if_fail (predecessor != NULL);
        g_return_if_fail (IS_NETWORK_ITEM (predecessor));

	item->predecessors = g_slist_remove (item->predecessors, predecessor);
}

static gint
get_depth (NetworkItem *item)
{
	gint depth;

	g_return_val_if_fail (item != NULL, -1);
	g_return_val_if_fail (IS_NETWORK_ITEM (item), -1);
	
	depth = 0;
	while (item->parent_item) {
		item = item->parent_item;
		depth++;
	}

	return depth;
}

static void
get_ancestors_with_same_parent (NetworkItem  *item,
				NetworkItem  *predecessor,
				NetworkItem **item_ancestor,
				NetworkItem **predecessor_ancestor)
{
	gint item_depth, predecessor_depth;
	
	g_return_if_fail (item != NULL);
	g_return_if_fail (predecessor != NULL);

	item_depth = get_depth (item);
	predecessor_depth = get_depth (predecessor);

	/* Get the ancestors at a common depth. */

	while (item_depth > predecessor_depth) {
		item  = item->parent_item;
		item_depth--;
	}

	while (predecessor_depth > item_depth) {
		predecessor = predecessor->parent_item;
		predecessor_depth--;
	}

	/* Bubble upwards until we get the same parent. */
	while (predecessor->parent_item != item->parent_item) {
		predecessor = predecessor->parent_item;
		item = item->parent_item;
	}

	*item_ancestor = item;
	*predecessor_ancestor = predecessor;
}

void
network_item_sort_predecessors (NetworkItem *item)
{
	GSList      *l;
	NetworkItem *item_to_move, *item_to_move_after;
	NetworkItem *predecessor;
	NetworkItem *item_ancestor;
	NetworkItem *predecessor_ancestor;
	gdouble      move_to_x;
	gdouble      x1, y1, x2, y2;
	
        g_return_if_fail (item != NULL);
        g_return_if_fail (IS_NETWORK_ITEM (item));

	item_to_move = item;
	item_to_move_after = NULL;
	move_to_x = 0;
	for (l = item->predecessors; l; l = l->next) {
		predecessor = l->data;

		/* We need to get the item's ancestor and the predecessors
		 * ancestor that has the same parent, and move the item's
		 * ancestor to the right of the predecessor's ancestor.
		 */

		get_ancestors_with_same_parent (item, predecessor, &item_ancestor, &predecessor_ancestor);

		if (predecessor_ancestor && predecessor_ancestor->task == NULL) {
			g_warning ("Trying to align with root!");
			predecessor_ancestor = NULL;
			continue;
		}

		gnome_canvas_item_get_bounds (GNOME_CANVAS_ITEM (predecessor_ancestor),
					      &x1, &y1,
					      &x2, &y2);
		
		if (x2 > move_to_x) {
			move_to_x = x2;
			item_to_move = item_ancestor;
			item_to_move_after = predecessor_ancestor;
		}
	}

	if (item_to_move && item_to_move_after) {
			item_to_move->col = MAX (item_to_move->col, item_to_move_after->col + 1);
	}
}

void
network_item_layout (NetworkItem *item)
{
	NETWORK_ITEM_CLASS (GTK_OBJECT (item)->klass)->layout (item);
}

void
network_item_move (NetworkItem *item, gdouble x, gdouble y)
{
	GnomeCanvasItem *canvas_item;
	gdouble          old_x, old_y;

	g_return_if_fail (item != NULL);
	g_return_if_fail (IS_NETWORK_ITEM (item));

	canvas_item = GNOME_CANVAS_ITEM (item);
	
	/* We have to do it like this to get the canvas group to move
	 * its children as well.
	 */
	gtk_object_get (GTK_OBJECT (item), "x", &old_x, "y", &old_y, NULL);
	gnome_canvas_item_move (canvas_item, x - old_x, y - old_y);

	d(g_print ("move (%d) to %g %g\n", item->task->taskId, x, y));
	
	gtk_signal_emit (GTK_OBJECT (item), signals[MOVED], NULL);
}

