/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2000-2001 CodeFactory AB
 * Copyright (C) 2000-2001 Richard Hult <rhult@codefactory.se>
 * Copyright (C) 2000-2001 Mikael Hallendal <micke@hallendal.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Richard Hult
 */


/* TODO: check for weird things with memory allocation/destruction.
 * Might be a mix of corba/g* memory somewhere in this directory.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <bonobo/bonobo-xobject.h>
#include <bonobo/bonobo-shlib-factory.h>
#include <liboaf/liboaf.h>
#include <gal/e-table/e-table.h>
#include <gal/e-table/e-cell-text.h>
#include <gal/e-table/e-cell-tree.h>

#include <gal/e-table/e-tree.h>
#include <gal/e-table/e-tree-table-adapter.h>
#include <gal/e-table/e-tree-model.h>
#include <gal/e-table/e-tree-memory.h>
#include <gal/e-table/e-tree-memory-callbacks.h>
#include <gal/widgets/e-unicode.h>

#include "libmrproject/GNOME_MrProject.h"
#include "util/type-utils.h"
#include "util/corba-utils.h"
#include "util/time-utils.h"
#include "client/e-table-cells/m-cell-date.h"
#include "client/e-table-cells/m-cell-filter.h"
#include "client/widgets/mr-hpaned.h"
#include "client/widgets/mr-message-box.h"
#include "client/widgets/goto-popup.h"
#include "client/components/shell-component/shell-component.h"
#include "client/manager-clients/task-manager-client.h"
#include "client/manager-clients/resource-manager-client.h"
#include "client/manager-clients/allocation-manager-client.h"
#include "gantt-component.h"
#include "gantt-model.h"
#include "gantt-scale.h"
#include "gantt-chart.h"
#include "gantt-item.h"
#include "gantt-header-item.h"
#include "gantt-print.h"
#include "e-tree-extensions.h"


#define DEBUG 0
#include "util/debug.h"

#define OAFIID "OAFIID:GNOME_MrProject_GanttComponent"

enum {
	COL_ID,
	COL_NAME,
	COL_START,
	COL_END,
	COL_DURATION,
	
	COL_LAST,

	COL_TYPE,
	COL_SUMMARY
};


static void gantt_component_init        (GanttComponent          *component);
static void gantt_component_class_init  (GanttComponentClass     *klass);
static void task_inserted_cb            (GanttComponent          *gantt_comp,
					 GM_Task                 *task,
					 GM_Id                    sibling_id,
					 GM_TaskOrderType         type,
					 gpointer                 user_data);
static void task_updated_cb             (GanttComponent          *gantt_comp,
					 GM_Task                 *task,
					 gpointer                 user_data);
static void tasks_removed_cb            (GanttComponent          *gantt_comp,
					 GSList                  *tasks,
					 gpointer                 user_data);
static void task_reparented_cb          (GanttComponent          *gantt_comp,
					 GM_Id                    task_id,
					 GM_Id                    new_parent_id,
					 gpointer                 user_data);
static void task_repositioned_cb        (GanttComponent          *gantt_comp,
					 GM_Id                    task_id,
					 GM_Id                    sibling_id,
					 GM_TaskOrderType         type,
					 gpointer                 user_data);
static void tasks_linked_cb             (GanttComponent          *gantt_comp,
					 GM_Dependency           *dependency);
static void tasks_unlinked_cb           (GanttComponent          *gantt_comp,
					 GM_Dependency           *dependency);
static void tasks_clear_cb              (GanttComponent          *gantt_comp);
static void resource_updated_cb         (GanttComponent          *gantt_comp,
					 GM_Resource             *reosurce) G_GNUC_UNUSED;
static void allocation_added_cb         (GanttComponent          *gantt_comp,
					 GM_Allocation           *allocation,
					 gpointer                 user_data) G_GNUC_UNUSED;
static void allocations_removed_cb      (GanttComponent          *gantt_comp,
					 GSList                  *allocations,
					 gpointer                 user_data) G_GNUC_UNUSED;;
static gchar *filter_date		(gconstpointer	          val);
static gchar *filter_duration		(gconstpointer            duration);
static gint get_selected_id             (GanttComponent          *gantt_comp);
static gboolean get_two_selected_ids    (GanttComponent          *component, 
					 gint                    *id1,
					 gint                    *id2);
static GM_Task *get_selected_task       (GanttComponent          *gantt_comp);
static ETableExtras *create_tree_extras (void);
static gint get_button_height           (GtkWidget               *widget);
static void paned_realize               (GtkWidget               *widget,
					 GtkTable                *e_tree);
static gboolean gc_add_to_shell         (ShellComponent          *shell_comp,
					 GNOME_MrProject_Shell    shell);
static void gc_remove_from_shell        (ShellComponent          *shell_comp);
static void gc_set_debug_flags          (ShellComponent          *shell_comp,
					 glong                    flags);
static void gc_shell_event              (ShellComponent          *shell_comp,
					 const gchar             *event_name,
					 CORBA_any               *any);
static BonoboControl *gc_control_new    (GanttComponent          *gantt_comp);
static void e_tree_right_click          (ETree                   *et,
					 int                      row,
					 ETreePath                path,
					 int                      col,
					 GdkEvent               *event,
					 gpointer                 user_data);
static gboolean e_tree_button_press     (GtkWidget               *widget,
					 GdkEvent                *event,
					 gpointer                 user_data);
static void e_tree_double_click         (ETree                   *et,
					 int                      row,
					 ETreePath                path,
					 int                      col,
					 GdkEvent                *event,
					 gpointer                 user_data);
static void gantt_row_double_clicked    (GanttItem               *gantt,
					 GnomeCanvasItem         *row,
					 GdkEvent                *event,
					 GanttComponent          *component);
static void gantt_row_clicked           (GanttItem               *item,
					 GnomeCanvasItem         *row_item,
					 GdkEvent                *event,
					 GanttComponent          *component);
static void e_tree_selection_change     (ETree                   *e_tree,               
					 GanttComponent          *gantt_comp);
static void gantt_control_deactivate    (BonoboControl           *control);
static void set_sensitivity_and_status  (GanttComponent          *component);
static GM_Task *gc_create_task          (GanttComponent          *gantt_comp,
					 GM_Task                 *parent_task);
static void gc_insert_sub_task          (GanttComponent          *gantt_comp);
static void gc_insert_task_below        (GanttComponent          *gantt_comp);
static void gc_delete_task              (GanttComponent          *gantt_comp);
static void load_begin_cb               (GanttComponent          *gantt_comp,
					 gpointer                 user_data);
static void load_end_cb                 (GanttComponent          *gantt_comp,
					 gpointer                 user_data);



#define PARENT_TYPE SHELL_COMPONENT_TYPE
static GtkObjectClass *parent_class;


/* Private members. */
struct _GanttComponentPriv {
	BonoboControl	        *control;
	gint		         node_changed_id;
	gint		         node_data_changed_id;

	TaskManagerClient       *tm_client;
	ResourceManagerClient   *rm_client;
	AllocationManagerClient *am_client;
	
	/* Widgets. */
	GtkWidget	        *gantt_widget;
	GanttModel              *gantt_model;
	GtkWidget               *paned;
	GtkWidget               *e_tree;
	GtkWidget	        *e_tree_hscroll;
	GtkWidget	        *chart;
	gboolean                 goto_dialog_shown;
	guint                    set_sensitivity_idle;

	gboolean                 loading;
};

static void
cmd_indent (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent       *gantt_comp;
	GanttComponentPriv   *priv;
	gint                  parent_id;
	GNOME_MrProject_Task *task;
	CORBA_Environment    ev;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;
	
	task = get_selected_task (gantt_comp);
	if (!task) {
		return;
	}

	/*
	 * Find the row ABOVE (view-wise) the selected row.
	 */
	parent_id = gantt_model_get_task_above (priv->gantt_model,
						task->taskId);

	if (parent_id != task->parentId) {
		CORBA_exception_init (&ev);
		
		task_mc_reparent_task (priv->tm_client, 
				       task->taskId, parent_id, 
				       &ev);

		CORBA_exception_free (&ev);
	}
}

static void
cmd_unindent (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent       *gantt_comp;
	GanttComponentPriv   *priv;
	gint                  parent_id;
	GNOME_MrProject_Task *task, *parent_task;
	CORBA_Environment     ev;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;
	
	task = get_selected_task (gantt_comp);
	if (!task) {
		return;
	}

	parent_id = task->parentId;
	parent_task = gantt_model_get_task (priv->gantt_model, parent_id);
	if (parent_task) {
		parent_id = parent_task->parentId;
	} else {
		parent_id = 0;
	}

	if (parent_id != task->parentId) {
		CORBA_exception_init (&ev);
		task_mc_reparent_task (priv->tm_client, task->taskId,
				       parent_id, &ev);

		CORBA_exception_free (&ev);
	}
}

static void
foreach_remove (ETreePath path, gpointer data)
{
	GSList **list = data;
	
	*list = g_slist_prepend (*list, path);
}

static void
cmd_remove (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent        *gantt_comp;
	GanttComponentPriv    *priv;
	CORBA_Environment      ev;
	GSList		      *list, *l, *l2;
	
	d(puts (__FUNCTION__));
	
	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	list = NULL;
	e_tree_selected_path_foreach (E_TREE (priv->e_tree),
				      foreach_remove,
				      &list);

	if (!list) {
		return;
	}

	l2 = NULL;

	CORBA_exception_init (&ev);

	for (l = list; l; l = l->next) {
		GNOME_MrProject_Task *task;
	
		task = e_tree_memory_node_get_data (
			E_TREE_MEMORY (priv->gantt_model->etm),	l->data);

		if (!task) {
			/* This is needed (for now) since the ETree's root
			 * is selected even though it is not visible.
			 */
			continue;
		}

		l2 = g_slist_prepend (l2, GINT_TO_POINTER (task->taskId));
	}

	task_mc_remove_tasks (priv->tm_client, l2, &ev);

	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
		g_warning ("Could not remove tasks from project.");
		CORBA_exception_init (&ev);
	}

	g_slist_free (l2);
	
	CORBA_exception_free (&ev);
}

static void
cmd_insert (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent *gantt_comp;

	gantt_comp = GANTT_COMPONENT (data);

	gc_insert_task_below (gantt_comp);
}

static void
cmd_insert_sub_task (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent *gantt_comp;
	
	gantt_comp = GANTT_COMPONENT (data);

	gc_insert_sub_task (gantt_comp);
}

static void
cmd_select_all (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;
	ESelectionModel    *selection;
	
	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	selection = e_tree_get_selection_model (E_TREE (priv->e_tree));

	/* Note: select_all selects the invisible root as well. */
	e_selection_model_select_all (selection);
}

static void
cmd_invert_selection (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;
	ESelectionModel    *selection;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	selection = e_tree_get_selection_model (E_TREE (priv->e_tree));
	e_selection_model_invert_selection (selection);
}

static void
up_or_down (GanttComponent *gantt_comp, gboolean up)
{
	GanttComponentPriv *priv;
	CORBA_Environment   ev;
	GNOME_MrProject_Id  id, sibling_id;
	
	priv = gantt_comp->priv;
	
	id = get_selected_id (gantt_comp);
	if (id == -1) {
		return;
	}

	/* Get the task's new sibling. */
	if (up) {
		sibling_id = gantt_model_get_prev_sibling (
			gantt_comp->priv->gantt_model,
			id);
	} else {
		sibling_id = gantt_model_get_next_sibling (
			gantt_comp->priv->gantt_model,
			id);
	}		

	if (sibling_id != -1) {
		CORBA_exception_init (&ev);

		task_mc_reposition_task (priv->tm_client,
					 id,
					 sibling_id, 
					 up ?
					 GNOME_MrProject_TASK_BEFORE :
					 GNOME_MrProject_TASK_AFTER,
					 &ev);

		if (BONOBO_EX (&ev)) {
			g_warning ("Could not move task.");
			CORBA_exception_free (&ev);
			return;
		}

		CORBA_exception_free (&ev);
	}
}

static void
cmd_up (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent *gantt_comp;

	gantt_comp = GANTT_COMPONENT (data);
	up_or_down (gantt_comp, TRUE);
}

static void
cmd_down (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent *gantt_comp;

	gantt_comp = GANTT_COMPONENT (data);
	up_or_down (gantt_comp, FALSE);
}

static GNOME_MrProject_TaskDialog
activate_task_dialog (GanttComponent *gantt_comp, GM_Id id)
{
	GanttComponentPriv         *priv;
	CORBA_Environment           ev;
	Bonobo_Control              control_co;
	GtkWidget                  *widget;
	GNOME_MrProject_TaskDialog  task_dialog_co;

	if (id == -1) {
		return NULL;
	}

	priv = gantt_comp->priv;

	CORBA_exception_init (&ev);
	control_co = bonobo_get_object ("OAFIID:GNOME_MrProject_TaskDialog",
					"IDL:Bonobo/Control:1.0",
					&ev);

	if (BONOBO_EX (&ev) || !control_co) {
		g_warning ("Could not activate Task dialog.");
		CORBA_exception_free (&ev);
		return NULL;
	}
	
	widget = bonobo_widget_new_control_from_objref (control_co, NULL);
	gtk_widget_show (widget);
	
	task_dialog_co = Bonobo_Unknown_queryInterface (
		control_co, "IDL:GNOME/MrProject/TaskDialog:1.0", &ev);

	bonobo_object_release_unref (control_co, NULL);

	if (BONOBO_EX (&ev) || !task_dialog_co) {
		g_warning ("Could not get TaskDialog interface.");
		CORBA_exception_free (&ev);
		return NULL;
	}

	GNOME_MrProject_TaskDialog_setShell (task_dialog_co,
					     SHELL_COMPONENT (gantt_comp)->shell,
					     &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not set shell on TaskDialog dialog.");
		CORBA_exception_free (&ev);
		bonobo_object_release_unref (task_dialog_co, NULL);
		return NULL;
	}

	GNOME_MrProject_TaskDialog_show (task_dialog_co, &ev);

	if (BONOBO_EX (&ev)) {
		g_warning ("Could not show TaskDialog dialog.");
		CORBA_exception_free (&ev);
		bonobo_object_release_unref (task_dialog_co, NULL);
		return NULL;
	}

	GNOME_MrProject_TaskDialog_setTaskId (task_dialog_co, id, &ev);

	if (BONOBO_EX (&ev)) {
		g_warning ("Could not set Task id on TaskDialog.");
		CORBA_exception_free (&ev);
		bonobo_object_release_unref (task_dialog_co, NULL);
		return NULL;
	}

	return task_dialog_co;
}

static void
cmd_edit (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;
	GM_Id               id;
	CORBA_Environment   ev;
	GM_TaskDialog       task_dialog_co;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	id = get_selected_id (gantt_comp);
	
	task_dialog_co = activate_task_dialog (gantt_comp, id);
	if (!task_dialog_co) {
		return;
	}
	
	CORBA_exception_init (&ev);

	GNOME_MrProject_TaskDialog_showPage (task_dialog_co,
					     GNOME_MrProject_PAGE_GENERAL,
					     &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not set task on TaskDialog.");
		CORBA_exception_free (&ev);
		bonobo_object_release_unref (task_dialog_co, NULL);
	}

	CORBA_exception_free (&ev);
}

static void
cmd_assign (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;
	GM_Id               id;
	CORBA_Environment   ev;
	GM_TaskDialog       task_dialog_co;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	id = get_selected_id (gantt_comp);
	
	task_dialog_co = activate_task_dialog (gantt_comp, id);
	if (!task_dialog_co) {
		return;
	}
	
	CORBA_exception_init (&ev);

	GNOME_MrProject_TaskDialog_showPage (task_dialog_co,
					     GNOME_MrProject_PAGE_RESOURCES,
					     &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not set task on TaskDialog.");
		CORBA_exception_free (&ev);
		bonobo_object_release_unref (task_dialog_co, NULL);
	}

	CORBA_exception_free (&ev);
}

static void
cmd_print (BonoboUIComponent *component, gpointer data, const char *cname)
{
	/*EPrintable *ep;*/
	GanttChart *chart;
	
	/*ep = e_tree_get_printable (E_TREE (GANTT_COMPONENT (data)->priv->e_tree));*/
	chart = GANTT_CHART (GANTT_COMPONENT (data)->priv->chart);

	print_gantt (GANTT_COMPONENT (data)->priv->gantt_model,
		     GANTT_ITEM (chart->gantt_item),
		     GANTT_HEADER_ITEM (chart->header_item),
		     TRUE);
}

static void
cmd_link (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;
	gint                id1, id2;
	CORBA_Environment   ev;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	if (!get_two_selected_ids (gantt_comp, &id1, &id2)) {
		return;
	}

	CORBA_exception_init (&ev);
	
	task_mc_link_tasks (priv->tm_client, id2, id1, 
			    GNOME_MrProject_DEPENDENCY_FS,
			    &ev);

	CORBA_exception_free (&ev);
}

static void
cmd_unlink (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;
	gint                id1, id2;
	CORBA_Environment   ev;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	if (!get_two_selected_ids (gantt_comp, &id1, &id2)) {
		return;
	}

	CORBA_exception_init (&ev);

	task_mc_unlink_tasks (priv->tm_client, id2, id1, &ev);

	CORBA_exception_free (&ev);
}

static void
cmd_goto_task (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent       *gantt_comp;
	GanttComponentPriv   *priv;
	GNOME_MrProject_Task *task;
	time_t                minor_t;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	d(g_print ("goto task\n "));
	
	task = get_selected_task (gantt_comp);
	if (task) {
		GanttChart *chart = GANTT_CHART (priv->chart);
		
		minor_t = gantt_scale_get_minor_tick (chart->gantt_scale);
		gantt_chart_scroll_to_time (chart, task->start - minor_t);
	}
}

static void
cmd_goto_date (BonoboUIComponent *component, gpointer data, const char *cname)
{
	/* Obsoleted by cmd_goto? */
}

static void
cmd_goto (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;
	GtkWidget          *dialog, *calendar;
	gint                ret;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	dialog = gnome_dialog_new (_("Goto Date"),
				   GNOME_STOCK_BUTTON_CLOSE,
				   NULL);
	calendar = gtk_calendar_new ();
	gtk_box_pack_start (GTK_BOX (GNOME_DIALOG (dialog)->vbox),
			    calendar, FALSE, FALSE, 0);
	gtk_widget_show (calendar);
	
	ret = gnome_dialog_run (GNOME_DIALOG (dialog));
	if (ret == 0) {
		gint   year, month, day;
		time_t t;

		gtk_calendar_get_date (GTK_CALENDAR (calendar),
				       &year,
				       &month,
				       &day);
		t = time_from_day (year, month, day);
		gantt_chart_scroll_to_time (GANTT_CHART (priv->chart), t);
		gtk_widget_destroy (dialog);
	}
}

static void
cmd_timescale (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	gantt_chart_customize_scale (GANTT_CHART (priv->chart));
}

static void
cmd_zoom (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	gantt_chart_zoom (GANTT_CHART (priv->chart));
}

static void
cmd_zoom_entire (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	gantt_chart_zoom_entire (GANTT_CHART (priv->chart));
}

static void
cmd_debug (BonoboUIComponent *component, gpointer data, const char *cname)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;
	CORBA_Environment   ev;
	CORBA_Object        mgr;
	GM_IdSeq           *ids;
	GSList             *list, *l;

	gantt_comp = GANTT_COMPONENT (data);
	priv = gantt_comp->priv;

	CORBA_exception_init (&ev);
	/*GNOME_MrProject_TaskManager_dumpTree (priv->task_mgr_co, &ev);*/

	mgr = task_mc_get_manager (priv->tm_client);
		
	ids = GNOME_MrProject_TaskManager_getAllTaskIdsSorted (mgr, &ev);

	list = corba_util_id_seq_to_list (ids);
	CORBA_free (ids);

	g_print ("Sorted ids: ");
	for (l = list; l; l = l->next) {
		g_print ("%d ", GPOINTER_TO_INT (l->data));
	}
	g_print ("\n");
	
	g_slist_free (list);
	
	CORBA_exception_free (&ev);
}

static BonoboUIVerb verbs [] = {
	BONOBO_UI_VERB ("GanttIndent",          cmd_indent),
	BONOBO_UI_VERB ("GanttUnindent",        cmd_unindent),
	BONOBO_UI_VERB ("GanttEdit",            cmd_edit),
	BONOBO_UI_VERB ("GanttAssign",          cmd_assign),
	BONOBO_UI_VERB ("GanttRemove",          cmd_remove),
	BONOBO_UI_VERB ("GanttInsert",          cmd_insert),
	BONOBO_UI_VERB ("GanttInsertSubTask",   cmd_insert_sub_task),
	BONOBO_UI_VERB ("GanttSelectAll",       cmd_select_all),
	BONOBO_UI_VERB ("GanttInvertSelection", cmd_invert_selection),
	BONOBO_UI_VERB ("GanttUp",              cmd_up),
	BONOBO_UI_VERB ("GanttDown",            cmd_down),
	BONOBO_UI_VERB ("FilePrint",            cmd_print),
	BONOBO_UI_VERB ("GanttGoto",            cmd_goto),
	BONOBO_UI_VERB ("GanttGotoTask",        cmd_goto_task),
	BONOBO_UI_VERB ("GanttGotoDate",        cmd_goto_date),
	BONOBO_UI_VERB ("GanttLink",            cmd_link),
	BONOBO_UI_VERB ("GanttUnlink",          cmd_unlink),
	BONOBO_UI_VERB ("GanttTimescale",       cmd_timescale),
	BONOBO_UI_VERB ("GanttZoom",            cmd_zoom),
	BONOBO_UI_VERB ("GanttZoomEntire",      cmd_zoom_entire),
	BONOBO_UI_VERB ("GanttDebug",           cmd_debug),
	BONOBO_UI_VERB_END
};

static void
task_inserted_cb (GanttComponent   *gantt_comp,
		  GM_Task          *task,
		  GM_Id             sibling_id,
		  GM_TaskOrderType  type,
		  gpointer          user_data)
{
	GanttComponentPriv   *priv;
	GNOME_MrProject_Task *copy;
	ETreePath             path;

	priv = gantt_comp->priv;

	copy = corba_util_task_duplicate (task);
	gantt_model_insert_task (priv->gantt_model,
				 task->parentId,
				 sibling_id,
				 type,
				 copy);

	/* Select the inserted task (if we're not loading). */
	if (!priv->loading) {
		path = gantt_model_get_path (priv->gantt_model, task->taskId);
		if (task->taskId != 0 && path) {
			gtk_widget_grab_focus (priv->e_tree);
			e_tree_set_cursor (E_TREE (priv->e_tree), path);
		}
	}
}

static void 
task_updated_cb (GanttComponent *gantt_comp,
		 GM_Task        *task,
		 gpointer        user_data)
{
	gantt_model_update_task (gantt_comp->priv->gantt_model,
				 task->taskId,
				 task);
}

static void
tasks_removed_cb (GanttComponent *gantt_comp,
		  GSList         *tasks,
		  gpointer        user_data)
{
	GanttComponentPriv *priv;
	GNOME_MrProject_Id  id, next_id, prev_id;
	ETreePath           path;
	ESelectionModel    *selection;

	priv = gantt_comp->priv;

	/* FIXME: Remove this when it doesn't crash in
	 * the selection change callback...?
	 */
	selection = e_tree_get_selection_model (E_TREE (priv->e_tree));
	e_selection_model_clear (selection);

	if (g_slist_length (tasks) == 1) {
		id = GPOINTER_TO_INT (tasks->data);
		prev_id = gantt_model_get_prev_sibling (priv->gantt_model, id);
		next_id = gantt_model_get_next_sibling (priv->gantt_model, id);
	} else {
		prev_id = 0;
		next_id = 0;
	}
		
	gantt_model_remove_tasks (gantt_comp->priv->gantt_model, tasks);

	path = NULL;

	if (next_id > 0) {
		path = gantt_model_get_path (priv->gantt_model, next_id);
	}
	if (path == NULL && prev_id > 0) {
		path = gantt_model_get_path (priv->gantt_model, prev_id);
	}

	if (path) {
		e_tree_set_cursor (E_TREE (priv->e_tree), path);
	}
}

static void
task_reparented_cb (GanttComponent *gantt_comp,
		    GM_Id           task_id,
		    GM_Id           new_parent_id,
		    gpointer        user_data)
{
	ETreePath        path;
	ESelectionModel *selection;

	d(puts (__FUNCTION__));

	selection = e_tree_get_selection_model (E_TREE (gantt_comp->priv->e_tree));
	e_selection_model_clear (selection);

	/* FIXME: below/above? */
	
	gantt_model_reparent_task (gantt_comp->priv->gantt_model,
				   new_parent_id,
				   task_id,
				   -1);

	path = gantt_model_get_path (gantt_comp->priv->gantt_model,
				     task_id);
	if (path) {
		e_tree_set_cursor (E_TREE (gantt_comp->priv->e_tree), path);
	}
}

static void
task_repositioned_cb (GanttComponent   *gantt_comp,
		      GM_Id             task_id,
		      GM_Id             sibling_id,
		      GM_TaskOrderType  type,
		      gpointer          user_data)
{
	ETreePath        path;
	ESelectionModel *selection;

	selection = e_tree_get_selection_model (E_TREE (gantt_comp->priv->e_tree));
	e_selection_model_clear (selection);

	gantt_model_reposition_task (gantt_comp->priv->gantt_model,
				     task_id,
				     sibling_id,
				     type);

	path = gantt_model_get_path (gantt_comp->priv->gantt_model, task_id);
	if (path) {
		e_tree_set_cursor (E_TREE (gantt_comp->priv->e_tree), path);
	}
}

static void
tasks_linked_cb (GanttComponent *gantt_comp,
		 GM_Dependency  *dependency)
{
	gantt_model_link (gantt_comp->priv->gantt_model, dependency);
}

static void
tasks_unlinked_cb (GanttComponent *gantt_comp,
		   GM_Dependency  *dependency)
{
	gantt_model_unlink (gantt_comp->priv->gantt_model, dependency);
}

static void
tasks_clear_cb (GanttComponent *gantt_comp)
{
	ESelectionModel *selection;

	/* Must have this, because of a bug (?) in e_tree/selection. */
	selection = e_tree_get_selection_model (E_TREE (gantt_comp->priv->e_tree));
	e_selection_model_clear (selection);

	gantt_model_remove_all_tasks (gantt_comp->priv->gantt_model);
}

static void
resource_updated_cb (GanttComponent *gantt_comp, GM_Resource *resource)
{
	GanttComponentPriv *priv;
	CORBA_Environment   ev;
	GSList             *tasks, *node;
	GM_Allocation      *allocation;
	
	g_return_if_fail (gantt_comp != NULL);
	g_return_if_fail (IS_GANTT_COMPONENT (gantt_comp));

	priv = gantt_comp->priv;

	CORBA_exception_init (&ev);

	tasks = allocation_mc_get_allocations_by_resource (priv->am_client,
							   resource->resourceId,
							   &ev);

	for (node = tasks; node; node = node->next) {
		allocation = (GM_Allocation *) node->data;
		
		gantt_model_update_allocated_resource (priv->gantt_model,
						       allocation->taskId,
						       resource);
	}

	g_slist_CORBA_free (tasks);
	
	CORBA_exception_free (&ev);
}

static void
allocation_added_cb (GanttComponent *gantt_comp,
		     GM_Allocation  *allocation,
		     gpointer        user_data)
{
	GanttComponentPriv *priv;
	GM_Resource        *resource;
	CORBA_Environment   ev;
	GM_Id               resource_id, task_id;
	
	g_return_if_fail (gantt_comp != NULL);
	g_return_if_fail (IS_GANTT_COMPONENT (gantt_comp));
	g_return_if_fail (allocation != NULL);

	d(puts (__FUNCTION__));

	priv        = gantt_comp->priv;
	task_id     = allocation->taskId;
	resource_id = allocation->resourceId;
	
	CORBA_exception_init (&ev);

	resource = resource_mc_get_resource (priv->rm_client, resource_id, 
					     &ev);
	
	if (BONOBO_EX (&ev)) {
		CORBA_exception_free (&ev);
		return;
	}

 	gantt_model_assign_allocation (priv->gantt_model, resource, task_id); 
	
	CORBA_free (resource);
	
	CORBA_exception_free (&ev);
}

static void
allocations_removed_cb (GanttComponent *gantt_comp,
			GSList         *allocations,
			gpointer        user_data)
{
	GanttComponentPriv *priv;
	GM_Allocation      *allocation;
	GSList             *node;
	
	g_return_if_fail (gantt_comp != NULL);
	g_return_if_fail (IS_GANTT_COMPONENT (gantt_comp));
	g_return_if_fail (allocations != NULL);

	d(puts (__FUNCTION__));
	
	priv = gantt_comp->priv;

	for (node = allocations; node; node = node->next) {
			allocation  = (GM_Allocation *) node->data;
			
			gantt_model_unassign_allocation (priv->gantt_model, 
							 allocation->resourceId, 
							 allocation->taskId);
	}
}

static void
task_changed (GanttModel     *model,
	      GM_Task        *task,
	      GanttComponent *gantt_comp)
{
	CORBA_Environment ev;

	g_return_if_fail (model != NULL);
	g_return_if_fail (task != NULL);
	g_return_if_fail (gantt_comp != NULL);

	CORBA_exception_init (&ev);

	task_mc_update_task (gantt_comp->priv->tm_client,
			     task->taskId, task, &ev);

	CORBA_exception_free (&ev);
}

static void
task_moved (GanttModel     *model,
	    GM_Task        *task,
	    time_t          t,
	    GM_TaskEdge     edge,
	    GanttComponent *gantt_comp)
{
	CORBA_Environment ev;

	g_return_if_fail (model != NULL);
	g_return_if_fail (task != NULL);
	g_return_if_fail (gantt_comp != NULL);

	CORBA_exception_init (&ev);

	task_mc_move_task (gantt_comp->priv->tm_client,
			   task->taskId,
			   t,
			   edge,
			   &ev);

	CORBA_exception_free (&ev);
}

static void
task_duration_changed (GanttModel     *model,
		       GM_Task        *task,
		       gint            duration,
		       gint            edge,
		       GanttComponent *gantt_comp)
{
	CORBA_Environment ev;

	g_return_if_fail (model != NULL);
	g_return_if_fail (task != NULL);
	g_return_if_fail (gantt_comp != NULL);

	CORBA_exception_init (&ev);
	
	task_mc_set_task_duration (gantt_comp->priv->tm_client,
				   task->taskId,
				   (time_t) duration,
				   (GM_TaskEdge) edge,
				   &ev);

	CORBA_exception_free (&ev);
}

static void
load_begin_cb (GanttComponent *gantt_comp,
	       gpointer        user_data)
{
	GanttComponentPriv *priv;
	
	priv = gantt_comp->priv;
	priv->loading = TRUE;
}

static void
load_end_cb (GanttComponent *gantt_comp,
	     gpointer        user_data)
{
	GanttComponentPriv *priv;

	priv = gantt_comp->priv;
	priv->loading = FALSE;

	set_sensitivity_and_status (gantt_comp);
}

#define IS_SUMMARY(_t) (_t->type == GNOME_MrProject_TASK_SUMMARY)

/* Tree callbacks. */
static int
tree_col_count (ETreeModel *etree, void *data)
{
	return COL_LAST;
}

static void *
tree_duplicate_value (ETreeModel *etm, int col, const void *val, void *data)
{
	return (void*) val;
}

static void
tree_free_value (ETreeModel *etm, int col, void *val, void *data)
{
}

static void *
tree_initialize_value (ETreeModel *etm, int col, void *data)
{
	return NULL;
}

static gboolean
tree_value_is_empty (ETreeModel *etm, int col, const void *val, void *data)
{
	switch (col) {
	case COL_ID:
		return ((int)val) == -1;

	case COL_NAME:
		return val == NULL;

	case COL_START:
	case COL_END:
	case COL_DURATION:
		return ((int)val) == -1;

	default:
		return TRUE;
	}
}

static char *
tree_value_to_string (ETreeModel *etm, int col, const void *val, void *data)
{
	switch (col) {
	case COL_ID:
		return g_strdup_printf ("%d", (int)val);

	case COL_NAME:
		return (char *) val;

	case COL_START:
	case COL_END:
		return filter_date (val);

	case COL_DURATION:
		return filter_duration (val);
		
	default: 
		return NULL;
	}
}

static void *
tree_value_at (ETreeModel *etm, ETreePath path, int col, void *data)
{
	GNOME_MrProject_Task *task;

	task = e_tree_memory_node_get_data (E_TREE_MEMORY (etm), path);
	if (task == NULL) {
		g_warning ("No Task at this path.");
		return NULL;
	}
	
	switch (col) {
	case COL_ID:
		return (void *) ((int)task->taskId);
		
	case COL_NAME: 
		return task->name;

	case COL_START: 	
		return (void *) ((int)task->start);
		
	case COL_END: 
		return (void *) ((int)task->end);

	case COL_DURATION: 
		return (void *) ((int)(task->end - task->start));
		
	case COL_SUMMARY:
		return (void*) IS_SUMMARY (task);
		
	default: 
		return NULL;
	}
} 

static void
tree_set_value_at (ETreeModel *etm, ETreePath path, int col, const void *val, void *data)
{
	GM_Task        *task;
	GanttComponent *component;
	GanttModel     *model;
	gint		duration, mult;
	gchar          *end_ptr;

	task = e_tree_memory_node_get_data (E_TREE_MEMORY (etm), path);
	if (task== NULL) {
		g_warning ("No Task at this path.");
		return;
	}

	component = GANTT_COMPONENT (data);
	model = component->priv->gantt_model;
	
	switch (col) {
	case COL_NAME: 
		CORBA_free (task->name);
		task->name = CORBA_string_dup (val);
		gantt_model_task_changed (model, task->taskId);
		break;

	case COL_START:
		/* We can't explicitly set the start/end times for a non-leaf. */
		if (IS_SUMMARY (task)) {
			return;
		}
		
		duration = task->end - task->start;

		task->start = (time_t) val;
		task->end = task->start + duration;

		gantt_model_task_changed (model, task->taskId);
		break;

	case COL_END:
		/* We can't explicitly set the start/end times for a non-leaf. */
		if (IS_SUMMARY (task)) 
			return;

		task->end = (time_t) val;
		
		/* Make sure the task doesn't get invisible cause of too small length. */
		if (task->start >= task->end) {
			task->end = task->start + 60 * 60;
		} 

		gantt_model_task_changed (model, task->taskId);
		break;

	case COL_DURATION:
		/* We can't explicitly set the start/end times for a non-leaf. */
		if (IS_SUMMARY (task)) 
			return;

		end_ptr = NULL;
		duration = strtol (val, &end_ptr, 10);
		if (duration == 0) {
			return;
		}

		/* Make the default a setting. */
		mult = 60 * 60;
		
		end_ptr = g_strstrip (end_ptr);
		
		/* Note: these are matched against user input to decide the unit
		 * on the duration entered. Enter 5 variations that you find appropiate
		 * for your language. If you don't want to use all of them, enter the same
		 * for the ones that you don't need (e.g. "weeks", "weeks", "weeks").
		 */
		if (*end_ptr != '\0') {
			if (!strcmp (end_ptr, _("w")) ||
			    !strcmp (end_ptr, _("weeks")) ||
			    !strcmp (end_ptr, _("week")) ||
			    !strcmp (end_ptr, _("wks")) ||
			    !strcmp (end_ptr, _("wk"))) {
				mult = 7 * 24 * 60 * 60;
			}
			else if (!strcmp (end_ptr, _("d")) ||
				 !strcmp (end_ptr, _("days")) ||
				 !strcmp (end_ptr, _("day")) ||
				 !strcmp (end_ptr, _("dys")) ||
				 !strcmp (end_ptr, _("dy"))) {
				mult = 24 * 60 * 60;
			}
			else if (!strcmp (end_ptr, _("h")) ||
				 !strcmp (end_ptr, _("hours")) ||
				 !strcmp (end_ptr, _("hour")) ||
				 !strcmp (end_ptr, _("hrs")) ||
				 !strcmp (end_ptr, _("hr"))) {
			mult = 60 * 60;
			}			
		}
		duration *= mult;
		
		task->end = task->start + duration;
		gantt_model_task_changed (model, task->taskId);
		break;
		
	case COL_SUMMARY:
		break;
		
	default:
		break;
	}
}

static gboolean
tree_is_editable (ETreeModel *etm, ETreePath path, int col, void *data)
{
	GNOME_MrProject_Task *task;

	task = e_tree_memory_node_get_data (E_TREE_MEMORY (etm), path);
	if (task == NULL) {
		g_warning ("No Task at this path.");
		return FALSE;
	}

	switch (col) {
	case COL_ID:
		return FALSE;

	case COL_NAME:
		return TRUE;

	case COL_START:
	case COL_END:
		return FALSE;
		/*return !IS_SUMMARY (task);*/
		
	case COL_DURATION:
		if (IS_SUMMARY (task)) {
			return FALSE;
		}

		return TRUE;

	default:
		return FALSE;
	}
}

static GdkPixbuf *
tree_icon_at (ETreeModel *etm, ETreePath path, void *data)
{
	/* No icon, since the cell tree renderer takes care of the +/- icons itself. */
	return NULL;
}

static gboolean
gc_add_to_shell (ShellComponent        *shell_comp,
		 GNOME_MrProject_Shell  shell)
{
	CORBA_Environment   ev;
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;

	d(puts (__FUNCTION__));

	CORBA_exception_init (&ev);

	g_return_val_if_fail (shell_comp != NULL, FALSE);
	g_return_val_if_fail (IS_GANTT_COMPONENT (shell_comp), FALSE);
	g_return_val_if_fail (shell != NULL, FALSE);
	
	gantt_comp = GANTT_COMPONENT (shell_comp);
	priv = gantt_comp->priv;

	/* Create and add a gantt control to the Shell. */
	priv->control = gc_control_new (gantt_comp);
	GNOME_MrProject_Shell_addControl (shell_comp->shell,
					  BONOBO_OBJREF (shell_comp),
					  BONOBO_OBJREF (priv->control),
					  OAFIID,
					  _("Gantt Chart"),
					  MRPROJECT_IMAGEDIR "gantt.png",
					  &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not add the Gantt chart to the Shell.");
		bonobo_object_unref (BONOBO_OBJECT (priv->control));
		priv->control = NULL;
		CORBA_exception_free (&ev);
		return FALSE;
	}

	/* Create and connect to a Task manager client object. */
	priv->tm_client = TASK_MANAGER_CLIENT (task_mc_new (shell, FALSE, &ev));
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not create task manager client.");
		bonobo_object_unref (BONOBO_OBJECT (priv->control));
		priv->control = NULL;
		CORBA_exception_free (&ev);
		return FALSE;
	}

	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "load_begin",
				   GTK_SIGNAL_FUNC (load_begin_cb),
				   GTK_OBJECT (gantt_comp));

	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "load_end",
				   GTK_SIGNAL_FUNC (load_end_cb),
				   GTK_OBJECT (gantt_comp));

	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "task_inserted",
				   GTK_SIGNAL_FUNC (task_inserted_cb),
				   GTK_OBJECT (gantt_comp));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "task_updated",
				   GTK_SIGNAL_FUNC (task_updated_cb),
				   GTK_OBJECT (gantt_comp));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "tasks_removed",
				   GTK_SIGNAL_FUNC (tasks_removed_cb),
				   GTK_OBJECT (gantt_comp));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "task_repositioned",
				   GTK_SIGNAL_FUNC (task_repositioned_cb),
				   GTK_OBJECT (gantt_comp));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "task_reparented",
				   GTK_SIGNAL_FUNC (task_reparented_cb),
				   GTK_OBJECT (gantt_comp));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "tasks_linked",
				   GTK_SIGNAL_FUNC (tasks_linked_cb),
				   GTK_OBJECT (gantt_comp));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "tasks_unlinked",
				   GTK_SIGNAL_FUNC (tasks_unlinked_cb),
				   GTK_OBJECT (gantt_comp));

	gtk_signal_connect_object (GTK_OBJECT (priv->tm_client),
				   "items_clear",
				   GTK_SIGNAL_FUNC (tasks_clear_cb),
				   GTK_OBJECT (gantt_comp));

	/* Resource manager client, ditto. */
	priv->rm_client = RESOURCE_MANAGER_CLIENT (resource_mc_new (shell, 
								    FALSE,
								    &ev));

	gtk_signal_connect_object (GTK_OBJECT (priv->rm_client),
				   "resource_updated",
				   GTK_SIGNAL_FUNC (resource_updated_cb),
				   GTK_OBJECT (gantt_comp));

	/* And allocation manager client. */
	priv->am_client = ALLOCATION_MANAGER_CLIENT (allocation_mc_new (shell,
									FALSE, 
									&ev));

	gtk_signal_connect_object (GTK_OBJECT (priv->am_client),
				   "allocation_added",
				   GTK_SIGNAL_FUNC (allocation_added_cb),
				   GTK_OBJECT (gantt_comp));
	
	gtk_signal_connect_object (GTK_OBJECT (priv->am_client),
				   "allocations_removed",
				   GTK_SIGNAL_FUNC (allocations_removed_cb),
				   GTK_OBJECT (gantt_comp));

	CORBA_exception_free (&ev);

	return TRUE;
}

static void
gc_remove_from_shell (ShellComponent *shell_comp)
{
	GanttComponentPriv *priv;
	
	d(puts (__FUNCTION__));

	priv = GANTT_COMPONENT (shell_comp)->priv;

	gtk_object_destroy (GTK_OBJECT (priv->tm_client));
	priv->tm_client = NULL;

	gtk_object_destroy (GTK_OBJECT (priv->rm_client));
	priv->rm_client = NULL;

	gtk_object_destroy (GTK_OBJECT (priv->am_client));
	priv->am_client = NULL;

}

static void
gc_set_debug_flags (ShellComponent *shell_comp,
		    glong           flags)
{
	GanttComponentPriv *priv;
	
	d(puts (__FUNCTION__));

	priv = GANTT_COMPONENT (shell_comp)->priv;
}

#define IS_EVENT(event_name, str) (!strncmp (event_name, str, sizeof (str)-1))

static void
gc_shell_event (ShellComponent *shell_comp,
		const gchar    *event_name,
		CORBA_any      *any)
{
	GanttComponentPriv *priv;
	gchar              *subtype, *kind;

	subtype = bonobo_event_subtype (event_name);
	kind = bonobo_event_kind (event_name);
		
	d(g_print ("GanttComp: got event '%s', subtype: '%s', kind: '%s'\n",
		 event_name,
		 subtype,
		 kind));

	priv = GANTT_COMPONENT (shell_comp)->priv;

	if (IS_EVENT (event_name, "Bonobo/Property:")) {
		/* We have a BonoboProperty event. */

		if (!strcmp (subtype, "Start")) {
			time_t t = *(time_t *) any->_value;

			if (t > 0) {
				gantt_chart_scroll_to_time (GANTT_CHART (priv->chart), t);
			}
		}
	} else {
		g_print ("GanttComponent: got unhandled event: %s\n", event_name);
	}

	g_free (kind);
	g_free (subtype);
}

static void
gc_destroy (GtkObject *object)
{
	GanttComponent     *gantt_comp;
	GanttComponentPriv *priv;

	d(puts (__FUNCTION__));
	
	gantt_comp = GANTT_COMPONENT (object);
	priv = gantt_comp->priv;

	/* Destroy the manager clients. */
	if (priv->tm_client) {
		gtk_object_destroy (GTK_OBJECT (priv->tm_client));
		priv->tm_client = NULL;
	}
	if (priv->rm_client) {
		gtk_object_destroy (GTK_OBJECT (priv->rm_client));
		priv->rm_client = NULL;
	}
	if (priv->am_client) {
		gtk_object_destroy (GTK_OBJECT (priv->am_client));
		priv->am_client = NULL;
	}
	
	if (priv->control) {
		/* Destroy the control as well. */
		bonobo_object_unref (BONOBO_OBJECT (priv->control));
		priv->control = NULL;
	}

	g_free (priv);
	gantt_comp->priv = NULL;

	(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void
gantt_component_class_init (GanttComponentClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ShellComponentClass *shell_component_class = (ShellComponentClass *) klass;

	parent_class = gtk_type_class (PARENT_TYPE);
	
	object_class->destroy = gc_destroy;

	shell_component_class->add_to_shell      = gc_add_to_shell;
	shell_component_class->remove_from_shell = gc_remove_from_shell;
	shell_component_class->set_debug_flags   = gc_set_debug_flags;
	shell_component_class->shell_event       = gc_shell_event;
}

static void
gantt_component_init (GanttComponent *gantt_comp)
{
	GanttComponentPriv  *priv;
	
	priv = g_new0 (GanttComponentPriv, 1);
	gantt_comp->priv = priv;
}

static void
set_pixmap (BonoboUIComponent *uic,
	    const char        *xml_path,
	    const char        *icon)
{
	char      *path;
	GdkPixbuf *pixbuf;

	path = g_concat_dir_and_file (MRPROJECT_IMAGEDIR, icon);

	pixbuf = gdk_pixbuf_new_from_file (path);
	if (pixbuf == NULL) {
		g_warning ("Cannot load image -- %s", path);
		g_free (path);
		return;
	}

	bonobo_ui_util_set_pixbuf (uic, xml_path, pixbuf);

	gdk_pixbuf_unref (pixbuf);

	g_free (path);
}

static void
goto_date_clicked (GtkWidget *widget, time_t t, GanttComponent *component)
{
	puts (__FUNCTION__);
	gantt_chart_scroll_to_time (GANTT_CHART (component->priv->chart), t);
}

static void
goto_today_clicked (GtkWidget *widget, GanttComponent *component)
{
	GanttChart *chart;
	time_t      minor_t;
	
	chart = GANTT_CHART (component->priv->chart);

	minor_t = gantt_scale_get_minor_tick (chart->gantt_scale);
	gantt_chart_scroll_to_time (chart, time (NULL) - minor_t);
}

static void
goto_selected_clicked (GtkWidget *widget, GanttComponent *component)
{
	GNOME_MrProject_Task *task;
	GanttChart           *chart;
	time_t                minor_t;

	task = get_selected_task (component);
	if (!task) {
		return;
	}

	chart = GANTT_CHART (component->priv->chart);

	minor_t = gantt_scale_get_minor_tick (chart->gantt_scale);
	gantt_chart_scroll_to_time (chart, task->start - minor_t);
}

static void
goto_start_clicked (GtkWidget *widget, GanttComponent *component)
{
	GanttChart           *chart;
	time_t                t;

	chart = GANTT_CHART (component->priv->chart);

	t = gantt_model_get_first_time (component->priv->gantt_model);
	if (t < 0) {
		return;
	}
	
	t -= gantt_scale_get_minor_tick (chart->gantt_scale);
	gantt_chart_scroll_to_time (chart, t);
}

static gboolean
goto_dialog_shown (GtkWidget *widget, GanttComponent *component)
{
	if (component->priv->goto_dialog_shown) {
		return FALSE;
	}

	component->priv->goto_dialog_shown = TRUE;
	return TRUE;
}

static void
goto_dialog_hidden (GtkWidget *widget, GanttComponent *component)
{
	component->priv->goto_dialog_shown = FALSE;
}

static void
gantt_control_activate (BonoboControl  *control,
			GanttComponent *gantt_comp)
{
	Bonobo_UIContainer  remote_ui_container;
	BonoboUIComponent  *ui_component;
	BonoboControl      *goto_control;
	GtkWidget          *goto_widget;

	ui_component = bonobo_control_get_ui_component (control);
	g_assert (ui_component != NULL);

	remote_ui_container = bonobo_control_get_remote_ui_container (control);
	bonobo_ui_component_set_container (ui_component, remote_ui_container);
	bonobo_object_release_unref (remote_ui_container, NULL);

	bonobo_ui_component_freeze (ui_component, NULL);
	bonobo_ui_component_add_verb_list_with_data (ui_component, verbs, gantt_comp);

	bonobo_ui_util_set_ui (ui_component,
			       GNOME_DATADIR,
			       "GNOME_MrProject_GanttComponent.ui",
			       "mrproject-gantt-component");

	goto_widget = goto_popup_new (_("Go to - Gantt"),
				      GOTO_TODAY |
				      GOTO_START |
				      GOTO_SELECTED);

	goto_control = bonobo_control_new (goto_widget);

	gtk_widget_show (goto_widget);

	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "date-clicked",
			    goto_date_clicked,
			    gantt_comp);

	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "today-clicked",
			    goto_today_clicked,
			    gantt_comp);

	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "start-clicked",
			    goto_start_clicked,
			    gantt_comp);

	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "selected_clicked",
			    goto_selected_clicked,
			    gantt_comp);
	
	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "dialog_shown",
			    GTK_SIGNAL_FUNC (goto_dialog_shown),
			    gantt_comp);
	
	gtk_signal_connect (GTK_OBJECT (goto_widget),
			    "dialog_hidden",
			    goto_dialog_hidden,
			    gantt_comp);
	
	bonobo_ui_component_object_set (ui_component,
                                        "/Toolbar/GanttGoto",
                                        BONOBO_OBJREF (goto_control),
                                        NULL);

	set_pixmap (ui_component, "/Toolbar/GanttIndent", "24_indent_task.png");
	set_pixmap (ui_component, "/Toolbar/GanttUnindent", "24_unindent_task.png");
	set_pixmap (ui_component, "/Toolbar/GanttUp", "24_task_up.png");
	set_pixmap (ui_component, "/Toolbar/GanttDown", "24_task_down.png");
	set_pixmap (ui_component, "/Toolbar/GanttInsert", "24_add_task.png");
	set_pixmap (ui_component, "/Toolbar/GanttAssign", "24_assign.png");
	set_pixmap (ui_component, "/Toolbar/GanttRemove", "24_delete_task.png");
	set_pixmap (ui_component, "/Toolbar/GanttLink", "24_chain.png");
	set_pixmap (ui_component, "/Toolbar/GanttUnlink", "24_unchain.png");
	set_pixmap (ui_component, "/Toolbar/GanttZoomEntire", "24_zoom_fit.png");
	set_pixmap (ui_component, "/Toolbar/FilePrint", "24_print.png");
	set_pixmap (ui_component, "/menu/File/FilePrint", "16_print.png");

	set_sensitivity_and_status (gantt_comp);
	
	bonobo_ui_component_thaw (ui_component, NULL);
}

static void
gantt_control_deactivate (BonoboControl *control)
{
	BonoboUIComponent *ui_component;

	ui_component = bonobo_control_get_ui_component (control);
	g_assert (ui_component != NULL);

	bonobo_ui_component_rm (ui_component, "/", NULL);
 	bonobo_ui_component_unset_container (ui_component);
}

static void
control_activate_cb (BonoboControl *control,
		     gboolean       activate,
		     gpointer       user_data)
{
	if (activate) {
		gantt_control_activate (control, user_data);
	} else {
		gantt_control_deactivate (control);
	}
}

static void
control_destroy_cb (BonoboControl  *control,
		    GanttComponent *gantt_comp)
{
	gantt_comp->priv->control = NULL;
}

static BonoboControl *
gc_control_new (GanttComponent *gantt_comp)
{
	GanttComponentPriv *priv;
	ETreeModel         *etm;
	ETreeTableAdapter  *etta;
	ETableExtras       *extras;
	GtkAdjustment      *adj;
	GtkWidget          *pad;
	GtkWidget	   *e_tree_vbox;
	gint                pad_height;

	priv = gantt_comp->priv;

	priv->paned = mr_hpaned_new ();
	
	e_tree_vbox = gtk_vbox_new (FALSE, 0);
	e_paned_add1 (E_PANED (priv->paned), e_tree_vbox);

	etm = e_tree_memory_callbacks_new (
		tree_icon_at,
		tree_col_count,
		NULL, /* has save id */
		NULL, /* get save id */
		NULL,
		NULL,
		tree_value_at,
		tree_set_value_at,
		tree_is_editable,
		tree_duplicate_value,
		tree_free_value,
		tree_initialize_value,
		tree_value_is_empty,
		tree_value_to_string,
		gantt_comp);
	e_tree_memory_set_expanded_default (E_TREE_MEMORY (etm), TRUE);

	extras = create_tree_extras ();

	priv->e_tree = e_tree_new_from_spec_file (
		etm,
		extras,
		MRPROJECT_DATADIR "gantt-component.etspec",
		MRPROJECT_DATADIR "gantt-component.etstate");

	gtk_object_unref (GTK_OBJECT (extras));
	
	e_tree_root_node_set_visible (E_TREE (priv->e_tree), FALSE);
	
	gtk_object_get (GTK_OBJECT (priv->e_tree),
			"ETreeTableAdapter", &etta,
			NULL);

	gtk_object_set (GTK_OBJECT (priv->e_tree),
			"horizontal_draw_grid", 1,
			"vertical_draw_grid", 1,
			NULL);

	priv->gantt_model = gantt_model_new (etm,
					     etta,
					     E_TREE (priv->e_tree));

	adj = NULL;
	priv->e_tree_hscroll = gtk_hscrollbar_new (adj);

	pad = gtk_hbox_new (0, FALSE);
	pad_height = get_button_height (priv->e_tree);
	gtk_widget_set_usize (pad, -1, pad_height);
	gtk_box_pack_start (GTK_BOX (e_tree_vbox),
			    pad,
			    FALSE,
			    TRUE,
			    0);
	gtk_box_pack_start (GTK_BOX (e_tree_vbox),
			    priv->e_tree,
			    TRUE,
			    TRUE,
			    0);
	gtk_box_pack_start (GTK_BOX (e_tree_vbox),
			    priv->e_tree_hscroll,
			    FALSE,
			    FALSE,
			    0);

	priv->chart = gantt_chart_new (priv->gantt_model, E_TABLE_MODEL (etta));
	e_paned_add2 (E_PANED (priv->paned), priv->chart);

	/* Share the same vertical adjustment between the views. */
	adj = gantt_chart_get_vadjustment (GANTT_CHART (priv->chart));
	gtk_widget_set_scroll_adjustments (priv->e_tree, NULL, adj);
	
	gtk_signal_connect (GTK_OBJECT (priv->paned), 
			    "realize", 
			    paned_realize, 
			    priv->e_tree);
	gtk_signal_connect (GTK_OBJECT (priv->e_tree),
			    "right-click",
			    e_tree_right_click,
			    gantt_comp);
	gtk_signal_connect (GTK_OBJECT (priv->e_tree),
			    "button-press-event",
			    GTK_SIGNAL_FUNC (e_tree_button_press),
			    gantt_comp);
	gtk_signal_connect (GTK_OBJECT (priv->e_tree), 
			    "double-click", 
			    e_tree_double_click, 
			    gantt_comp);
	gtk_signal_connect (GTK_OBJECT (GANTT_CHART (priv->chart)->gantt_item),
			    "row-clicked", 
			    gantt_row_clicked, 
			    gantt_comp);
	gtk_signal_connect (GTK_OBJECT (GANTT_CHART (priv->chart)->gantt_item),
			    "row-double-clicked", 
			    gantt_row_double_clicked, 
			    gantt_comp);
	gtk_signal_connect (GTK_OBJECT (priv->e_tree), 
			    "selection_change", 
			    e_tree_selection_change,
			    gantt_comp);

	priv->control = bonobo_control_new (priv->paned);
	gtk_widget_show_all (bonobo_control_get_widget (priv->control));

	gtk_signal_connect (GTK_OBJECT (priv->gantt_model),
			    "task_changed",
			    task_changed,
			    gantt_comp);

	gtk_signal_connect (GTK_OBJECT (priv->gantt_model),
			    "task_moved",
			    task_moved,
			    gantt_comp);
	
	gtk_signal_connect (GTK_OBJECT (priv->gantt_model),
			    "task_duration_changed",
			    task_duration_changed,
			    gantt_comp);
	
	gtk_signal_connect (GTK_OBJECT (priv->control), "activate",
			    GTK_SIGNAL_FUNC (control_activate_cb),
			    gantt_comp);

	gtk_signal_connect (GTK_OBJECT (priv->control), "destroy",
			    GTK_SIGNAL_FUNC (control_destroy_cb),
			    gantt_comp);

	bonobo_control_set_automerge (priv->control, FALSE);
	
	return priv->control;
}

static ShellComponent *
gantt_component_new (void)
{
	ShellComponent *component;

	component = gtk_type_new (GANTT_COMPONENT_TYPE);

	return shell_component_construct (component,
					  "=Bonobo/Property:change:Start");
}

static BonoboObject *
gc_factory (BonoboGenericFactory *this,
	    const char           *object_id,
	    void                 *data)
{
	g_return_val_if_fail (object_id != NULL, NULL);

	if (!strcmp (object_id, OAFIID)) {
		return BONOBO_OBJECT (gantt_component_new ());
	} else {
		g_warning ("Failing to manufacture a '%s'", object_id);
	}
	
	return NULL;
}

BONOBO_OAF_SHLIB_FACTORY_MULTI ("OAFIID:GNOME_MrProject_GanttComponentFactory",
				"Mr Project gantt component factory",
				gc_factory,
				NULL);

BONOBO_X_TYPE_FUNC (GanttComponent,
		    PARENT_TYPE,
		    gantt_component);

static void
paned_realize (GtkWidget *widget, GtkTable *e_tree)
{
	gint paned_width;

	/* 
	 * Workaround a bug somewhere between GtkWidget and ETable... 
	 * Initial size allocation messes up without this. /rh
	 */
	gtk_table_set_row_spacing (e_tree, 0, 1);
	gtk_table_set_row_spacing (e_tree, 0, 0);

	paned_width = MAX (300, widget->allocation.width / 3);
	e_paned_set_position (E_PANED (widget), paned_width);
}

static ETableExtras *
create_tree_extras (void)
{
	ETableExtras *extras;
	ECell        *cell, *popup_cell;

	extras = e_table_extras_new ();
	
	cell = m_cell_filter_new (NULL,
				  GTK_JUSTIFY_LEFT,
				  filter_date,
				  (MCellFilterFreeFunc) g_free);
	gtk_object_set (GTK_OBJECT (cell),
			"editable", TRUE,
			"bold_column", COL_SUMMARY,
			NULL);
	
	popup_cell = m_cell_date_new (NULL, GTK_JUSTIFY_LEFT);
	e_cell_popup_set_child (E_CELL_POPUP (popup_cell), cell);
	gtk_object_unref (GTK_OBJECT (cell));
	
 	e_table_extras_add_cell (extras, "date_popup", popup_cell);

	cell = m_cell_filter_new (NULL,
				  GTK_JUSTIFY_LEFT,
				  (MCellFilterGetFunc) filter_duration,
				  (MCellFilterFreeFunc) g_free);
	gtk_object_set (GTK_OBJECT (cell),
			"editable", TRUE,
			"bold_column", COL_SUMMARY,
			NULL);
	e_table_extras_add_cell (extras, "render_duration", cell);
	
	cell = e_cell_text_new (NULL, GTK_JUSTIFY_LEFT);
	gtk_object_set (GTK_OBJECT (cell),
			"bold_column", COL_SUMMARY,
			NULL);
	e_table_extras_add_cell (extras,
				 "render_name", 
				 e_cell_tree_new (NULL, NULL, TRUE, cell));
	
	return extras;
}

static gint
get_button_height (GtkWidget *widget)
{
	gint height;

	gtk_widget_ensure_style (widget);
	height = widget->style->font->ascent + widget->style->font->descent;
	height += 2 * (1 + widget->style->klass->ythickness);

	return height;
}

static gint
get_selected_id (GanttComponent *component)
{
	ETreeModel *etm;
	ETreePath   selected_path;
	GM_Task    *task;

	etm = E_TREE_MODEL (component->priv->gantt_model->etm);
	
	selected_path = e_tree_get_cursor (E_TREE (component->priv->e_tree));
	if (!selected_path) {
		return -1;
	}

	task = e_tree_memory_node_get_data (E_TREE_MEMORY (etm), selected_path);
	if (!task) {
		return -1;
	}

	return task->taskId;
}

typedef struct {
	ETreePath *root_path;
	GSList    *list;
} GetTwoSelectionData;

static void
get_two_selection_foreach (ETreePath path, gpointer data)
{
	GetTwoSelectionData *sdata = data;

	/* Appearantly, we will get the root path here even when it's
	 * invisible... Work around, not so nice :( */
	if (sdata->root_path == path) {
		return;
	}
	
	sdata->list = g_slist_append (sdata->list, path);
}

static gboolean
get_two_selected_ids (GanttComponent *component, gint *id1, gint *id2)
{
	ETreeMemory          *etm;
	GNOME_MrProject_Task *task1, *task2;
	GetTwoSelectionData  sdata;
	GSList               *list;

	etm = E_TREE_MEMORY (component->priv->gantt_model->etm);

	sdata.root_path = gantt_model_get_root_path (component->priv->gantt_model);
	sdata.list = NULL;
	
	e_tree_selected_path_foreach (E_TREE (component->priv->e_tree),
				      get_two_selection_foreach,
				      &sdata);

	if (!sdata.list || g_slist_length (sdata.list)!= 2) {
		return FALSE;
	}

	task1 = task2 = NULL;
	for (list = sdata.list; list; list = list->next) {
		if (!task1) {
			task1 = e_tree_memory_node_get_data (etm, list->data);
		} else {
			task2 = e_tree_memory_node_get_data (etm, list->data);
		}
	} 

	g_slist_free (sdata.list);

	*id1 = task1->taskId;
	*id2 = task2->taskId;
	return TRUE;
}

static GNOME_MrProject_Task *
get_selected_task (GanttComponent *component)
{
	ETreeModel *etm;
	ETreePath   selected_path;
	GM_Task    *task;

	etm = E_TREE_MODEL (component->priv->gantt_model->etm);
	
	selected_path = e_tree_get_cursor (E_TREE (component->priv->e_tree));
	if (!selected_path) {
		return NULL;
	}

	task = e_tree_memory_node_get_data (E_TREE_MEMORY (etm), selected_path);
	return task;
}

static void
gantt_component_edit_task_activate (GtkMenuItem    *menu_item,
				    GanttComponent *component)
{
	g_return_if_fail (component != NULL);
	g_return_if_fail (IS_GANTT_COMPONENT (component));
	
	cmd_edit (NULL, component, NULL);
}

/* If have_task is FALSE, disable the options that need a task. */
static GtkWidget *
gc_setup_right_click_menu (GanttComponent *component, gboolean have_task)
{
	GtkWidget *menu;
	GtkWidget *menu_item;
	
	menu = gtk_menu_new ();
	
	menu_item = gtk_menu_item_new_with_label (_("Edit task..."));
	gtk_signal_connect (GTK_OBJECT (menu_item),
			    "activate",
			    gantt_component_edit_task_activate,
			    component);
	gtk_widget_show (menu_item);
	gtk_menu_append (GTK_MENU (menu), menu_item);
	gtk_widget_set_sensitive (menu_item, have_task);

	menu_item = gtk_menu_item_new ();
	gtk_widget_show (menu_item);
	gtk_menu_append (GTK_MENU (menu), menu_item);
	
	menu_item = gtk_menu_item_new_with_label (_("Insert subtask"));
	gtk_signal_connect_object (GTK_OBJECT (menu_item),
				   "activate",
				   gc_insert_sub_task,
				   GTK_OBJECT (component));
	gtk_widget_show (menu_item);
	gtk_menu_append (GTK_MENU (menu), menu_item);
	gtk_widget_set_sensitive (menu_item, have_task);

	menu_item = gtk_menu_item_new_with_label (_("Insert task below"));
	gtk_signal_connect_object (GTK_OBJECT (menu_item),
				   "activate",
				   gc_insert_task_below,
				   GTK_OBJECT (component));
	gtk_widget_show (menu_item);
	gtk_menu_append (GTK_MENU (menu), menu_item);

	menu_item = gtk_menu_item_new ();
	gtk_widget_show (menu_item);
	gtk_menu_append (GTK_MENU (menu), menu_item);

	menu_item = gtk_menu_item_new_with_label (_("Delete task"));
	gtk_signal_connect_object (GTK_OBJECT (menu_item),
				   "activate",
				   gc_delete_task,
				   GTK_OBJECT (component));
	gtk_widget_show (menu_item);
	gtk_menu_append (GTK_MENU (menu), menu_item);
	gtk_widget_set_sensitive (menu_item, have_task);

	return menu;
}

/* Note: This only works for the first right-click. After adding
 * tasks, ETree seems to not send events anymore...
 */
static gboolean
e_tree_button_press (GtkWidget *widget,
		     GdkEvent  *event,
		     gpointer   user_data)
{
	GtkWidget      *menu;
	GdkEventButton *button_event;

	button_event = (GdkEventButton *) event;

	if (button_event->button == 3) {
		menu = gc_setup_right_click_menu (user_data, FALSE);
		gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
				button_event->button, button_event->time);
		return TRUE;
	}

	return FALSE;
}

static void
e_tree_right_click (ETree     *et,
		    int        row,
		    ETreePath  path,
		    int        col,
		    GdkEvent  *event,
		    gpointer   user_data)
{
	GtkWidget      *menu;
	GdkEventButton *button_event;

	menu = gc_setup_right_click_menu (user_data, TRUE);
	button_event = (GdkEventButton *) event;
	
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
			button_event->button, button_event->time);
}

static void
e_tree_double_click (ETree     *et,
		     int        row,
		     ETreePath  path,
		     int        col,
		     GdkEvent  *event,
		     gpointer   user_data)
{
	cmd_edit (NULL, user_data, NULL);
}

static void
gantt_row_double_clicked (GanttItem       *gantt,
			  GnomeCanvasItem *row,
			  GdkEvent        *event,
			  GanttComponent  *component)
{
	GM_Task           *task;
	GM_Id              id;
	CORBA_Object       task_dialog_co;
	CORBA_Environment  ev;

	gtk_object_get (GTK_OBJECT (row), "task", &task, NULL);
	if (!task) {
		return;
	}

	id = task->taskId;
	
	task_dialog_co = activate_task_dialog (component, id);
	if (!task_dialog_co) {
		return;
	}
	
	CORBA_exception_init (&ev);

	GNOME_MrProject_TaskDialog_showPage (task_dialog_co,
					     GNOME_MrProject_PAGE_GENERAL,
					     &ev);
	if (BONOBO_EX (&ev)) {
		g_warning ("Could not set task on TaskDialog.");
		CORBA_exception_free (&ev);
		bonobo_object_release_unref (task_dialog_co, NULL);
	}

	CORBA_exception_free (&ev);
}

static void
gantt_row_clicked (GanttItem       *gantt_item,
		   GnomeCanvasItem *row_item,
		   GdkEvent        *event,
		   GanttComponent  *component)
{
	/* There seems to be no way to select rows programatically
	 * in ETree so this will have to wait.
	 */
#if 0
	if (!(event->button.state & GDK_CONTROL_MASK)) {
		gantt_item_unselect_all (gantt_item);
	}
	
	gantt_item_select_row_item (gantt_item, row_item);
#endif
}

static gchar *
filter_date (gconstpointer val)
{
	static gchar  buf[64];
	struct tm    *tm;

	tm = localtime ((time_t *) &val);
	buf[0] = '\0';
	strftime (buf, 16, "%a %e %b", tm); /* Fre 1 apr */
	return e_utf8_from_locale_string (buf);
}

static gchar *
filter_duration (gconstpointer val)
{
	gint     duration;
	gdouble  days;
	gint     hours;
	gchar   *filtered_time;
	
	duration = GPOINTER_TO_INT (val);
	days = duration / (60.0 * 60.0 * 24.0);
	duration -= floor (days) * (60 * 60 * 24);
	hours = duration / (60 * 60);

	/* FIXME: How do we want to format the duration really? */
	
	if (days >= 2) {
		filtered_time = g_strdup_printf (_("%d %s"), (int) floor (days + 0.5), _("days"));
	}
	else if (days < 1 && hours > 0) {
		filtered_time = g_strdup_printf (_("%d %s"), hours, _("hrs"));
	}
	else if (days >= 1 && days <= 2) {
		filtered_time = g_strdup (_("1 day"));
	} else {
		filtered_time = g_strdup (_("0 days"));
	}
	
	return filtered_time;
}

static void
set_status_message (GanttComponent *component)
{
	GanttComponentPriv *priv;
	GSList             *row_numbers;
	BonoboUIComponent  *ui_component;
	gchar              *str;
	gint                length;

	d(puts (__FUNCTION__));
	
	priv = component->priv;

	ui_component = bonobo_control_get_ui_component (priv->control);
	
	row_numbers = e_tree_extension_get_selected_row_numbers (
		E_TREE (priv->e_tree));

	/* FIXME: the above does not seem to work correctly when the selection
	 * changed by inverting the selection... weird.
	 */
	
	gantt_chart_select (GANTT_CHART (priv->chart), row_numbers);

	length = g_slist_length (row_numbers);

	d(g_print ("selected count %d\n", length));
	
	if (length > 1) {
		str = g_strdup_printf (_("%d tasks selected"), length);
	}
	else if (length == 0) {
		str = g_strdup (" "); /* "" does not seem to work. */
	}
	else { /* We have exactly one. */
		ETreePath *path;
		GM_Task   *task;
		gint       row;
		gchar     *tmp;

		row = GPOINTER_TO_INT (row_numbers->data);

		path = e_tree_node_at_row (
			E_TREE (priv->e_tree),
			row);
		if (path) {
			task = e_tree_memory_node_get_data (
				E_TREE_MEMORY (priv->gantt_model->etm),
				path);
			
			tmp = e_utf8_to_locale_string (task->name);
			str = g_strdup_printf (_("Selected task: %s"), tmp);
			g_free (tmp);
		}
		else {
			str = g_strdup (" ");
		}
	}	

	bonobo_ui_component_set_status (ui_component,
					str,
					NULL);
	g_free (str);

	g_slist_free (row_numbers);
}

static gboolean
real_set_sensitivity (gpointer user_data)
{
	GanttComponent     *component;
	GanttComponentPriv *priv;
	ESelectionModel    *selection;
	BonoboUIComponent  *uic;
	gint		    count;
	gchar              *one, *two, *many;

	component = GANTT_COMPONENT (user_data);
	priv = component->priv;

	selection = e_tree_get_selection_model (E_TREE (priv->e_tree));
	count = e_selection_model_selected_count (selection);

	one = (count == 1) ? "1" : "0";
	two = (count == 2) ? "1" : "0";
	many = (count > 0) ? "1" : "0";

	uic = bonobo_control_get_ui_component (priv->control);

	if (bonobo_ui_component_get_container (uic) != CORBA_OBJECT_NIL) {
		bonobo_ui_component_freeze (uic, NULL);

		set_status_message (component);

		bonobo_ui_component_set_prop (uic, "/commands/GanttIndent",
					      "sensitive", one, NULL);
		bonobo_ui_component_set_prop (uic, "/commands/GanttUnindent",
					      "sensitive", one, NULL);
		bonobo_ui_component_set_prop (uic, "/commands/GanttRemove",
					      "sensitive", many, NULL);
		bonobo_ui_component_set_prop (uic, "/commands/GanttUp",
					      "sensitive", one, NULL);
		bonobo_ui_component_set_prop (uic, "/commands/GanttDown",
					      "sensitive", one, NULL);
		bonobo_ui_component_set_prop (uic, "/commands/GanttEdit",
					      "sensitive", one, NULL);
		bonobo_ui_component_set_prop (uic, "/commands/GanttLink",
					      "sensitive", two, NULL);
		bonobo_ui_component_set_prop (uic, "/commands/GanttUnlink",
					      "sensitive", two, NULL);
		bonobo_ui_component_set_prop (uic, "/commands/GanttGotoTask",
					      "sensitive", one, NULL);
		bonobo_ui_component_set_prop (uic, "/commands/GanttAssign",
					      "sensitive", one, NULL);
		bonobo_ui_component_thaw (uic, NULL);

		priv->set_sensitivity_idle = 0;
	}

	return FALSE;
}

static void
set_sensitivity_and_status (GanttComponent *component)
{
	GanttComponentPriv *priv;

	priv = component->priv;

	if (priv->loading) {
		return;
	}
	
	/* FIXME: This is just a workaround. We would probably
	 * want to have a way to insert lots of tasks with one event
	 * instead. For now, this reduces some flickering.
	 */
	if (priv->set_sensitivity_idle == 0) {
		priv->set_sensitivity_idle = 
			gtk_timeout_add (75, real_set_sensitivity, component);
	}
}

static GM_Task *
gc_create_task (GanttComponent *gantt_comp, GM_Task *parent_task)
{
	GanttComponentPriv   *priv;
	GNOME_MrProject_Task *task;
	time_t                first_visible_time;
	CORBA_Environment     ev;

	priv = gantt_comp->priv;

	CORBA_exception_init (&ev);

	task = task_mc_create_task (priv->tm_client, &ev);
	if (BONOBO_EX (&ev) || !task) {
		g_log_exception (&ev, "Gantt Component");
		CORBA_exception_free (&ev);
		return NULL;
	}

	if (parent_task) {
		task->start = parent_task->start;
		task->end = parent_task->end;
	} else {
		/* Get the dates currently shown in the chart. */
		first_visible_time = gantt_chart_get_first_visible_time (
			GANTT_CHART (priv->chart));
		
		task->start = first_visible_time + 8 * 3600;
		task->end = task->start + 8 * 3600;
	}

	CORBA_exception_free (&ev);

	return task;
}

static void
gc_insert_sub_task (GanttComponent *gantt_comp)
{
	GanttComponentPriv   *priv;
	GNOME_MrProject_Task *parent_task, *task;
	gint                  pos;
	GNOME_MrProject_Id    id;
	CORBA_Environment     ev;

	priv        = gantt_comp->priv;
	parent_task = get_selected_task (gantt_comp);

	if (parent_task) {
		id = parent_task->taskId;
		pos = -1;
	} else {
		id = 0;
		pos = -1;
	}

	/* Create and insert a new row. */
	CORBA_exception_init (&ev);

	task = gc_create_task (gantt_comp, parent_task);
	if (!task) {
		return;
	}

	task_mc_insert_task (priv->tm_client, task, id, &ev);
	if (BONOBO_EX (&ev)) {
		CORBA_free (task);
		g_log_exception (&ev, "Gantt Component");
	}

	CORBA_free (task);
	CORBA_exception_free (&ev);
}

static void
gc_insert_task_below (GanttComponent *gantt_comp)
{
	GanttComponentPriv   *priv;
	GanttModel           *gantt_model;
	GNOME_MrProject_Task *selected_task, *task;
	GNOME_MrProject_Id    parent_id, sibling_id;
	CORBA_Environment     ev;

	priv = gantt_comp->priv;
	gantt_model = priv->gantt_model;

	CORBA_exception_init (&ev);

	parent_id = 0;
	sibling_id = -1;
	
	selected_task = get_selected_task (gantt_comp);
	if (selected_task) {
		parent_id = selected_task->parentId;
		sibling_id = selected_task->taskId;
	}

	/* We don't want to inherit a span so send NULL as parent. */
	task = gc_create_task (gantt_comp, NULL);
	if (!task) {
		return;
	}

	task_mc_insert_task_full (priv->tm_client,
				  task,
				  parent_id,
				  sibling_id,
				  GNOME_MrProject_TASK_AFTER,
				  &ev);
		
	if (BONOBO_EX (&ev)) {
		g_log_exception (&ev, "Gantt Component");
		CORBA_exception_free (&ev);			
		CORBA_free (task);
		return;
	}

	CORBA_free (task);
	CORBA_exception_free (&ev);			
}

static void
gc_delete_task (GanttComponent *gantt_comp)
{
	GanttComponentPriv   *priv;
	GNOME_MrProject_Id    id;
	CORBA_Environment     ev;
	GSList               *l; 

	priv = gantt_comp->priv;
	id = get_selected_id (gantt_comp);

	if (id > 0) {
		CORBA_exception_init (&ev);

		l = g_slist_prepend (NULL, GINT_TO_POINTER (id));
		
		task_mc_remove_tasks (priv->tm_client, l, &ev);
		g_slist_free (l);

		if (BONOBO_EX (&ev)) {
			g_log_exception (&ev, "Gantt Component");
			CORBA_exception_free (&ev);
			return;
		}
		
		CORBA_exception_free (&ev);
	}
}

static void
e_tree_selection_change (ETree *e_tree, GanttComponent *component)
{
	d(puts (__FUNCTION__));
	set_sensitivity_and_status (component);	
}
