/*	alg.c
 *
 *	Detect changes in a video stream.
 *	Copyright 2001 by Jeroen Vreeken (pe1rxq@amsat.org)
 *	This software is distributed under the GNU public license version 2
 *	See also the file 'COPYING'.
 *
 */
#include "motion.h"
#include "video.h"
#include "conf.h"
#include "alg.h"

/* locate the center of the movement. */
struct coord alg_locate_center (char *out, int width, int height)
{
	int x, y, centc=0;
	struct coord cent;

	cent.x=0;
	cent.y=0;
	/* Locate movement */
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			if (*(out++) + *(out++) + *(out++)) {
				cent.x+=x;
				cent.y+=y;
				centc++;
			}
		}
	}
	if (centc) {
		cent.x=cent.x/centc;
		cent.y=cent.y/centc;
	}
	return (cent);
}

/* draw a box around the movement */
void alg_locate (struct coord cent, char *out, char *new, int width, int height)
{
	int x, y, maxx=0, maxy=0, minx=width, miny=height;
	int centc=0, xdist=0, ydist=0;
	char *tmp=out;

	centc=0;
	for (y=0; y<height; y++) {
		for (x=0; x<width; x++) {
			if (*(out++) + *(out++) + *(out++)) {
				if (x > cent.x) xdist+=x-cent.x;
				if (y > cent.y) ydist+=y-cent.y;
				if (x < cent.x) xdist+=cent.x-x;
				if (y < cent.y) ydist+=cent.y-y;
				centc++;
			}
		}	
	}
	if (centc) {
		minx=cent.x-xdist/centc*2;
		maxx=cent.x+xdist/centc*2;
		miny=cent.y-ydist/centc*2;
		/* Make the box a litle bigger in y direction to make sure the
		 * heads fit in */
		maxy=cent.y+ydist/centc*3;
	}
	if (minx < 0) minx=0;
	if (miny < 0) miny=0;
	if (maxx > width-1) maxx=width-1;
	if (maxy > height-1) maxy=height-1;
	out=tmp;
	
	/* Draw a box around the movement */
	for (x=minx; x<=maxx; x++) {
		new[(x+width*miny)*3+0]=~new[(x+width*miny)*3+0];
		new[(x+width*miny)*3+1]=~new[(x+width*miny)*3+1];
		new[(x+width*miny)*3+2]=~new[(x+width*miny)*3+2];
		new[(x+width*maxy)*3+0]=~new[(x+width*maxy)*3+0];
		new[(x+width*maxy)*3+1]=~new[(x+width*maxy)*3+1];
		new[(x+width*maxy)*3+2]=~new[(x+width*maxy)*3+2];
		out[(x+width*miny)*3+0]=~out[(x+width*miny)*3+0];
		out[(x+width*miny)*3+1]=~out[(x+width*miny)*3+1];
		out[(x+width*miny)*3+2]=~out[(x+width*miny)*3+2];
		out[(x+width*maxy)*3+0]=~out[(x+width*maxy)*3+0];
		out[(x+width*maxy)*3+1]=~out[(x+width*maxy)*3+1];
		out[(x+width*maxy)*3+2]=~out[(x+width*maxy)*3+2];
	}
	for (y=miny; y<=maxy; y++) {
		new[(minx+y*width)*3+0]=~new[(minx+y*width)*3+0];
		new[(minx+y*width)*3+1]=~new[(minx+y*width)*3+1];
		new[(minx+y*width)*3+2]=~new[(minx+y*width)*3+2];
		new[(maxx+y*width)*3+0]=~new[(maxx+y*width)*3+0];
		new[(maxx+y*width)*3+1]=~new[(maxx+y*width)*3+1];
		new[(maxx+y*width)*3+2]=~new[(maxx+y*width)*3+2];
		out[(minx+y*width)*3+0]=~out[(minx+y*width)*3+0];
		out[(minx+y*width)*3+1]=~out[(minx+y*width)*3+1];
		out[(minx+y*width)*3+2]=~out[(minx+y*width)*3+2];
		out[(maxx+y*width)*3+0]=~out[(maxx+y*width)*3+0];
		out[(maxx+y*width)*3+1]=~out[(maxx+y*width)*3+1];
		out[(maxx+y*width)*3+2]=~out[(maxx+y*width)*3+2];
	}
}


/*	The heart of the motion detector
 *	Basicly: ref-new and count the pixels that are over a noise threshold
 */
int alg_diff_standard (struct images *imgs)
{
	int i, diffs=0;
	long int level=0;
	int noise=conf.noise;
	int width=imgs->width;
	int height=imgs->height;
	char *ref=imgs->ref;
	char *new=imgs->new;
	char *out=imgs->out;
	unsigned char *mask=imgs->mask;


	/* If the average level of the picture is to low, compensate by 
	 * lowering the noise threshold
	 */
	if (conf.nightcomp) {
		for (i=0; i<width*height*3; i++) {
			level+=(unsigned char)new[i];
		}
		if (level)
			level/=width*height*3;
		if (level < noise)
			noise/=2;
	}

	for (i=0; i<width*height*3; i++) {
		*out=*ref-*new;
		if (mask)
			*out=((int)((char)*out*mask[i])/255);
		if (*out >  noise ||
		    *out < -noise ) {
			if (!conf.realmotion) *out=*new;
			diffs++;
		} else {
			if (!conf.realmotion) *out=0;
		}
		out++;
		ref++;
		new++;
	}

	return diffs;
}

/*
	Very fast diff function, does not do nightcompensation or mask
	overlaying.
*/
int alg_diff_fast (struct images *imgs)
{
	int i, diffs=0;
	int noise=conf.noise;
	int height=imgs->height;
	int width=imgs->width;
	char *out=imgs->out;
	char *ref=imgs->ref;
	char *new=imgs->new;

	memset(out, 0, width*height*3);
	for (i=0; i<width*height*3; i+=40) {
		*out=*ref-*new;
		if (*out >  noise ||
		    *out < -noise ) {
			if (!conf.realmotion) *out=*new;
			diffs++;
		} else {
			if (!conf.realmotion) *out=0;
		}
		out+=40;
		ref+=40;
		new+=40;
	}
	diffs=diffs*40;
	return diffs;
}

/*
	diff_hybrid uses diff_fast to quickly decide if there is anything worth
	sending to diff_standard.
*/
int alg_diff_hybrid (struct images *imgs)
{
	int diffs;
	
	diffs=alg_diff_fast(imgs);
	if (diffs > conf.max_changes/2)
		diffs=alg_diff_standard(imgs);
	return diffs;
}

int alg_diff (struct images *imgs)
{
	return alg_diff_hybrid(imgs);
}


/* Detect a sudden massive change in the picture.
   It is assumed to be the light being switched on or a camera displacement.
   In any way the user doesn't think it is worth capturing.
 */
int alg_lightswitch (int diffs, int dev, int pipe, int mpipe, struct images *imgs)
{
	int i, j;
	
	/* is 2/3 of the image right?  */
	if (diffs > imgs->width*imgs->height*2) {
		printf("lightswitch detected\n");
		/* wait for the device to settle and update the
		 * reference frame to prevent the next frames
		 * from being detected
		 * if 1/3 of the frame is still moving asume that the camera
		 * has not yet compensated.
		 */
		i=10;
		while(i && (imgs->new=vid_next(dev, imgs->new, imgs->width, imgs->height))) {
			if (alg_diff(imgs)
			    > imgs->width*imgs->height)
				i=10;
			for (j=0; j<imgs->width*imgs->height*3; j++) {
				imgs->ref[j]=imgs->new[j];
			}
			if (conf.vidpipe)
				vid_putpipe(pipe, imgs->new, imgs->width, imgs->height);
			if (conf.motionvidpipe)
				vid_putpipe(mpipe, imgs->out, imgs->width, imgs->height);
			i--;
		}
		return 0;
	}
	return diffs;
}
