# -*- Mode: Python; test-case-name: morituri.test.test_image_cue -*-
# vi:si:et:sw=4:sts=4:ts=4

import os
import copy
import unittest

from morituri.image import toc

from morituri.test import common

class CureTestCase(unittest.TestCase):
    def setUp(self):
        self.toc = toc.TocFile(os.path.join(os.path.dirname(__file__),
            u'cure.toc'))
        self.toc.parse()
        self.assertEquals(len(self.toc.table.tracks), 13)

    def testGetTrackLength(self):
        t = self.toc.table.tracks[0]
        # first track has known length because the .toc is a single file
        self.assertEquals(self.toc.getTrackLength(t), 28324)
        # last track has unknown length
        t = self.toc.table.tracks[-1]
        self.assertEquals(self.toc.getTrackLength(t), -1)

    def testIndexes(self):
        # track 2, index 0 is at 06:16:45 or 28245
        # track 2, index 1 is at 06:17:49 or 28324
        # FIXME: cdrdao seems to get length of FILE 1 frame too many,
        # and START value one frame less
        t = self.toc.table.tracks[1]
        self.assertEquals(t.getIndex(0).relative, 28245)
        self.assertEquals(t.getIndex(1).relative, 28324)

    def _getIndex(self, t, i):
        track = self.toc.table.tracks[t - 1]
        return track.getIndex(i)

    def _assertAbsolute(self, t, i, value):
        index = self._getIndex(t, i)
        self.assertEquals(index.absolute, value)

    def _assertPath(self, t, i, value):
        index = self._getIndex(t, i)
        self.assertEquals(index.path, value)

    def _assertRelative(self, t, i, value):
        index = self._getIndex(t, i)
        self.assertEquals(index.relative, value)

    def testSetFile(self):
        self._assertAbsolute(1, 1, 0)
        self._assertAbsolute(2, 0, 28245)
        self._assertAbsolute(2, 1, 28324)
        self._assertPath(1, 1, "data.wav")

        self.toc.table.absolutize()
        self.toc.table.clearFiles()

        self._assertAbsolute(1, 1, 0)
        self._assertAbsolute(2, 0, 28245)
        self._assertAbsolute(2, 1, 28324)
        self._assertAbsolute(3, 1, 46110)
        self._assertAbsolute(4, 1, 66767)
        self._assertPath(1, 1, None)
        self._assertRelative(1, 1, None)

        # adding the first track file with length 28324 to the table should
        # relativize from absolute 0 to absolute 28323, right before track 2,
        # index 1
        self.toc.table.setFile(1, 1, 'track01.wav', 28324)
        self._assertPath(1, 1, 'track01.wav')
        self._assertRelative(1, 1, 0)
        self._assertPath(2, 0, 'track01.wav')
        self._assertRelative(2, 0, 28245)

        self._assertPath(2, 1, None)
        self._assertRelative(2, 1, None)

    def testConvertCue(self):
        self.toc.table.absolutize()
        cue = self.toc.table.cue()
        ref = open(os.path.join(os.path.dirname(__file__), 'cure.cue')).read()
        self.assertEquals(cue, ref)

        # we verify it because it has failed in readdisc in the past
        self.assertEquals(self.toc.table.getAccurateRipURL(),
            'http://www.accuraterip.com/accuraterip/'
            '3/c/4/dBAR-013-0019d4c3-00fe8924-b90c650d.bin')

# Bloc Party - Silent Alarm has a Hidden Track One Audio
class BlocTestCase(unittest.TestCase):
    def setUp(self):
        self.toc = toc.TocFile(os.path.join(os.path.dirname(__file__),
            u'bloc.toc'))
        self.toc.parse()
        self.assertEquals(len(self.toc.table.tracks), 13)

    def testGetTrackLength(self):
        t = self.toc.table.tracks[0]
        # first track has known length because the .toc is a single file
        # the length is from Track 1, Index 1 to Track 2, Index 1, so
        # does not include the htoa
        self.assertEquals(self.toc.getTrackLength(t), 19649)
        # last track has unknown length
        t = self.toc.table.tracks[-1]
        self.assertEquals(self.toc.getTrackLength(t), -1)

    def testIndexes(self):
        t = self.toc.table.tracks[0]
        self.assertEquals(t.getIndex(0).relative, 0)
        self.assertEquals(t.getIndex(1).relative, 15220)

    # This disc has a pre-gap, so is a good test for .CUE writing
    def testConvertCue(self):
        #self.toc.table.absolutize()
        self.failUnless(self.toc.table.hasTOC())
        cue = self.toc.table.cue()
        ref = open(os.path.join(os.path.dirname(__file__),
            'bloc.cue')).read()
        self.assertEquals(cue, ref)

    def testCDDBId(self):
        self.toc.table.absolutize()
        # cd-discid output:
        # ad0be00d 13 15370 35019 51532 69190 84292 96826 112527 132448
        # 148595 168072 185539 203331 222103 3244

        self.assertEquals(self.toc.table.getCDDBDiscId(), 'ad0be00d')

    def testAccurateRip(self):
        # we verify it because it has failed in readdisc in the past
        self.toc.table.absolutize()
        self.assertEquals(self.toc.table.getAccurateRipURL(),
            'http://www.accuraterip.com/accuraterip/'
            'e/d/2/dBAR-013-001af2de-0105994e-ad0be00d.bin')

# The Breeders - Mountain Battles has CDText
class BreedersTestCase(unittest.TestCase):
    def setUp(self):
        self.toc = toc.TocFile(os.path.join(os.path.dirname(__file__),
            u'breeders.toc'))
        self.toc.parse()
        self.assertEquals(len(self.toc.table.tracks), 13)

    def testCDText(self):
        cdt = self.toc.table.cdtext
        self.assertEquals(cdt['PERFORMER'], 'THE BREEDERS')
        self.assertEquals(cdt['TITLE'], 'MOUNTAIN BATTLES')

        t = self.toc.table.tracks[0]
        cdt = t.cdtext
        self.assertRaises(AttributeError, getattr, cdt, 'PERFORMER')
        self.assertEquals(cdt['TITLE'], 'OVERGLAZED')

    def testConvertCue(self):
        self.toc.table.absolutize()
        self.failUnless(self.toc.table.hasTOC())
        cue = self.toc.table.cue()
        ref = open(os.path.join(os.path.dirname(__file__),
            'breeders.cue')).read()
        self.assertEquals(cue, ref)

# Ladyhawke has a data track
class LadyhawkeTestCase(unittest.TestCase):
    def setUp(self):
        self.toc = toc.TocFile(os.path.join(os.path.dirname(__file__),
            u'ladyhawke.toc'))
        self.toc.parse()
        self.assertEquals(len(self.toc.table.tracks), 13)
        #import code; code.interact(local=locals())
        self.failIf(self.toc.table.tracks[-1].audio)

    def testCDDBId(self):
        self.toc.table.absolutize()
        self.assertEquals(self.toc.table.getCDDBDiscId(), 'c60af50d')
        # output from cd-discid:
        # c60af50d 13 150 15687 31841 51016 66616 81352 99559 116070 133243
        # 149997 161710 177832 207256 2807

class CapitalMergeTestCase(unittest.TestCase):
    def setUp(self):
        self.toc1 = toc.TocFile(os.path.join(os.path.dirname(__file__),
            u'capital.1.toc'))
        self.toc1.parse()
        self.assertEquals(len(self.toc1.table.tracks), 11)
        self.failUnless(self.toc1.table.tracks[-1].audio)

        self.toc2 = toc.TocFile(os.path.join(os.path.dirname(__file__),
            u'capital.2.toc'))
        self.toc2.parse()
        self.assertEquals(len(self.toc2.table.tracks), 1)
        self.failIf(self.toc2.table.tracks[-1].audio)

        self.table = copy.deepcopy(self.toc1.table)
        self.table.merge(self.toc2.table)


    def testCDDBId(self):
        self.table.absolutize()
        self.assertEquals(self.table.getCDDBDiscId(), 'b910140c')
        # output from cd-discid:
        # b910140c 12 24320 44855 64090 77885 88095 104020 118245 129255 141765 164487 181780 209250 4440

    def testMusicBrainz(self):
        # URL to submit: http://mm.musicbrainz.org/bare/cdlookup.html?toc=1+11+197850+24320+44855+64090+77885+88095+104020+118245+129255+141765+164487+181780&tracks=11&id=MAj3xXf6QMy7G.BIFOyHyq4MySE-
        self.assertEquals(self.table.getMusicBrainzDiscId(),
            "MAj3xXf6QMy7G.BIFOyHyq4MySE-")

class UnicodeTestCase(unittest.TestCase):
    def setUp(self):
        path = u'Jos\xe9Gonz\xe1lez.toc'
        self._performer = u'Jos\xe9 Gonz\xe1lez'
        self.toc = toc.TocFile(os.path.join(os.path.dirname(__file__), path))
        self.toc.parse()
        self.assertEquals(len(self.toc.table.tracks), 10)

    def testGetTrackLength(self):
        t = self.toc.table.tracks[0]
        # first track has known length because the .toc is a single file
        self.assertEquals(self.toc.getTrackLength(t), 12001)
        # last track has unknown length
        t = self.toc.table.tracks[-1]
        self.assertEquals(self.toc.getTrackLength(t), -1)

    def testGetTrackPerformer(self):
        t = self.toc.table.tracks[0]
        self.assertEquals(t.cdtext['PERFORMER'], self._performer)
