# copyright (C) 1997-2001 Jean-Luc Fontaine (mailto:jfontain@free.fr)
# this program is free software: please read the COPYRIGHT file enclosed in this package or use the Help Copyright menu

set rcsId {$Id: system.tcl,v 1.6 2001/01/27 15:08:19 jfontain Exp $}


package provide system [lindex {$Revision: 1.6 $} 1]
package require network 1
package require miscellaneous 1

namespace eval system {

    array set data {
        updates 0
        0,label version 0,type ascii 0,message {kernel version}
        1,label date 1,type clock 1,message {kernel build date}
        2,label time 2,type clock 2,message {kernel build time}
        3,label {CPU vendor} 3,type ascii 3,message {processor vendor identification}
        4,label {CPU model} 4,type ascii 4,message {processor model name}
        5,label {CPU speed} 5,type real 5,message {processor speed in megahertz}
        6,label {CPU MIPS} 6,type real 6,message {processor speed in bogomips}
        7,label {up time} 7,type dictionary 7,message {system uptime in d(ays), h(ours), m(inutes) and s(econds)}
        8,label {idle time} 8,type dictionary 8,message {system idle time in d(ays), h(ours), m(inutes) and s(econds)}
        9,label users 9,type integer 9,message {number of users currently logged on}
        10,label processes 10,type integer 10,message {number of processes}
        pollTimes {60 10 20 30 120 300 600}
        views {{indices {0 1 2 3 4 5 6 7 8 9 10} swap 1}}
        switches {-r 1 --remote 1}
    }

    set file [open system.htm]
    set data(helpText) [read $file]                                                           ;# initialize HTML help data from file
    close $file

    proc initialize {optionsName} {
        upvar $optionsName options
        variable remote
        variable channel

        if {![catch {set locator $options(--remote)}]||![catch {set locator $options(-r)}]} {                   ;# remote monitoring
            foreach {remote(protocol) remote(user) remote(host)} [network::parseRemoteLocator $locator] {}
            network::checkRemoteOutputEmptiness $remote(protocol) $remote(user) $remote(host)
            set data(identifier) system($remote(host))
            set file [open "| /usr/bin/$remote(protocol) -n -l $remote(user) $remote(host) cat /proc/sys/kernel/osrelease"]
            fileevent $file readable {set ::system::remote(busy) 0}
            vwait ::system::remote(busy)                                                               ;# do not hang user interface
        } else {
            set file [open /proc/sys/kernel/osrelease]
        }
        regexp {^[\d\.]+} [gets $file] version                                     ;# ignore extra characters, such as in 2.2.0-pre1
        if {[info exists remote]} {
            # detect errors early (but avoid write on pipe with no readers errors by reading whole data)
            if {[catch {read $file} message]||[catch {close $file} message]} {
                error "on remote host $remote(host) as user $remote(user): $message"
            }
        } else {
            set channel(release) $file
            set channel(version) [open /proc/sys/kernel/version]
            set channel(uptime) [open /proc/uptime]
            set channel(load) [open /proc/loadavg]
            set channel(cpu) [open /proc/cpuinfo]
        }
        if {[package vcompare $version 2]<0} {                                                               ;# check kernel version
            error {at least a version 2 kernel is needed}
        }
    }

    proc update {} {
        variable remote
        variable data
        variable channel

        if {[info exists remote]} {
            if {[info exists channel(remote)]} {
                gets $channel(remote) line(release)
                gets $channel(remote) line(version)
                gets $channel(remote) line(uptime)
                gets $channel(remote) line(load)
                while {[gets $channel(remote) string]>=0} {
                    lappend lines $string
                }
                read $channel(remote)                        ;# avoid write on pipe with no readers errors by reading remaining data
                if {[catch {close $channel(remote)} message]} {                          ;# communication error can be detected here
                    flashMessage "system error: $message"
                    unset line                                                                 ;# consider data corrupted as a whole
                    set lines {}
                }
                unset channel(remote)
                set remote(busy) 0
            } else {                                                       ;# start data gathering process in a non blocking fashion
                if {$remote(busy)} return                                           ;# core invocation while waiting for remote data
                set remote(busy) 1
                set file [open\
                    "| /usr/bin/$remote(protocol) -n -l $remote(user) $remote(host)\
                    cat /proc/sys/kernel/osrelease /proc/sys/kernel/version /proc/uptime /proc/loadavg /proc/cpuinfo"\
                ]
                # do not hang GUI, allow other modules updates
                fileevent $file readable "set ::system::channel(remote) $file; ::system::update"
                return                                                                                       ;# wait for remote data
            }                                                                                 ;# else continue below to process data
        } else {
            seek $channel(release) 0                                                                ;# rewind before retrieving data
            gets $channel(release) line(release)
            seek $channel(version) 0
            gets $channel(version) line(version)
            seek $channel(uptime) 0
            gets $channel(uptime) line(uptime)
            seek $channel(load) 0
            gets $channel(load) line(load)
            seek $channel(cpu) 0
            while {[gets $channel(cpu) string]>=0} {
                lappend lines $string
            }
        }
        if {[info exists line]} {
            set data(0,0) $line(release)
            # ignore heading (#n, SMP, ...) and day. example : #1 Fri Oct 6 22:01:22 CEST 2000
            set data(0,1) "[lrange $line(version) end-4 end-3], [lindex $line(version) end]"                      ;# month day, year
            set data(0,2) [lindex $line(version) end-2]                                                                  ;# HH:MM:SS
            set data(0,7) [formattedTime [expr {round([lindex $line(uptime) 0])}]]
            set data(0,8) [formattedTime [expr {round([lindex $line(uptime) 1])}]]
            scan $line(load) {%*f %*f %*f %u/%u} data(0,9) data(0,10)
            foreach string $lines {
                if {![regexp {^(.+?)\s+:\s+(.+?)$} $string dummy variable value]} continue
                switch $variable {
                    vendor_id {set data(0,3) $value}
                    {model name} {set data(0,4) $value}
                    {cpu MHz} {set data(0,5) $value}
                    bogomips {set data(0,6) $value}
                }
            }
        } else {                                                                                                       ;# data error
            array set data {0,0 ? 0,1 ? 0,2 ? 0,3 ? 0,4 ? 0,5 ? 0,6 ? 0,7 ? 0,8 ? 0,9 ? 0,10 ?}
        }
        incr data(updates)
    }

}
