/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <time.h>
#include <stdio.h>
/* other libs */
#include <apr.h>
#include <apr_pools.h>
#include <apr_strings.h>
/* local */
#include <libbttracker.h>

const btt_infohash new_btt_infohash = {
    BT_EMPTY_INFOHASH,     /* infohash */
    "\0",                  /* filename */
    0,                     /* file_size */
    0, 0, 0,               /* max uploaded/downloaded/left */
    0,                     /* min_left */
    0,                     /* hits */
    0, 0, 0,               /* peers, seeds, shields */
    0, 0, 0,               /* starts, stops, completes */
    0, 0, 0,               /* first_t, last_t, register_t */
    0, 0,                  /* first_peer_t, last_peer_t */
    0, 0                   /* first_seed_t, last_seed_t */
};

#ifdef BTT_WITH_LIBXML2
#define BT_INFOHASH_OFFSET(n) ( \
    (long)(&new_btt_infohash.n) - (long)(&new_btt_infohash) \
)
#define BT_INFOHASH_NODE(name,format,item) { \
    (xmlChar*)name, format, BT_INFOHASH_OFFSET(item) \
}

btt_xml_node infohash_nodes[] = {
    BT_INFOHASH_NODE("Filename",        "%s",           filename),
    BT_INFOHASH_NODE("Filesize",        BT_SIZE_T_FMT,  filesize),
    BT_INFOHASH_NODE("MaxUploaded",     BT_OFF_T_FMT,   max_uploaded),
    BT_INFOHASH_NODE("MaxDownloaded",   BT_OFF_T_FMT,   max_downloaded),
    BT_INFOHASH_NODE("MaxLeft",         BT_SIZE_T_FMT,  max_left),
    BT_INFOHASH_NODE("MinLeft",         BT_SIZE_T_FMT,  min_left),
    BT_INFOHASH_NODE("Hits",            "%"PRIu64,      hits),
    BT_INFOHASH_NODE("Peers",           "%"PRIu32,      peers),
    BT_INFOHASH_NODE("Seeds",           "%"PRIu32,      seeds),
    BT_INFOHASH_NODE("Shields",         "%"PRIu32,      shields),
    BT_INFOHASH_NODE("Starts",          "%"PRIu32,      starts),
    BT_INFOHASH_NODE("Stops",           "%"PRIu32,      stops),
    BT_INFOHASH_NODE("Completes",       "%"PRIu32,      completes),
    BT_INFOHASH_NODE("FirstT",          BT_TIME_T_FMT,  first_t),
    BT_INFOHASH_NODE("LastT",           BT_TIME_T_FMT,  last_t),
    BT_INFOHASH_NODE("RegisterT",       BT_TIME_T_FMT,  register_t),
    BT_INFOHASH_NODE("FirstPeerT",      BT_TIME_T_FMT,  first_peer_t),
    BT_INFOHASH_NODE("LastPeerT",       BT_TIME_T_FMT,  last_peer_t),
    BT_INFOHASH_NODE("FirstSeedT",      BT_TIME_T_FMT,  first_seed_t),
    BT_INFOHASH_NODE("LastSeedT",       BT_TIME_T_FMT,  last_seed_t),
    { NULL, NULL, 0 }
};
#endif

int btt_infohash2table(apr_pool_t* p, btt_infohash* infohash, char** result) {
    time_t now = time(NULL);

    char* rv = apr_psprintf(
        p,
        "<TR><TH>Infohash:</TH><TD>%s</TH></TR>\n"
        "<TR><TH>Filename:</TH><TD>%s</TD></TR>\n"
        "<TR><TH>File Size:</TH><TD>%s</TD></TR>\n"
        "<TR><TH>Announce Hits:</TH><TD>%"PRIu64"</TD></TR>\n"
        "<TR><TH>Max Uploaded/Downloaded:</TH><TD>%s/%s</TD></TR>\n"
        "<TR><TH>Min/Max Left:</TH><TD>%s/%s</TD></TR>\n"
        "<TR><TH>Peers/Seeds/Shields:</TH>"
        "<TD>%"PRIu32"/%"PRIu32"/%"PRIu32"</TD></TR>\n"
        "<TR><TH>Total "
        "<CODE>started</CODE>/<CODE>stopped</CODE>/<CODE>completed</CODE>"
        "Events:</TH>"
        "<TD>%"PRIu32"/%"PRIu32"/%"PRIu32"</TD></TR>\n"
        "<TR><TH>Date Added:</TH><TD>%s</TD></TR>\n"
        "<TR><TH>Date Registered:</TH><TD>%s</TD></TR>\n"
        "<TR><TH>Last Activity:</TH><TD>"BT_TIME_T_FMT"s</TD></TR>\n"
        "<TR><TH>First/Last Peer:</TH><TD>%s/%s</TD></TR>\n"
        "<TR><TH>First/Last Seed:</TH><TD>%s/%s</TD></TR>\n"
        "<TR><TH>Server Time:</TH><TD>%s</TD></TR>\n",
        bt_str_infohash(p, infohash->infohash), infohash->filename, 
        bt_nice_size(p, infohash->filesize), infohash->hits,
        bt_nice_size(p, infohash->max_uploaded),
        bt_nice_size(p, infohash->max_downloaded),
        bt_nice_size(p, infohash->min_left),
        bt_nice_size(p, infohash->max_left),
        infohash->peers, infohash->seeds, infohash->shields,
        infohash->starts, infohash->stops, infohash->completes,
        bt_nice_date(p, infohash->first_t),
        bt_nice_date(p, infohash->register_t),
        now - infohash->last_t,
        bt_strinterval(p, now - infohash->first_peer_t),
        bt_strinterval(p, now - infohash->last_peer_t),
        bt_strinterval(p, now - infohash->first_seed_t),
        bt_strinterval(p, now - infohash->last_seed_t),
        bt_nice_date(p, now)
    );
 
    *result = rv;
    return strlen(rv);
}


int btt_infohash2text(apr_pool_t* p, btt_infohash* infohash, char** result) {
    time_t now = time(NULL);

    char* rv = apr_psprintf(
        p,
        "Infohash:            %s\n"
        "Filename:            %s\n"
        "File Size:           %s\n"
        "Announce Hits:       %"PRIu64"\n"
        "Max Uploaded:        %s\n"
        "Mac Downloaded:      %s\n"
        "Min/Max Left:        %s/%s\n"
        "Peers:               %"PRIu32"\n"
        "  Seeds:             %"PRIu32"\n"
        "  Shields:           %"PRIu32"\n"
        "Events:\n"
        "  started:           %"PRIu32"\n"
        "  stopped:           %"PRIu32"\n"
        "  completed:         %"PRIu32"\n"
        "Date Added:          %s\n"
        "Date Registered:     %s\n"
        "Last Activity:       "BT_TIME_T_FMT"s\n"
        "First/Last Peer:     "BT_TIME_T_FMT"s/"BT_TIME_T_FMT"s\n"
        "First/Last Seed:     "BT_TIME_T_FMT"s/"BT_TIME_T_FMT"s\n"
        "Server Time:         %s\n"
        "\n",
        bt_str_infohash(p, infohash->infohash), infohash->filename, 
        bt_nice_size(p, infohash->filesize), infohash->hits,
        bt_nice_size(p, infohash->max_uploaded),
        bt_nice_size(p, infohash->max_downloaded),
        bt_nice_size(p, infohash->min_left), bt_nice_size(p, infohash->max_left),
        infohash->peers, infohash->seeds, infohash->shields,
        infohash->starts, infohash->stops, infohash->completes,
        bt_nice_date(p, infohash->first_t),
        bt_nice_date(p, infohash->register_t),
        now - infohash->last_t,
        now - infohash->first_peer_t, now - infohash->last_peer_t,
        now - infohash->first_seed_t, now - infohash->last_seed_t,
        bt_nice_date(p, now)
    );
 
    *result = rv;
    return strlen(rv);
}


int btt_infohash2scrape_verbose(
    apr_pool_t* p, btt_infohash* infohash, char** result
) {
    int complete;
    int len;

    if(infohash->shields < infohash->seeds)
        complete = infohash->seeds - infohash->shields;
    else
        complete = infohash->seeds;

    char* rv = apr_psprintf(
        p,
        "20:....................d8:completei%"PRIu32"e"
        "10:downloadedi%"PRIu32"e10"
        ":incompletei%"PRIu32"e4:name%zu:%s4:sizei"BT_SIZE_T_FMT"ee",
        complete, infohash->completes, infohash->peers - infohash->seeds,
        strlen(infohash->filename), infohash->filename, infohash->filesize
    );

    len = strlen(rv);
    memcpy(&(rv[3]), infohash->infohash, BT_INFOHASH_LEN);
    *result = rv;
    return len;
}

int btt_infohash2tr(
    apr_pool_t* p, btt_infohash* infohash, char* detail_url, char** result
) {
    char shieldstr[10] = "";
    const char* tr_class;
    char* rv;
    char* url = NULL;
    u_int32_t seeds, downloaders;
    time_t now = time(NULL);
    int len = 0;
 
    if(!infohash->peers)
        tr_class = "unpeered";
    else if(infohash->peers == infohash->seeds)
        tr_class = "allseeds";
    else if(infohash->seeds)
        tr_class = "seeded";
    else
        tr_class = "unseeded";

    seeds = infohash->seeds - infohash->shields;
    downloaders = infohash->peers - infohash->seeds;

    if(infohash->shields)
        sprintf(shieldstr, " (+%"PRIu32")", infohash->shields);

    if(detail_url && *detail_url) {
        len = strlen(detail_url);
        if(detail_url[len - 1] == '.') {
            if(infohash->filename[0]) {
                url = apr_pstrdup(p, detail_url);
                url[len - 1] = 0;
                url = apr_psprintf(
                    p, "%s%s.torrent", url,
                    bt_uri_escape(
                        p, infohash->filename, strlen(infohash->filename)
                    )
                );
            }
        } else if(detail_url[len - 1] == '/') {
            if(infohash->filename[0]) {
                url = apr_psprintf(
                    p, "%s%s", detail_url,
                    bt_uri_escape(
                        p, infohash->filename, strlen(infohash->filename))
                );
            }
        } else {
            url = apr_psprintf(
                p, "%s?info_hash=%s", detail_url,
                bt_uri_escape(p, infohash->infohash, BT_INFOHASH_LEN)
            );
        }
    }

    if(url && *url) {
        rv = apr_psprintf(
            p,
            "<TR CLASS=\"%s\">\n"
            " <TD><A HREF=\"%s\"><CODE>%s</CODE></A></TD>\n"
            " <TD>%s</TD>\n"
            " <TD CLASS=\"numeric\">"BT_TIME_T_FMT"s</TD>\n"
            " <TD CLASS=\"numeric\">%"PRIu32"%s</TD>\n"
            " <TD CLASS=\"numeric\">%"PRIu32"</TD>\n"
            " <TD CLASS=\"numeric\">%"PRIu32"</TD>\n"
            "</TR>\n",
            tr_class,
            url, bt_str_infohash(p, infohash->infohash), 
            infohash->filename ? infohash->filename : "-",
            now - infohash->last_t,
            seeds, shieldstr, downloaders, infohash->completes
        );
    } else {
        rv = apr_psprintf(
            p,
            "<TR CLASS=\"%s\">\n"
            " <TD><CODE>%s</CODE></TD>\n"
            " <TD>%s</TD>\n"
            " <TD CLASS=\"numeric\">"BT_TIME_T_FMT"s</TD>\n"
            " <TD CLASS=\"numeric\">%"PRIu32"%s</TD>\n"
            " <TD CLASS=\"numeric\">%"PRIu32"</TD>\n"
            " <TD CLASS=\"numeric\">%"PRIu32"</TD>\n"
            "</TR>\n",
            tr_class,
            bt_str_infohash(p, infohash->infohash), 
            infohash->filename ? infohash->filename : "-",
            now - infohash->last_t,
            seeds, shieldstr, downloaders, infohash->completes
        );
    }
 
    *result = rv;
    return strlen(rv);
}

int btt_infohash2scrape(apr_pool_t* p, btt_infohash* infohash, char** result) {
    uint32_t complete;
    int len;
 
    if(infohash->shields < infohash->seeds)
        complete = infohash->seeds - infohash->shields;
    else
        complete = infohash->seeds;

    char* rv = apr_psprintf(
        p,
        "20:....................d8:completei%"PRIu32"e"
        "10:downloadedi%"PRIu32"e10"
        ":incompletei%"PRIu32"ee",
        complete, infohash->completes, infohash->peers - infohash->seeds
    );

    len = strlen(rv);
    memcpy(&(rv[3]), infohash->infohash, BT_INFOHASH_LEN);
    
    *result = rv;
    return len;
}

#ifdef BTT_WITH_LIBXML2
xmlNodePtr btt_infohash2nodes(
    apr_pool_t* p, xmlDocPtr document, btt_infohash *infohash
) {
    xmlNodePtr hashnode = xmlNewDocNode(
        document, NULL, (xmlChar*)"Infohash", NULL
    );
    xmlSetProp(
        hashnode, (xmlChar*)"ID",
        (xmlChar*)bt_str_infohash(p, infohash->infohash)
    );
    btt_struct2xml(hashnode, infohash_nodes, infohash);
    return hashnode;
}

xmlDocPtr btt_infohash2xmlDoc(apr_pool_t* p, btt_infohash *infohash) {
    xmlDocPtr document = xmlNewDoc((xmlChar*)"1.0");
    xmlNodePtr hashnode = btt_infohash2nodes(p, document, infohash);
    xmlDocSetRootElement(document, hashnode);
    return document;
}

int btt_infohash2xml(apr_pool_t* p, btt_infohash* infohash, char** result) {
    xmlChar* xmlresult;
    int junk;
    char* rv;
    char* pos;
    xmlDocPtr document = btt_infohash2xmlDoc(p, infohash);
    xmlDocDumpFormatMemory(document, &xmlresult, &junk, 1);
    pos = index((char*)xmlresult, '\n');
    pos++;
    rv = apr_pstrdup(p, pos);
    xmlFree(xmlresult);
    xmlFreeDoc(document);
    *result = rv;
    return strlen(rv);
}

btt_infohash btt_xml2infohash(apr_pool_t* p, xmlNodePtr top) {
    btt_infohash rv = new_btt_infohash;
    char* hash;
    xmlAttrPtr id = xmlHasProp(top, BAD_CAST "ID");
    xmlChar* idstr;
    if(id) {
        idstr = xmlNodeGetContent(id->children);
        btt_xml2struct(top, infohash_nodes, &rv);
        hash = bt_infohash_str(p, (char*)idstr);
        xmlFree(idstr);
        memcpy(rv.infohash, hash, BT_INFOHASH_LEN);
    } else {
        fprintf(stderr, "bt_xml2infohash: Failed to get ID property of node!\n");
        rv = new_btt_infohash;
    }
    return rv;    
}
#endif
