/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2004, 2005, 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* libc */
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <inttypes.h>
/* php */
#include "php.h"
#include "SAPI.h"
/* suppress incompatible Apache regex symbols */
#ifndef _PCREPOSIX_H
#define _PCREPOSIX_H
#endif
#include "php_apache.h"
/* mod_bt */
#include <libbttracker.h>
#include "apache2/mod_bt.h"
/* apache2 */
#include "apr.h"
#include "apr_strings.h"
#include "httpd.h"
#include "http_config.h"
#include "http_log.h"
#include "http_request.h"
#include "http_protocol.h"
#include "http_core.h"
#include "util_filter.h"
#include "mpm.h"
/* libdb */
#include <db.h>

zend_module_entry php_mod_bt_module_entry;

static btt_tracker* get_tracker(php_struct* ctx)
{
 modbt_config_t* cfg = ap_get_module_config(ctx->r->server->module_config, &bt_module);
 return cfg->tracker;
}

ZEND_FUNCTION(tracker_config)
{
 php_struct *ctx = SG(server_context);
 btt_tracker* tracker = get_tracker(ctx);

 if(ZEND_NUM_ARGS())
 {
  WRONG_PARAM_COUNT;
 }

 if(array_init(return_value)==FAILURE)
 {
  RETURN_FALSE;
 }
 
 add_assoc_stringl(return_value, "stylesheet", tracker->c->stylesheet, strlen(tracker->c->stylesheet), 1);
 add_assoc_stringl(return_value, "db_dir", tracker->c->db_dir, strlen(tracker->c->db_dir), 1);
 add_assoc_long(return_value, "flags", tracker->c->flags);
 add_assoc_long(return_value, "random_retry", tracker->c->random_retry);
 add_assoc_long(return_value, "return_max", tracker->c->return_max);
 add_assoc_long(return_value, "return_interval", tracker->c->return_interval);
 add_assoc_long(return_value, "return_peer_factor", tracker->c->return_peer_factor);
 add_assoc_long(return_value, "hash_watermark", tracker->c->hash_watermark);
 add_assoc_long(return_value, "hash_min_age", tracker->c->hash_min_age);
 add_assoc_long(return_value, "hash_max_age", tracker->c->hash_max_age);
 if(tracker->c->parent_server)
  add_assoc_stringl(return_value, "parent_server", tracker->c->parent_server, strlen(tracker->c->parent_server), 1);
}

ZEND_FUNCTION(tracker_stats)
{
 php_struct *ctx = SG(server_context);
 btt_tracker* tracker = get_tracker(ctx);

 if(ZEND_NUM_ARGS())
 {
  WRONG_PARAM_COUNT;
 }

 if(array_init(return_value)==FAILURE)
 {
  RETURN_FALSE;
 }

 add_assoc_long(return_value, "num_children", tracker->s->num_children);
 add_assoc_long(return_value, "num_requests", tracker->s->num_requests);
 add_assoc_long(return_value, "num_hashes", tracker->s->num_hashes);
 add_assoc_long(return_value, "num_peers", tracker->s->num_peers);
 add_assoc_long(return_value, "announces", tracker->s->announces);
 add_assoc_long(return_value, "scrapes", tracker->s->scrapes);
 add_assoc_long(return_value, "full_scrapes", tracker->s->full_scrapes);
 add_assoc_long(return_value, "bad_announces", tracker->s->bad_announces);
 add_assoc_long(return_value, "bad_scrapes", tracker->s->bad_scrapes); 
 add_assoc_long(return_value, "start_t", tracker->s->start_t);
 add_assoc_long(return_value, "master_pid", tracker->s->master_pid);
 add_assoc_long(return_value, "server_time", tracker->s->server_time);
}

ZEND_FUNCTION(tracker_flags)
{
 int i;
	
 if(ZEND_NUM_ARGS())
 {
  WRONG_PARAM_COUNT;
 }

 if(array_init(return_value)==FAILURE)
 {
  RETURN_FALSE;
 }
 
 for(i=0;btt_tracker_flags[i].flag;i++)
  add_index_stringl(return_value, btt_tracker_flags[i].flag, btt_tracker_flags[i].config_name, strlen(btt_tracker_flags[i].config_name), 1);
}

zval* php_mod_bt_convert_infohash(apr_pool_t* p, btt_infohash* hash)
{
 zval* rv = NULL;
 MAKE_STD_ZVAL(rv);
 char* hash_str;
 char* size_str;

 if(array_init(rv) == FAILURE)
 {
/*  php_error_docref(NULL TSRMLS_CC, E_WARNING, "php_mod_bt_convert_infohash(): array_init() failed"); */
  return NULL;
 }

 hash_str = bt_str_infohash(p, hash->infohash);
 add_assoc_stringl(rv, "infohash", hash_str, strlen(hash_str), 1);
 add_assoc_stringl(rv, "filename", hash->filename, strlen(hash->filename), 1);
 size_str = apr_psprintf(p, BT_SIZE_T_FMT, hash->filesize);
 add_assoc_stringl(rv, "filesize", size_str, strlen(size_str), 1);
 size_str = apr_psprintf(p, BT_OFF_T_FMT, hash->max_uploaded);
 add_assoc_stringl(rv, "max_uploaded", size_str, strlen(size_str), 1);
 size_str = apr_psprintf(p, BT_OFF_T_FMT, hash->max_downloaded);
 add_assoc_stringl(rv, "max_downloaded", size_str, strlen(size_str), 1);
 size_str = apr_psprintf(p, BT_SIZE_T_FMT, hash->max_left);
 add_assoc_stringl(rv, "max_left", size_str, strlen(size_str), 1);
 size_str = apr_psprintf(p, BT_SIZE_T_FMT, hash->min_left);
 add_assoc_stringl(rv, "min_left", size_str, strlen(size_str), 1);
 add_assoc_long(rv, "hits", hash->hits);
 add_assoc_long(rv, "peers", hash->peers);
 add_assoc_long(rv, "seeds", hash->seeds);
 add_assoc_long(rv, "shields", hash->shields);
 add_assoc_long(rv, "starts", hash->starts);
 add_assoc_long(rv, "stops", hash->stops);
 add_assoc_long(rv, "completes", hash->completes);
 add_assoc_long(rv, "first_t", hash->first_t);
 add_assoc_long(rv, "last_t", hash->last_t);
 add_assoc_long(rv, "register_t", hash->register_t);
 add_assoc_long(rv, "first_peer_t", hash->first_peer_t);
 add_assoc_long(rv, "last_peer_t", hash->last_peer_t);
 add_assoc_long(rv, "first_seed_t", hash->first_seed_t);
 add_assoc_long(rv, "last_seed_t", hash->last_seed_t);

 return rv;
}

zval* php_mod_bt_infohash(btt_tracker* tracker, char *in_hash)
{
 zval* rv = NULL;
 apr_pool_t* p = NULL;
 btt_infohash* hash = NULL;
 DB_TXN* txn = NULL;
 DBT hash_key;
 char *infohash;
 
 apr_pool_create(&p, NULL);
 
 infohash = bt_infohash_str(p, in_hash);
 
 if((btt_txn_start(tracker, NULL, &txn, 0)) != 0)
 {
/*  php_error_docref(NULL TSRMLS_CC, E_WARNING, "php_mod_bt_infohash(): txn_start() failed"); */
  apr_pool_destroy(p);
  return NULL;
 }
 
 bzero(&hash_key, sizeof(hash_key));

 hash_key.data = infohash;
 hash_key.size = BT_INFOHASH_LEN;
 hash_key.ulen = BT_INFOHASH_LEN;
 hash_key.flags = DB_DBT_USERMEM;
 
 if(!(hash = btt_txn_load_hash(tracker, p, txn, &hash_key, 0, 0, 0)))
 {
  txn->abort(txn);
  apr_pool_destroy(p);
  return NULL;
 }
 
 if((txn->commit(txn, 0)) != 0)
 {
  txn->abort(txn);
  apr_pool_destroy(p);
/*  php_error_docref(NULL TSRMLS_CC, E_WARNING, "php_mod_bt_infohash(): txn->commit() failed"); */
  return NULL;
 }

 rv = php_mod_bt_convert_infohash(p, hash);
 apr_pool_destroy(p);
 return rv;
}

ZEND_FUNCTION(tracker_infohash)
{
 zval* rv;
 zval** hasharg;
 php_struct *ctx = SG(server_context);
 btt_tracker* tracker = get_tracker(ctx);

 if(ZEND_NUM_ARGS() != 1)
 {
  php_error_docref(NULL TSRMLS_CC, E_WARNING, "php_mod_bt_infohash(): no hash specified");
  WRONG_PARAM_COUNT;
 }

 if(zend_get_parameters_ex(1, &hasharg) == FAILURE)
 {
  php_error_docref(NULL TSRMLS_CC, E_WARNING, "php_mod_bt_infohash(): no hash specified");
  WRONG_PARAM_COUNT;
 }
 
 if(Z_STRLEN_PP(hasharg) != (BT_INFOHASH_LEN * 2))
 {
  php_error_docref(NULL TSRMLS_CC, E_WARNING, "php_mod_bt_infohash(): bad infohash length");
  RETURN_FALSE;
 }

 if((rv = php_mod_bt_infohash(tracker, Z_STRVAL_PP(hasharg))))
 {
  *return_value = *rv;
 }
 else
 {
  RETURN_FALSE;
 }
}

ZEND_FUNCTION(tracker_infohashes)
{
 php_struct *ctx = SG(server_context);
 btt_tracker* tracker = get_tracker(ctx);
 DB_TXN* txn = NULL;
 DBC* cur = NULL;
 DBT key;
 DBT val;
 apr_pool_t*	p = NULL;
 char key_data[BT_INFOHASH_LEN + 1];
 char* key_str;
 key_data[BT_INFOHASH_LEN] = 0;
 btt_infohash	val_data;
 int ret = 0;
 int n = 0;
 zval* chash;

 if(ZEND_NUM_ARGS())
 {
  WRONG_PARAM_COUNT;
 }

 if(array_init(return_value)==FAILURE)
 {
  RETURN_FALSE;
 }

 if((ret = btt_txn_start(tracker, NULL, &txn, 0)) != 0)
 {
  tracker->db.env->err(tracker->db.env, ret, "tracker_infohashes(): bt_txn_start()");
  goto err;
 }
	
 if((ret = tracker->db.hashes->cursor(tracker->db.hashes, txn, &cur, 0)) != 0)
 {
  tracker->db.env->err(tracker->db.env, ret, "tracker_infohashes(): cursor()");
  goto err;
 }

 bzero(&key, sizeof(key));
 bzero(&val, sizeof(val));
	
 key.data = key_data;
 key.size = 0;
 key.ulen = BT_INFOHASH_LEN;
 key.flags = DB_DBT_USERMEM;
	
 val.data = &val_data;
 val.size = 0;
 val.ulen = sizeof(val_data);
 val.flags = DB_DBT_USERMEM;
	
 apr_pool_create(&p, NULL);
	
 while(!ret)
 {
  if((ret = cur->c_get(cur, &key, &val, DB_NEXT)) == 0)
  {
   if((chash = php_mod_bt_convert_infohash(p, &val_data)))
   {
   	key_str = bt_str_infohash(p, key_data);
    add_assoc_zval(return_value, key_str, chash);
    n++;
   }
  }
  else
  {
   if(ret != DB_NOTFOUND)
   {
    tracker->db.env->err(tracker->db.env, ret, "tracker_infohashes(): c_get()");
    goto err;
   }
  }
 }
	
 cur->c_close(cur);
 cur = NULL;
	
 if((ret = txn->commit(txn, 0)) != 0)
 {
  tracker->db.env->err(tracker->db.env, ret, "tracker_infohashes(): commit()");
  goto err;
 }
 
 txn = NULL;
 goto done;

 err:

 if(cur)
 {
  cur->c_close(cur);
  cur = NULL;
 }
 
 if(txn)
 {
  txn->abort(txn);
  txn = NULL;
 }

 if(p)
 {
  apr_pool_destroy(p);
  p = NULL;
 }
 
 RETURN_FALSE;
 
 done:

 if(p)
 {
  apr_pool_destroy(p);
  p = NULL;
 }
}

zval* php_mod_bt_convert_peer(apr_pool_t* p, btt_peer* peer)
{
 zval* rv = NULL;
 MAKE_STD_ZVAL(rv);
 char* hash_str;
 char* peer_str;
 char* ip_str;
 char* size_str;

 if(array_init(rv) == FAILURE)
 {
/*  php_error_docref(NULL TSRMLS_CC, E_WARNING, "php_mod_bt_convert_peer(): array_init() failed"); */
  return NULL;
 }

 hash_str = bt_str_infohash(p, peer->infohash);
 peer_str = bt_str_peerid(p, peer->peerid);
 
 add_assoc_stringl(rv, "peerid", peer_str, strlen(peer_str), 1);
 add_assoc_stringl(rv, "infohash", hash_str, strlen(hash_str), 1);
 add_assoc_stringl(rv, "ua", peer->ua, strlen(peer->ua), 1);
 add_assoc_stringl(rv, "event", peer->event, strlen(peer->event), 1);
 add_assoc_long(rv, "flags", peer->flags);

 ip_str = apr_psprintf(p, "%s:%u", inet_ntoa(peer->address.sin_addr), ntohs(peer->address.sin_port));
 add_assoc_stringl(rv, "address", ip_str, strlen(ip_str), 1);

 ip_str = apr_psprintf(p, "%s:%u", inet_ntoa(peer->real_address.sin_addr), ntohs(peer->real_address.sin_port));
 add_assoc_stringl(rv, "real_address", ip_str, strlen(ip_str), 1);

 add_assoc_long(rv, "first_t", peer->first_t);
 add_assoc_long(rv, "last_t", peer->last_t);
 add_assoc_long(rv, "first_serve_t", peer->first_serve_t);
 add_assoc_long(rv, "last_serve_t", peer->last_serve_t);
 add_assoc_long(rv, "complete_t", peer->complete_t);
 add_assoc_long(rv, "return_interval", peer->return_interval);
 add_assoc_long(rv, "hits", peer->hits);
 add_assoc_long(rv, "serves", peer->serves);
 add_assoc_long(rv, "num_want", peer->num_want);
 add_assoc_long(rv, "num_got", peer->num_got);
 add_assoc_long(rv, "announce_bytes", peer->announce_bytes);
 size_str = apr_psprintf(p, BT_OFF_T_FMT, peer->uploaded);
 add_assoc_stringl(rv, "uploaded", size_str, strlen(size_str), 1);
 size_str = apr_psprintf(p, BT_SIZE_T_FMT, peer->downloaded);
 add_assoc_stringl(rv, "downloaded", size_str, strlen(size_str), 1);
 size_str = apr_psprintf(p, BT_SIZE_T_FMT, peer->left);
 add_assoc_stringl(rv, "left", size_str, strlen(size_str), 1);
 
 return rv;
}

ZEND_FUNCTION(tracker_peers)
{
 apr_pool_t* p = NULL;
 DB_TXN* txn = NULL;
 DBT key;
 DBT val;
 DBC* cur = NULL;
 char *infohash;
 int ret = 0;
 int n = 0;

 zval** hasharg;
 zval* cpeer;
 php_struct *ctx = SG(server_context);
 btt_tracker* tracker = get_tracker(ctx);
 btt_peer val_data;
 char* key_str;

 if(ZEND_NUM_ARGS() != 1)
 {
  php_error_docref(NULL TSRMLS_CC, E_WARNING, "tracker_peers(): no hash specified");
  WRONG_PARAM_COUNT;
 }

 if(zend_get_parameters_ex(1, &hasharg) == FAILURE)
 {
  php_error_docref(NULL TSRMLS_CC, E_WARNING, "tracker_peers(): no hash specified");
  WRONG_PARAM_COUNT;
 }
 
 if(Z_STRLEN_PP(hasharg) != (BT_INFOHASH_LEN * 2))
 {
  php_error_docref(NULL TSRMLS_CC, E_WARNING, "tracker_peers(): bad infohash length");
  RETURN_FALSE;
 }

 if(array_init(return_value)==FAILURE)
 {
  RETURN_FALSE;
 }

 apr_pool_create(&p, NULL);
 infohash = bt_infohash_str(p, Z_STRVAL_PP(hasharg));
 
 if((ret = btt_txn_start(tracker, NULL, &txn, 0)) != 0)
 {
  tracker->db.env->err(tracker->db.env, ret, "tracker_peers: bt_txn_start()");
  goto err;
 }

 if((ret = tracker->db.index->cursor(tracker->db.index, txn, &cur, 0)) != 0)
 {
  tracker->db.env->err(tracker->db.env, ret, "tracker_peers: cursor()");
  goto err;
 }
	
 key.data = infohash;
 key.size = BT_INFOHASH_LEN;
 key.ulen = BT_INFOHASH_LEN;
 key.flags = DB_DBT_USERMEM;
	
 val.data = &val_data;
 val.size = 0;
 val.ulen = sizeof(val_data);
 val.flags = DB_DBT_USERMEM;

 ret = cur->c_get(cur, &key, &val, DB_SET);
	
 while(!ret)
 {
  if((cpeer = php_mod_bt_convert_peer(p, &val_data)))
  {
   key_str = bt_str_peerid(p, val_data.peerid);
   add_assoc_zval(return_value, key_str, cpeer);
   n++;
  }

  ret = cur->c_get(cur, &key, &val, DB_NEXT_DUP);
 }
	
 if(ret != DB_NOTFOUND)
 {
  tracker->db.env->err(tracker->db.env, ret, "tracker_peers(): c_get()");
  goto err;
 }
	
 cur->c_close(cur);
 cur = NULL;
	
 if((ret = txn->commit(txn, 0)) != 0)
 {
  tracker->db.env->err(tracker->db.env, ret, "tracker_peers(): commit()");
  goto err;
 }

 txn = NULL;
 
 goto done;

 err:
 
 if(cur)
 {
  cur->c_close(cur);
  cur = NULL;
 }
 
 if(txn)
 {
  txn->abort(txn);
  txn = NULL;
 }
 
 if(p)
 {
  apr_pool_destroy(p);
  p = NULL;
 }
 
 RETURN_FALSE;

 done:
 if(p)
 {
  apr_pool_destroy(p);
  p = NULL;
 }
}

ZEND_FUNCTION(tracker_peer_flags)
{
 int i;
	
 if(ZEND_NUM_ARGS())
 {
  WRONG_PARAM_COUNT;
 }

 if(array_init(return_value)==FAILURE)
 {
  RETURN_FALSE;
 }
 
 for(i=0;btt_peer_flags[i].flag;i++)
  add_index_stringl(return_value, btt_peer_flags[i].flag, btt_peer_flags[i].config_name, strlen(btt_peer_flags[i].config_name), 1);
}

static zend_function_entry php_mod_bt_functions[] =
{
 ZEND_FE(tracker_config, NULL)
 ZEND_FE(tracker_stats, NULL)
 ZEND_FE(tracker_flags, NULL)
 ZEND_FE(tracker_infohash, NULL)
 ZEND_FE(tracker_infohashes, NULL)
 ZEND_FE(tracker_peers, NULL)
 ZEND_FE(tracker_peer_flags, NULL)
 { NULL, NULL, NULL }
};

/* compiled module information */
zend_module_entry php_mod_bt_module_entry =
{
    STANDARD_MODULE_HEADER,
    "mod_bt BitTorrent Tracker API",
    php_mod_bt_functions, /* functions */
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL,
    NO_VERSION_YET,
    STANDARD_MODULE_PROPERTIES
};

/* implement standard "stub" routine to introduce ourselves to Zend */
#if COMPILE_DL
ZEND_GET_MODULE(php_mod_bt)
#endif






/* declaration of functions to be exported */
/*
ZEND_FUNCTION(first_module);
*/

/* compiled function list so Zend knows what's in this module */
/*
zend_function_entry firstmod_functions[] =
{
    ZEND_FE(first_module, NULL)
    {NULL, NULL, NULL}
};
*/

/* compiled module information */
/*
zend_module_entry firstmod_module_entry =
{
    STANDARD_MODULE_HEADER,
    "First Module",
    firstmod_functions,
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL,
    NO_VERSION_YET,
    STANDARD_MODULE_PROPERTIES
};
*/

/* implement standard "stub" routine to introduce ourselves to Zend */
/*
#if COMPILE_DL
ZEND_GET_MODULE(firstmod)
#endif
*/

/* implement function that is meant to be made available to PHP */
/*
ZEND_FUNCTION(first_module)
{
    long parameter;

    if (zend_parse_parameters(ZEND_NUM_ARGS() TSRMLS_CC, "l", &parameter) == FAILURE) {
        return;
    }

    RETURN_LONG(parameter);
}
*/
