#include "udm_config.h"

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <signal.h>
#include <sys/types.h>
#include <errno.h>

#ifdef HAVE_WINSOCK_H
#include <winsock.h>
#endif
#ifdef HAVE_IO_H   /* for Win */
#include <io.h>
#endif
#ifdef HAVE_PROCESS_H
#include <process.h>
#endif

#ifdef HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_SYS_PARAM_H
#include <sys/param.h>
#endif
#ifdef HAVE_SYS_WAIT_H
#include <sys/wait.h>
#endif
#ifdef   HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif
#ifdef HAVE_MMAN_H
#include <sys/mman.h>
#endif
#ifdef HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif
#ifdef HAVE_NETINET_IN_SYSTM_H
#include <netinet/in_systm.h>
#endif
#ifdef HAVE_NETINET_IP_H
#include <netinet/ip.h>
#endif
#ifdef HAVE_NETINET_TCP_H
#include <netinet/tcp.h>
#endif
#ifdef HAVE_NETDB_H
#include <netdb.h>
#endif

#include "udm_common.h"
#include "udm_spell.h"
#include "udmsearch.h"

#include <arpa/inet.h>


#define UDM_SPELL_MAXCLNTS 32
#ifndef MSG_WAITALL
#define MSG_WAITALL 0
#endif

const char *config_name= UDM_CONF_DIR "/spelld.conf";
UDM_AGENT *Indexer;

static char pidname[1024];
static UDM_ENV * Conf=NULL;

static void exitproc(void){
	unlink(pidname);
}

static char time_pid[100];
static char * time_pid_info(void){
	struct tm * tim;
	time_t t;
	t=time(NULL);
	tim=localtime(&t);
	strftime(time_pid,sizeof(time_pid),"%a %d %T",tim);
	sprintf(time_pid+strlen(time_pid)," [%d]",(int)getpid());
	return(time_pid);
}

/********************* SIG Handlers ****************/
static void sighandler(int sign);

static void init_signals(void){
	/* Set up signals handler*/
	signal(SIGPIPE,sighandler);
	signal(SIGHUP,sighandler);
	signal(SIGINT,sighandler);
	signal(SIGTERM,sighandler);
}


static void sighandler(int sign){

	fprintf(stderr,"%s Signal %d arrived\n",time_pid_info(),sign);
	switch(sign){
		case SIGPIPE:
			fprintf(stderr,"%s FATAL: This should not ever happen! Broken pipe!\n",time_pid_info());
			break;
		case SIGHUP: /* reload config */
			UdmFreeAgent(Indexer);
			UdmFreeEnv(Conf);
			Conf = UdmAllocEnv();
			UdmLoadConfig(Conf,config_name,0,UDM_FLAG_SPELL);
			if(UdmEnvErrCode(Conf)){
			  fprintf(stderr,"%s\n",UdmEnvErrMsg(Conf));
			  UdmFreeEnv(Conf);
			  unlink(pidname);
			  exit(1);
			}
			Indexer = UdmAllocAgent(Conf, 0, UDM_OPEN_MODE_READ);
			if (Conf->ispell_mode & UDM_ISPELL_MODE_DB) {
			  if (UdmDBImportAffixes(Indexer,Indexer->charset)) exit(1); /* fixme: Indexer.charset undefined */
			  if (UdmImportDictionaryFromDB(Indexer)) exit(1);
			  if(Conf->nspell)UdmSortDictionary(Indexer->Conf);
			  UdmSortAffixes(Indexer->Conf);
			}

			fprintf(stderr, "%s spelld - langs: %d ", time_pid_info(), Conf->nLang);
			fprintf(stderr, "nspell: %d ", Conf->nspell);
			fprintf(stderr, "naffixes: %d\n", Conf->naffixes);
			break;
		default:
			fprintf(stderr,"%s Shutdown\n",time_pid_info());
			exit(0);
	}
	init_signals();
}

#define senderror(line) fprintf(stderr,"%s send() @%d error %d %s\n",time_pid_info(),line,errno,strerror(errno)); \
      exit(1);


/*************************************************************/

int main(int argc, char **argv) {
  struct sockaddr_in server_addr, client_addr;
  int s, nport;
  int pid_fd;
  int i, j, on = 1;
  char pidbuf[1024];
  char buf[1024];


  UdmInit(); /* Initialize library */
  Conf = UdmAllocEnv();

  UdmLoadConfig(Conf,config_name,0,UDM_FLAG_SPELL);

  if(UdmEnvErrCode(Conf)){
    fprintf(stderr,"%s\n",UdmEnvErrMsg(Conf));
    UdmFreeEnv(Conf);
    exit(1);
  }

  /* Check that another instance isn't running */
  /* and create PID file.                       */

  sprintf(pidname,"%s%s",Conf->vardir,"spelld.pid");
  pid_fd = open(pidname,O_CREAT|O_EXCL|O_WRONLY,0644);
  if(pid_fd < 0){
    fprintf(stderr,"%s Can't create '%s': %s\n", time_pid_info(), pidname, strerror(errno));
    if(errno == EEXIST){
      fprintf(stderr,"It seems that another spelld is already running!\n");
      fprintf(stderr,"Remove '%s' if it is not true.\n",pidname);
    }
    exit(1);
  }
  sprintf(pidbuf,"%d\n",(int)getpid());
  write(pid_fd,&pidbuf,strlen(pidbuf));

  atexit(&exitproc);
  init_signals();


  if (Conf->ispell_mode & UDM_ISPELL_MODE_SERVER) {
    fprintf(stderr, "It's impossible to user IspellMode server in spelld config !\nTry IspellMode text or db.\n");
    UdmFreeEnv(Conf);
    exit(1);
  }

  /* load ispell data in IspellMode db */
  Indexer = UdmAllocAgent(Conf, 0, UDM_OPEN_MODE_READ);
  if (Conf->ispell_mode & UDM_ISPELL_MODE_DB) {
    if (UdmDBImportAffixes(Indexer,Indexer->charset)) return(1); /* fixme: Indexer.charset undefined */
    if (UdmImportDictionaryFromDB(Indexer)) return(1);
    if(Conf->nspell)UdmSortDictionary(Indexer->Conf);
    UdmSortAffixes(Indexer->Conf);
  }

  fprintf(stderr, "%s spelld - langs: %d ", time_pid_info(), Conf->nLang);
  fprintf(stderr, "nspell: %d ", Conf->nspell);
  fprintf(stderr, "naffixes: %d\n", Conf->naffixes);

  nport = UDM_SPELL_PORT;
  nport = htons((u_short) nport);

  if ((s = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    fprintf(stderr,"%s socket() error %d\n",time_pid_info(),errno);
    exit(1);
  }
  
  if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&on, sizeof(on)) != 0){
	fprintf(stderr,"%s setsockopt() error %d\n",time_pid_info(),errno);
	exit(1);
  }


  bzero(&server_addr, sizeof(server_addr));
  server_addr.sin_family = AF_INET;
  server_addr.sin_addr.s_addr = htonl(INADDR_ANY);
  server_addr.sin_port = nport;

  if (bind(s, (struct sockaddr *)&server_addr, sizeof(server_addr)) == -1) {
    fprintf(stderr,"%s bind() error %d %s\n",time_pid_info(),errno,strerror(errno));
    exit(1);
  }

  if (listen(s, UDM_SPELL_MAXCLNTS) == -1) {
    fprintf(stderr,"%s listen() error %d %s\n",time_pid_info(),errno,strerror(errno));
    exit(1);
  }

  while(1) {
    socklen_t addrlen;
    size_t wlen;
    int ns, nbytes, pid;
    int status;
    UDM_AFFIX *Affix;
    
    bzero(&client_addr, addrlen = sizeof(client_addr));

    if ((ns = accept(s, (struct sockaddr *) &client_addr, &addrlen)) == -1) {
      fprintf(stderr,"%s accept() error %d %s\n",time_pid_info(),errno,strerror(errno));
      exit(1);
    }

    fprintf(stderr, "Client = %s\n", inet_ntoa(client_addr.sin_addr));
    
    if ((pid = fork() ) == -1) {
      fprintf(stderr,"%s fork() error %d %s\n",time_pid_info(),errno,strerror(errno));
      exit(1);
    }
    
    if (pid == 0) { /* child process */
      close(s);
      
      if ((nbytes = recvall(ns, buf, 4, 0))) {
	if (strncmp(buf, "HELO", 4) == 0) { /* indexer get all data */

	  if (send(ns, (void *)&Conf->nLang, sizeof(Conf->nLang), 0) < 0) {
	    senderror(__LINE__-1);
	  }

	  Affix = (UDM_AFFIX *)Conf->Affix;
	  if (send(ns, (void *)&Conf->naffixes, sizeof(Conf->naffixes), 0) < 0) {
	    senderror(__LINE__-1);
	  }
	  for(i = 0; i < Conf->naffixes; i++) {
	    if (send(ns, (void *)&Affix[i], sizeof(UDM_AFFIX), 0) < 0) {
	      senderror(__LINE__-1);
	    }
	  }
	  if (send(ns, (void *)&Conf->nspell, sizeof(Conf->nspell), 0) < 0) {
	    senderror(__LINE__-1);
	  }
	  for(j = 0; j < Conf->nspell; j++) {
	    if (send(ns, (void *)&Conf->Spell[j], sizeof(UDM_SPELL), 0) < 0) {
	      senderror(__LINE__-1);
	    }
	    wlen = strlen(Conf->Spell[j].word);
	    if (send(ns, (void *)&wlen, sizeof(wlen), 0) < 0) {
	      senderror(__LINE__-1);
	    }
	    if (send(ns, (void *)Conf->Spell[j].word, wlen, 0) < 0) {
	      senderror(__LINE__-1);
	    }
	  }
	  for(i = 0; i < Conf->nLang; i++) {
	    if (send(ns, (void *)&Conf->PrefixTree[i], sizeof(Tree_struct), 0) < 0) {
	      senderror(__LINE__-1);
	    }
	    if (send(ns, (void *)&Conf->SuffixTree[i], sizeof(Tree_struct), 0) < 0) {
	      senderror(__LINE__-1);
	    }
	    if (send(ns, (void *)&Conf->SpellTree[i], sizeof(Tree_struct), 0) < 0) {
	      senderror(__LINE__-1);
	    }
	  }
	} else if (strncmp(buf, "CLIL", 4) == 0) { /* search.cgi request to Normalize word */
	  size_t wordlen;
	  char **ww;
	  
	  /*fprintf(stderr, "CLIL received\n");*/
	  if (recvall(ns, buf, 2, MSG_WAITALL) < 0) { /* lang */
	    close(ns);
	    exit(1);
	  }
	  buf[2] = 0;
	  Indexer->spellang = UdmSelectSpellLang(Conf, buf);
	  /*fprintf(stderr, "Lang [%s] received\n", buf);*/
	  if (recvall(ns, &wordlen, sizeof(wordlen), MSG_WAITALL) < 0) { /* word length */
	    close(ns);
	    exit(1);
	  }
	  /*fprintf(stderr, "Word length: %d\n", wordlen);*/
	  if (recvall(ns, buf, wordlen, MSG_WAITALL) < 0) { /* word */
	    close(ns);
	    exit(1);
	  }
	  buf[wordlen] = 0;
	  fprintf(stderr, "Request: %s\nReply:", buf);
	  ww = UdmNormalizeWord(Indexer, buf);
	  if (ww != NULL)
	    while (*ww != 0) {
	      wordlen = strlen(*ww);
	      if (send(ns, (void *)&wordlen, sizeof(wordlen), 0) < 0) {
		senderror(__LINE__-1);
	      }
	      if (send(ns, (void *)*ww, wordlen, 0) < 0) {
		senderror(__LINE__-1);
	      }
	      fprintf(stderr, " %s", *ww);
	      ww++;
	    }
	  wordlen = 0;
	  if (send(ns, (void *)&wordlen, sizeof(wordlen), 0) < 0) {
	    senderror(__LINE__-1);
	  }
	  fprintf(stderr, "\n");
	  /* END of transmission */

	} else {
	  fprintf(stderr,"%s [HELO] error %d %s\n",time_pid_info(),errno,buf /*strerror(errno)*/);
	  close(ns);
	  exit(1);
	}
      } else {
	fprintf(stderr,"%s recv() [HELO] error %d %s\n",time_pid_info(),errno,strerror(errno));
	close(ns);
	exit(1);
      }
      

      close(ns);
      exit(0);
      }
    /* parent process */
    close(ns);
    wait(&status);
  }


}

