/*   implementation of the own backend to use UdmSearch without third-party
 *   libraries
 */

#include "udm_config.h"

#ifdef HAVE_FILES
#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

#include "udm_common.h"
#include "udm_spell.h"
#include "udm_db.h"
#include "udm_charset.h"
#include "udm_robots.h"
#include "udm_mutex.h"
#include "udm_utils.h"
#include "udm_log.h"
#include "udm_conf.h"
#include "udm_crc32.h"
#include "udm_cache.h"
#include "udm_server.h"
#include "udm_xmalloc.h"
#include "udm_searchtool.h"
#include "udm_boolean.h"
#include "udm_stopwords.h"


static const char udmver[]= PACKAGE "-" VERSION "/UdmDB";
__INDLIB__ const char * UdmVersion(){
	return(udmver);
}


char * UdmDBEscStr(int DBType,char *x,const char *y){
	return(strcpy(x,y));
}


/************************* Backen functions *****************/
#define MAXMULTI	32
typedef struct txt_db {
	int connected;
	FILE * dict;
	FILE * url;
	FILE * robots;
	FILE * stopwords;
	int  crcdict[MAXMULTI];
	int  open_mode;
	int  errcode;
	char errmsg[1024];
} DB;

static void CloseDB(void * vdb){
DB * db;
int i;
	db=(DB*)vdb;
	if(db->dict)fclose(db->dict);
	if(db->url)fclose(db->url);
	if(db->robots)fclose(db->robots);
	if(db->stopwords)fclose(db->stopwords);
	for(i=0;i<MAXMULTI;i++)
		if(db->crcdict[i]>=0)
			close(db->crcdict[i]);
}

void UdmFreeDB(UDM_AGENT * Indexer){
        if(Indexer->db){                                                        
                CloseDB((DB*)(Indexer->db));                                    
                free(Indexer->db);                                              
        }                                                                       
}

static int InitDB(UDM_AGENT * Indexer){
	const char * fmode;
	int  flags,perm;
	int  i;
	DB * db=Indexer->db;
	
	if(db->connected)
		return(0);
	else
		db->connected=1;

	for(i=0;i<MAXMULTI;i++)
		db->crcdict[i]=-1;
	switch(db->open_mode){
		case UDM_OPEN_MODE_READ:
			fmode="r";
			flags=O_RDONLY|UDM_BINARY;
			break;
		case UDM_OPEN_MODE_WRITE:
		default:
			fmode="w";
			flags=O_WRONLY|O_CREAT|O_TRUNC|UDM_BINARY;
			perm=UDM_IWRITE;
	}

	if(Indexer->Conf->DBMode==UDM_DBMODE_SINGLE_CRC){
		char fname[UDMSTRSIZ]="";
		sprintf(fname,"%s%s",Indexer->Conf->vardir,"dict");
		db->crcdict[0]=open(fname,flags,perm);
		if(db->crcdict[0]<0){
			sprintf(db->errmsg,"Can't open dict file '%s' (%s)", fname, strerror(errno));
			db->errcode=1;
			return(1);
		}
		sprintf(fname,"%s%s",Indexer->Conf->vardir,"url.txt");
		db->url=fopen(fname,fmode);
		if(!db->url){
			sprintf(db->errmsg,"Can't open URL file '%s' (%s)", fname, strerror(errno));
			db->errcode=1;
			return(1);
		}
	}else
	if(Indexer->Conf->DBMode==UDM_DBMODE_MULTI_CRC){
		int fnum;
		char fname[UDMSTRSIZ];
		for(fnum=1;fnum<MAXMULTI;fnum++){	
			sprintf(fname,"%sdict%02d",Indexer->Conf->vardir,fnum);
			db->crcdict[fnum]=open(fname,flags,perm);
			if(db->crcdict[fnum]<0){
				sprintf(db->errmsg,"Can't open dict file '%s' (%s)", fname, strerror(errno));
				db->errcode=1;
				return(1);
			}
		}
		sprintf(fname,"%s%s",Indexer->Conf->vardir,"url.txt");
		db->url=fopen(fname,fmode);
		if(!db->url){
			sprintf(db->errmsg,"Can't open URL file '%s' (%s)", fname, strerror(errno));
			db->errcode=1;
			return(1);
		}
	}else{
		char fname[UDMSTRSIZ];
		sprintf(fname,"%s%s",Indexer->Conf->vardir,"dict.txt");
		db->dict=fopen(fname,fmode);
		if(!db->dict){
			sprintf(db->errmsg,"Can't open dict file '%s' (%s)", fname, strerror(errno));
			db->errcode=1;
			return(1);
		}
		sprintf(fname,"%s%s",Indexer->Conf->vardir,"url.txt");
		db->url=fopen(fname,fmode);
		if(!db->url){
			sprintf(db->errmsg,"Can't open URL file '%s' (%s)", fname, strerror(errno));
			db->errcode=1;
			return(1);
		}
	}
	return(0);
}

void * UdmAllocDB(UDM_AGENT * Indexer, int mode){
DB * db;
	db=(DB*)UdmXmalloc(sizeof(DB));
	db->dict=NULL;
	db->url=NULL;
	db->robots=NULL;
	db->stopwords=NULL;
	db->errcode=0;
	db->errmsg[0]=0;
	db->open_mode=mode;
	db->connected=0;
	return((void*)db);
}
char * UdmDBErrorMsg(void *db){
	return(((DB*)db)->errmsg);
}
int UdmDBErrorCode(void *db){
	return(((DB*)db)->errcode);
}


/*************  Words ****************************/
/*
typedef struct cstruct {
	int		url_id;
	udmcrc32_t	wrd_id;
	char		weight;
} UDM_CRCWORD;
*/

int UdmDeleteWordFromURL(UDM_AGENT* Indexer,int url_id){
	if(Indexer->Conf->DBMode==UDM_DBMODE_CACHE){
		int i;
		i=UdmDeleteURLFromCache(Indexer,url_id);
		return(i);
	}
	return(0);
}
int UdmDeleteCrossWordFromURL(UDM_AGENT* Indexer,int url_id, int ref_id){
	return(0);
}
int UdmStoreCrossWords(UDM_AGENT * Indexer,int url_id){
	return 0;
}
int UdmStoreWords(UDM_AGENT * Indexer, int url_id, int site_id, const char * cat, const char * tag, int status){
	if (InitDB(Indexer)){
		UdmLog(Indexer, UDM_LOG_ERROR, "Error: %s", ((DB*)(Indexer->db))->errmsg);
		exit(1);
	}
	if(Indexer->Conf->DBMode==UDM_DBMODE_SINGLE_CRC){
		int fd,i;
		size_t bytes;
		UDM_CRCWORD * cw;
		if(Indexer->nwords){
			fd=((DB*)(Indexer->db))->crcdict[0];
			bytes=sizeof(UDM_CRCWORD)*Indexer->nwords;
			cw=(UDM_CRCWORD*)UdmXmalloc(bytes);
			for(i=0;i<Indexer->nwords;i++){
				if(Indexer->Word[i].count){
					cw[i].url_id=url_id;
					cw[i].weight=Indexer->Word[i].count;
					cw[i].wrd_id=UdmStrCRC32(Indexer->Word[i].word);
				}
			}
			if((bytes!=write(fd,(void *)cw,bytes))){
				sprintf(((DB*)(Indexer->db))->errmsg,"Can't write to dict file (%s)", strerror(errno));
				((DB*)(Indexer->db))->errcode=1;
				return(1);
			}
			free(cw);
		}
	}else
	if(Indexer->Conf->DBMode==UDM_DBMODE_MULTI_CRC){
		int fd,i,j,len;
		size_t bytes;
		UDM_CRCWORD * cw;
		if(Indexer->nwords){
			bytes=sizeof(UDM_CRCWORD)*Indexer->nwords;
			cw=(UDM_CRCWORD*)UdmXmalloc(bytes);
			for(len=1;len<MAXMULTI;len++){
				j=0;
				for(i=0;i<Indexer->nwords;i++)
				if((Indexer->Word[i].count)&&
				    len==strlen(Indexer->Word[i].word)){
					cw[j].url_id=url_id;
					cw[j].weight=Indexer->Word[i].count;
					cw[j].wrd_id=UdmStrCRC32(Indexer->Word[i].word);
					j++;
				}
				fd=((DB*)(Indexer->db))->crcdict[len];
				bytes=sizeof(UDM_CRCWORD)*j;
				if((bytes!=write(fd,(void *)cw,bytes))){
					sprintf(((DB*)(Indexer->db))->errmsg,"Can't write to dict file (%s)", strerror(errno));
					((DB*)(Indexer->db))->errcode=1;
					return(1);
				}
			}
			free(cw);
		}
	}else
	if(Indexer->Conf->DBMode){
		int res;
		res=UdmStoreWordsCache(Indexer,url_id,site_id,cat,tag);
	}else{
		FILE *f;
		int i;
		f=((DB*)(Indexer->db))->dict;
		for(i=0;i<Indexer->nwords;i++){
			if(Indexer->Word[i].count){
				fprintf(f,"%d\t%d\t%s\n",url_id,
					Indexer->Word[i].count,
					Indexer->Word[i].word);
			}
		}
	}
	return(IND_OK);
}

/************** stopwords **********************/

int UdmLoadStopList(UDM_AGENT * Indexer,const char * table){
	return(IND_OK);
}

/************ URLs stuff *******************************/

static int n_urls=0;
static UDM_DOCUMENT* URLs=NULL;


UDM_DOCUMENT * UdmGetDocInfo(UDM_AGENT* Indexer,int expired_first){
int i;

UDM_DOCUMENT * Result;

	for(i=0;i<n_urls;i++){
		if(!URLs[i].indexed){
			URLs[i].indexed=1;
			Result=(UDM_DOCUMENT *)UdmXmalloc(sizeof(UDM_DOCUMENT));
			Result->content_type=NULL;
			Result->title=NULL;
			Result->text=NULL;
			Result->last_index_time=0;
			Result->next_index_time=0;
			Result->last_mod_time=0;
			Result->content=NULL;
			Result->keywords=NULL;
			Result->description=NULL;
			Result->url=strdup(URLs[i].url);
			Result->url_id=URLs[i].url_id;
			Result->size=0;
			Result->status=0;
			Result->hops=URLs[i].hops;
			Result->crc32=0;
			return(Result);
		}
	}
	return(NULL);
}
int UdmFindURL(UDM_AGENT* Indexer,char *url){
	return 0;
}

int UdmAddURL(UDM_AGENT* Indexer,char *url,int referrer,int hops, char * msg_id, char * tag, char * cat){
	if(!n_urls){
		URLs=(UDM_DOCUMENT *)UdmXmalloc(sizeof(UDM_DOCUMENT));
	}else{
		URLs=(UDM_DOCUMENT *)UdmXrealloc(URLs,sizeof(UDM_DOCUMENT)*(n_urls+1));
	}
	URLs[n_urls].url_id=n_urls+1;
	URLs[n_urls].indexed=0;
	URLs[n_urls].hops=hops;
	URLs[n_urls].referrer=referrer;
	URLs[n_urls].url=strdup(url);
	n_urls++;
	return(IND_OK);
}
int UdmDeleteUrl(UDM_AGENT* Indexer,int url_id){
	return(IND_OK);
}
int UdmUpdateUrl(UDM_AGENT* Indexer,int url_id,int status,int period){
	return(IND_OK);
}

#define XVAL(x)		(!x?"-":!x[0]?"-":x)
int UdmLongUpdateUrl(UDM_AGENT* Indexer,
	int url_id,int status,int is_changed,
	int size,int period,char * tag,
	time_t last_mod_time,
	char *text_escaped,
	char *title_escaped,
	char *content_type,
	char *keywords_escaped,
	char *descript_escaped,
	udmcrc32_t crc32,
	char *lang,
	char *category
#ifdef NEWS_EXT
	,
	char *hd_date,
	char *hd_subj,
	char *hd_from,
	char *hd_group,
	char *hd_ref,
	char *msg_id
#endif
	){
	
	int i;
	FILE *f;

	if (InitDB(Indexer)){
	    UdmLog(Indexer, UDM_LOG_ERROR, "Error: %s", ((DB*)(Indexer->db))->errmsg);
	    exit(1);
	}

	f=((DB*)(Indexer->db))->url;

	for(i=0;i<n_urls;i++)
	if(URLs[i].url_id==url_id){

		fprintf(f,"%d\t%s\t%d\t%d\t%s\t%s\t%u\t%s\t%li\t%s\t%s\t%s\t%s\n",
			url_id,
			XVAL(tag),status,size,
			XVAL(content_type),
			XVAL(lang),
			crc32,
			XVAL(URLs[i].url),
			last_mod_time,
			XVAL(text_escaped),
			XVAL(title_escaped),
			XVAL(keywords_escaped),
			XVAL(descript_escaped));
		break;
	}
	return(IND_OK);
}



/************* Clones ***************************/
int UdmFindOrigin(UDM_AGENT* Indexer, udmcrc32_t crc32, int size){
	return(0);
}

int UdmUpdateClone(UDM_AGENT* Indexer,int url_id,int status,int period,
	char *content_type,time_t last_mod_time,udmcrc32_t crc32){

	return(IND_OK);
}

UDM_RESULT * UdmCloneList(UDM_AGENT * Indexer, udmcrc32_t crc32){
	return(NULL);
}

#ifdef NEWS_EXT
int UdmRegisterChild(UDM_AGENT *Indexer, int parent_id, int child_id){
	return 0;
}
int UdmFindMsgID(UDM_AGENT *Indexer, const char * msg_id){
	return 0;
}
#endif


/************* Clearing **************************/
int UdmDeleteAllFromDict(UDM_AGENT* Indexer){
	return(IND_OK);
}
int UdmDeleteAllFromUrl(UDM_AGENT* Indexer){
	return(IND_OK);
}
int UdmClearDB(UDM_AGENT* Indexer){
	return(IND_OK);
}

/*************** Servers *************************/

int UdmLoadServerTable(UDM_AGENT * Indexer, char * name, int flags){
	return(IND_OK);
}

/************** Robots.txt stuff *****************/
int UdmLoadRobots(UDM_AGENT *Indexer){
	return(IND_OK);
}
int UdmDeleteRobotsFromHost(UDM_AGENT* Indexer,char *hostinfo){
	return(IND_OK);
}
int UdmAddRobotsToHost(UDM_AGENT* Indexer,char *hostinfo,char *s){
	if(Indexer->Conf->LockProc)Indexer->Conf->LockProc(UDM_LOCK,UDM_LOCK_ROBOTS);
	if(!Indexer->Conf->Robots.nrobots){
		Indexer->Conf->Robots.robots=(UDM_ROBOT *)UdmXmalloc(sizeof(UDM_ROBOT));
	}else{
		Indexer->Conf->Robots.robots=(UDM_ROBOT *)UdmXrealloc(Indexer->Conf->Robots.robots,(Indexer->Conf->Robots.nrobots+1)*sizeof(UDM_ROBOT));
	}	
	Indexer->Conf->Robots.robots[Indexer->Conf->Robots.nrobots].hostinfo=strdup(hostinfo);
	Indexer->Conf->Robots.robots[Indexer->Conf->Robots.nrobots].path=strdup(s);
	Indexer->Conf->Robots.nrobots++;
	if(Indexer->Conf->LockProc)Indexer->Conf->LockProc(UDM_UNLOCK,UDM_LOCK_ROBOTS);
	return(IND_OK);
}



/**************** search stuff *******************/

#define WF_ONE(q,w,n)	(((w>>n)&(0x01))*((q)->wf[n]))
#define WF_ALL(q,w)	(WF_ONE(q,w,0)+WF_ONE(q,w,1)+WF_ONE(q,w,2)+WF_ONE(q,w,3)+WF_ONE(q,w,4)+WF_ONE(q,w,5)+WF_ONE(q,w,6)+WF_ONE(q,w,7))
#define QWF(q,w)	((q->weight_factor)?(WF_ALL(q,w)):(w&0xFFFF))

static UDM_SEARCHWORD * UdmFindBuiltIn(UDM_AGENT * query, char * text){
	int i,j,url_id;
	UDM_SEARCHWORD * wrd=NULL;
	char str[4048];

	UdmPrepare(query,text);

	/* Now find each word */
	if(query->Conf->DBMode==UDM_DBMODE_SINGLE_CRC){
		UDM_CRCWORD cw[256];
		int bytes, wnum;
		while((bytes=read(((DB*)(query->db))->crcdict[0],&cw,sizeof(cw)))){
			wnum=bytes/sizeof(UDM_CRCWORD);
			for(j=0;j<wnum;j++){
				for(i=0;i<query->words_in_query;i++){
					if(query->cwords[i]==cw[j].wrd_id){
						int fweight;
						fweight=QWF(query,cw[j].weight);
						
						if(fweight){
							if(!query->total_found){
								wrd=(UDM_SEARCHWORD*)UdmXmalloc((query->total_found+1)*sizeof(UDM_SEARCHWORD));
							}else{
								wrd=(UDM_SEARCHWORD*)UdmXrealloc(wrd,(query->total_found+1)*sizeof(UDM_SEARCHWORD));
							}
							wrd[query->total_found].url_id=cw[j].url_id;
							wrd[query->total_found].count=1<<(query->wordorders[i]);
							wrd[query->total_found].weight=fweight;
							wrd[query->total_found].pos=(((unsigned int)cw[j].weight>>16));
							query->total_found++;
						}
					}
				}
			}
		}
	}else
	if(query->Conf->DBMode==UDM_DBMODE_MULTI_CRC){
		UDM_CRCWORD cw[256];
		int bytes,len,fd,wnum;

		for(i=0;i<query->words_in_query;i++){
			len=strlen(query->words[i]);
			if(len>=MAXMULTI)len=MAXMULTI-1;
			fd=((DB*)(query->db))->crcdict[len];
			lseek(fd,(off_t)0,SEEK_SET);
			while((bytes=read(fd,&cw,sizeof(cw)))){
				wnum=bytes/sizeof(UDM_CRCWORD);
				for(j=0;j<wnum;j++){
					if(query->cwords[i]==cw[j].wrd_id){
						int fweight;
						fweight=QWF(query,cw[j].weight);
						
						if(fweight){
							if(!query->total_found){
								wrd=(UDM_SEARCHWORD*)UdmXmalloc((query->total_found+1)*sizeof(UDM_SEARCHWORD));
							}else{
								wrd=(UDM_SEARCHWORD*)UdmXrealloc(wrd,(query->total_found+1)*sizeof(UDM_SEARCHWORD));
							}
							wrd[query->total_found].url_id=cw[j].url_id;
							wrd[query->total_found].count=1<<(query->wordorders[i]);
							wrd[query->total_found].weight=fweight;
							wrd[query->total_found].pos=(((unsigned int)cw[j].weight>>16));
							query->total_found++;
						}
					}
				}
			}
		}
	}else{
		/* UDM_DBMODE_SINGLE */
		while(fgets(str,sizeof(str),((DB*)(query->db))->dict)){
			char * w, * lasttok=NULL;
			int weight;
			
			w=UdmGetToken(str,"\t",&lasttok);
			if(!w)continue;url_id=atoi(w);
			w=UdmGetToken(NULL,"\t",&lasttok);
			if(!w)continue;weight=atoi(w);
			w=UdmGetToken(NULL,"\t\n",&lasttok);
			if(!w)continue;
			
			for(i=0;i<query->words_in_query;i++){
				int matches=0;
				int fweight=0;
				
				switch(query->word_match){
					case UDM_MATCH_BEGIN:
						matches=!strncmp(w,query->words[i],strlen(query->words[i]));
						break;
					case UDM_MATCH_END:
						if(strlen(query->words[i])<=strlen(w)){
							char *ending;
							ending=w+strlen(w)-strlen(query->words[i]);
							matches=!strcmp(ending,query->words[i]);
						}
						break;
					case UDM_MATCH_SUBSTR:
						matches=(strstr(w,query->words[i])!=NULL);
						break;
					case UDM_MATCH_WORD:
					default:
						matches=!strcmp(w,query->words[i]);
						break;
				}
				if(matches)fweight=QWF(query,weight);
				if(matches&&fweight){
					if(!query->total_found){
						wrd=(UDM_SEARCHWORD*)malloc((query->total_found+1)*sizeof(UDM_SEARCHWORD));
					}else{
						wrd=(UDM_SEARCHWORD*)realloc(wrd,(query->total_found+1)*sizeof(UDM_SEARCHWORD));
					}
					wrd[query->total_found].url_id=url_id;
					wrd[query->total_found].count=1<<query->wordorders[i];
					wrd[query->total_found].weight=fweight;
					wrd[query->total_found].pos=(((unsigned int)weight>>16));
					query->total_found++;
				}
			}
		}
	}

	return wrd;
}


UDM_RESULT * UdmFind(UDM_AGENT * query, char * text){
	char str[UDMSTRSIZ];
	int url_id;
	int i,j;
	UDM_SEARCHWORD * wrd=NULL;
	UDM_RESULT * Res;
	int first;
	int skip=0;
	unsigned long ticks;

	if(InitDB(query))return(NULL);
	ticks=UdmStartTimer();

	UdmLoadStopList(query,"");
	if(UdmDBErrorCode(query->db))return(NULL);

	if(query->Conf->DBMode==UDM_DBMODE_CACHE){
		wrd=UdmFindCache(query,text);
	}else{
		wrd=UdmFindBuiltIn(query,text);
	}

	/* Allocate result */
	Res=(UDM_RESULT*)malloc(sizeof(UDM_RESULT));
	Res->Doc=NULL;
	Res->total_found=0;
	Res->num_rows=0;
	strcpy(Res->wordinfo,query->wordinfo);

	/* Return if nothing was found */
	if(!query->total_found){
		Res->work_time=UdmStartTimer()-ticks;
		return(Res);
	}

	/* Now let's sort in url_id order then group results */
	if(query->total_found){
		UdmSortSearchWordsByURL(wrd,query->total_found);
		UdmGroupByURL(query,wrd);
	}

	/* Sort by the weight */
	if(query->page_number<FAST_PRESORT_PAGES){
		size_t topcount;
		topcount=query->page_size*(query->page_number+1)-1;
		if(topcount>=query->total_found)topcount=query->total_found-1;
		UdmWrdTopSort(wrd,query->total_found,topcount);
	}else{
		UdmSortSearchWordsByWeight(wrd,query->total_found);
	}

	/* Copy SEARCHWORD to DOC structure */
	Res->Doc=(UDM_DOCUMENT*)UdmXmalloc(sizeof(UDM_DOCUMENT)*(query->total_found));
	for(i=0;i<query->total_found;i++){
		Res->Doc[i].url_id=wrd[i].url_id;
		Res->Doc[i].rating=wrd[i].pos;
		Res->Doc[i].content_type=NULL;
		Res->Doc[i].crc32=0;
		Res->Doc[i].url=NULL;
		Res->Doc[i].last_mod_time=0;
		Res->Doc[i].text=NULL;
		Res->Doc[i].title=NULL;
		Res->Doc[i].keywords=NULL;
		Res->Doc[i].description=NULL;
	}
	free(wrd);

	/* Now apply URL limits */
	while(fgets(str,sizeof(str),((DB*)(query->db))->url)){
		url_id=atoi(str);
		for(i=0;i<query->total_found-skip;i++){
			if(url_id==Res->Doc[i].url_id){
				char * tag=NULL, *w, *lasttok=NULL;
				int matched=1;
				w=UdmGetToken(str,"\t",&lasttok);
				for(j=0;w;j++){
					switch(j){
						case  1: tag=strdup(w);break;
						case  3: Res->Doc[i].size=atoi(w);break;
						case  4: Res->Doc[i].content_type=strdup(w);break;
						case  6: Res->Doc[i].crc32=(udmcrc32_t)strtol(w, NULL, 10);break;
						case  7: Res->Doc[i].url=strdup(w);break;
						case  8: Res->Doc[i].last_mod_time=atol(w);break;
						case  9: Res->Doc[i].text=strdup(w);break;
						case 10: Res->Doc[i].title=strdup(w);break;
						case 11: Res->Doc[i].keywords=strdup(w);break;
						case 12: Res->Doc[i].description=strdup(w);break;
					}
					
					w=UdmGetToken(NULL,"\t",&lasttok);
					
				}

				/* Check "ul" subsection match  */
				if(query->Conf->ul)matched=(strstr(Res->Doc[i].url,query->Conf->ul)!=NULL);
				/* Check "tag" subsection match */
				if((matched)&&(query->Conf->ttag)&&(tag))matched=(!strcmp(query->Conf->ttag,tag));

				if(!matched){
					/* Skip this URL */
					UDM_FREE(Res->Doc[i].content_type);
					UDM_FREE(Res->Doc[i].url);
					UDM_FREE(Res->Doc[i].text);
					UDM_FREE(Res->Doc[i].title);
					UDM_FREE(Res->Doc[i].keywords);
					UDM_FREE(Res->Doc[i].description);
					skip++;
					memmove(&(Res->Doc[i]),&(Res->Doc[i+1]),(query->total_found-i-1)*sizeof(UDM_DOCUMENT));
				}
			}
		}
	}
	
	Res->total_found =query->total_found-skip;
	for(i=0;i<Res->total_found;i++){
		Res->Doc[i].order=i+1;
	}
	first=query->page_number*query->page_size;
	if(first>query->total_found-skip)first=query->total_found-skip;

	Res->num_rows = query->total_found-skip-first;
	if((query->total_found-skip-first)>query->page_size)
	{
		Res->num_rows = query->page_size;
	}
	if(first>0){
		for(i=0;i<first;i++){
			UDM_FREE(Res->Doc[i].content_type);
			UDM_FREE(Res->Doc[i].url);
			UDM_FREE(Res->Doc[i].text);
			UDM_FREE(Res->Doc[i].title);
			UDM_FREE(Res->Doc[i].keywords);
			UDM_FREE(Res->Doc[i].description);
		}
		memmove(&(Res->Doc[0]),&(Res->Doc[first]),Res->num_rows*sizeof(UDM_DOCUMENT));
		/* FIXME realloc required */
	}
	Res->work_time=UdmStartTimer()-ticks;
	Res->first=first+1;
	Res->last=Res->first+Res->num_rows-1;
	query->total_found=query->total_found-skip;
	query->total_found=Res->total_found;
	return(Res);
}



/***************** Misc functions ***************/

int * UdmGetURLList(UDM_AGENT * query){
	return(NULL);
}

int UdmInsertSpell(UDM_AGENT *Indexer,const char *flag,const char *lang,const char *word){
    return(0);
}

UDM_SPELL * UdmFindWordDB(UDM_AGENT * Indexer,const char *word) {
    return(NULL);
}

int UdmInsertAffix(UDM_AGENT * Indexer,int flag,const char *lang,const char *mask,const char *find,const char *repl,const char *type){
    return(0);
}

int UdmImportDictionaryFromDB(UDM_AGENT *Indexer) {
    return(0);
}

int UdmDBImportAffixes(UDM_AGENT * Indexer, int LCharset){
    return(0);
}

__INDLIB__ int UdmClearSpell(UDM_AGENT * Indexer){
    return(0);
}


UDM_CATEGORY * UdmCatPath(UDM_AGENT * Indexer,const char * category){
	return(NULL);
}

UDM_CATEGORY * UdmCatList(UDM_AGENT * Indexer,const char * category){
	return(NULL);
}

int UdmBuildExtIndex(UDM_AGENT * query){
	return(IND_OK);
}

__INDLIB__ int  UdmGetStatistics(UDM_AGENT * Indexer){
	if(!Indexer->Conf->StatInfo)return(IND_OK);
	Indexer->Conf->StatInfo(Indexer->handle,0,0,0,"Stats not implemented in built-in database");
	Indexer->Conf->StatInfo(Indexer->handle,-1,0,0,"");
	return(IND_OK);
}
__INDLIB__ int UdmMarkForReindex(UDM_AGENT* Indexer){
	return(IND_OK);
}
__INDLIB__ int UdmAddTagLimit(UDM_ENV * Conf,char * tag){
	Conf->ttag=strdup(tag);
	return(0);
}
__INDLIB__ int UdmAddStatusLimit(UDM_ENV * Conf,int status){
	return(0);
}
__INDLIB__ int UdmAddURLLimit(UDM_ENV * Conf,char *URL){
	UDM_FREE(Conf->ul);
	Conf->ul=strdup(URL);
	return(0);
}
__INDLIB__ int UdmAddLangLimit(UDM_ENV * Conf,char *URL){
	return(0);
}

__INDLIB__ int UdmAddCatLimit(UDM_ENV * Conf,char * cat){
	return(0);
}

__INDLIB__ int UdmClearURLLimit(UDM_ENV * Conf){
	UDM_FREE(Conf->ul);
	UDM_FREE(Conf->ttag);
	return(0);
}
__INDLIB__ int UdmClearLimits(UDM_ENV * Conf){
	UDM_FREE(Conf->ul)
	return(0);
}
__INDLIB__ int UdmGetReferers(UDM_AGENT* Indexer){
	return(IND_OK);
}
__INDLIB__ int UdmAddTimeLimit(UDM_ENV * Conf,struct udm_stl_info_t * stl){
 return 0;
}
__INDLIB__ int UdmGetDocCount(UDM_AGENT * Indexer){
	return 0;
}


__INDLIB__ UDM_URLSTATE * UdmGetURLState(UDM_AGENT * Indexer, const char * hostinfo,time_t lmt){
	return NULL;
}


void UdmFreeUrlRes(UDM_ENV *Conf) {
  return;
}


#endif
