#################################################################
# connect.tcl - callback based layer on top of socket
# 
# Copyright (C) 2002 Mark Patton
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
###################################################################

# connect starts a socket connection to a host. It uses the Thread extension
# if present. The actual connection is managed through a callback interface.
# The attempt event in the callback gives a token that can be used with
# connect_cancel to stop the connection attempt

# args has these switches
#   -timeout secs      max time to wait for a connection, defaults to 10
#   -max_attempts num  number of connection attempts to make, defaults to 1
#   -attempt_wait secs time to wait after a timeout to attempt reconnection
#   -thread bool       use threads, defaults to true if Thread package present

# callback sig is {data event args} where the events are:
# event types:
#   attempt tok num
#   failure msg
#   timeout
#   success chan

set _con(script_loaded) 0
set _con(seq) 0

proc connect {host port callback data args} {
    global _con

    array set arg [list -max_attempts 1 -thread 0 -timeout 10 -attempt_wait 5]
    array set arg $args
    set arg(-thread) \
	    [expr {$arg(-thread) && ![catch {package require Thread 2.5}]}]
    set arg(-timeout) [expr {$arg(-timeout) * 1000}]
    set arg(-attempt_wait) [expr {$arg(-attempt_wait) * 1000}]
    set id [incr _con(seq)]

    set script [list _connect $id $host $port $callback $data \
	    $arg(-max_attempts) $arg(-timeout) $arg(-attempt_wait)]

    if {$arg(-thread)} {
	set thr_id [thread::create -preserved]
	lappend script [list [thread::id] $thr_id]

	thread::send -async $thr_id $_con(script)
	thread::send -async $thr_id $script
    } else {
	if {!$_con(script_loaded)} {
	    eval $_con(script)
	    set _con(script_loaded) 1
	}

	eval $script
    }

    return
}

proc connect_cancel {tok} {
    global _con

    foreach {id thr_id} $tok {}

    if {[llength $tok] == 2} {
	thread::release $thr_id
    } else {
	catch {
	    after cancel $_con($id,timeout_id)
	    close $_con($id,chan)
	}
	array unset _con $id,*
    }
}

set _con(script) {
    proc _connect_cleanup {id} {
	global _con

	if {$_con($id,slave_thread)} {
	    thread::release
	}

	array unset _con $id,*
    }

    proc _connect_callback {id args} {
	global _con
	
	set script [linsert $args 0 $_con($id,callback) $_con($id,data)]

	if {$_con($id,slave_thread)} {
	    thread::send -async $_con($id,master) $script
	} else {
	    eval $script
	}
    }

    proc _connect_timeout {id} {
	global _con

	close $_con($id,chan)
	_connect_callback $id timeout
	    
	if {$_con($id,attempt) < $_con($id,max_attempts)} {
	    incr _con($id,attempt)
	    after $_con($id,attempt_wait) [list _connect_real $id]
	} else {
	    _connect_callback $id failure "all connection attempts timed out"
	    _connect_cleanup $id
	}
    }

    proc _connect_complete {id} {
	global _con
	
	after cancel $_con($id,timeout_id)
	fileevent $_con($id,chan) readable ""

	set err [fconfigure $_con($id,chan) -error]
	
	if {$err eq ""} {
	    if {$_con($id,slave_thread)} {
		thread::transfer $_con($id,master) $_con($id,chan)
	    }
	 
	    _connect_callback $id success $_con($id,chan)
	} else {
	    catch {close $_con($id,chan)}
	    _connect_callback $id failure $err
	}
	
	_connect_cleanup $id
    }
    
    proc _connect_real {id} {
	global _con

	set tok [list $id]
	if {$_con($id,slave_thread)} {
	    lappend tok $_con($id,slave)
	}

	_connect_callback $id attempt $tok $_con($id,attempt)


	if {![info exists _con($id,timeout)]} {
	    # canceled in the callback
	    return
	}

	set _con($id,timeout_id) \
		[after $_con($id,timeout) [list _connect_timeout $id]]


	set error [catch {socket -async $_con($id,host) $_con($id,port)} res]

	if {$error} {
	    after cancel $_con($id,timeout_id)
	    _connect_callback $id failure $res
	    _connect_cleanup $id
	} else {

	    set _con($id,chan) $res
	    fileevent $_con($id,chan) readable [list _connect_complete $id]
	}
    }

    proc _connect {id host port cb data attempts timeout wait {thrd_ids {}}} {
	global _con
	
	array set _con [list $id,host $host $id,port $port $id,attempt 1 \
		$id,timeout $timeout $id,max_attempts $attempts          \
		$id,callback $cb $id,data $data $id,attempt_wait $wait   \
		$id,master [lindex $thrd_ids 0]                          \
		$id,slave [lindex $thrd_ids 1]                           \
		$id,slave_thread [llength $thrd_ids]]

	_connect_real $id
    }
}

