# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Report printer.

# load libs
import wx
from wx.html import HtmlEasyPrinting
import os.path
import string
from math import fmod


class mReportPrinter(HtmlEasyPrinting):
    """ Simple HTML printing class for report. """

    # ----
    def __init__(self, parent):
        HtmlEasyPrinting.__init__(self, name='mMass report')
        self.parent = parent
        self.config = parent.config.cfg

        # set defaults
        self.printData = self.GetPrintData()
        self.printData.SetOrientation(wx.PORTRAIT)
        self.printData.SetQuality(wx.PRINT_QUALITY_MEDIUM)
        self.pageSetup = self.GetPageSetupData()
        self.pageSetup.SetMarginBottomRight((20, 20))
        self.pageSetup.SetMarginTopLeft((20, 20))
        self.pageSetup.SetPrintData(self.printData)
    # ----


    # ----
    def onPrint(self, docData):
        """ Format data to HTML and print page. """

        # get formated data
        path = docData.getPath()
        reportHTML = self.formatReport(docData)

        # print report
        self.PrintText(reportHTML, os.path.basename(path))
    # ----


    # ----
    def onPreview(self, docData):
        """ Format data to HTML and preview page. """

        # get formated data
        path = docData.getPath()
        reportHTML = self.formatReport(docData)

        # preview report
        self.PreviewText(reportHTML, os.path.basename(path))
    # ----


    # ----
    def setPageSetup(self, pageSetup):
        """ Set current page setup. """

        printDataTmp = pageSetup.GetPrintData()
        self.printData.SetBin(printDataTmp.GetBin())
        self.printData.SetCollate(printDataTmp.GetCollate())
        self.printData.SetColour(printDataTmp.GetColour())
        self.printData.SetDuplex(printDataTmp.GetDuplex())
        self.printData.SetNoCopies(printDataTmp.GetNoCopies())
        self.printData.SetOrientation(printDataTmp.GetOrientation())
        self.printData.SetPaperId(printDataTmp.GetPaperId())
        self.printData.SetPrinterName(printDataTmp.GetPrinterName())
        self.printData.SetQuality(printDataTmp.GetQuality())
        self.pageSetup.SetMarginBottomRight(pageSetup.GetMarginBottomRight())
        self.pageSetup.SetMarginTopLeft(pageSetup.GetMarginTopLeft())
        self.pageSetup.SetPrintData(self.printData)
    # ----


    # ----
    def formatReport(self, docData):
        """ Format data to HTML. """

        buff = '<html><body>'

        # add header
        buff += '<hr size="1" noshade><center><b>mMass sample report</b></center><hr size="1" noshade>'

        # add basic file info
        buff += '<font size=-2><table cellspacing="10">'
        buff += '<tr><td align="right"><b>measurement&nbsp;date:</b>&nbsp;</td><td>%s</td></tr>' % docData.getDescription('date')
        buff += '<tr><td align="right"><b>operator:</b>&nbsp;</td><td>%s</td></tr>' % docData.getDescription('operator')
        buff += '<tr><td align="right"><b>contact:</b>&nbsp;</td><td>%s</td></tr>' % docData.getDescription('contact')
        buff += '<tr><td align="right"><b>institution:</b>&nbsp;</td><td>%s</td></tr>' % docData.getDescription('institution')
        buff += '<tr><td align="right"><b>instrument:</b>&nbsp;</td><td>%s</td></tr>' % docData.getDescription('instrument')
        buff += '<tr><td align="right"><b>document&nbsp;path:</b>&nbsp;</td><td>%s</td></tr>' % docData.getPath()
        buff += '</table></font>'
        buff += '<hr size="1" noshade>'

        # add peaklist
        peaklist = docData.getPeaks()
        buff += self.formatPeaklist(peaklist)
        buff += '<hr size="1" noshade>'

        # add sequence
        seqTitle = docData.getSequenceTitle()
        sequence = docData.getParsedSequence()
        if sequence:
            buff += self.formatSequence(sequence, seqTitle)
            buff += '<hr size="1" noshade>'

        # add footer
        buff += '<font size=-2><center>Created by mMass - http://mmass.biographics.cz/</center></font>'

        buff += '</body></html>'

        return buff
    # ----


    # ----
    def formatPeaklist(self, peaklist):
        """ Convert peaklist data to html table. """

        # add table head
        buff = '<font size=-2><table cellspacing="10">'
        buff += '<tr>\
                    <td>&nbsp;</td>\
                    <td align="center">#</td>\
                    <td align="center"><b>m/z</b></td>\
                    <td align="center"><b>intensity</b></td>\
                    <td align="center"><b>annotations</b></td>\
                </tr>'

        # add peaks
        peakNo = 0
        for row in peaklist:
            peakNo += 1
            buff += '<tr>'

            # mark annotated peaks
            if row[2] != '':
                buff += '<td valign="top">&bull;</td>'
            else:
                buff += '<td>&nbsp;</td>'

            # add peak number
            buff += '<td align="right" valign="top">' + str(peakNo) + '&nbsp;</td>'

            # paste data for each peak
            digitsFormat = '%0.' + `self.config['common']['digits']` + 'f'
            for col in range(len(row)):
                    
                # mass
                if col == 0:
                    mass = digitsFormat % row[0]
                    buff += '<td align="right" valign="top">' + str(mass) + '&nbsp;</td>'

                # intensity
                if col == 1:
                    intensity = digitsFormat % row[1]
                    buff += '<td align="right" valign="top">' + str(intensity) + '&nbsp;&nbsp;&nbsp;</td>'

                # annotations
                if col == 2:
                    annotation = row[2]
                    buff += '<td align="left" valign="top">' + annotation + '</td>'

            buff += '</tr>'

        buff += '</table></font>'

        return buff
    # ----


    # ----
    def formatSequence(self, sequence, title):
        """ Convert sequence data to html. """

        buff = '<font size=-2><table cellspacing="10">'
        buff += '<tr><td align="right" valign="top"><b>sequence&nbsp;title:</b>&nbsp;</td><td>%s</td></tr>' % title
        
        # format sequence
        buff += '<tr><td align="right" valign="top"><b>sequence:</b>&nbsp;</td><td><code>'
        modifications = ''
        for i, amino in enumerate(sequence):

            # add amino and check for modifications
            if len(amino) > 3:
                buff += '<u>%s</u>' % amino[0]
                mods = string.join(amino[3:], '; ')
                modifications += '<tr><td>%s%d - %s</td></tr>' % (amino[0], i+1, mods)
            else:
                buff += amino[0]

            # add space every 10 amino
            if fmod(i+1, 10) == 0:
                buff += ' '

        buff += '</code></td></tr>'

        # format modifications
        if modifications != '':
            buff += '<tr><td align="right" valign="top"><b>modifications:</b>&nbsp;</td><td>'
            buff += '<table cellspacing="10">%s</table>' % modifications
            buff += '</td></tr>'

        buff += '</table></font>'

        return buff
    # ----
