# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Load and parse peaklists data from file.

# load libs
import wx
import os
import string

# load modules
import document_mmass
import document_mzdata
import document_mzxml
import document_bruker_flex
import document_ascii_xy


class mPeaklistImporter:
    """ Import peaklist data from ASCII file. """

    # ----
    def __init__(self, parent):
        self.parent = parent
    # ----


    # ----
    def showDialog(self, lastDir='.', multiple=False):

        # get last dir
        if lastDir == '':
            lastDir = '.'

        # available formats
        wildcard = "All files|*.*|" \
                    "All supported documents|fid;*.msd;*.mzData;*.mzdata;*.mzXML;*.mzxml;*.xml;*.XML;*.txt|" \
                    "mMass document (*.msd)|*.msd|" \
                    "mzData (*.mzData; *.xml)|*.mzData;*.mzdata;*.xml;*.XML|" \
                    "mzXML (*.mzXML; *.xml)|*.mzXML;*.mzxml;*.xml;*.XML|" \
                    "Bruker Flex Series (fid)|fid;FID"

        # set style
        if multiple:
            style = wx.OPEN|wx.MULTIPLE|wx.FILE_MUST_EXIST
        else:
            style = wx.OPEN|wx.FILE_MUST_EXIST

        # raise import dialog
        dlg = wx.FileDialog(self.parent, "Import Peaklist", lastDir, "", wildcard, wx.OPEN|wx.FILE_MUST_EXIST)
        if dlg.ShowModal() == wx.ID_OK:
            if multiple:
                paths = dlg.GetPaths()
                paths.sort()
            else:
                paths = dlg.GetPath()
            dlg.Destroy()
            return paths
        else:
            dlg.Destroy()
            return False
    # ----


    # ----
    def importData(self, path):
        """ Show import dialog and import data. """

        # init data
        data = {}
        data['errors'] = ''
        data['path'] = path
        data['docType'] = ''
        data['scanID'] = ''

        data['operator'] = ''
        data['contact'] = ''
        data['institution'] = ''
        data['date'] = ''
        data['instrument'] = ''
        data['notes'] = ''

        data['spectrum'] = []
        data['peaklist'] = []
        data['sequence'] = []
        data['seqtitle'] = ''
        data['modifications'] = []

        docData = None

        # check if file exists
        if not os.path.exists(path):
            message = "File does not exists!"
            dlg = wx.MessageDialog(self.parent, message, "Import Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False

        # open file
        else:

            # busy cursor start
            wx.Yield()
            wx.BeginBusyCursor()

            # try to load data by file extension
            ext = os.path.splitext(path)[1]
            ext = string.lower(ext)

            if ext == '.msd':
                loader = document_mmass.mMassDoc()
            elif ext in ['.mzxml']:
                loader = document_mzxml.mzXMLDoc(self.parent)
            elif ext in ['.mzdata', '.xml']:
                loader = document_mzdata.mzDataDoc(self.parent)
            elif ext == '':
                loader = document_bruker_flex.brukerFlexDoc()
            else:
                loader = document_ascii_xy.mASCIIDoc()

            docData = loader.getElement('peaklist', path)
            if docData == None:
                    wx.EndBusyCursor()
                    return None

            # Bruker flex series
            if not docData:
                loader = document_bruker_flex.brukerFlexDoc()
                docData = loader.getElement('peaklist', path)

            # ASCII XY
            if not docData:
                loader = document_ascii_xy.mASCIIDoc()
                docData = loader.getElement('peaklist', path)

            # mzXML Document
            if not docData:
                loader = document_mzxml.mzXMLDoc(self.parent)
                docData = loader.getElement('peaklist', path)
                if docData == None:
                    wx.EndBusyCursor()
                    return None

            # mzData Document
            if not docData:
                loader = document_mzdata.mzDataDoc(self.parent)
                docData = loader.getElement('peaklist', path)
                if docData == None:
                    wx.EndBusyCursor()
                    return None

            # mMass Document
            if not docData:
                loader = document_mmass.mMassDoc()
                docData = loader.getElement('peaklist', path)

            # busy cursor end
            wx.EndBusyCursor()

        # check peaklist
        if docData and docData.has_key('peaklist') and docData['peaklist']:
            for x, peak in enumerate(docData['peaklist']):
                data['peaklist'].append([peak[0], peak[1], peak[2], 0])
        else:
            message = "Document contains no peaklist or incorrect data format."
            dlg = wx.MessageDialog(self.parent, message, "Import Error", wx.OK|wx.ICON_ERROR)
            dlg.ShowModal()
            dlg.Destroy()
            return False

        return data
    # ----
