/* -*- Mode: C++; indent-tabs-mode: true; c-basic-offset: 4-*- */

/*
 *This file is part of MlView
 *
 *MlView is free software; you can redistribute it
 *and/or modify it under the terms of
 *the GNU General Public License as published
 * by the Free Software Foundation; either version 2,
 *or (at your option) any later version.
 *
 *MlView is distributed in the hope that it will be useful,
 *but WITHOUT ANY WARRANTY; 
 *without even the implied warranty of MERCHANTABILITY or
 *FITNESS FOR A PARTICULAR PURPOSE.
 *See the GNU General Public License for more details.
 *
 *You should have received a copy of the GNU General Public License
 *along with MlView; 
 *see the file COPYING. If not, write to the 
 *Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *Boston, MA 02111-1307, USA.
 *
 *See COPYRIGHT file for copyright informations.
 */

/**
 *@file
 *The definition of the #AppContext class.
 *
 *The context is the way by which the application container 
 *and the applications communicate (here, the application is MlViewEditor).
 *Both applications and the application container can store information in it.
 *The application container insures the consistency of the 
 *application context during the life time of the application. 
 *
 */
#include <map>
#include <stdarg.h>
/*#include <libgnomeui/libgnomeui.h>*/
#include <string.h>
#include "mlview-ustring.h"
#include <mlview-marshal.h>
#include "mlview-app-context.h"
#include "mlview-file-descriptor.h"
#include "mlview-utils.h"
#include "mlview-xml-document.h"
#include "mlview-clipboard.h"
#include "mlview-safe-ptr-utils.h"
#include "mlview-prefs.h"
#include "mlview-prefs-category-sizes.h"

namespace mlview {

/**
 *Some icons.
 */
#define ELEMENT_ICON      "mlview/xml-element-node.png"
#define OPEN_ELEMENT_ICON "mlview/xml-element-node-open.png"
#define TEXT_ICON         "mlview/xml-text-node.png"
#define ROOT_ICON         "mlview/xml-root.png"
#define OPEN_ROOT_ICON    "mlview/xml-root-open.png"
#define COMMENT_ICON      "mlview/xml-comment-node.png"
#define PI_ICON           "mlview/xml-pi-node.png"
#define ENTITY_REF_ICON   "mlview/xml-entity-ref-node.png"


/**
 *The private data members of AppContext.
 */
struct AppContextPriv
{

	/**The place to store some elements of the context.*/
	GHashTable *context_elements;

	/**A cache for the xpms.
	*We have choosen this method
	*to enable pixmap designers to
	*change the pixmaps and have
	*mlview load the new pixmap dynamically.
	*/
	GHashTable *pixmaps_cache;

	/**
	*mlview application bitmaps cache.
	*one the bitmap have been loaded from the disk, they
	*are cached in this hash table.
	*/
	GHashTable *bitmaps_cache;

	/**
	*The place where the error messages are
	*put before being displayed.
	*/
	gchar *error_msg_buffer;

	/**
	*The title of the error dialog box
	*/
	gchar *error_dialog_title;


	/**
	 *A cache for the instance of GtkFileChooser (for gtk+ >=2.4)
	 *components should use this instance of file chooser.         
	 */
	GtkFileChooser *file_chooser ;
	/**
	 *The xml catalog loaded by mlview
	 */
	xmlCatalog *xml_catalog;

	/*
	 *the last attribute id number 
	 *part generated by mlview
	 */
	gint last_id;

	/**
	 *the refcount on 
	 *the type icons
	 */
	guint type_icons_refcnt;

	/**
	 *the type icons
	 */
	struct TypeIcons *type_icons;

	SafePtr<Clipboard, ObjectRef, ObjectUnref> clipboard_ptr ;


	sigc::signal0<void>signal_application_initialized ;

	sigc::signal2<void, GtkWidget*, GdkEvent*> signal_contextual_menu_requested ;

	sigc::signal2<void, Object*, Object*> signal_view_swapped ;

	sigc::signal1<void, gpointer> signal_document_name_changed ;

	sigc::signal0<void> signal_view_undo_state_changed ;

	AppContextPriv () :
		context_elements (NULL),
		pixmaps_cache (NULL),
		bitmaps_cache (NULL),
		error_msg_buffer (NULL),
		error_dialog_title (NULL),
		file_chooser (NULL),
		xml_catalog (NULL),
		last_id (0),
		type_icons_refcnt (0),
		type_icons (NULL),
		clipboard_ptr (NULL)
	{}
};


/**
 *the macro to access the private part of AppContext object
 */

static void
pixmaps_cache_foreach_func (gchar * a_key,
                            GdkPixmap * a_value,
                            gchar * user_data);

static void
bitmaps_cache_foreach_func (gchar * a_key,
                            GdkBitmap * a_value,
                            gchar * user_data);

static void
pixmaps_cache_foreach_func (gchar * a_key,
                            GdkPixmap * a_value,
                            gchar * user_data)
{
	gdk_pixmap_unref (a_value);
}

static void
bitmaps_cache_foreach_func (gchar * a_key,
                            GdkBitmap * a_value,
                            gchar * user_data)
{
	gdk_bitmap_unref (a_value);
}

AppContext::AppContext ()
{
	m_priv = new AppContextPriv ();

	m_priv->context_elements = g_hash_table_new (g_str_hash, g_str_equal) ;

	m_priv->pixmaps_cache = g_hash_table_new (g_str_hash, g_str_equal);

	m_priv->bitmaps_cache = g_hash_table_new (g_str_hash, g_str_equal);
	/*
	 *initialise the list of available encodings
	 *supported by mlview and ref count it.
	 */
	mlview_utils_init_available_encodings_list ();

	mlview_utils_ref_available_encodings ();
}

AppContext::AppContext (AppContext const &a_context)
{
	THROW ("this method is forbiden") ;
}

AppContext&
AppContext::operator= (AppContext const &a_context)
{
	THROW ("this method is forbiden") ;
}

AppContext::~AppContext ()
{
	if (!m_priv) {
		return ;
	}

	if (m_priv->context_elements) {
		g_hash_table_destroy (m_priv-> context_elements);
	}
	if (m_priv->pixmaps_cache) {
		g_hash_table_foreach (m_priv->
		                      pixmaps_cache, (GHFunc)
		                      pixmaps_cache_foreach_func,
		                      NULL);
		m_priv->pixmaps_cache = NULL;
	}
	if (m_priv->file_chooser) {
		gtk_widget_destroy (GTK_WIDGET
		                    (m_priv->file_chooser)) ;
		m_priv->file_chooser = NULL ;
	}
	if (m_priv->bitmaps_cache) {
		g_hash_table_foreach (m_priv->
		                      bitmaps_cache, (GHFunc)
		                      bitmaps_cache_foreach_func,
		                      NULL);
		m_priv->bitmaps_cache = NULL;
	}

	if (m_priv->xml_catalog) {
		xmlFreeCatalog (m_priv->xml_catalog);
		m_priv->xml_catalog = NULL;
	}

	delete m_priv ;
	m_priv = NULL ;
}


//******************
//Public methods
// ******************

/**
 *The instance getter of the AppContext class.
 *Actually, if you want to use a 'singleton pattern', use this method.
 *All the invocations to this method will allways return the same object ;
 *
 *@return the singleton object. The firts time this method is called, creates
 *a new object and returns it. subsequent calls will always 
 *returns the same object.
 */
AppContext *
AppContext::get_instance (void)
{
	static AppContext *app_context = NULL;

	if (!app_context) {
		app_context = new AppContext () ;
	}
	return app_context;
}


/**
 *a getter of the last attribute id automatically generated.
 *This method is used in the automatic generation of attributes ids.
 *
 *@returns a pointer to the last id stored 
 *the method set_last_id() ;
 *You should manipulate this pointer
 *with great care or else you can break the attribute id storage.
 */
gint *
AppContext::get_last_id_ptr ()
{
	THROW_IF_FAIL (m_priv != NULL);
	return &m_priv->last_id;
}


/**
 *The getter of the last generated attribute id.
 *This method is used in the automatic generation of attributes ids.
 *
 *@return the last generated attribute id.
 */
gint
AppContext::get_last_id ()
{
	THROW_IF_FAIL (m_priv != NULL);

	return m_priv->last_id;
}


/**
 *A setter of the last generated attribut id.
 *This method is used in the automatic attribute id generation process.
 *
 *@param a_new_id the new id.
 */
void
AppContext::set_last_id (gint a_new_id)
{
	THROW_IF_FAIL (m_priv != NULL);

	m_priv->last_id = a_new_id;
}


/**
 *Stores a key/value pair in the context.
 *This method is used tho stores objects in the context.
 *If you want to store strings, (for settings for ex) 
 *use AppContext::set_settings_value() method instead.
 *
 *@param a_element_name the key of the object to store. 
 *Note that this string is not copied so this
 *pointer must remain valid during all the lifetime of the context. 
 *When the context is destroyed, 
 *a_element_name is _not_ destroyed. 
 *So the caller must manage the life time of this string.
 *This pointer is not destroyed upon
 *destruction of the context. 
 *It is up to the caller to manage the life time of this pointer. If
 *it is NULL, a NULL value is stored.
 */
void
AppContext::set_element (const gchar * a_element_name,
						 gpointer a_context_element)
{
	THROW_IF_FAIL (m_priv != NULL);
	THROW_IF_FAIL (a_element_name != NULL);

	if (m_priv->context_elements == NULL)
		m_priv->context_elements =
		    g_hash_table_new (g_str_hash,
		                      g_str_equal);

	g_hash_table_insert (m_priv->
	                     context_elements,
	                     (gchar *) a_element_name,
	                     a_context_element);
}


/**
 *Gets an object stored in the context using the 
 *AppContext::set_element() method.
 *@param a_element_name the key of the object stored.
 *@return the element stored or NULL if nothing has been stored. 
 *If a NULL value has been stored
 *using this key, (using the AppContext::set_element() method) 
 *a NULL value is also returned.
 */
gpointer
AppContext::get_element (const gchar * a_element_name)
{
	THROW_IF_FAIL (m_priv != NULL);

	return g_hash_table_lookup (m_priv->
	                            context_elements,
	                            a_element_name);
}

/**
 *Emits the "contextual-menu-requested" signal to notify that
 *an editing widget has been requested for a contextual menu.
 *@param a_source the widget that received the contextual menu
 *request.
 *@param MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
AppContext::notify_contextual_menu_request (GtkWidget *a_source_widget,
											GdkEvent *a_event)
{
	THROW_IF_FAIL (a_source_widget && GTK_IS_WIDGET (a_source_widget)) ;

	signal_contextual_menu_requested ().emit (a_source_widget, a_event) ;
	return MLVIEW_OK ;
}

/**
 *Emits a "view-swapped" signal to notify a new view becoming
 *the "currently selected view"
 *@param a_old_view the previous "current selected view". Should
 *be an implementation of #MlViewIView
 *@param a_new_view the new "current selected view". Should be
 *an implementation of the #MlViewIView
 *@return MLVIEW_OK upon successful completion, an error code
 *otherwise.
 */
enum MlViewStatus
AppContext::notify_view_swapped (Object* a_old_view, Object *a_new_view)
{
	THROW_IF_FAIL (m_priv) ;

	signal_view_swapped ().emit (a_old_view, a_new_view) ;

	return MLVIEW_OK ;
}

/**
 *Notifies listeners that the application started.
 *This actually makes the #AppContext instance
 *to emit the "application-initialized" signal.
 *@return MLVIEW_OK upon succesful completion, an error code
 *otherwise.
 */
enum MlViewStatus
AppContext::notify_application_initialized ()
{
	THROW_IF_FAIL (m_priv) ;

	signal_application_initialized ().emit () ;
	return MLVIEW_OK ;
}

/**
 *Notifies listeners that the name of the document being edited has
 *changed.
 *@param a_doc the document which name has changed.
 *@return MLVIEW_OK upon successful completion.
 */
enum MlViewStatus
AppContext::notify_document_name_changed (gpointer a_doc)
{
	THROW_IF_FAIL (m_priv) ;
	THROW_IF_FAIL (a_doc && MLVIEW_IS_XML_DOCUMENT (a_doc)) ;

	if (a_doc) {
		signal_document_name_changed ().emit (a_doc) ;
	}
	return MLVIEW_OK ;
}


enum MlViewStatus
AppContext::notify_view_undo_state_changed ()
{
	THROW_IF_FAIL (m_priv) ;

	signal_view_undo_state_changed ().emit () ;
	return MLVIEW_OK ;
}

/**
 *Pushes a message on the app bar. 
 *@param a_msg_format the format of the message (same parameter as printf)
 */
void
AppContext::sbar_push_message (const gchar * a_msg_format, ...)
{
	va_list params;
	gchar *msg = NULL;

	/*
	 *FIXME: this is broken since I dumped
	 *the GnomeApp stuffs.
	 */
	va_start (params, a_msg_format);
	msg = g_strdup_vprintf (a_msg_format, params);
	va_end (params);
	if (msg) {
		g_free (msg);
		msg = NULL;
	}
}


/**
 *Pops the last message pushed on the message stack.
 *The effect is that the message pushed before the 
 *last message has been pushed, will be
 *displayed. 
 */
void
AppContext::sbar_pop_message ()
{
	/*
	 *FIXME: this is broken since I dumped
	 *the GnomeApp stuffs.
	 */
}


/**
 *Sets the message that is displayed when no message is pushed on the
 *status bar message stack. 
 *
 *@param a_msg_format the format of the message
 *(as in the standard printf function)
 */
void
AppContext::sbar_set_default_message (const gchar *a_msg_format, ...)
{
	va_list params;
	gchar *msg = NULL;

	/*
	 *FIXME: this is broken since I dumped
	 *the GnomeApp stuffs.
	 */
	va_start (params, a_msg_format);
	msg = g_strdup_vprintf (a_msg_format, params);
	va_end (params);


	if (msg) {
		g_free (msg);
		msg = NULL;
	}
}


/**
 *Displays a message to the user. 
 *It opens a Modal dialog with the message on it.
 *The user just has to click OK. 
 *
 *@param a_msg_format the format of the message (as in the printf funtion)
 */
void
AppContext::message (const gchar * a_msg_format, ...)
{
	va_list params;
	gchar *msg = NULL;

	THROW_IF_FAIL (a_msg_format) ;
	va_start (params, a_msg_format);
	msg = g_strdup_vprintf (a_msg_format, params);
	va_end (params);

	mlview_utils_display_message_dialog ((const gchar*)msg) ;

	if (msg) {
		g_free (msg);
		msg = NULL;
	}
}


/**
 *Sets the title of the next error message that will be displayed by a call
 *to the function AppContext::error() or 
 *AppContext::display_buffered_error().
 *After a call to one of these two functions, 
 *all subsequent calls will display an error dialog
 *with no title. 
 *
 *@param a_title the new title of the error 
 *message dialog thatwill be used to display the 
 *content of the error message buffer.
 *look at definition of AppContext::bufferize_error(). 
 */
void
AppContext::set_error_dialog_title (const gchar * a_title)
{
	if (!m_priv)
		return;

	if (m_priv->error_dialog_title) {
		g_free (m_priv->error_dialog_title);
		m_priv->error_dialog_title = NULL;
	}

	if (a_title)
		m_priv->error_dialog_title =
		    g_strdup (a_title);
	else
		m_priv->error_dialog_title = NULL;
}


/**
 *Getter of error message dialog title.
 *
 *@return a pointer to the title of the next 
 *error message that will be displayed. 
 */
gchar *
AppContext::get_error_dialog_title ()
{
	if (!m_priv)
		return NULL;

	return m_priv->error_dialog_title;
}

void
AppContext::bufferize_error (const gchar *a_msg_format, va_list a_params)
{
	gchar * temp_err_msg = g_strdup_vprintf (a_msg_format, a_params);

	if (temp_err_msg) {

		if (!m_priv->error_msg_buffer)

			m_priv->error_msg_buffer =
			    g_strdup (temp_err_msg);
		else {
			gchar *old_str =
			    m_priv->
			    error_msg_buffer;

			m_priv->error_msg_buffer =
			    g_strconcat
			    (m_priv->
			     error_msg_buffer, temp_err_msg,
			     NULL);
			g_free (old_str);
			old_str = NULL ;
		}
	}
	if (temp_err_msg) {
		g_free (temp_err_msg);
		temp_err_msg = NULL ;
	}
}

/**
 *Bufferizes the error message given by a_msg_format and the other parameters.
 *The buffered error message can be displayed to the user by calling 
 *the function
 *@AppContext::display_buffered_error() 
 *with the same context in parameter.
 *Note that if this function is called with a NULL context param, the
 *formated error message is sent to stderr and is not buffered. Subsquent call
 *to AppContext::display_buffered_error() will not display anything. 
 *
 *@param a_msg_format the format string. 
 *Plays the same role asthe format string used in the function fprintf.
 *@param ... the parameters that may be used in the format string. 
 *
 *
 */
void
AppContext::bufferize_error (const gchar * a_msg_format, ...)
{
	va_list params;

	va_start (params, a_msg_format);
	bufferize_error (a_msg_format, params) ;
	va_end (params);
}


/**
 *Displays the error message that has been buffered by calling the function
 *AppContext::bufferize_error() 
 */
void
AppContext::display_buffered_error ()
{

	if (!m_priv || !m_priv->error_msg_buffer)
		return;

	error (m_priv->error_msg_buffer);

	g_free (m_priv->error_msg_buffer);

	m_priv->error_msg_buffer = NULL;
}

/**
 *Tells wether if there are error messages to be displayed or not.
 *Note the messages that may have been stored in the 
 *message buffer have been stored
 *using the function AppContext::bufferize_error() .
 *
 *@return TRUE if the error buffer is empty and false if not. 
 */
gboolean
AppContext::error_buffer_is_empty ()
{
	THROW_IF_FAIL (m_priv != NULL);

	return (m_priv->error_msg_buffer == NULL
	        || !strcmp (m_priv->
	                    error_msg_buffer, ""));
}


/**
 *Displays a warning message. It actually displays a modal 
 *dialog showing the message.
 *The user just has to click OK. 
 *
 *@param a_msg_format the format of the warning message to display.
 *If NULL, stderr is used to display.
 */
void
AppContext::warning (const gchar * a_msg_format, ...)
{
	va_list params;
	gchar *warning_msg = NULL;


	va_start (params, a_msg_format);
	warning_msg = g_strdup_vprintf (a_msg_format, params);
	va_end (params);

	if (warning_msg) {
		mlview_utils_display_warning_dialog ((const gchar*)warning_msg) ;
	} else if (warning_msg) {
		g_printerr (warning_msg);
	}

	if (warning_msg) {
		g_free (warning_msg);
		warning_msg = NULL ;
	}
}

void
AppContext::error (const gchar *a_msg_format, va_list a_params)
{
	gchar *err_msg = g_strdup_vprintf (a_msg_format, a_params);

	/*Add a title to the error message if needed */
	if (err_msg && m_priv->error_dialog_title) {
		gchar *tmp_str = err_msg;

		err_msg = g_strconcat
		          (m_priv->error_dialog_title,
		           "\n", tmp_str, NULL);
		g_free (tmp_str);

		/*delete the error dialog title */
		g_free (m_priv->error_dialog_title);
		m_priv->error_dialog_title = NULL;
	}

	mlview_utils_display_error_dialog ((const gchar*)err_msg);

	if (err_msg) {
		g_free (err_msg);
		err_msg = NULL ;
	}
}

/**
 *Displays a (fatal) error to the user. It actually 
 *shows a modal dialog box to the user.
 *The user just has to click OK. 
 *
 *@param a_msg_format the format of the 
 *message (as in the standard printf funtion)
 */
void
AppContext::error (const gchar * a_msg_format, ...)
{
	va_list params;

	va_start (params, a_msg_format);
	error (a_msg_format, params) ;
	va_end (params);
}

/**
 * Save main window size settings
 */
void
AppContext::save_window_state (gint width, gint height)
{
    THROW_IF_FAIL (m_priv);

    mlview::PrefsCategorySizes *prefs =
	dynamic_cast<mlview::PrefsCategorySizes*> (
	    mlview::Preferences::get_instance ()
	    ->get_category_by_id (mlview::PrefsCategorySizes::CATEGORY_ID));
    THROW_IF_FAIL (prefs);

    prefs->set_main_window_width (width);
    prefs->set_main_window_height (height);
}

/**
 * Save TreeView paned positions state
 */
void
AppContext::save_treeview_state (gint treeview_height, gint completion_box_size)
{
    THROW_IF_FAIL (m_priv);

    mlview::PrefsCategorySizes *prefs =
	dynamic_cast<mlview::PrefsCategorySizes*> (
	    mlview::Preferences::get_instance ()
	    ->get_category_by_id (mlview::PrefsCategorySizes::CATEGORY_ID));
    THROW_IF_FAIL (prefs);

    prefs->set_tree_editor_size (treeview_height);
    prefs->set_completion_box_size (completion_box_size);
}

GtkFileChooser *
AppContext::get_file_chooser (const gchar *a_title,
			      FileChooserMode a_mode)
{
	THROW_IF_FAIL (m_priv) ;

	/*if (!m_priv->file_chooser) {*/
	GtkWidget * parent_window = NULL ;

	parent_window = (GtkWidget*) get_element ("MlViewMainWindow") ;

	m_priv->file_chooser = GTK_FILE_CHOOSER (gtk_file_chooser_dialog_new
	                                 (a_title,
	                                  GTK_WINDOW (parent_window),
	                                  GTK_FILE_CHOOSER_ACTION_OPEN,
	                                  GTK_STOCK_CANCEL,
	                                  GTK_RESPONSE_CANCEL,
	                                  (a_mode == MLVIEW_FILE_CHOOSER_OPEN_MODE) ? GTK_STOCK_OPEN : GTK_STOCK_SAVE,
	                                  GTK_RESPONSE_OK,
	                                  NULL));
	THROW_IF_FAIL (m_priv->file_chooser != NULL) ;
	g_signal_connect (G_OBJECT (m_priv->file_chooser),
	                  "delete-event",
	                  G_CALLBACK (gtk_widget_hide_on_delete),
	                  NULL) ;
	gtk_window_set_modal (GTK_WINDOW (m_priv->file_chooser),
	                      TRUE) ;
	gtk_file_chooser_set_local_only
	(m_priv->file_chooser, TRUE);
	gtk_file_chooser_set_select_multiple
	(GTK_FILE_CHOOSER (m_priv->file_chooser),
	 FALSE);
	/*}*/
	THROW_IF_FAIL (m_priv->file_chooser != NULL) ;
	gtk_window_set_title (GTK_WINDOW (m_priv->file_chooser),
	                      a_title) ;
	if (a_mode == MLVIEW_FILE_CHOOSER_OPEN_MODE)
		gtk_file_chooser_set_action (m_priv->file_chooser, GTK_FILE_CHOOSER_ACTION_OPEN);
	else
		gtk_file_chooser_set_action (m_priv->file_chooser, GTK_FILE_CHOOSER_ACTION_SAVE);

	return m_priv->file_chooser ;
}

/**
 *Associates an xml catalog to the current instance of AppContext.
 *When the application context is destroyed, 
 *the instance of xml catalog is _NOT_ destroyed.
 *@param a_xml_catalog the new xml catalog
 */
void
AppContext::set_xml_catalog (xmlCatalog * a_xml_catalog)
{
	THROW_IF_FAIL (m_priv != NULL);

	m_priv->xml_catalog = a_xml_catalog;
}


/**
 *Decrements the reference count of the structure which
 *countains the type icons used to represent different types
 *of XML nodes used in some components of mlview.
 */
void
AppContext::type_icons_unref ()
{
	THROW_IF_FAIL (m_priv);

	THROW_IF_FAIL (m_priv->type_icons_refcnt);

	m_priv->type_icons_refcnt--;

	if (!m_priv->type_icons_refcnt) {
		unload_type_icons ();
	}
}


/**
 *Getter of the xml catalog associated to the current instance of
 *AppContext.
 *
 *@param the current instance of AppContext .
 *@return the instance of xmlCatalog associated to the current instance of
 *AppContext.
 */
xmlCatalog *
AppContext::get_xml_catalog ()
{
	THROW_IF_FAIL (m_priv != NULL);
	THROW_IF_FAIL (m_priv != NULL);

	return (m_priv->xml_catalog);
}

Clipboard *
AppContext::get_clipboard ()
{
	THROW_IF_FAIL (m_priv) ;

	if (!m_priv->clipboard_ptr) {
		m_priv->clipboard_ptr = new Clipboard () ;
	}
	return m_priv->clipboard_ptr ;
}

/**
 *Presents a dialog to the user asking for the name
 *of an internal subset
 *@param a_name out parameter. The name entered by the user, or
 *NULL if she entered nothing.
 *@return TRUE if the user entered a name, FALSE otherwise.
 */
gboolean
AppContext::ask_internal_subset_node_name (gchar **a_name)
{
	GtkWidget *dialog;
	GtkWidget *hbox;
	GtkWidget *image;
	GtkWidget *label;
	GtkWidget *entry;
	gint res;
	gboolean ret = FALSE;

	THROW_IF_FAIL (a_name);

	*a_name = NULL;

	dialog = gtk_dialog_new_with_buttons
	         (_("Name of the internal subset node"),
	          NULL, GTK_DIALOG_MODAL,
	          GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	          GTK_STOCK_OK, GTK_RESPONSE_OK,
	          NULL);

	gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
	gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);

	hbox = gtk_hbox_new (FALSE, 6);
	image = gtk_image_new_from_stock (GTK_STOCK_DIALOG_QUESTION,
	                                  GTK_ICON_SIZE_DIALOG);
	gtk_misc_set_alignment (GTK_MISC (image), 0.5, 0.0);
	label = gtk_label_new (_("Internal subset node name:"));
	entry = gtk_entry_new ();
	gtk_entry_set_activates_default (GTK_ENTRY (entry), TRUE);

	gtk_box_pack_start (GTK_BOX (hbox), image, FALSE, FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), label, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), entry, TRUE, TRUE, 0);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox),
	                    hbox, FALSE, FALSE, 0);

	gtk_widget_show_all (dialog);

	res = gtk_dialog_run (GTK_DIALOG (dialog));
	switch (res) {
	case GTK_RESPONSE_OK:
		*a_name = g_strdup (gtk_entry_get_text (GTK_ENTRY (entry)));
		ret = TRUE;
		break;
	case GTK_RESPONSE_CANCEL:
	case GTK_RESPONSE_DELETE_EVENT:
		break;
	default:
		g_assert_not_reached ();
	}

	gtk_widget_destroy (dialog);
	return ret;
}

//**************
//protected
//**************

/**
 *Unloads the type icons used to represent different types
 *of XML nodes in some components of mlview.
 */
void
AppContext::unload_type_icons ()
{
	struct TypeIcons *icons = NULL;

	THROW_IF_FAIL (m_priv);

	icons = m_priv->type_icons;

	if (icons) {
		if (icons->element) {
			g_object_unref (G_OBJECT (icons->element));
			icons->element = NULL ;
		}
		if (icons->open_element) {
			g_object_unref (G_OBJECT (icons->open_element));
			icons->open_element = NULL ;
		}
		if (icons->text) {
			g_object_unref (G_OBJECT (icons->text));
			icons->text = NULL ;
		}
		if (icons->root) {
			g_object_unref (G_OBJECT (icons->root));
			icons->root = NULL ;
		}
		if (icons->open_root) {
			g_object_unref (G_OBJECT (icons->open_root));
			icons->open_root = NULL ;
		}
		if (icons->comment) {
			g_object_unref (G_OBJECT (icons->comment));
			icons->comment = NULL ;
		}
		if (icons->pi) {
			g_object_unref (G_OBJECT (icons->pi));
			icons->pi = NULL ;
		}
		if (icons->entity_ref) {
			g_object_unref (G_OBJECT (icons->entity_ref));
			icons->entity_ref = NULL ;
		}

		g_free (icons);

		icons = NULL;
		m_priv->type_icons = NULL;
	}
}

/**
 *Loads the type icons used to represent different types 
 *of XML nodes in some components of mlview.
 *@return a TypeIcons struct which contains the icons.
 */
struct TypeIcons *
AppContext::load_type_icons ()
{
	gchar *path = NULL;
	struct TypeIcons *type_icons = NULL;

	THROW_IF_FAIL (m_priv);

	type_icons = (struct TypeIcons*) g_try_malloc
	             (sizeof (struct TypeIcons));

	if (!type_icons)
	{
		mlview_utils_trace_debug ("malloc failed, system may be out of memory");

		return NULL;
	}

	path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
	                                  ELEMENT_ICON, TRUE, NULL);

	if (!path)
	{
		mlview_utils_trace_debug ("lookup of icon file failed:") ;
		mlview_utils_trace_debug (ELEMENT_ICON) ;
	} else
	{
		type_icons->element = gdk_pixbuf_new_from_file (path, NULL);
		g_free (path);
	}

	path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
	                                  OPEN_ELEMENT_ICON, TRUE, NULL);
	if (!path)
	{
		mlview_utils_trace_debug ("lookup of icon file failed:") ;
		mlview_utils_trace_debug (OPEN_ELEMENT_ICON) ;
	} else
	{
		type_icons->open_element = gdk_pixbuf_new_from_file (path, NULL);
		g_free (path);
	}

	path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
	                                  TEXT_ICON, TRUE, NULL);
	if (!path)
	{
		mlview_utils_trace_debug ("lookup of icon file failed:") ;
		mlview_utils_trace_debug (TEXT_ICON) ;
	} else
	{
		type_icons->text = gdk_pixbuf_new_from_file (path, NULL);
		g_free (path);
	}

	path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
	                                  ROOT_ICON, TRUE, NULL);
	if (!path)
	{
		mlview_utils_trace_debug ("lookup of icon file failed:") ;
		mlview_utils_trace_debug (ROOT_ICON) ;
	} else
	{
		type_icons->root = gdk_pixbuf_new_from_file (path, NULL);
		g_free (path);
	}

	path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
	                                  OPEN_ROOT_ICON, TRUE, NULL) ;
	if (!path)
	{
		mlview_utils_trace_debug ("lookup of icon file failed:") ;
		mlview_utils_trace_debug (OPEN_ROOT_ICON) ;
	} else
	{
		type_icons->open_root = gdk_pixbuf_new_from_file (path, NULL);
		g_free (path);
	}

	path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
	                                  COMMENT_ICON, TRUE, NULL);
	if (!path)
	{
		mlview_utils_trace_debug ("lookup of icon file failed:") ;
		mlview_utils_trace_debug (COMMENT_ICON) ;
	} else
	{
		type_icons->comment = gdk_pixbuf_new_from_file (path, NULL);
		g_free (path);
	}

	path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
	                                  ENTITY_REF_ICON, TRUE, NULL);
	if (!path)
	{
		mlview_utils_trace_debug ("lookup of icon file failed:") ;
		mlview_utils_trace_debug (ENTITY_REF_ICON) ;
	} else
	{
		type_icons->entity_ref = gdk_pixbuf_new_from_file (path, NULL);
		g_free (path);
	}

	path = gnome_program_locate_file (NULL, GNOME_FILE_DOMAIN_APP_DATADIR,
	                                  PI_ICON, TRUE, NULL);
	if (!path)
	{
		mlview_utils_trace_debug ("lookup of icon file failed:") ;
		mlview_utils_trace_debug (PI_ICON) ;
	} else
	{
		type_icons->pi = gdk_pixbuf_new_from_file (path, NULL);
		g_free (path);
	}

	return type_icons;
}

/**
 *Returns the type icons used to represent different types
 *of XML nodes in some components of mlview.
 *@return a TypeIcons struct which contains the icons.
 */
struct TypeIcons *
AppContext::type_icons_ref ()
{
	THROW_IF_FAIL (m_priv);

	if (m_priv->type_icons_refcnt)
	{
		THROW_IF_FAIL (m_priv->type_icons);

		m_priv->type_icons_refcnt++;
	} else
	{
		THROW_IF_FAIL (!m_priv->type_icons);

		m_priv->type_icons = load_type_icons ();

		THROW_IF_FAIL (m_priv->type_icons);

		m_priv->type_icons_refcnt++;
	}

	return m_priv->type_icons;
}

sigc::signal0<void>&
AppContext::signal_application_initialized ()
{
	return m_priv->signal_application_initialized  ;
}


sigc::signal2<void, GtkWidget*, GdkEvent*>&
AppContext::signal_contextual_menu_requested ()
{
	return m_priv->signal_contextual_menu_requested ;
}

sigc::signal2<void, Object*, Object*>&
AppContext::signal_view_swapped ()
{
	return m_priv->signal_view_swapped ;
}

sigc::signal1<void, gpointer>&
AppContext::signal_document_name_changed ()
{
	return m_priv->signal_document_name_changed ;
}

sigc::signal0<void>&
AppContext::signal_view_undo_state_changed ()
{
	return m_priv->signal_view_undo_state_changed ;
}

}//end namespace mlview
