/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include <QScrollArea>
#include <QVBoxLayout>
#include <QTabWidget>
#include <QLabel>
#include <QSettings>
#include <QDialogButtonBox>
#include <QGroupBox>
#include <QMessageBox>

#include "ExportDialog.h"
#include "MvQGui.h"
#include "uPlot.h"
#include "ObjectList.h"
#include "MvRequestUtil.hpp"

ExportDialog::ExportDialog(int currentFrame, int totalFrames, QWidget *parent) :
			   QFileDialog(parent,"Export"),
			   current_  (currentFrame),
			   total_    (totalFrames),
			   comboBox_ (0),
			   leFs3_    (0),
			   cbFs1_    (0),
			   cbFs2_    (0),
			   cbFs3_    (0)
{
	// Set widget mode behaviour to "Save As"
	this->setAcceptMode(QFileDialog::AcceptSave);

	// Restore previous gui settings
	QSettings settings("ECMWF","uPlotExportDialog");
	settings.beginGroup("main");
        selectFile(settings.value("fileName").toString());
	QStringList fout = settings.value("formats").toStringList();
	settings.endGroup();

	// Set filter options to select which files to be shown
	this->setFilter(tr("All Files (*.*);;Images (*.png *.ps *.eps *.jpg *.kml *.gif *.svg *.pdf)"));

	// Get the main layout
	QGridLayout *mainLayout = static_cast<QGridLayout*>(layout());
	int row = mainLayout->rowCount();
	int col = mainLayout->columnCount();

	// Build label for the Output format
	QLabel *label = new QLabel(this);
	label->setText(tr("Output formats:"));
	mainLayout->addWidget(label,row,0);

	// Retrieve list of output formats from the resources file
	// Build 'check box combo' with the list of output formats
	comboBox_ = new MvQCheckBoxCombo(this);
	MvRequest reqof;
	ObjectList::FindAll("output_format",reqof);
	while ( reqof )
	{
		// Check if output format has been previously selected
		QString str = (const char*)reqof("class");
		bool flag = fout.contains(str);

		// Add output format to the user interface
		comboBox_->addItem((const char*)reqof("class"),flag);

		reqof.advance();
	}

	mainLayout->addWidget(comboBox_,row,1,1,1);

	// Build Frame Selection widget
	buildFrameSelection( row-2, col-1, mainLayout );

	// Build the advanced Options button
	row++;
	bAdvd_ = new QPushButton("Options >>");
	bAdvd_->setCheckable(true);
	mainLayout->addWidget(bAdvd_,row,col-1);

bAdvd_->setVisible(false);  //Remove later

#if 0	// Build tab for output formats
	row++;
	QTabWidget *tab = new QTabWidget();
	mainLayout->addWidget(tab,row,0,10,col-1);
	tab->hide();

 //Removed temporarily until bAdvd_->setVisible(false) is turned to true
	// Build output formats user interface
	reqof.rewind();
	while ( reqof )
	{
		// Get driver object resources
		MvRequest reqDef = ObjectList::Find("object",(const char*)reqof("output"));
		if ( strcmp(reqDef.getVerb(), "EMPTY") == 0 )
		{
			reqof.advance();
			continue;
		}

		// Get definition file request
		const char* defName = reqDef("definition_file");
		MvRequest defReq = read_language_file(defName);

		// Read output format request
		// Try to first search the icon file in the Metview/Preferences directory.
		// If not found, try the Metview/Defaults directory, If not found, build
		// a default request
		MvRequest reqOut = ObjectList::UserPreferencesRequest ( (const char*)reqof("output") );

		reqOut = ObjectList::ExpandRequest ( reqOut, EXPAND_DEFAULTS );

		// Merge the Interface request with the current user request
		MvRequest interReq;
		MvRequest uiReq;
		int n = 0;
		defReq.iterInitParam();
		while(defReq.iterGetNextParamInterface(interReq))
		{
			MvRequest req((const char*)interReq("interface"));
			const char* caux = reqOut.getParameter(n);
			req("name") = caux;
			req("values") = reqOut(caux);
			CopyAndRemoveParameters ( interReq, req, "interface" );
			uiReq = uiReq + req;
			n++;
		}

		// Create output format request to be used 
//		tab->addTab(scroll1,(const char*)req("class"));

	QWidget *w1 = new QWidget;
	QScrollArea *scroll = new QScrollArea;
//	QVBoxLayout *layout1 = new QVBoxLayout(w1);
	QVBoxLayout *layout1 = new QVBoxLayout;
	layout1->setSizeConstraint(QLayout::SetMinAndMaxSize);
	w1->setLayout(layout1);
	scroll->setWidget(w1);

	MvQGui *tGui = new MvQGui(layout1);
	tGui->buildFromDefinition(uiReq);

	tab->addTab(scroll,(const char*)reqof("class"));

//	connect(macroGui_,SIGNAL(editFinished()),
//		this,SLOT(slotRunMacro()));

//	filterSplitter->addWidget(scroll);

		reqof.advance();
	}

	// Build tab buttons
	QWidget *wt = new QWidget;
	QVBoxLayout *vt = new QVBoxLayout(wt);
	vt->addWidget(new QPushButton("Save as default"));
	vt->addWidget(new QPushButton("Load default"));
	vt->addWidget(new QPushButton("Save as..."));
	wt->hide();
	mainLayout->addWidget(wt,row,col-1,1,1);

	// Define functions
	connect(bAdvd_,SIGNAL(toggled(bool)), tab, SLOT(setVisible(bool)));
	connect(bAdvd_,SIGNAL(toggled(bool)), wt, SLOT(setVisible(bool)));
	connect(bAdvd_,SIGNAL(clicked(bool)), this, SLOT(slotAdvancedButton(bool)));

//	layout()->setSizeConstraint(QLayout::SetFixedSize);
#endif

	// Move Save and Cancel buttons to the bottom.
	// Save and Cancel buttons are defined together in a QDialogButtonBox.
	// Therefore, only plays with one of the buttons.
	row++;
	QLayoutItem* liCancel = mainLayout->itemAtPosition (row-4, col-1 );
	mainLayout->removeItem(liCancel);
	QDialogButtonBox* bb = static_cast<QDialogButtonBox*>(liCancel->widget());
	bb->setOrientation(Qt::Horizontal);
	mainLayout->addItem(liCancel,row,2);
}

// Destructor
ExportDialog::~ExportDialog()
{
	// Save current gui settings
	QSettings settings("ECMWF","uPlotExportDialog");
	settings.beginGroup("main");
        settings.setValue("fileName",selectedFiles());
	QStringList formats = comboBox_->getSelectedValues();
	if ( formats.size() )
		settings.setValue("formats",formats);
	else
		settings.setValue("formats","PS");
	settings.endGroup();

	delete comboBox_;
	delete bAdvd_;
	delete leFs3_;
	delete cbFs1_;
	delete cbFs2_;
	delete cbFs3_;
}

// Callback from the SAVE button
void ExportDialog::accept()
{	
	// Get filename from the user interface
	QStringList s = selectedFiles();
	string filename = s.at(0).toStdString();

	// Get output formats
	QStringList formats = comboBox_->getSelectedValues();
	if ( formats.size() == 0 )
	{
		cout << "Send an pop up error message: No output formats selected" << endl;
	}

	// Create output requests
	MvRequest reqOut;
	getOutputFormatsReq ( filename, formats, reqOut );

	// Get frames selection
	if ( !getFrameSelection( reqOut ) )
		return;

	// Send export info to be processed
	emit sendExpDialogInfo(&reqOut);

	// Close the window
	int res=0;
	done(res);
//	hide();
}

void ExportDialog::reject()
{
	hide();
}

void ExportDialog::setVisible(bool b)
{
	QDialog::setVisible(b);
}

void ExportDialog::getOutputFormatsReq ( string fname, QStringList& fout, MvRequest& reqOut )
{
	// Get all valid output formats
	MvRequest reqof;
	ObjectList::FindAll("output_format",reqof);

	// Build selected output formats
	for ( int i = 0; i < fout.size(); i++ )
	{
		// Get output format
		string sclasse = fout.at(i).toStdString();
		MvRequest oneReq = ObjectList::Find("output_format",sclasse.c_str());

		// Create request
		MvRequest req((const char*)oneReq("output"));
		req("OUTPUT_NAME") = fname.c_str();
		reqOut = reqOut + req;
	}

	return;
}

void ExportDialog::slotAdvancedButton(bool toggle)
{
	(toggle) ? bAdvd_->setText ("Options <<") : bAdvd_->setText ("Options >>");
}

void ExportDialog::slotRangeFrameSelection(bool checked)
{
	// Enable/Disable line edit
	leFs3_->setEnabled(checked);
}

void ExportDialog::buildFrameSelection( int row, int col, QGridLayout* mainLayout )
{
	// Group Box with radio buttons and one line edit
	QGroupBox* gbFs = new QGroupBox( tr("Frame Selection") );
	cbFs1_ = new QRadioButton( tr("&Current") );
	cbFs2_ = new QRadioButton( tr("&All") );
	cbFs3_ = new QRadioButton( tr("&Range") );
	cbFs1_->setChecked(true);

	// Line edit with a validator
	leFs3_ = new QLineEdit( );
	leFs3_->setToolTip(tr("Specify one or more frame ranges, e.g. 1-3,6,7"));
	leFs3_->setEnabled(false);
	QRegExp rx("^\\d{1,}(-\\d{1,})?(,\\d{1,}|,\\d{1,}-\\d{1,})*");
	QValidator *validator = new QRegExpValidator(rx, this);
	leFs3_->setValidator(validator);

	// Set layout
	QGridLayout* glFs = new QGridLayout( gbFs );
	glFs->addWidget( cbFs1_, 0, 0 );
	glFs->addWidget( cbFs2_, 1, 0 );
	glFs->addWidget( cbFs3_, 2, 0 );
	glFs->addWidget( leFs3_, 2, 1 );

	// Conexions
	connect(cbFs3_,SIGNAL(toggled(bool)), this, SLOT(slotRangeFrameSelection(bool)));

	// Attach widget to the parent
	mainLayout->addWidget(gbFs,row,col,3,1);

	return;
}

bool ExportDialog::getFrameSelection( MvRequest& req )
{
	// Three options: CURRENT, ALL, RANGE
	// ALL: nothing need to be done
	if ( cbFs2_->isChecked() )
		return true;

	// CURRENT: get the current frame numbeer
	int i;
	QList<int> list;
	if ( cbFs1_->isChecked() )
	{
		list.append(current_+1);  // start from 1, not from 0
	}
	else	// RANGE
	{
		// Get the list of range entries (split by comma)
		QStringList list1 = leFs3_->text().split(",", QString::SkipEmptyParts);

		// Analyse all entries
		for ( i = 0; i < list1.size(); i++)
		{
			// Check if it is a single value or a range of values
			QStringList list2 = list1.at(i).split("-", QString::SkipEmptyParts);
			if ( list2.size() == 1 ) // single value
				list.append(list2.at(0).toInt());
			else
			{
				int i1 = list2.at(0).toInt();
				int i2 = list2.at(1).toInt();
				if ( i1 > i2 )  // invert sequence
				{
					int iaux = i1;
					i1 = i2;
					i2 = iaux;
				}
				for ( int j = i1; j <= i2; j++ )
					list.append(j);
			}
		}

		// Check if the range of values is valid
		qSort(list.begin(), list.end());
		if ( list.at(list.size()-1) > total_ )
		{
			ostrstream info;
			info << "Enter values from 1 to " << total_ << ends;
			errorMessage("Invalid range!",(const char*)info.str());
			return false;
		}
	}

	// Save list of frames selection for all drivers
	while ( req )
	{
		for ( i = 0; i < list.size(); i++)
			req.addValue("OUTPUT_FRAME_LIST",list.at(i));

		req.advance();
	}

	req.rewind();

	return true;
}

void ExportDialog::errorMessage (const char* msg, const char* info)
{
	QMessageBox msgBox;
	msgBox.setIcon(QMessageBox::Critical);
	msgBox.setWindowTitle(tr("Export Error"));
	msgBox.setText(QString::fromUtf8(msg));
	if ( info )
		msgBox.setInformativeText(QString::fromUtf8(info));

	msgBox.exec();
}
